/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#ifndef AD7124_H
#define AD7124_H

#include <stdbool.h>
#include <stdint.h>
#include "hal_spi.h"

/* Регистр конфигурации */
#define AD7124_CFG_REG_BIPOLAR    (1 << 11)          /* Полярность сигнала */
#define AD7124_CFG_REG_BURNOUT(x) (((x) & 0x3) << 9) /* Выбор источника тока */
#define AD7124_CFG_REG_REF_BUFP   (1 << 8)           /* Положительный источник питания буферизуется */
#define AD7124_CFG_REG_REF_BUFM   (1 << 7)           /* Отрицательный источник питания буферизуется */
#define AD7124_CFG_REG_AIN_BUFP   (1 << 6)           /* Включить буфер для положительного входа */
#define AD7124_CFG_REG_AINN_BUFM  (1 << 5)           /* Включить буфер для отрицательного входа */
#define AD7124_CFG_REG_REF_SEL(x) (((x) & 0x3) << 3) /* Выбор источника питания */
#define AD7124_CFG_REG_PGA(x)     (((x) & 0x7) << 0) /* Программируемый массив усилителей */

/* Регистр фильтра */
#define AD7124_FILT_REG_FILTER(x)      (((x) & 0x7) << 21)
#define AD7124_FILT_REG_REJ60          (1 << 20)
#define AD7124_FILT_REG_POST_FILTER(x) (((x) & 0x7) << 17)
#define AD7124_FILT_REG_SINGLE_CYCLE   (1 << 16)
#define AD7124_FILT_REG_FS(x)          (((x) & 0x7FF) << 0)

/* Биты регистра ADC_Control */
#define AD7124_ADC_CTRL_REG_DOUT_RDY_DEL  (1 << 12)
#define AD7124_ADC_CTRL_REG_CONT_READ     (1 << 11)
#define AD7124_ADC_CTRL_REG_DATA_STATUS   (1 << 10)
#define AD7124_ADC_CTRL_REG_CS_EN         (1 << 9)
#define AD7124_ADC_CTRL_REG_REF_EN        (1 << 8)
#define AD7124_ADC_CTRL_REG_POWER_MODE(x) (((x) & 0x3) << 6)
#define AD7124_ADC_CTRL_REG_MODE(x)       (((x) & 0xF) << 2)
#define AD7124_ADC_CTRL_REG_CLK_SEL(x)    (((x) & 0x3) << 0)

/* Биты регистра статуса */
#define AD7124_STATUS_REG_RDY          (1 << 7)
#define AD7124_STATUS_REG_ERROR_FLAG   (1 << 6)
#define AD7124_STATUS_REG_POR_FLAG     (1 << 4)
#define AD7124_STATUS_REG_CH_ACTIVE(x) ((x) & 0xF)

/* Выставление регистров каналов */
#define AD7124_CH_MAP_REG_CH_ENABLE (1 << 15)
#define AD7124_CH_MAP_REG_SETUP(x)  (((x) & 0x7) << 12)
#define AD7124_CH_MAP_REG_AINP(x)   (((x) & 0x1F) << 5)
#define AD7124_CH_MAP_REG_AINM(x)   (((x) & 0x1F) << 0)

/* Операции чтения и записи */
#define RD_REGISTER(x) (((x) & (0x3F)) | (1 << 6))
#define WR_REGISTER(x) (((x) & (0x3F)) & ~(1 << 6))

/* Максимальное число каналов и настроек */
#define AD7124_MAX_CHANNEL_COUNT 16
#define AD7124_MAX_SETUP_COUNT   8

/* Множители для однополярного и биполярного режима */
#define AD7124_MULTIPLIER_BIPOLAR  0x7FFFFFUL
#define AD7124_MULTIPLIER_UNIPOLAR 0xFFFFFFUL

/* Communication Register bits */
#define AD7124_COMM_REG_WEN   (0 << 7)
#define AD7124_COMM_REG_WR    (0 << 6)
#define AD7124_COMM_REG_RD    (1 << 6)
#define AD7124_COMM_REG_RA(x) ((x) & 0x3F)

/* Номинал внутреннего источника питания */
#define AD7124_INTERNAL_VOLTAGE_REFERENCE 2.5

/*! @brief Источники питания AD7124 */
typedef enum {
    AD7124_REFERENCE_INTERNAL = 2 /* Внутренний */
} ad7124_ref_sel;

/*! @brief Данные из регистра ID */
typedef enum {
    AD7124_4_ID  = 0x04,
    AD7124_4B_ID = 0x06,
    AD7124_4E_ID = 0x07
} ad7124_id;

/*! @brief Конфигурация усиления */
typedef enum {
    AD7124_GAIN_1   = 0, /* Усиление 1.   При VREF=2.5V входной диапазон ±2.5V     */
    AD7124_GAIN_2   = 1, /* Усиление 2.   При VREF=2.5V входной диапазон ±1.25V    */
    AD7124_GAIN_4   = 2, /* Усиление 4.   При VREF=2.5V входной диапазон ±625mV    */
    AD7124_GAIN_8   = 3, /* Усиление 8.   При VREF=2.5V входной диапазон ±312.5V   */
    AD7124_GAIN_16  = 4, /* Усиление 16.  При VREF=2.5V входной диапазон ±156.25mV */
    AD7124_GAIN_32  = 5, /* Усиление 32.  При VREF=2.5V входной диапазон ±78.125mV */
    AD7124_GAIN_64  = 6, /* Усиление 64.  При VREF=2.5V входной диапазон ±39.06mV  */
    AD7124_GAIN_128 = 7  /* Усиление 128. При VREF=2.5V входной диапазон ±19.53mV  */
} ad7124_gain;

/*! @brief Источник постоянного тока */
typedef enum {
    AD7124_BURNOUT_CURRENT_OFF   = 0,
    AD7124_BURNOUT_CURRENT_500NA = 1,
    AD7124_BURNOUT_CURRENT_2UA   = 2,
    AD7124_BURNOUT_CURRENT_4UA   = 3
} ad7124_burnout_current;

/*! @brief Каналы ad7124 */
typedef enum {
    AD7124_CHANNEL_0  = 0,
    AD7124_CHANNEL_1  = 1,
    AD7124_CHANNEL_2  = 2,
    AD7124_CHANNEL_3  = 3,
    AD7124_CHANNEL_4  = 4,
    AD7124_CHANNEL_5  = 5,
    AD7124_CHANNEL_6  = 6,
    AD7124_CHANNEL_7  = 7,
    AD7124_CHANNEL_8  = 8,
    AD7124_CHANNEL_9  = 9,
    AD7124_CHANNEL_10 = 10,
    AD7124_CHANNEL_11 = 11,
    AD7124_CHANNEL_12 = 12,
    AD7124_CHANNEL_13 = 13,
    AD7124_CHANNEL_14 = 14,
    AD7124_CHANNEL_15 = 15
} ad7124_channel;

/*! @brief Аналоговые входы AIN */
typedef enum {
    AD7124_AIN0_INPUT = 0,
    AD7124_AIN1_INPUT = 1,
    AD7124_AIN2_INPUT = 2,
    AD7124_AIN3_INPUT = 3,
    AD7124_AIN4_INPUT = 4,
    AD7124_AIN5_INPUT = 5,
    AD7124_AIN6_INPUT = 6,
    AD7124_AIN7_INPUT = 7,
    AD7124_AVSS       = 17,
    AD7124_INTERNAL   = 18,
    AD7124_DGND       = 19
} ad7124_input_sel;

/*! @brief Доступные настройки */
typedef enum {
    AD7124_SETUP_0 = 0,
    AD7124_SETUP_1 = 1,
    AD7124_SETUP_2 = 2,
    AD7124_SETUP_3 = 3,
    AD7124_SETUP_4 = 4,
    AD7124_SETUP_5 = 5,
    AD7124_SETUP_6 = 6,
    AD7124_SETUP_7 = 7
} ad7124_setups;

/*! @brief Доступные фильтры */
typedef enum {
    AD7124_FILTER_0 = 0,
    AD7124_FILTER_1 = 1,
    AD7124_FILTER_2 = 2,
    AD7124_FILTER_3 = 3,
    AD7124_FILTER_4 = 4,
    AD7124_FILTER_5 = 5,
    AD7124_FILTER_6 = 6,
    AD7124_FILTER_7 = 7
} ad7124_filters;

/*! @brief Доступные типы фильтров */
typedef enum {
    AD7124_SINC_4_FILTER      = 0,
    AD7124_SINC_3_FILTER      = 2,
    AD7124_SINC_4_FAST_FILTER = 4,
    AD7124_SINC_3_FAST_FILTER = 5,
    AD7124_POST_FILTER        = 7,
} ad7124_filters_type;

/*! @brief Доступные post фильтры */
typedef enum {
    AD7124_NO_POST_FILTER = 0
} ad7124_post_filter_type;

/*! @brief Источники тактирования AD7124 */
typedef enum {
    AD7124_INTERNAL_CLK                 = 0, /* Внутренний на 614.4 kHz. Не доступен на CLK pin */
    AD7124_INTERNAL_CLK_WITH_OUT_OUTPUT = 1, /* Внутренний на 614.4 kHz. Доступен на CLK pin */
    AD7124_EXTERNAL_CLK                 = 2, /* Внешний на 614.4 kHz */
    AD7124_EXTERNAL_DIV_4_CLK           = 3  /* Деление внешней частоты на 4 */
} ad7124_clk_sourse;

/*! @brief Регистры AD7124 */
typedef enum {
    AD7124_REG_STATUS      = 0,
    AD7124_REG_ADC_CONTROL = 1,
    AD7124_REG_DATA        = 2,
    AD7124_REG_IOCON1      = 3,
    AD7124_REG_IOCON2      = 4,
    AD7124_REG_ID          = 5,
    AD7124_REG_ERROR       = 6,
    AD7124_REG_ERROR_EN    = 7,
    AD7124_REG_MCLK_COUNT  = 8,
    AD7124_REG_CHANNEL_0   = 9,
    AD7124_REG_CHANNEL_1   = 10,
    AD7124_REG_CHANNEL_2   = 11,
    AD7124_REG_CHANNEL_3   = 12,
    AD7124_REG_CHANNEL_4   = 13,
    AD7124_REG_CHANNEL_5   = 14,
    AD7124_REG_CHANNEL_6   = 15,
    AD7124_REG_CHANNEL_7   = 16,
    AD7124_REG_CHANNEL_8   = 17,
    AD7124_REG_CHANNEL_9   = 18,
    AD7124_REG_CHANNEL_10  = 19,
    AD7124_REG_CHANNEL_11  = 20,
    AD7124_REG_CHANNEL_12  = 21,
    AD7124_REG_CHANNEL_13  = 22,
    AD7124_REG_CHANNEL_14  = 23,
    AD7124_REG_CHANNEL_15  = 24,
    AD7124_REG_CONFIG_0    = 25,
    AD7124_REG_CONFIG_1    = 26,
    AD7124_REG_CONFIG_2    = 27,
    AD7124_REG_CONFIG_3    = 28,
    AD7124_REG_CONFIG_4    = 29,
    AD7124_REG_CONFIG_5    = 30,
    AD7124_REG_CONFIG_6    = 31,
    AD7124_REG_CONFIG_7    = 32,
    AD7124_REG_FILTER_0    = 33,
    AD7124_REG_FILTER_1    = 34,
    AD7124_REG_FILTER_2    = 35,
    AD7124_REG_FILTER_3    = 36,
    AD7124_REG_FILTER_4    = 37,
    AD7124_REG_FILTER_5    = 38,
    AD7124_REG_FILTER_6    = 39,
    AD7124_REG_FILTER_7    = 40,
    AD7124_REG_OFFSET_0    = 41,
    AD7124_REG_OFFSET_1    = 42,
    AD7124_REG_OFFSET_2    = 43,
    AD7124_REG_OFFSET_3    = 44,
    AD7124_REG_OFFSET_4    = 45,
    AD7124_REG_OFFSET_5    = 46,
    AD7124_REG_OFFSET_6    = 47,
    AD7124_REG_OFFSET_7    = 48,
    AD7124_REG_GAIN_0      = 49,
    AD7124_REG_GAIN_1      = 50,
    AD7124_REG_GAIN_2      = 51,
    AD7124_REG_GAIN_3      = 52,
    AD7124_REG_GAIN_4      = 53,
    AD7124_REG_GAIN_5      = 54,
    AD7124_REG_GAIN_6      = 55,
    AD7124_REG_GAIN_7      = 56
} ad7124_registers;

/*! @brief Режим преобразования */
typedef enum {
    AD7124_CONTINUOUS_MODE  = 0, /* Непрерывное преобразование */
    AD7124_SINGLE_CONV_MODE = 1, /* Одиночное преобразование */
    AD7124_STANDBY_MODE     = 2  /* Режим ожидания */
} ad7124_operating_mode;

/*! @brief Режим питания АЦП */
typedef enum {
    AD7124_LOW_POWER  = 0,
    AD7124_MID_POWER  = 1,
    AD7124_FULL_POWER = 2
} ad7124_power_mode;

/*! @brief Статусы возврата из функций ad7124 */
enum ad7124_status {
    AD7124_STATUS_OK   =  0, /* Успех  */
    AD7124_STATUS_FAIL =  1  /* Провал */
};

/*!
 * @brief Конфигурация ad7124
 *
 * Можно выбрать одну из 8-ми настроек для АЦП,
 * источник питания, коэффициент усиления,
 * полярность и величину источника тока.
 *
 * @param setup   Номер настройки
 * @param ref     Источник питания
 * @param gain    Коэффициент усиления
 * @param bipolar Выбор полярности: биполярная (1) или однополярная (0)
 * @param burnout Величина тока
 * @param base    Базовый адрес SPI
 *
 * @retval #AD7124_STATUS_OK
 * @retval #AD7124_STATUS_FAIL
 */
enum ad7124_status AD7124_SetConfigure(ad7124_setups setup,
    ad7124_ref_sel ref, ad7124_gain gain, bool bipolar,
    ad7124_burnout_current burnout, SPI_Type *base);

/*!
 * @brief Настройка фильтра
 *
 * Конфигурация одного из 8-ми фильтров.
 *
 * @param filter       Номер фильтра
 * @param filter_type  Тип фильтра
 * @param post_filter  Тип post фильтра
 * @param fs           Скорости фильтрации входных данных
 * @param rej60        Фильтрация частот 50Hz и/или 60Hz
 * @param single_cycle Применить (1) фильтр для одного преобразования
 * @param base         Базовый адрес SPI
 *
 * @retval #AD7124_STATUS_OK
 * @retval #AD7124_STATUS_FAIL
 */
enum ad7124_status AD7124_SetFilter(ad7124_filters filter,
    ad7124_filters_type filter_type, ad7124_post_filter_type post_filter,
    uint16_t fs, bool rej60, bool single_cycle, SPI_Type *base);

/*!
 * @brief Установка канала ad7124
 *
 * Имеется 16 независимых каналов,для каждого из каналов
 * можно установить одну из 8-ми настроек.
 *
 * @param channel Номер канала от 0 до 15
 * @param setup   Номер одной из 8-ми настроек для выбранного канала
 * @param ainp    Положительный аналоговый вход (AINP)
 * @param ainm    Отрицательный аналоговый вход (AINM)
 * @param enable  Включает (1) конфигурацию для канала
 * @param base    Базовый адрес SPI
 *
 * @retval #AD7124_STATUS_OK
 * @retval #AD7124_STATUS_FAIL
 */
enum ad7124_status AD7124_SetChannel(ad7124_channel channel,
    ad7124_setups setup, ad7124_input_sel ainp,
    ad7124_input_sel ainm, bool enable, SPI_Type *base);

/*!
 * @brief Установка control регистров
 *
 * Выбор режима АЦП, источника тактирования,
 * режим питания и режим преобразования.
 *
 * @param mode        Режим работы АЦП
 * @param power       Режим питания
 * @param ref_en      Источник питания, внутренний (1) или внешний (0) источник питания
 * @param cont_read   Непрерываное чтение, вкл (1) или выкл (0) режима непрерываного чтения
 * @param data_status Отображение номера активного канала, вкл (1) или выкл (0)
 * @param clk_sel     Источник тактирования
 * @param base        Базовый адрес SPI
 *
 * @retval            #AD7124_STATUS_OK
 * @retval            #AD7124_STATUS_FAIL
 */
enum ad7124_status AD7124_SetControl(ad7124_operating_mode mode,
    ad7124_power_mode power, bool ref_en, bool cont_read,
    bool data_status, ad7124_clk_sourse clk_sel, SPI_Type *base);

/*!
 * @brief Включение и выключение канала
 *
 * @param channel Номер канала
 * @param enable  Включение (1) или выключение (0) канала
 * @param base    Базовый адрес SPI
 *
 * @retval #AD7124_STATUS_OK
 * @retval #AD7124_STATUS_FAIL
 */
enum ad7124_status AD7124_EnableChannel(ad7124_channel channel,
        bool enable, SPI_Type *base);

/*!
 * @brief Установка режима преобразования
 *
 * @param mode Режим преобразования
 * @param base Базовый адрес SPI
 *
 * @retval #AD7124_STATUS_OK
 * @retval #AD7124_STATUS_FAIL
 */
enum ad7124_status AD7124_SetMode(ad7124_operating_mode mode,
        SPI_Type *base);

/*!
 * @brief Чтение регистра DATA АЦП
 *
 * @param base Базовый адрес SPI
 *
 * @return 24-битное значение регистра DATA
 */
int32_t AD7124_GetData(SPI_Type *base);

/*!
 * @brief Быстрое чтение регистра DATA АЦП
 *
 * Чтение данных в режиме "Continuous Read Mode",
 * т.е. без взаимодействия с регистром Communication.
 *
 * @param base Базовый адрес SPI
 *
 * @return 24-битное значение регистра DATA
 */
int32_t AD7124_GetDataFast(SPI_Type *base);

/*!
 * @brief Ожидание готовности данных в регистре DATA
 *
 * Если данные готовы для считывания, то бит RDY = 0.
 *
 * @param rdy_field Битовое поле для проверки статуса
 * @param base      Базовый адрес SPI
 */
void AD7124_CheckStatusRegister(int32_t rdy_field, SPI_Type *base);

/*!
 * @brief Ожидание готовности данных в регистре DATA
 *
 * Если данные готовы для считывания, то пин DOUT/RDY = 0.
 */
void AD7124_CheckRDY(void);

/*!
 * @brief Сброс АЦП ad7124
 *
 * @param base Базовый адрес SPI
 */
void AD7124_Reset(SPI_Type *base);

/*!
 * @brief Чтение регистра ID
 *
 * @param base Базовый адрес SPI
 *
 * @return 8-битное значение регистра ID
 */
uint8_t AD7124_ReadIdRegister(SPI_Type *base);

/*!
 * @brief Преобразование значений регистров АЦП в значение напряжения в вольтах
 *
 * Данные из АЦП в зависимости от режима работы преобразуются в вольты.
 *
 * @param value   Значение из регистра данных АЦП
 * @param gain    Режим усиления
 * @param vref    Напряжение источника питания АЦП
 * @param bipolar Биполярный (1) или однополярный (0) режим работы
 *
 * @return        Значение напряжения в вольтах
 */
double AD7124_ToVoltage(int32_t value, uint8_t gain,
    float vref, bool bipolar);

/*!
 * @brief Проверка соединения связи с AD7124
 *
 * Чтение регистра ID, если ID равен 0x04 или 0x06, то AD7124 работает
 *
 * @param base Базовый адрес SPI
 *
 * @retval 0 - связь не установлена
 * @retval 1 - связь установлена
 */
bool AD7124_CheckWorking(SPI_Type *base);

/*!
 * @brief Чтение регистра
 *
 * @param reg  Адрес регистра
 * @param base Базовый адрес SPI
 *
 * @return 64-битное значение
 */
uint32_t AD7124_ReadRegister(uint8_t reg, SPI_Type *base);

/*!
 * @brief Конфигурация АЦП в 6-ти канальном режиме
 *
 * В этом режиме АЦП работает на максимальной скорости.
 *
 * @param base Базовый адрес SPI
 */
void AD7124_Configuration6ChannelsFullSpeed(SPI_Type *base);

#endif /* AD7124_H */
