/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "bsp/board_api.h"
#include "tusb.h"

#include "hal_jtm.h"
#include "hal_rwc.h"
#include "hal_wdt.h"

#include "embedded_cli.h"
#include "cli_time_support.h"

#define TEST_TCAL      129  /*!< Значение калибровочного параметра TCAL для партии микросхем MPW */
#define TEST_WCAL      1311 /*!< Значение калибровочного параметра WCAL для партии микросхем MPW */
#define TEST_WTCONF    0x3D /*!< Значение калибровочного параметра WTCONF для партии микросхем MPW */
#define TEST_WTCALCONF 0    /*!< Значение калибровочного параметра WTCALCONF для партии микросхем MPW */

#define TUD_BUFFER_SIZE 64U
#define MESSAGE_BUFFER_SIZE 256U

#define REBOOT_TIME 2U

static EmbeddedCli *cli;

volatile bool esc_pressed = false;    /*!< Флаг нажатия ESC */
volatile bool arrow_sequence = false; /*!< Флаг идентификации последовательности символов для стрелок */

char message_buff[MESSAGE_BUFFER_SIZE] = {0}; /*!< Буфер для хранения строки на печать */

jtm_status_t JTM_status;      /*!< Статус JTM модуля */
enum rwc_status RWC_status;   /*!< Статус RWC модуля */

/*------------- Прототипы -------------*/

/*!
 * @brief CDC-обработчик.
 *
 * Функция инициализирует печать через tiny USB.
 *
 */
static void cdc_task(void);

/* Объявления функций обратных вызовов терминальных команд. */

/*!
 * @brief Функция печати символа для CLI.
 *
 * @param cli Структура CLI.
 * @param c   Символ для отображения в CLI.
 */
void EmbeddedCliWriteChar(EmbeddedCli *cli, char c);

/*!
 * @brief Функция-обработчик для неизвестной команды.
 *
 * @param cli Структура CLI.
 * @param cmd Структура команды.
 */
void EmbeddedCliOnCommand(EmbeddedCli *cli, CliCommand *cmd);

/* Блок объявления пользовательских команд. */

/*!
 * @brief Функция-обработчик команды whoami.
 *
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnWhoAmI(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды cpuinfo.
 *
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnCPUinfo(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды timenow.
 *
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnTimenow(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды settime.
 *
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnSetTime(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды reboot.
 *
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnReboot(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция обработки символа для CLI.
 *
 * Выполняет анализ переданного символа.
 *
 * @param sym Обрабатываемый символ.
 */
void AnalyzeChar(uint8_t input_sym)
{
    char sym;

    if (input_sym == 0) {
        return;
    } else {
        sym = (char) input_sym;
    }

    /* Если введен ненулевой символ. */
    if (sym != '\0') {
        /*
         * Если ранее был введен ESC, проверяем следующий символ
         * на последовательность UpArrow/DownArrow.
         */
        if (esc_pressed && (sym == '[')) {
            arrow_sequence = true;
            return;
        } else if (arrow_sequence && (sym == 'A' || sym == 'B')) {
            /*
             * Если введена полная последовательность UpArrow/DownArrow,
             * происходит передача символов в CLI.
             */
            embeddedCliReceiveChar(cli, '\x1B');
            embeddedCliReceiveChar(cli, '[');
            embeddedCliReceiveChar(cli, sym);
            embeddedCliProcess(cli);
            return;
        } else {
            /* Сброс всех флагов. */
            esc_pressed = false;
            arrow_sequence = false;
        }

        /*
         * Общая проверка введенного символа: если введен ESC, то
         * для следующего символа будет выполнена особая проверка.
         */
        if (sym == 27U) {
            esc_pressed = true;
            return;
        } else {
            /* Передача введенного символа в CLI. */
            embeddedCliReceiveChar(cli, sym);
            embeddedCliProcess(cli);
        }
    }
}

/*!
 * @brief Функция задержки tiny-USB.
 *
 * В цикле запускает основную функцию стека tiny-USB @ref tud_task.
 *
 * @param count Количество циклов операций стека tiny-USB.
 *
 */
void TUD_delay(uint32_t count)
{
    uint32_t iter = 0;
    while (iter < count) {
        tud_task();
        iter++;
    }
}

/*!
 * @brief Получение размера сообщения.
 *
 * Функция возвращает количество символов, отличных от нулевого символа ('\0').
 *
 * @return Размер текущего сообщения.
 */
uint32_t GetRealMessageSize()
{
    uint32_t length = 0;
    while (1) {
        if (message_buff[length] != 0) {
            length++;
        } else {
            return length;
        }
    }
}

/*!
 * @brief Отправка сообщения в CLI.
 *
 * Функция запускает отправку сообщения, после чего очищает
 * буфер отправки.
 *
 */
void SendMessage()
{
    TUD_delay(1);
    uint32_t count = GetRealMessageSize();

    for (uint32_t i = 0; i < count; i++) {
        EmbeddedCliWriteChar(NULL, message_buff[i]);
    }

    TUD_delay(1);
    printf("\r\n[INFO - PRINTED MESSAGE]:\r\n%s\r\n", message_buff);
}

/*------------- MAIN -------------*/
int main(void)
{
    static jtm_config_t jtm_config;    /*!< Параметры JTM */
    struct rwc_config rwc_config;      /*!< Параметры RWC */

    board_init();

    /* Конфигурация tiny USB. */
    tusb_rhport_init_t dev_init = {
        .role = TUSB_ROLE_DEVICE,
        .speed = TUSB_SPEED_AUTO,
    };
    tusb_init(BOARD_TUD_RHPORT, &dev_init);

    tud_cdc_n_write_flush(0);
    tud_cdc_n_read_flush(0);

    /* Конфигурация JTM. */
    jtm_config.tcal = TEST_TCAL;
    jtm_config.wcal = TEST_WCAL;
    jtm_config.wtconf = TEST_WTCONF;
    jtm_config.wtcalconf = TEST_WTCALCONF;

    JTM_Init(JTM_Secure, &jtm_config);

    /* Конфигурация RWC. */

    /* Сброс и запрет прерываний. */
    NVIC_ClearPendingIRQ(RWC_ALARM_IRQn);
    NVIC_DisableIRQ(RWC_ALARM_IRQn);
    RWC_EnableAlarmTimerInterruptFromDPD(RWC_Secure, 0);
    RWC_InterruptClear(RWC_Secure);

    /* Получение конфигурации по умолчанию и инициализация таймера. */
    RWC_GetDefaultConfig(&rwc_config);
    RWC_Init(RWC_Secure, rwc_config);

    /* Конфигурация CLI. */
    cli = embeddedCliNewDefault();

    /* Проверка инициализации CLI. */
    if (cli == NULL) {
        printf("\r\n[INFO]: CLI was not created. Check sizes!\r\n");
        return -1;
    }

    cli->onCommand = &EmbeddedCliOnCommand;
    cli->writeChar = &EmbeddedCliWriteChar;

    /* Непосредственное добавление команд в CLI. */
    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "whoami",
        "Return name of board/device.",
        false,
        NULL,
        EmbeddedCliOnWhoAmI
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "cpuinfo",
        "Print information about SoC.",
        false,
        NULL,
        EmbeddedCliOnCPUinfo
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "timenow",
        "Print current time of RWC timer.",
        false,
        NULL,
        EmbeddedCliOnTimenow
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "settime",
        "Set time/date of RWC timer\r\n\tSupported modes:\r\n\t  1) settime date dd.mm.yyyy,\r\n\t  2) settime time hh:mm:ss,\r\n\t  3) settime dd.mm.yyyy hh:mm:ss.",
        true,
        NULL,
        EmbeddedCliOnSetTime
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "reboot",
        "Reboot the CPU",
        false,
        NULL,
        EmbeddedCliOnReboot
    });

    /* Ожидание готовности tinyUSB CDC-устройства. */
    while(!tud_ready()) {
        tud_task();
    }

    TUD_delay(100000);

    sprintf(message_buff, "%s USB Embedded CLI example\r\n", BOARD_NAME);
    SendMessage();
    TUD_delay(100000);

    sprintf(message_buff, "CLI was started. Enter your commands\r\n");
    SendMessage();
    TUD_delay(100000);

    sprintf(message_buff, "Use command help to see all available commands\r\n");
    SendMessage();
    TUD_delay(100000);

    /* Инициирование вывода "пригласительного" символа '<'. */
    embeddedCliReceiveChar(cli, '\0');
    embeddedCliProcess(cli);

    while (1) {
        if (tud_task_event_ready()) {
            tud_task();
        }
        cdc_task();
    }

    return 0;
}

static void cdc_task()
{
    uint8_t res = tud_cdc_n_available(0);
    if (res > 0) {

        uint8_t buf[TUD_BUFFER_SIZE] = {0};
        uint32_t count = tud_cdc_read(buf, sizeof(buf));

        for (uint32_t i = 0; i < count; i++) {
            if (buf[i] > 0 || buf[i] == ' ') {
                AnalyzeChar(buf[i]);
            }

            if (tud_cdc_write_available() == 0) {
                tud_cdc_n_write_flush(0);
            }

            if (tud_task_event_ready()) {
                tud_task();
            }

        }

        tud_cdc_n_write_flush(0);
    }
}

/* Определения функций обратных вызовов исполняемых терминальных команд. */

void EmbeddedCliOnCommand(EmbeddedCli *cli, CliCommand *cmd)
{
    UNUSED(cli);
    sprintf(message_buff, "Unknown command \"%s\". ", cmd->name);
    SendMessage();

    sprintf(message_buff, "Use command help to see all available commands\r\n");
    SendMessage();
}

void EmbeddedCliWriteChar(EmbeddedCli *cli, char c)
{
    UNUSED(cli);
    tud_cdc_n_write_char(0, c);
    if (tud_cdc_write_available() == 0) {
        tud_cdc_n_write_flush(0);
        TUD_delay(1);
    }
    tud_cdc_n_write_flush(0);
}

void EmbeddedCliOnWhoAmI(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    sprintf(message_buff, "%s\r\n", BOARD_NAME);
    SendMessage();
}

void EmbeddedCliOnCPUinfo(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    int32_t value;
    uint32_t sys_clk_mhz = CLKCTR_GetSysClk(CLOCK_BASE) / 1000000;
    uint32_t f_clk_mhz = CLKCTR_GetFClk(CLOCK_BASE) / 1000000;

    sprintf(message_buff, "%-15s: Armv8-M\r\n", "Architecture");
    SendMessage();

    sprintf(message_buff, "%-15s: Cortex-M33\r\n", "Model name");
    SendMessage();

    sprintf(message_buff, "%-15s: %ldMHz\r\n", "Core 0 (freq.)", sys_clk_mhz);
    SendMessage();

    sprintf(message_buff, "%-15s: %ldMHz\r\n", "Core 1 (freq.)", f_clk_mhz);
    SendMessage();

    JTM_status = JTM_GetParameterValue(JTM_Secure, JTM_Temperature, &value);
    sprintf(message_buff, "%-15s: %.03f\r\n", "Temperature", value / 1000.);
    SendMessage();

    JTM_status = JTM_GetParameterValue(JTM_Secure, JTM_Vcasn, &value);
    sprintf(message_buff, "%-15s: %.03f\r\n", "V CASN", value / 1000.);
    SendMessage();

    JTM_status = JTM_GetParameterValue(JTM_Secure, JTM_Vcore, &value);
    sprintf(message_buff, "%-15s: %.03f\r\n", "V CORE (VDDC)", value / 1000.);
    SendMessage();
}

void EmbeddedCliOnTimenow(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    rtc_datetime_t time_now1, time_now2; /*!< Структура даты и времени для контроля чтения из RWC */
    uint32_t iter = 0;                   /*!< Количество попыток записи в RWC */

    /* Чтение данных из RWC. */
    do {
        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now1);
        if (RWC_status != RWC_Status_Ok) {
            sprintf(message_buff ,"RWC_GetDatetime error : %d\r\n", RWC_status);
            SendMessage();
            while (1)
                ;
        }

        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now2);
        if (RWC_status != RWC_Status_Ok) {
            sprintf(message_buff, "RWC_GetDatetime error : %d\r\n", RWC_status);
            SendMessage();
            while (1)
                ;
        }

        iter++;
    } while (CompareTime(&time_now1, &time_now2) == 0 && iter < 10);

    /* Вывод ошибки, если за 10 попыток не удалось корректно считать время. */
    if (CompareTime(&time_now1, &time_now2) == 0) {
        sprintf(message_buff, "CompareTime error : read different values from the same register\r\n");
        SendMessage();
        return;
    }

    sprintf (message_buff, "Date: %02d.%02d.%02d\r\nTime: %02d:%02d:%02d\r\n",
        time_now1.day,
        time_now1.month,
        time_now1.year,
        time_now1.hour,
        time_now1.minute,
        time_now1.second);
    SendMessage();
}

void EmbeddedCliOnReboot(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    sprintf(message_buff, "Rebooting in %u second(-s)...\r\n\r\n", REBOOT_TIME);
    SendMessage();

    struct wdt_config config; /* Конфигурация watch-dog таймера */

    uint32_t frequency = CLKCTR_GetSysClk(CLOCK_BASE);

    /* Установка таймера на REBOOT_TIME секунд с разрешением прерывания и сброса. */
    WDT_GetDefaultConfig(&config);
    config.load = REBOOT_TIME * frequency;
    config.inten = WDT_IntenEnable;
    config.resen = WDT_ResenEnable;
    WDT_Init(SWDT_Secure, &config);

    while (1) {};
}
void EmbeddedCliOnSetTime(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(context);

    rtc_datetime_t time_now1, time_now2;    /*!< Структура даты и времени для контроля чтения/записи RWC */
    uint32_t iter = 0;                      /*!< Количество попыток записи чтения и записи RWC */
    enum dtime_status_t status;             /*!< Возвращаемое значение функции установки времени/даты */

    /* Чтение данных из RWC. */
    do {
        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now1);
        if (RWC_status != RWC_Status_Ok) {
            sprintf(message_buff, "RWC_GetDatetime error : %d\r\n", RWC_status);
            SendMessage();
            return;
        }

        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now2);
        if (RWC_status != RWC_Status_Ok) {
            sprintf(message_buff, "RWC_GetDatetime error : %d\r\n", RWC_status);
            SendMessage();
            return;
        }
        iter++;
    } while (CompareTime(&time_now1, &time_now2) == 0 && iter < 10);

    /* Вывод ошибки, если за 10 попыток не удалось корректно считать время. */
    if (CompareTime(&time_now1, &time_now2) == 0) {
        sprintf(message_buff, "CompareTime error : read different values from single register\r\n");
        SendMessage();
        return;
    }

    /* Выполнение функции обработки введенных пользователем параметров
     * и проверка возвращаемого значения.
     */
    status = SetDatetime(&time_now1, args);
    if (status != Dtime_Status_SetDateOk && status != Dtime_Status_SetTimeOk) {
        sprintf(message_buff, "SetDatetime error : %s\r\n", InfoDtimeStatus(&status));
        SendMessage();
    }

    iter = 0;

    /* Запись новых значений даты/времени в RWC. */
    do {
        RWC_status = RWC_SetDatetime(RWC_Secure, &time_now1);
        if (RWC_status != RWC_Status_Ok) {
            sprintf(message_buff, "RWC_SetDatetime error : %d\r\n", RWC_status);
            SendMessage();
            return;
        }

        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now2);
        if (RWC_status != RWC_Status_Ok) {
            sprintf(message_buff, "RWC_GetDatetime error : %d\r\n", RWC_status);
            SendMessage();
            return;
        }
        iter++;
    } while (CompareTime(&time_now1, &time_now2) == 0 && iter < 10);

    sprintf(message_buff, "Succesfull!\r\n");
    SendMessage();

    EmbeddedCliOnTimenow(NULL, NULL, NULL);
}
