/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup can_driver Драйвер модуля CAN
 *
 * @brief Драйвер ввода-вывода по последовательному шинному интерфейсу CAN
 *
 * Драйвер содержит функции управления контроллером CAN микросхемы ELIOT1.
 */

/*!
 * @addtogroup can_driver
 * @{
 */

/*!
 * @file hal_can.h
 *
 * @brief Интерфейс драйвера модуля ввода-вывода по интерфейсу CAN
 */

#ifndef HAL_CAN_H
#define HAL_CAN_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"

/*!
 * @brief Версия драйвера CAN
 */
#define HAL_CAN_DRIVER_VERSION (MAKE_VERSION(1, 1, 0))

/*!
 * @brief Коды возврата функций драйвера CAN
 */
typedef enum _can_status {
    CAN_Status_Ok              = 0U, /*!< Успешно */
    CAN_Status_Fail            = 1U, /*!< Провал */
    CAN_Status_InvalidArgument = 2U, /*!< Неверный аргумент */
    CAN_Status_TxBusy          = 3U, /*!< Передатчик занят */
    CAN_Status_RxEmpty         = 4U, /*!< Приемный буфер пуст */
    CAN_Status_TxIdle          = 5U, /*!< Заданное число кадров выдано */
    CAN_Status_RxIdle          = 6U, /*!< Заданное число кадров принято */
    CAN_Status_RxBusy          = 7U, /*!< Уже запущен прием */
} can_status_t;

/*!
 * @brief Виды ошибок на линии CAN
 *
 * Перечисление содержит виды ошибок, которые определяет контроллер CAN.
 * Одно из данных значений устанавливается контроллером в поле koer
 * структуры принятого кадра.
 */
typedef enum _can_kind_of_error {
    CAN_ErrorNone       = 0U, /*!< Нет ошибки */
    CAN_ErrorBitError   = 1U, /*!< Ошибка бита */
    CAN_ErrorFormError  = 2U, /*!< Ошибка формы */
    CAN_ErrorStuffError = 3U, /*!< Ошибка вставки дополнительных битов (бит-стаффинга) */
    CAN_ErrorAckError   = 4U, /*!< Ошибка подтверждения */
    CAN_ErrorCrcError   = 5U, /*!< Ошибка контрольной суммы */
    CAN_ErrorOtherError = 6U, /*!< Прочие ошибки: доминантные биты после передачи собственного признака ошибки, признак ошибки был слишком длинным, доминантный бит при передаче признака Passive-Error после определения ошибки подтверждения */
    CAN_ErrorReserved   = 7U, /*!< Не используется */
} can_kind_of_error_t;

/*!
 * @brief Флаги прерываний и состояний CAN
 *
 * Флаги в перечислении делятся на флаги прерываний, на которые можно
 * зарегистрировать обработчик прерывания, и флаги состояний, которые только
 * сигнализируют об определенном состоянии контроллера CAN и на которые нельзя
 * прикрепить обработчик. Флаги состояния имеют суффикс State в названии и
 * слово "состояние" в русском пояснении.
 */
typedef enum _can_flag {
    CAN_FlagAbortState            =  0U, /*!< Состояние отмененной передачи */
    CAN_FlagError                 =  1U, /*!< Флаг ошибки */
    CAN_FlagTransmissionSecondary =  2U, /*!< Выполнена передача из низкоприоритетного буфера */
    CAN_FlagTransmissionPrimary   =  3U, /*!< Выполнена передача из высокоприоритетного буфера */
    CAN_FlagTransmitBufferFull    =  4U, /*!< Буфер выдачи заполнен */
    CAN_FlagRBAlmostFull          =  5U, /*!< Буфер приема почти заполнен */
    CAN_FlagRBFull                =  6U, /*!< Буфер приема заполнен */
    CAN_FlagRBOverrun             =  7U, /*!< Переполнение буфера приема */
    CAN_FlagReceive               =  8U, /*!< Выполнен прием кадра */
    CAN_FlagBusError              =  9U, /*!< Ошибка шины данных(BUS ERROR) */
    CAN_FlagArbitrationLost       = 10U, /*!< Проигран арбитраж на шине данных */
    CAN_FlagErrorPassiveInterrupt = 11U, /*!< Переход в пассивный режим */
    CAN_FlagErrorPassiveState     = 12U, /*!< Состояние пассивного режима */
    CAN_FlagErrorWarningState     = 13U, /*!< Состояние, в котором количество ошибок достигло уровня предупреждения */
    CAN_FlagTimeTriggered         = 14U, /*!< Сработал триггер TTCAN */
    CAN_FlagTriggerError          = 15U, /*!< Ошибка триггера TTCAN */
    CAN_FlagWatchTriggerError     = 16U, /*!< Сработал следящий триггер TTCAN */
    CAN_FlagsNumber                      /*!< Константа - количество флагов состояния */
} can_flag_t;

/*!
 * @brief Режимы работы по протоколу CAN FD (есть отличия в расчете контрольной
 *        суммы и формировании битов стаффинга
 */
typedef enum _canfd_mode {
    CAN_BoschFd = 0U, /*!< Режим Bosch CAN FD */
    CAN_IsoFd   = 1U, /*!< Режим ISO CAN FD */
} canfd_mode_t;

/*!
 * @brief Дисциплина выдачи из низкоприоритетного буфера
 */
typedef enum _can_stb_discipline {
    CAN_Fifo     = 0U, /*!< Выдача в порядке поступления (по очереди) */
    CAN_Priority = 1U, /*!< Выдача в соответствии с приоритетом кадра */
} can_stb_discipline_t;

/*!
 * @brief Размер данных кадра CAN, указываемый в поле DLC
 */
typedef enum _can_bytes_in_datafield {
    CAN_0ByteDatafield  =  0U, /*!< 0 байт */
    CAN_1ByteDatafield  =  1U, /*!< 1 байт */
    CAN_2ByteDatafield  =  2U, /*!< 2 байтa */
    CAN_3ByteDatafield  =  3U, /*!< 3 байтa */
    CAN_4ByteDatafield  =  4U, /*!< 4 байтa */
    CAN_5ByteDatafield  =  5U, /*!< 5 байт */
    CAN_6ByteDatafield  =  6U, /*!< 6 байт */
    CAN_7ByteDatafield  =  7U, /*!< 7 байт */
    CAN_8ByteDatafield  =  8U, /*!< 8 байт */
    CAN_12ByteDatafield =  9U, /*!< 12 байт */
    CAN_16ByteDatafield = 10U, /*!< 16 байт */
    CAN_20ByteDatafield = 11U, /*!< 20 байт */
    CAN_24ByteDatafield = 12U, /*!< 24 байтa */
    CAN_32ByteDatafield = 13U, /*!< 32 байтa */
    CAN_48ByteDatafield = 14U, /*!< 48 байт */
    CAN_64ByteDatafield = 15U, /*!< 64 байтa */
} can_bytes_in_datafield_t;

#if defined(__CC_ARM)
#pragma anon_unions
#endif

/*!
 * @brief Структура буфера передачи кадра CAN
 */
typedef struct _can_tx_buffer_frame {
    struct {
        uint32_t                 id    : 29; /*!< Идентификатор кадра CAN */
        uint32_t                       :  2; /*!< Выравнивание */
        bool                     ttsen :  1; /*!< Включение отметок времени передачи (CiA 603) */
    };
    struct {
        can_bytes_in_datafield_t dlc   :  4; /*!< Длина поля данных */
        bool                     brs   :  1; /*!< Разрешение переключения скорости передачи (для CAN FD) */
        bool                     fdf   :  1; /*!< Признак формата CAN FD */
        bool                     rtr   :  1; /*!< Признак кадра удаленного запроса */
        bool                     ide   :  1; /*!< Признак расширенного идентификатора */
        uint32_t                       : 24; /*!< Выравнивание */
    };
    uint8_t                      data[64];   /*!< Поле данных кадра CAN */
} can_tx_buffer_frame_t;

/*!
 * @brief Структура буфера приема кадра CAN
 */
typedef struct _can_rx_buffer_frame {
    struct {
        uint32_t                 id         : 29; /*!< Идентификатор кадра CAN */
        uint32_t                            :  2; /*!< Выравнивание */
        bool                     esi        :  1; /*!< Признак состояния ошибки узла, передавшего кадр */
    };
    struct {
        can_bytes_in_datafield_t dlc        :  4; /*!< Длина поля данных */
        bool                     brs        :  1; /*!< Разрешение переключения скорости передачи (для CAN FD) */
        bool                     fdf        :  1; /*!< Признак формата CAN FD */
        bool                     rtr        :  1; /*!< Признак кадра удаленного запроса */
        bool                     ide        :  1; /*!< Признак расширенного идентификатора */
        uint32_t                            :  4; /*!< Выравнивание */
        bool                     tx         :  1; /*!< Буфер активен */
        can_kind_of_error_t      koer       :  3; /*!< Вид ошибки */
        uint32_t                 cycle_time : 16; /*!< Время приема кадра по таймеру TTCAN */
    };
    uint8_t                      data[64];        /*!< Поле данных кадра CAN */
    uint64_t                     rts;             /*!< Отметка времени приема кадра (CiA 603) */
} can_rx_buffer_frame_t;

/*!
 * @brief Фильтр принятых кадров CAN
 */
typedef struct _can_frame_filter {
    uint32_t id               : 29; /*!< Идентификатор принятого кадра */
    uint32_t                  :  3; /*!< Выравнивание */
    uint32_t mask             : 29; /*!< Маска битов проверки принятого кадра. Для каждого бита идентификатора принятого кадра он проверяется на равенство с битом, заданном в фильтре, если соответствующий бит маски установлен; иначе не проверяется (значение бита идентификатора принятого кадра может быть любым) */
    uint32_t accepted_ide     :  1; /*!< Значение признака расширенного кадра, если его проверка включена (#enable_ide_check) */
    uint32_t enable_ide_check :  1; /*!< Проверять ли при фильтрации признак расширенного кадра */
    uint32_t                  :  1; /*!< Выравнивание */
} can_frame_filter_t;

/*!
 * @brief Количество входных фильтров CAN
 */
#define CAN_NB_OF_FILTERS 4

/*!
 * @brief Конфигурация фильтрации принятых кадров CAN
 */
typedef struct _can_frame_filter_config {
    can_frame_filter_t filter[CAN_NB_OF_FILTERS]; /*!< Массив настроек фильтров CAN */
    uint8_t            nb_filters_used;           /*!< Количество задействованных фильтров */
} can_frame_filter_config_t;

/*!
 * @brief Символьные константы значений делителя блока TTCAN
 */
typedef enum _ttcan_timer_prescaler {
    CAN_TTCANDiv1 = 0U, /*!< Делитель отсутствует */
    CAN_TTCANDiv2 = 1U, /*!< Делитель равен 2 */
    CAN_TTCANDiv4 = 2U, /*!< Делитель равен 4 */
    CAN_TTCANDiv8 = 3U, /*!< Делитель равен 8 */
} ttcan_timer_prescaler_t;

/*!
 * @brief Тип триггера TTCAN
 */
typedef enum _ttcan_trigger_type {
    CAN_TriggerImmediate          = 0U, /*!< Передача запускается сразу */
    CAN_TriggerTime               = 1U, /*!< Триггер только устанавливает флаг прерывания */
    CAN_TriggerSingleShotTransmit = 2U, /*!< Триггер предназначен для использования в окнах выдачи с эксклюзивным доступом только одного узла */
    CAN_TriggerTransmitStart      = 3U, /*!< Триггер предназначен для запуска передачи в окнах выдачи, в котором могут выдавать несколько узлов */
    CAN_TriggerTransmitStop       = 4U, /*!< Триггер предназначен для останова передачи в окнах выдачи, в котором могут выдавать несколько узлов */
} ttcan_trigger_type_t;

/*!
 * @brief Временные параметры передачи битов CAN
 */
typedef struct _ttcan_config {
    ttcan_timer_prescaler_t prescaler;                /*!< Предделитель счетчика TTCAN */
    uint8_t                 transmit_trigger_pointer; /*!< Указатель на слот передачи */
    ttcan_trigger_type_t    trigger_type;             /*!< Тип триггера */
    uint8_t                 transmit_enable_window;   /*!< Ширина окна разрешения передачи */
    uint8_t                 trigger_time0;            /*!< Время по циклическому таймеру TTCAN для триггера 0 */
    uint8_t                 trigger_time1;            /*!< Время по циклическому таймеру TTCAN для триггера 1 */
    uint8_t                 watch_trigger_time0;      /*!< Время по циклическому таймеру TTCAN для следящего триггера 0 */
    uint8_t                 watch_trigger_time1;      /*!< Время по циклическому таймеру TTCAN для следящего триггера 1 */
    uint32_t                reference_id;             /*!< Идентификатор референсного кадра */
    bool                    reference_ide;            /*!< Признак расширенного идентификатора у референсного кадра */
} ttcan_config_t;

/*!
 * @brief Временные параметры передачи битов CAN
 */
typedef struct _can_timing_config {
    uint8_t prescaler;                     /*!< Делитель системной частоты */
    uint8_t sjw;                           /*!< Максимально допустимый скачок при синхронизации */
    uint8_t seg1;                          /*!< Время сегмента 1 */
    uint8_t seg2;                          /*!< Время сегмента 2 */
    uint8_t data_prescaler;                /*!< Делитель системной частоты для сегмента данных (CAN FD) */
    uint8_t data_sjw;                      /*!< Максимально допустимый скачок при синхронизации сегмента данных (CAN FD) */
    uint8_t data_seg1;                     /*!< Время сегмента 1 (CAN FD) */
    uint8_t data_seg2;                     /*!< Время сегмента 2 (CAN FD) */
    uint8_t delay_compensation_enable;     /*!< Включение компенсации задержки передатчика (CAN FD) */
    uint8_t secondary_sample_point_offset; /*!< Смещение второй точки чтения для компенсации задержки (CAN FD) */
} can_timing_config_t;

/*!
 * @brief Параметры высокоприоритетного буфера выдачи
 */
typedef struct _can_ptb_config {
    bool tx_single_shot; /*!< Включение режима единичной выдачи */
} can_ptb_config_t;

/*!
 * @brief Параметры низкоприоритетного буфера выдачи
 */
typedef struct _can_stb_config {
    bool                 tx_single_shot; /*!< Включение режима единичной выдачи */
    can_stb_discipline_t tx_discipline;  /*!< Дисциплина выдачи кадров из низкоприоритетного буфера */
} can_stb_config_t;

/*!
 * @brief Параметры буфера приема
 */
typedef struct _can_rxb_config {
    uint32_t almost_full_level; /*!< Количество кадров в приемном буфере, при котором он считает почти полным */
    bool     self_acknowledge;  /*!< Режим подтверждения приема своих же кадров */
    bool     prohibit_overflow; /*!< При заполненной очереди новый кадр не принимается */
} can_rxb_config_t;

/*!
 * @brief Структура конфигурации контроллера CAN
 */
typedef struct _can_config {
    bool                      enable_listen_only;  /*!< Работа только в режиме прослушки шины данных */
    bool                      enable_loopback_int; /*!< Включение внутренней петли контроллера CAN */
    bool                      enable_loopback_ext; /*!< Включение внешней петли контроллера CAN */
    can_ptb_config_t          ptb_config;          /*!< Параметры высокоприоритетного буфера */
    can_stb_config_t          stb_config;          /*!< Параметры низкоприоритетного буфера */
    can_rxb_config_t          rxb_config;          /*!< Параметры буфера приема */
    can_timing_config_t       timing_config;       /*!< Битовая времянка */
    canfd_mode_t              can_fd_mode;         /*!< Режим работы по CAN FD */
    can_frame_filter_config_t filter_config;       /*!< Установки входных фильтров кадров CAN */
} can_config_t;

/*!
 * @brief Структура для передачи кадра CAN в неблокирующем режиме (по
 *        прерыванию)
 */
typedef struct _can_tx_transfer {
    can_tx_buffer_frame_t   *frames;    /*!< Указатель на буфер с кадрами для передачи */
    size_t                  nb_frames;  /*!< Количество кадров для передачи */
} can_tx_transfer_t;

/*!
 * @brief Структура для приема кадра CAN в неблокирующем режиме (по прерыванию)
 */
typedef struct _can_rx_transfer {
    can_rx_buffer_frame_t   *frames;    /*!< Указатель на буфер для приема кадра CAN */
    size_t                  nb_frames;  /*!< Количество кадров для приема */
} can_rx_transfer_t;

/*!
 * @brief Декларация типа дескриптора драйвера CAN
 */
typedef struct _can_handle can_handle_t;

/*!
 * @brief Функция обратного вызова CAN
 */
typedef void (*can_transfer_callback_t)(CAN_Type *base, can_handle_t *handle,
    can_status_t status, can_flag_t interrupt_flag, void *user_data);

/*!
 * @brief Структура дескриптора драйвера CAN
 */
struct _can_handle {
    volatile const can_tx_buffer_frame_t *tx_frames_prim;        /*!< Адрес оставшихся кадров для отправки через высокоприоритетный буфер */
    volatile size_t                      tx_nb_frames_rest_prim; /*!< Количество оставшихся кадров для отправки через высокоприоритетный буфер */
    size_t                               tx_nb_frames_all_prim;  /*!< Количество кадров для отправки через высокоприоритетный буфер */

    volatile const can_tx_buffer_frame_t *tx_frames_sec;         /*!< Адрес оставшихся кадров для отправки через низкоприоритетный буфер */
    volatile size_t                      tx_nb_frames_rest_sec;  /*!< Количество оставшихся кадров для отправки через низкоприоритетный буфер */
    size_t                               tx_nb_frames_all_sec;   /*!< Количество кадров для отправки через низкоприоритетный буфер */

    volatile can_rx_buffer_frame_t       *rx_frames;             /*!< Адрес оставшихся кадров для приема */
    volatile size_t                      rx_nb_frames_rest;      /*!< Количество оставшихся кадров для приема */
    size_t                               rx_nb_frames_all;       /*!< Количество кадров для приема */

    can_transfer_callback_t              callback;               /*!< Функция обратного вызова */
    void                                 *user_data;             /*!< Параметр функции обратного вызова */
};

/*!
 * @name Инициализация и деинициализация
 * @{
 */

/*!
 * @brief Инициализация драйвера CAN
 *
 * Функция инициализирует модуль CAN в соответствии с заданными
 * пользовательскими параметрами.
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с параметрами конфигурации
 */
can_status_t CAN_Init(CAN_Type *base, const can_config_t *config);

/*!
 * @brief Деинициализация драйвера CAN
 *
 * Функция деинициализирует контроллер CAN.
 *
 * @param base Базовый адрес контроллера
 */
void CAN_Deinit(CAN_Type *base);

/*!
 * @brief Получение параметров драйвера CAN по умолчанию
 *
 * @param config          Структура с параметрами конфигурации
 * @param source_clock_hz Опорная частота в Гц
 */
void CAN_GetDefaultConfig(can_config_t *config, uint32_t source_clock_hz);

/*!
 * @brief Переключение контроллера CAN в рабочий режим
 *
 * Данную функцию необходимо вызвать после инициализации, чтобы запустить
 * процесс подключения контроллера CAN к шине данных.
 *
 * @param base Базовый адрес контроллера
 */
void CAN_EnterNormalMode(CAN_Type *base);

/*!
 * @brief Переключение контроллера CAN в режим ожидания
 *
 * @param base Базовый адрес контроллера
 */
void CAN_EnterStandbyMode(CAN_Type *base);

/*!
 * @}
 */

/*!
 * @name Конфигурирование настроек приемопередачи
 * @{
 */

/*!
 * @brief Расчет рекомендуемых временных параметров (битовых таймингов) для
 *        указанных скоростей обмена в сегменте управления и данных
 *
 * @param baudrate        Скорость CANFD в битах/с при передаче управляющей
 *                        части кадра
 * @param baudrate_data   Скорость данных в битах/с (для CANFD)
 * @param source_clock_hz Опорная частота в Гц
 * @param pconfig         Структура с временными параметрами
 *
 * @retval true  Конфигурация найдена
 * @retval false Не удалось найти правильную конфигурацию
 */
bool CAN_CalculateImprovedTimingValues(uint32_t baudrate,
    uint32_t baudrate_data, uint32_t source_clock_hz,
    can_timing_config_t *pconfig);

/*!
 * @brief Установка временных параметров (битовых таймингов) CAN
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с временными параметрами
 */
void CAN_SetArbitrationTimingConfig(CAN_Type *base,
    const can_timing_config_t *config);

/*!
 * @brief Установка параметров высокоприоритетного буфера выдачи
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с параметрами высокоприоритетного буфера выдачи
 *               кадров
 */
void CAN_SetPrimaryTxBufferConfig(CAN_Type *base,
    const can_ptb_config_t *config);

/*!
 * @brief Установка параметров низкоприоритетного буфера выдачи
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с параметрами низкоприоритетного буфера выдачи кадров
 */
void CAN_SetSecondaryTxBufferConfig(CAN_Type *base,
    const can_stb_config_t *config);

/*!
 * @brief Установка параметров буфера приема
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с параметрами буфера приема кадров
 */
void CAN_SetRxBufferConfig(CAN_Type *base, const can_rxb_config_t *config);

/*!
 * @brief Установка параметров фильтрации кадров при приеме
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с параметрами фильтрации кадров
 */
void CAN_SetFilterConfig(CAN_Type *base,
    const can_frame_filter_config_t *config);

/*!
 * @brief Установка параметров работы контроллера в режиме TTCAN
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с параметрами TTCAN
 */
void CAN_SetTTCANConfig(CAN_Type *base, const ttcan_config_t *config);

/*!
 * @}
 */

/*!
 * @name Флаги статусов и прерываний
 * @{
 */

/*!
 * @brief Получение состояния флага состояния/прерывания
 *
 * @param base   Базовый адрес контроллера
 * @param idx    Номер флага состояния/прерывания
 * @param status По переданному указателю функция записывает свой статус
 *               выполнения. При передаче нулевого указателя запись не
 *               производится.
 *
 * @retval true  Состояние активно
 * @retval false Состояние неактивно
 */
bool CAN_GetStatusFlag(CAN_Type *base, can_flag_t idx, can_status_t *status);

/*!
 * @brief Получение маски активных прерываний
 *
 * @param base Базовый адрес контроллера
 *
 * @return Маска активных прерываний
 */
uint32_t CAN_GetStatusFlagMask(CAN_Type *base);

/*!
 * @brief Сброс флага состояния/прерывания
 *
 * @param base Базовый адрес контроллера
 * @param idx  Номер флага состояния/прерывания
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_InvalidArgument
 */
can_status_t CAN_ClearStatusFlag(CAN_Type *base, can_flag_t idx);

/*!
 * @brief Сброс флагов прерываний по маске
 *
 * @note Позиции битов для маски флагов прерываний указаны в перечислении
 *       #can_flag_t.
 *
 * @param base Базовый адрес контроллера
 * @param mask Маска флагов прерываний, которые необходимо сбросить
 */
void CAN_ClearStatusFlagMask(CAN_Type *base, uint32_t mask);

/*!
 * @}
 */

/*!
 * @name Управление прерываниями
 * @{
 */

/*!
 * @brief Разрешение прерывания
 *
 * @param base Базовый адрес контроллера
 * @param idx  Номер флага состояния/прерывания
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_InvalidArgument
 */
can_status_t CAN_EnableInterrupt(CAN_Type *base, can_flag_t idx);

/*!
 * @brief Разрешение прерываний по маске
 *
 * @param base Базовый адрес контроллера
 * @param mask Маска флагов прерываний, которые необходимо разрешить
 */
void CAN_EnableInterruptMask(CAN_Type *base, uint32_t mask);

/*!
 * @brief Запрос - разрешено ли прерывание CAN
 *
 * @param base   Базовый адрес CAN
 * @param idx    Номер флага состояния/прерывания
 * @param status По переданному указателю функция записывает свой статус
 * выполнения. При передаче нулевого указателя запись не производится.
 *
 * @retval true  Прерывание разрешено
 * @retval false Прерывание запрещено
 */
bool CAN_IsInterruptEnabled(CAN_Type *base, can_flag_t idx,
    can_status_t *status);

/*!
 * @brief Запрос маски разрешенных прерываний CAN
 *
 * Запрос маски разрешенных прерываний CAN; единицы в соответствующих
 * разрядах соответствуют включенным прерываниям.
 *
 * @param base Базовый адрес CAN
 *
 * @return Маска разрешенных прерываний
 *
 */
uint32_t CAN_GetEnabledInterruptMask(CAN_Type *base);

/*!
 * @brief Запрет прерывания
 *
 * @param base Базовый адрес контроллера
 * @param idx  Номер флага состояния/прерывания
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_InvalidArgument
 */
can_status_t CAN_DisableInterrupt(CAN_Type *base, can_flag_t idx);

/*!
 * @brief Запрет прерываний по маске
 *
 * @param base Базовый адрес контроллера
 * @param mask Маска флагов прерываний, которые необходимо запретить
 */
void CAN_DisableInterruptMask(CAN_Type *base, uint32_t mask);

/*!
 * @}
 */

/*!
 * @name Прямое управление выдачей и приемом
 * @{
 */

/*!
 * @brief Получение признака требования выдачи для высокоприоритетного буфера
 *
 * Данная функция возвращает состояние флага требования выдачи для
 * высокоприоритетного буфера. Если флаг установлен, то буфер занят.
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Признак выдачи активен (буфер занят)
 * @retval false Признак сброшен
 */
bool CAN_IsPrimaryTransmitRequestPending(CAN_Type *base);

/*!
 * @brief Получение признака требования выдачи для низкоприоритетного буфера
 *
 * Данная функция возвращает состояние флага требования выдачи для ячейки
 * низкоприоритетного буфера (при использовании приоритетного порядка выдачи из
 * низкоприоритетного буфера). Если флаг установлен, то ячейка занята.
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Признак выдачи активен (буфер занят)
 * @retval false Признак сброшен
 */
bool CAN_IsSecondaryTransmitRequestPending(CAN_Type *base);

/*!
 * @brief Получение признака опустошения низкоприоритетного буфера
 *
 * Данная функция сообщает, пуст ли низкоприоритетный буфер.
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Буфер пуст
 * @retval false В буфере есть кадры
 */
bool CAN_IsSecondaryTxBufferEmpty(CAN_Type *base);

/*!
 * @brief Получение признака заполнения низкоприоритетного буфера выше середины
 *
 * Данная функция сообщает, пуст ли низкоприоритетный буфер до уровня более
 * половины всех доступных ячеек.
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Буфер заполнен выше середины
 * @retval false Буфер заполнен меньше середины
 */
bool CAN_IsSecondaryTxBufferMoreThanHalfFull(CAN_Type *base);

/*!
 * @brief Получение признака заполнения низкоприоритетного буфера.
 *
 * Данная функция сообщает, заполнен ли низкоприоритетный буфер.
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Очередь пуста
 * @retval false В очереди есть кадры
 */
bool CAN_IsSecondaryTxBufferFull(CAN_Type *base);

/*!
 * @brief Запись кадра в высокоприоритетный буфер передачи.
 *
 * Данная функция записывает кадр в высокоприоритетный буфер (PTB, Primary
 * Transmit Buffer). Выдача из данного буфера "вклинивается" в выдачу из
 * низкоприоритетного буфера. Высокоприоритетный буфер может содержать только
 * один кадр CAN. Функция не блокирующая.
 *
 * @param base     Базовый адрес контроллера
 * @param ptxframe Кадр для выдачи
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_TxBusy
 * @retval #CAN_Status_InvalidArgument
 */
can_status_t CAN_WritePrimaryTxBuffer(CAN_Type *base,
    const can_tx_buffer_frame_t *ptxframe);

/*!
 * @brief Запись кадра в низкоприоритетный буфер передачи
 *
 * Данная функция записывает кадр в низкоприоритетный буфер (STB, Secondary
 * Transmit Buffer), работающий в режиме выдачи кадров в соответствии с их
 * приоритетом.
 *
 * @param base     Базовый адрес контроллера
 * @param ptxframe Кадр для выдачи
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_TxBusy
 */
can_status_t CAN_WriteSecondaryTxBuffer(CAN_Type *base,
    const can_tx_buffer_frame_t *ptxframe);

/*!
 * @brief Отмена выдачи кадра из высокоприоритетного буфера
 *
 * @param base Базовый адрес контроллера
 */
void CAN_AbortPrimaryTxBuffer(CAN_Type *base);

/*!
 * @brief Отмена выдачи кадров из низкоприоритетного буфера
 *
 * Данная функция отменяет выдачу кадров из низкоприотетного буфера (STB,
 * Secondary Transmit Buffer). Если буфер работает в режиме приоритетной выдачи,
 * то отменяемые кадры определяются маской, если соответстветствующий номеру
 * ячейки бит установлен, то выполняется отмена выдачи. Если буфер работает в
 * режиме очереди, то выполняется полная очистка буфера, независимо от значения,
 * переданного в маске.
 *
 * @param base Базовый адрес контроллера
 */
void CAN_AbortSecondaryTxBuffer(CAN_Type *base);

/*!
 * @brief Получение признака опустошения буфера приема
 *
 * Данная функция сообщает, пуст ли буфер приема.
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Буфер пуст
 * @retval false В буфере есть кадры
 */
bool CAN_IsRxBufferEmpty(CAN_Type *base);

/*!
 * @brief Получение признака заполнения буфера приема до границы "почти полный"
 *
 * Данная функция сообщает, заполнен ли буфер приема до состояния "почти
 * полный".
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Буфер заполнен до состояния "почти полный"
 * @retval false Буфер не заполнен до состояния "почти полный"
 */
bool CAN_IsRxBufferAlmostFull(CAN_Type *base);

/*!
 * @brief Получение признака заполнения буфера приема
 *
 * Данная функция сообщает, заполнен ли буфер приема.
 *
 * @param base Базовый адрес контроллера
 *
 * @retval true  Буфер заполнен
 * @retval false Буфер не заполнен
 */
bool CAN_IsRxBufferFull(CAN_Type *base);

/*!
 * @brief Чтение принятого кадра из приемной очереди
 *
 * Функция считывает один принятый кадр CAN из приемной очереди. Считанный кадр
 * при этом удаляется из очереди.
 *
 * @param base     Базовый адрес контроллера
 * @param prxframe Принятый кадр
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_RxEmpty
 */
can_status_t CAN_ReadRxBuffer(CAN_Type *base, can_rx_buffer_frame_t *prxframe);

/*!
 * @}
 */

/*!
 * @name Транзакционные передача и прием
 * @{
 */

/*!
 * @brief Блокирующая выдача кадра через высокоприоритетный буфер выдачи
 *
 * При использовании блокирующей выдачи не нужно устанавливать обработчик
 * соответствующего события.
 *
 * @param base      Базовый адрес контроллера
 * @param ptxframe  Буфер с кадрами для выдачи
 * @param nb_frames Количество кадров для выдачи
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_Fail
 */
can_status_t CAN_TransferSendPrimaryBlocking(CAN_Type *base,
    can_tx_buffer_frame_t *ptxframe, size_t nb_frames);

/*!
 * @brief Блокирующая выдача кадра через низкоприоритетный буфер выдачи
 *
 * При использовании блокирующей выдачи не нужно устанавливать обработчик
 * соответствующего события.
 *
 * @param base      Базовый адрес контроллера
 * @param ptxframe  Буфер с кадрами для выдачи
 * @param nb_frames Количество кадров для выдачи
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_Fail
 */
can_status_t CAN_TransferSendSecondaryBlocking(CAN_Type *base,
    can_tx_buffer_frame_t *ptxframe, size_t nb_frames);

/*!
 * @brief Блокирующий прием кадра
 *
 * При использовании блокирующей выдачи не нужно устанавливать обработчик
 * соответствующего события.
 *
 * @param base      Базовый адрес контроллера
 * @param prxframe  Буфер для приёма кадров
 * @param nb_frames Максимальное количество кадров в буфере для приёма
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_Fail
 */
can_status_t CAN_TransferReceiveFifoBlocking(CAN_Type *base,
    can_rx_buffer_frame_t *prxframe, size_t nb_frames);

/*!
 * @brief Инициализация обработчика событий CAN
 *
 * @param base      Базовый адрес контроллера
 * @param handle    Обработчик
 * @param callback  Функция обратного вызова
 * @param user_data Аргумент функции обратного вызова
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_Fail
 */
can_status_t CAN_TransferCreateHandle(CAN_Type *base, can_handle_t *handle,
    can_transfer_callback_t callback, void *user_data);

/*!
 * @brief Неблокирующая выдача через высокоприоритетный буфер
 *
 * Функция выдает кадр CAN по прерыванию. Это неблокирующая функция, она
 * возвращает управление сразу. По выдачи кадра в шину данных вызывает функцию
 * обратного вызова.
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 * @param xfer   Структура для передачи кадра в неблокирующем режиме
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_Fail
 * @retval #CAN_Status_TxBusy
 */
can_status_t CAN_TransferSendPrimaryNonBlocking(CAN_Type *base,
    can_handle_t *handle, can_tx_transfer_t *xfer);

/*!
 * @brief Возвращает количество кадров, отправленных в шину данных через
 * высокоприоритетный буфер
 *
 * Эта функция возвращает количество кадров, отправленных в шину данных по
 * прерыванию через высокоприоритетный буфер.
 *
 * @param base      Базовый адрес контроллера
 * @param handle    Обработчик
 * @param nb_frames Указатель для возврата количества отправленных кадров
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_TxIdle
 */
can_status_t CAN_TransferGetSentPrimaryCount(CAN_Type *base,
    can_handle_t *handle, uint32_t *nb_frames);

/*!
 * @brief Отмена выдачи из высокоприоритетного буфера
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 */
void CAN_TransferAbortSendPrimary(CAN_Type *base, can_handle_t *handle);

/*!
 * @brief Неблокирующая выдача через низкоприоритетный буфер
 *
 * Функция выдает кадр CAN по прерыванию. Это неблокирующая функция, она
 * возвращает управление сразу. По выдачи кадра в шину вызывается функция
 * обратного вызова.
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 * @param xfer   Структура для передачи кадра в неблокирующем режиме
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_Fail
 * @retval #CAN_Status_TxBusy
 */
can_status_t CAN_TransferSendSecondaryNonBlocking(CAN_Type *base,
    can_handle_t *handle, can_tx_transfer_t *xfer);

/*!
 * @brief Возвращает количество кадров, отправленных в шину данных через
 * низкоприоритетный буфер
 *
 * Эта функция возвращает количество кадров, отправленных в шину данных по
 * прерыванию через низкоприоритетный буфер.
 *
 * @param base      Базовый адрес контроллера
 * @param handle    Обработчик
 * @param nb_frames Указатель для возврата количества отправленных кадров
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_TxIdle
 */
can_status_t CAN_TransferGetSentSecondaryCount(CAN_Type *base,
    can_handle_t *handle, uint32_t *nb_frames);

/*!
 * @brief Отмена выдачи из низкоприоритетного буфера
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 */
void CAN_TransferAbortSendSecondary(CAN_Type *base, can_handle_t *handle);

/*!
 * @brief Неблокирующий прием
 *
 * Функция принимает кадр CAN по прерыванию. Это неблокирующая функция, она
 * возвращает управление сразу. По приему кадра из шину вызывается функция
 * обратного вызова.
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 * @param xfer   Структура для передачи кадра в неблокирующем режиме
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_InvalidArgument
 * @retval #CAN_Status_RxBusy
 */
can_status_t CAN_TransferReceiveFifoNonBlocking(CAN_Type *base,
    can_handle_t *handle, can_rx_transfer_t *xfer);

/*!
 * @brief Возвращает количество принятых кадров по прерыванию
 *
 * @param base      Базовый адрес контроллера
 * @param handle    Обработчик
 * @param nb_frames Указатель для возврата количества отправленных кадров
 *
 * @retval #CAN_Status_Ok
 * @retval #CAN_Status_RxIdle
 */
can_status_t CAN_TransferGetReceivedCount(CAN_Type *base,
    can_handle_t *handle, uint32_t *nb_frames);

/*!
 * @brief Отмена приема
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 */
void CAN_TransferAbortReceive(CAN_Type *base, can_handle_t *handle);

/*!
 * @brief Установка обработчика на прерывания от CAN, не связанные с
 *        приемом/выдачей
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 */
void CAN_TransferHandleIRQ(CAN_Type *base, can_handle_t *handle);

/*!
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* HAL_CAN_H */

/*!
 * @}
 */
