/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup clkctr_driver Драйвер модуля CLKCTR
 *
 * @brief Драйвер модуля управления частотами
 *
 * Драйвер модуля CLKCTR позволяет настраивать частоты тактирования.
 */

/*!
 * @addtogroup clkctr_driver
 * @{
 */

/*!
 * @file hal_clkctr.h
 *
 * @brief Интерфейс драйвера модуля CLKCTR
 */

#ifndef HAL_CLKCTR_H
#define HAL_CLKCTR_H

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#include "hal_rwc.h"

/*!
 * @name Экстремальные значения коэффициентов делителей частот     \
 *       параметров PLL (делитель частоты = коэффициент деления + 1)
 * @{
 */
#define CLKCTR_MAX_SYSCLK_DIV     31 /*!< Максимальное для SYSCLK */
#define CLKCTR_MAX_FCLK_DIV       31 /*!< Максимальное для FCLK */
#define CLKCTR_MAX_GNSSCLK_DIV     7 /*!< Максимальное для GNSSCLK */
#define CLKCTR_MAX_QSPICLK_DIV    31 /*!< Максимальное для QSPICLK */
#define CLKCTR_MAX_MCOCLK_DIV     31 /*!< Максимальное для MCOCLK */
#define CLKCTR_MAX_I2SCLK_DIV     31 /*!< Максимальное для I2SCLK */
#define CLKCTR_MIN_SYSCLK_DIV      0 /*!< Минимальное для SYSCLK */
#define CLKCTR_MIN_FCLK_DIV        0 /*!< Минимальное для FCLK */
#define CLKCTR_MIN_GNSSCLK_DIV     0 /*!< Минимальное для GNSSCLK */
#define CLKCTR_MIN_QSPICLK_DIV     0 /*!< Минимальное для QSPICLK */
#define CLKCTR_MIN_MCOCLK_DIV      0 /*!< Минимальное для MCOCLK */
#define CLKCTR_MIN_I2SCLK_DIV      0 /*!< Минимальное для I2SCLK */

#define PLL_MAX_MULTIPLIER     0x176 /*!< Максимальное для PLL */
#define PLL_MIN_MULTIPLIER       0x0 /*!< Минимальное для PLL */

#define CLKCTR_NR_MAN_MAX (CLKCTR_PLLCFG_NR_MAN_Msk \
    >> CLKCTR_PLLCFG_NR_MAN_Pos)     /*!< Максимальное значение NR_MAN */
#define CLKCTR_NF_MAN_MAX (CLKCTR_PLLCFG_NF_MAN_Msk \
    >> CLKCTR_PLLCFG_NF_MAN_Pos)     /*!< Максимальное значение NF_MAN */
#define CLKCTR_OD_MAN_MAX (CLKCTR_PLLCFG_OD_MAN_Msk \
    >> CLKCTR_PLLCFG_OD_MAN_Pos)     /*!< Максимальное значение OD_MAN */
#define CLKCTR_MAN_MAX (CLKCTR_PLLCFG_MAN_Msk \
    >> CLKCTR_PLLCFG_MAN_Pos)        /*!< Максимальное значение MAN */
#define CLKCTR_SEL_MAX (CLKCTR_PLLCFG_SEL_Msk \
    >> CLKCTR_PLLCFG_SEL_Pos)        /*!< Максимальное значение SEL */
/*!
 * @}
 */

/*!
 * @brief Подаваемая на MCO частота
 */
enum clkctr_mcoclk {
    CLKCTR_MCOClkTypeHFIClk  = 0, /*!< Частота внутреннего генератора */
    CLKCTR_MCOClkTypeRTCClk  = 1, /*!< Частота таймера реального времени */
    CLKCTR_MCOClkTypeLPClk   = 2, /*!< Частота таймера реального времени */
    CLKCTR_MCOClkTypeMainClk = 3, /*!< Основная частота */
    CLKCTR_MCOClkTypePLLClk  = 4, /*!< Частота от PLL */
    CLKCTR_MCOClkTypeSysClk  = 5, /*!< Системная частота */
    CLKCTR_MCOClkTypeFClkInt = 6, /*!< Внутренняя частота процессора F_INTCLK */
    CLKCTR_MCOClkTypeFClk    = 7, /*!< Внутрення частота после динамического управления */
};

/*!
 * @brief Подаваемая на LPCLK частота
 */
enum clkctr_lpclk {
    CLKCTR_LPClkTypeRTCClk = 0, /*!< От таймера реального времени */
    CLKCTR_LPClkType500    = 1, /*!< От XTI/500 */
};

/*!
 * @brief Подаваемая на RTCCLK частота
 */
enum clkctr_rtcclk {
    CLKCTR_RTCClkTypeLFI = RWC_RTCClkTypeLFI, /*!< От внутреннего RC-осциллятора */
    CLKCTR_RTCClkTypeLFE = RWC_RTCClkTypeLFE, /*!< От внешнего RC-осциллятора */
};

/*!
 * @brief Подаваемая на MAINCLK частота
 */
enum clkctr_mainclk {
    CLKCTR_MainClkTypeHFIClk = 0,                        /*!< Внутренний генератор */
    CLKCTR_MainClkTypeXTIClk = 1,                        /*!< Внешний генератор */
    CLKCTR_MainClkTypePLLClk = 2,                        /*!< Блок PLL */
    CLKCTR_MainClkTypeMax    = CLKCTR_MainClkTypePLLClk, /*!< Максимально допустимое значение*/
};

/*!
 * @brief Опорная частота PLL
 */
enum clkctr_pllref {
    CLKCTR_PLLRefTypeHFIClk = 0, /*!< Внутренний генератор */
    CLKCTR_PLLRefTypeXTIClk = 1, /*!< Внешний генератор */
};

/*!
 * @brief Подаваемая на USBCLK частота
 */
enum clkctr_usbclk {
    CLKCTR_USBClkTypeHFIClk = 0, /*!< Внутренний генератор */
    CLKCTR_USBClkTypeXTIClk = 1, /*!< Внешний генератор */
};

/*!
 * @brief Подаваемая на I2SCLK частота
 */
enum clkctr_i2sclk {
    CLKCTR_I2SClkTypeSysClk = 0, /*!< Внутренний генератор */
    CLKCTR_I2SClkTypeI2SClk = 1, /*!< Внешний вход (PA15) */
};

/*!
 * @brief Частота, определенная как HFICLK для MAINCLK
 *        (CLKCTR_MainClkTypeHFICLK)
 *
 * @note В зависимости от режима работы процессора (задается входом PMUDIS)
 *       при переключении MAINCLK на HFICLK частота может быть переключена
 *       как на HFICLK, так и на XTICLK.
 */
enum clkctr_hfi_for_main_type {
    CLKCTR_HFIForMainTypeHFI = 0, /*!< HFI */
    CLKCTR_HFIForMainTypeXTI = 1, /*!< XTI */
};

/*!
 * @brief Генератор/осциллятор/вход, для которого групповой функцией
 *        устанавливается частота
 */
enum clkctr_extern_freq {
    CLKCTR_ExternFreqXTI       = 0, /*!< Внешний высокочастотный генератор (вход XTI) */
    CLKCTR_ExternFreqXTI32     = 1, /*!< Внешний низкочастотный осциллятор (вход XTI32) */
    CLKCTR_ExternFreqHFI       = 2, /*!< Внутренний высокочастотный осциллятор HFI */
    CLKCTR_ExternFreqLFI       = 3, /*!< Внутренний высокочастотный осциллятор LFI */
    CLKCTR_ExternFreqI2SExtClk = 4, /*!< Внешняя частота I2S_EXTCLK (PA15) */
};

/*!
 * @brief Имя внутренней частоты тактирования микросхемы, для которой
 *        производится действие групповой функцией
 */
enum clkctr_int_freq {
    CLKCTR_IntFreqXTIClk     =  0, /*!< Для внешнего высокочастотного генератора */
    CLKCTR_IntFreqHFIClk     =  1, /*!< Для внутреннего высокочастотного генератора */
    CLKCTR_IntFreqRTCClk     =  2, /*!< Для генератора реального времени */
    CLKCTR_IntFreqLPClk      =  3, /*!< Для генератора реального времени для тактирования таймеров */
    CLKCTR_IntFreqPLLClk     =  4, /*!< Для выходной частоты PLL */
    CLKCTR_IntFreqPLLRefClk  =  5, /*!< Для опорной частоты PLL */
    CLKCTR_IntFreqMainClk    =  6, /*!< Для главной частоты */
    CLKCTR_IntFreqUSBClk     =  7, /*!< Для частоты USB PHY */
    CLKCTR_IntFreqFClkInt    =  8, /*!< Для внутренней частоты FCLK_INT */
    CLKCTR_IntFreqFClk       =  9, /*!< Для частоты CPU1 и SRAM3 */
    CLKCTR_IntFreqSysClk     = 10, /*!< Для системной частоты CPU0 и SRAM0-2, AHB/APB */
    CLKCTR_IntFreqGNSSClk    = 11, /*!< Для частоты тактирования GNSS */
    CLKCTR_IntFreqQSPIClk    = 12, /*!< Для частоты тактирования QSPI */
    CLKCTR_IntFreqI2SClk     = 13, /*!< Для частоты тактирования I2S */
    CLKCTR_IntFreqMCOClk     = 14, /*!< Для частоты тактирования MCO (выход на PA15) */
    CLKCTR_IntFreqHFIForMain = 15, /*!< Для частоты тактирования MAINCLK при переключении на HFI */
};

/*!
 * @brief Тип тактирования
 */
enum clkctr_clk_force_type {
    CLKCTR_ClkForceTypeDynamic = 0, /*!< Динамический */
    CLKCTR_ClkForceTypeForce   = 1, /*!< Принудительный */
};

/*!
 * @brief Домены, к которым может быть применено динамическое тактирование
 */
enum clkctr_device_clk_force {
    CLKCTR_SysSysClkForce    =  1, /*!< Системный домен общей подсистемы */
    CLKCTR_SysFClkForce      =  2, /*!< Системный домен частоты FCLK */
    CLKCTR_SRAMSysClkForce   =  3, /*!< Домен подсистемы памяти SRAM */
    CLKCTR_SRAMFClkForce     =  4, /*!< Домен SRAM FCLK */
    CLKCTR_CPUSysClkForce    =  5, /*!< Домен подсистемы CPU */
    CLKCTR_CPUFClkForce      =  6, /*!< Домен CPU FCLK */
    CLKCTR_CryptoSysClkForce =  7, /*!< Домен криптоблока */
    CLKCTR_CPUDBGPikClkForce =  8, /*!< Домен отладки CPU */
    CLKCTR_BasePikClkForce   =  9, /*!< Домен базовый */
    CLKCTR_SMCClkForce       = 10, /*!< Домен блока тактирования SMC */
    CLKCTR_GMSSysClkForce    = 11, /*!< Домен системного блока GMS */
};

/*!
 * @brief Делители блока
 */
struct clkctr_div {
    uint32_t clkctr_fclk_div;    /*!< Делитель частоты FCLK */
    uint32_t clkctr_sysclk_div;  /*!< Делитель частоты SYSCLK */
    uint32_t clkctr_gnssclk_div; /*!< Делитель частоты GNSSCLK */
    uint32_t clkctr_qspiclk_div; /*!< Делитель частоты QSPICLK */
    uint32_t clkctr_i2sclk_div;  /*!< Делитель частоты I2SCLK */
    uint32_t clkctr_mco_div;     /*!< Делитель частоты MCO */
};

/*!
 * @brief Коэффициенты PLL
 */
struct clkctr_pll_cfg {
    uint32_t lock;   /*!< Признак готовности PLL, при записи игнорируется */
    uint32_t nr_man; /*!< Значение делителя - 1; может быть [0:15] */
    uint32_t nf_man; /*!< Значение множителя - 1; может быть [0:8191] */
    uint32_t od_man; /*!< Значение делителя - 1; может быть [0:15] */
    uint32_t man;    /*!< Тип установки множителей: 0 - используется поле sel, 1 - используются поля nr_man, nf_man, od_man */
    uint32_t sel;    /*!< Значение предустановленного множителя - 1; может быть [0:511], реально используется [0:374] */
};

/*!
 * @brief Статусы драйвера CLKCTR
 */
enum clkctr_status {
    CLKCTR_Status_Ok              = 0, /*!< Нет ошибок */
    CLKCTR_Status_InvalidArgument = 1, /*!< Недопустимый аргумент */
    CLKCTR_Status_CheckError      = 2, /*!< Получена ошибка от оборудования */
    CLKCTR_Status_VerifyError     = 3, /*!< Верификация не прошла */
    CLKCTR_Status_ConfigureError  = 4, /*!< Недопустимая конфигурация или ошибка в описании оборудования */
};

/*!
 * @name Минимально и максимально допустимые внешние частоты блока CLKCTR
 * @{
 */
#define CLKCTR_XTI_MIN               1 /*!< Минимальное значение частоты XTI */
#define CLKCTR_XTI_MAX        50000000 /*!< Максимальное значение частоты XTI */
#define CLKCTR_XTI32_MIN         30000 /*!< Минимальное значение частоты XTI32 */
#define CLKCTR_XTI32_MAX         34000 /*!< Максимальное значение частоты XTI32 */
#define CLKCTR_HFI_MIN         4000000 /*!< Минимальное значение частоты HFI */
#define CLKCTR_HFI_MAX        25000000 /*!< Максимальное значение частоты HFI */
#define CLKCTR_LFI_MIN           32112 /*!< Минимальное значение частоты LFI */
#define CLKCTR_LFI_MAX           33423 /*!< Максимальное значение частоты LFI */
#define CLKCTR_I2S_EXTCLK_MIN        1 /*!< Минимальное значение частоты I2S_EXTCLK */
#define CLKCTR_I2S_EXTCLK_MAX 50000000 /*!< Максимальное значение частоты I2S_EXTCLK */
/*!
 * @}
 */

/*!
 * @name Минимально и максимально допустимые внутренние частоты блока \
 *       CLKCTR
 * @{
 */
#define CLKCTR_PLLREF_MIN        30000 /*!< Минимальное значение опорной частоты PLL */
#define CLKCTR_PLLREF_MAX     50000000 /*!< Максимальное значение опорной частоты PLL */
#define CLKCTR_PLLCLK_MIN      1880000 /*!< Минимальное значение выходной частоты PLL полный диапазон */
#define CLKCTR_PLLCLK_MAX    375000000 /*!< Максимальное значение выходной частоты PLL полный диапазон */
#define CLKCTR_PLLCLK_OD_MIN  30000000 /*!< Минимальное значение выходной частоты PLL без учета делителя OD */
#define CLKCTR_PLLCLK_OD_MAX 375000000 /*!< Максимальное значение выходной частоты PLL без учета делителя OD */
#define CLKCTR_FCLK_MIN              1 /*!< Минимальное значение опорной частоты FCLK */
#define CLKCTR_FCLK_MAX      150000000 /*!< Максимальное значение опорной частоты FCLK */
#define CLKCTR_SYSCLK_MIN            1 /*!< Минимальное значение опорной частоты SYSCLK */
#define CLKCTR_SYSCLK_MAX     50000000 /*!< Максимальное значение опорной частоты SYSCLK */
#define CLKCTR_QSPICLK_MIN           1 /*!< Минимальное значение опорной частоты QSPICLK */
#define CLKCTR_QSPICLK_MAX    96000000 /*!< Максимальное значение опорной частоты QSPICLK */
#define CLKCTR_GNSSCLK_MIN           1 /*!< Минимальное значение опорной частоты GNSSCLK */
#define CLKCTR_GNSSCLK_MAX    80000000 /*!< Максимальное значение опорной частоты GNSSCLK */
#define CLKCTR_I2SCLK_MIN            1 /*!< Минимальное значение опорной частоты I2SCLK */
#define CLKCTR_I2SCLK_MAX     25000000 /*!< Максимальное значение опорной частоты I2SCLK */
/*!
 * @}
 */

/*!
 * @name Служебные определения
 * @{
 */
#define HFI_FREQUENCY       15100000 /*!< Частота внутреннего генератора "по умолчанию" */
#define CLKCTR_FREQ_NOT_SET        0 /*!< Частота не установлена или недоступна */
/*!
 * @}
 */

/*!
 * @name Устаревшие функции для поддержки ранних примеров
 * @{
 */

/*!
 * @brief Установка целочисленного множителя PLL
 *
 * @note Для изменения делителей перед вызовом этой функции нужно вызвать
 *       @ref CLKCTR_SetSysDiv.
 *
 * @note Для использования внутреннего HFI в качестве частоты внешнего
 *       генератора следует передать 0 через xti_hz.
 *
 * @param base    Блок частот
 * @param xti_hz  Частота внешнего генератора в Гц
 * @param pll_mul Целочисленный множитель PLL
 */
void CLKCTR_SetPll(CLKCTR_Type * base, uint32_t xti_hz, uint32_t pll_mul);

/*!
 * @brief Установка множителей PLL в ручном режиме
 *
 * @note Для изменения делителей перед вызовом этой функции нужно вызвать
 *       @ref CLKCTR_SetSysDiv.
 *
 * @note Для использования внутреннего HFI в качестве частоты внешнего
 *       генератора следует передать 0 через xti_hz.
 *
 * @note Умноженная частота перед делителем OD не должна превышать 375МГц
 *       ((xti_hz / nr) * nf) <= 375000000
 *
 * @note Предделитель выходной частоты OD может иметь значения 1 или
 *       четные значение в диапозоне 2-16.
 *
 * @param base    Блок частот
 * @param xti_hz  Частота внешнего генератора в Гц
 * @param nr      Предделитель опорной частоты
 * @param nf      Множитель частоты
 * @param od      Предделитель выходной частоты PLL
 *
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 * @retval #CLKCTR_Status_Ok
 */
enum clkctr_status CLKCTR_SetPllMan(CLKCTR_Type * base, uint32_t xti_hz,
    uint32_t nr, uint32_t nf, uint32_t od);

/*!
 * @brief Установка делителей
 *
 * @param base        Блок частот
 * @param fclk_div    Делитель частоты FCLK
 * @param sysclk_div  Делитель частоты SYSCLK
 * @param gnssclk_div Делитель частоты GNSSCLK
 * @param qspiclk_div Делитель частоты QSPICLK
 */
void CLKCTR_SetSysDiv(CLKCTR_Type * base, uint16_t fclk_div,
    uint16_t sysclk_div, uint16_t gnssclk_div, uint16_t qspiclk_div);

/*!
 * @}
 */

/*!
 * @name Функции установки и получения частот генераторов/осцилляторов
 * @{
 */

/*!
 * @brief Установка значения частоты, подаваемой на вход XTI
 *
 * @note Частота определена как XTICLK.
 *
 * @param base      Блок частот
 * @param frequency Значение частоты в Гц
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetXTI(CLKCTR_Type * base, uint32_t frequency);

/*!
 * @brief Извлечение значения частоты, подаваемой на вход XTI
 *
 * @note Частота определена как XTICLK.
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц или #CLKCTR_FREQ_NOT_SET
 */
uint32_t CLKCTR_GetXTI(CLKCTR_Type * base);

/*!
 * @brief Установка значения частоты, подаваемой на вход XTI32
 *
 * @note Частота определена как LFE.
 *
 * @param base Блок частот
 * @param frequency Значение частоты в Гц
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetXTI32(CLKCTR_Type * base, uint32_t frequency);

/*!
 * @brief Извлечение значения частоты, подаваемой на вход XTI32
 *
 * @note Частота определена как LFE.
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц или #CLKCTR_FREQ_NOT_SET
 */
uint32_t CLKCTR_GetXTI32(CLKCTR_Type * base);

/*!
 * @brief Установка значения частоты внутреннего генератора APC
 *
 * @note Частота определена как HFI.
 *
 * @note Установка частоты происходит без учета подстройки HFITRIM.
 *
 * @param base      Блок частот
 * @param frequency Значение частоты в Гц
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetHFI(CLKCTR_Type * base, uint32_t frequency);

/*!
 * @brief Извлечение значения частоты внутреннего генератора APC
 *
 * @note Частота определена как HFI.
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц или #CLKCTR_FREQ_NOT_SET
 */
uint32_t CLKCTR_GetHFI(CLKCTR_Type * base);

/*!
 * @brief Установка значения частоты внутреннего генератора RWC
 *
 * @note Частота определена как LFI.
 *
 * @param base      Блок частот
 * @param frequency Значение частоты в Гц
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetLFI(CLKCTR_Type * base, uint32_t frequency);

/*!
 * @brief Извлечение значения частоты внутреннего генератора RWC
 *
 * @note Частота определена как LFI.
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц или #CLKCTR_FREQ_NOT_SET
 */
uint32_t CLKCTR_GetLFI(CLKCTR_Type * base);

/*!
 * @brief Установка значения частоты, поступаемой на PA15
 *
 * @note Частота определена как I2S_EXTCLK.
 *
 * @note Ножка порта PA15 не переконфигурируется.
 *
 * @param base      Блок частот
 * @param frequency Значение частоты в Гц
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetI2SExtClk(CLKCTR_Type * base, uint32_t frequency);

/*!
 * @brief Извлечение значения частоты, поступаемой на PA15
 *
 * @note Частота определена как I2S_EXTCLK.
 *
 * @param base    Блок частот
 *
 * @return Значение частоты в Гц или #CLKCTR_FREQ_NOT_SET
 */
uint32_t CLKCTR_GetI2SExtClk(CLKCTR_Type * base);

/*!
 * @}
 */

/*!
 * @name Функции установки и получения делителей, кроме PLL
 * @{
 */

/*!
 * @brief Извлечение всех делителей блока CLKCTR
 *
 * @param base     Блок частот
 * @param divisors Структура с делителями блока
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 */
enum clkctr_status CLKCTR_GetAllDiv(CLKCTR_Type * base,
    struct clkctr_div *divisors);

/*!
 * @brief Извлечение значения делителя MCODiv
 *
 * @return Значение делителя
 */
uint32_t CLKCTR_GetMCODiv(CLKCTR_Type * base);

/*!
 * @brief Установка значения делителя MCODiv
 *
 * @note Допустимые значения делителя - от 1 до 32.
 *
 * @param base  Блок частот
 * @param value Значение делителя
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetMCODiv(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Извлечение значения делителя FCLKDiv
 *
 * @param base    Блок частот
 *
 * @return Значение делителя
 */
uint32_t CLKCTR_GetFClkDiv(CLKCTR_Type * base);

/*!
 * @brief Установка значения делителя FCLKDiv
 *
 * @note Допустимые значения делителя - от 1 до 32.
 *
 * @param base  Блок частот
 * @param value Значение делителя
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetFClkDiv(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Извлечение значения делителя SYSCLKDiv
 *
 * @param base    Блок частот
 *
 * @return Значение делителя
 */
uint32_t CLKCTR_GetSysClkDiv(CLKCTR_Type * base);

/*!
 * @brief Установка значения делителя SYSCLKDiv
 *
 * @note Допустимые значения делителя - от 1 до 32.
 *
 * @param base  Блок частот
 * @param value Значение делителя
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSysClkDiv(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Извлечение значения делителя GNSSCLKDiv
 *
 * @param base    Блок частот
 *
 * @return Значение делителя
 */
uint32_t CLKCTR_GetGNSSClkDiv(CLKCTR_Type * base);

/*!
 * @brief Установка значения делителя GNSSCLKDiv
 *
 * @note Допустимые значения делителя - от 1 до 8.
 *
 * @param base  Блок частот
 * @param value Значение делителя
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetGNSSClkDiv(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Извлечение значения делителя QSPICLKDiv
 *
 * @param base Блок частот
 *
 * @return Значение делителя
 */
uint32_t CLKCTR_GetQSPIClkDiv(CLKCTR_Type * base);

/*!
 * @brief Установка значения делителя QSPICLKDiv
 *
 * @note Допустимые значения делителя - от 1 до 32.
 *
 * @param base  Блок частот
 * @param value Значение делителя
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetQSPIClkDiv(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Извлечение значения делителя I2SCLKDiv
 *
 * @param base    Блок частот
 *
 * @return Значение делителя
 */
uint32_t CLKCTR_GetI2SClkDiv(CLKCTR_Type * base);

/*!
 * @brief Установка значения делителя I2SCLKDiv
 *
 * @note Допустимые значения делителя - от 1 до 32.
 *
 * @param base  Блок частот
 * @param value Значение делителя
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetI2SClkDiv(CLKCTR_Type * base, uint32_t value);

/*!
 * @}
 */

/*!
 * @name Функции установки и получения коэффициентов PLL
 * @{
 */

/*!
 * @brief Извлечение коэффициентов PLL
 *
 * @param base   Блок частот
 * @param config Структура коэффициентов PLL
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 */
enum clkctr_status CLKCTR_GetPLLConfig(CLKCTR_Type * base,
    struct clkctr_pll_cfg *config);

/*!
 * @brief Установка коэффициентов PLL
 *
 * Функция устанавливает коэффициенты PLL и ждет завершения конфигурации.
 *
 * @note В момент использования функции не должно производиться
 *       тактирование от PLL.
 *
 * @param base   Блок частот
 * @param config Структура коэффициентов PLL
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetPLLConfig(CLKCTR_Type * base,
    struct clkctr_pll_cfg config);

/*!
 * @}
 */

/*!
 * @name Функции извлечения частот тактирования
 * @{
 */

/*!
 * @brief Извлечение значения частоты XTICLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetXTIClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты HFICLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetHFIClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты RTCCLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetRTCClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты LPCLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetLPClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты PLLCLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц или #CLKCTR_FREQ_NOT_SET, если PLL не
 *         вышла в рабочий режим
 */
uint32_t CLKCTR_GetPLLClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения опорной частоты PLL
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetPLLRef(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты MAINCLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetMainClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты USBCLK, подаваемого на USB PHY
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetUSBClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты FCLK_INT
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetFClkInt(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты FCLK
 *
 * @note Динамическое тактирование не учитывается.
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetFClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты SYSCLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetSysClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты GNSSCLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetGNSSClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты QSPICLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetQSPIClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты I2SCLK
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetI2SClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты MCOCLK, подаваемой на вывод PA15
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetMCOClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение значения частоты HFICLK для MAINCLK
 *
 * @note HFICLK для MAINCLK зависит от значения сигнала на внешней ножке
 *       PMUDIS.
 *
 * @param base Блок частот
 *
 * @return Значение частоты в Гц
 */
uint32_t CLKCTR_GetHFIClkForMainClk(CLKCTR_Type * base);

/*!
 * @}
 */

/*!
 * @name Функции выбора и извлечения источников частот
 * @{
 */

/*!
 * @brief Выбор источника частоты MCO
 *
 * @note Индексы источника частоты перечислены в @ref clkctr_mcoclk.
 *
 * @param base  Блок частот
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchMCOClk(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Выбор источника опорной частоты PLL
 *
 * @note В момент использования функции не должно производиться
 *       тактирование от PLL.
 *
 * @note Индексы источника частоты перечислены в @ref clkctr_pllref.
 *
 * @param base  Блок частот
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchPLLRef(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Выбор источника частоты MAINCLK
 *
 * @note Индексы источника частоты перечислены в @ref clkctr_mainclk.
 *
 * @param base  Блок частот
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchMainClk(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Выбор источника частоты RTCCLK
 *
 * @note Индексы источника частоты перечислены в @ref clkctr_rtcclk.
 *
 * @param base  Блок частот
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchRTCClk(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Выбор источника частоты LPCLK
 *
 * Функция переключает источник частоты LPCLK; переключение связано с
 * изменением частоты MAINCLK при выбранном источнике
 * @ref CLKCTR_MainClkTypeHFIClk.
 *
 * @note Индексы источника частоты перечислены в @ref clkctr_lpclk.
 *
 * @param base  Блок частот
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchLPClk(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Выбор источника частоты USBCLK
 *
 * @note Индексы источника частоты перечислены в @ref clkctr_usbclk.
 *
 * @param base  Блок частот
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchUSBClk(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Выбор источника частоты I2SCLK
 *
 * @note Индексы источника частоты перечислены в @ref clkctr_i2sclk.
 *
 * @param base  Блок частот
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchI2SClk(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Выбор режима работы процессора
 *
 * Значение, подаваемое на вход PMUDIS, определяет, какие выбраны источники
 * частоты LPCLK и MAINCLK при выбранном источнике
 * @ref CLKCTR_MainClkTypeHFIClk.
 *
 * @note Физически выбор определяется уровнем напряжения на входе
 *       микросхемы.
 *
 * @note Допустимые значения, подаваемые на вход PMUDIS - 0 и 1;
 *       0 соответствует логическому нулю на физическом входе PMUDIS, а !0
 *       - единице.
 *
 * @param base  Блок частот
 * @param value Значение, подаваемое на вход PMUDIS
 *
 * @retval #CLKCTR_Status_Ok
 */
enum clkctr_status CLKCTR_SetSwitchPMUDIS(CLKCTR_Type * base, uint32_t value);

/*!
 * @brief Извлечение источника частоты MCO
 *
 * @param base Блок частот
 *
 * @return Источник частоты
 */
uint32_t CLKCTR_GetSwitchMCOClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение источника частоты PLL
 *
 * @param base Блок частот
 *
 * @return Источник частоты
 */
uint32_t CLKCTR_GetSwitchPLLRef(CLKCTR_Type * base);

/*!
 * @brief Извлечение источника частоты MAINCLK
 *
 * @param base Блок частот
 *
 * @return Источник частоты
 */
uint32_t CLKCTR_GetSwitchMainClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение источника частоты RTCCLK
 *
 * @param base Блок частот
 *
 * @return Источник частоты
 */
uint32_t CLKCTR_GetSwitchRTCClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение источника частоты LPCLK
 *
 * @param base Блок частот
 *
 * @return Источник частоты
 */
uint32_t CLKCTR_GetSwitchLPClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение источника частоты USBCLK
 *
 * @param base Блок частот
 *
 * @return Источник частоты
 */
uint32_t CLKCTR_GetSwitchUSBClk(CLKCTR_Type * base);

/*!
 * @brief Извлечение источника частоты I2SCLK
 *
 * @param base Блок частот
 *
 * @return Источник частоты
 */
uint32_t CLKCTR_GetSwitchI2SClk(CLKCTR_Type * base);

/*!
 * @brief Определение режима работы процессора
 *
 * Функция возвращает значение, поданное на вход PMUDIS, которое определяет
 * выбранные источники частоты LPCLK и MAINCLK при выбранном источнике
 * @ref CLKCTR_MainClkTypeHFIClk.
 *
 * @param base Блок частот
 *
 * @retval 0 Логический 0 на входе PMUDIS
 * @retval 1 Логическая 1 на входе PMUDIS
 */
uint32_t CLKCTR_GetSwitchPMUDIS(CLKCTR_Type * base);

/*!
 * @}
 */

/*!
 * @name Прочие функции
 * @{
 */

/*!
 * @brief Установка индекса коэффициента подстройки частоты осциллятора HFI
 *
 * @note Допустимые значения индекса коэффициента подстройки частоты -
 *       от 0 до 0x1f.
 *
 * @param base  Блок частот
 * @param index Индекс коэффициента подстройки частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetHFITrim(CLKCTR_Type * base, uint32_t index);

/*!
 * @brief Извлечение индекса коэффициента подстройки частоты осциллятора
 *        HFI
 *
 * @param base Блок частот
 *
 * @return Индекс коэффициента подстройки частоты
 */
uint32_t CLKCTR_GetHFITrim(CLKCTR_Type * base);

/*!
 * @brief Извлечение разрешения тактового сигнала MCO
 *
 * @param base  Блок частот
 *
 * @return 0 Вывод запрещен
 * @return 1 Вывод разрешен
 */
uint32_t CLKCTR_GetMCOEn(CLKCTR_Type * base);

/*!
 * @brief Установка или запрет разрешения тактового сигнала MCO
 *
 * @param base   Блок частот
 * @param enable 0 - запрещает вывод частоты, !0 - разрешает
 *
 * @return #CLKCTR_Status_Ok
 */
enum clkctr_status CLKCTR_SetMCOEn(CLKCTR_Type * base, uint32_t enable);

/*!
 * @brief Извлечение типа тактирования для модуля
 *
 * @param base     Блок частот
 * @param device   Модуль
 * @param clk_type Тип тактирования
 *
 * @return #CLKCTR_Status_Ok
 * @return #CLKCTR_Status_InvalidArgument
 */
enum clkctr_status CLKCTR_GetClkForce(CLKCTR_Type * base,
    enum clkctr_device_clk_force device,
    enum clkctr_clk_force_type *clk_type);

/*!
 * @brief Установка типа тактирования для модуля
 *
 * @param base     Блок частот
 * @param device   Модуль
 * @param clk_type Тип тактирования
 *
 * @return #CLKCTR_Status_Ok
 * @return #CLKCTR_Status_InvalidArgument
 */
enum clkctr_status CLKCTR_SetClkForce(CLKCTR_Type * base,
    enum clkctr_device_clk_force device,
    enum clkctr_clk_force_type clk_type);

/*!
 * @}
 */

/*!
 * @name Функции-обертки
 * @{
 */

/*!
 * @brief Установка значения частоты, подаваемой на выбранный вход
 *
 * @param base      Блок частот
 * @param input     Вход, для которого устанавливается частота
 * @param frequency Значение частоты в Гц
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetFrequency(CLKCTR_Type * base,
    enum clkctr_extern_freq input, uint32_t frequency);

/*!
 * @brief Извлечение значения частоты, подаваемой на выбранный вход
 *
 * @param base      Блок частот
 * @param input     Вход, для которого определяется частота
 *
 *
 * @return Значение частоты в Гц или #CLKCTR_FREQ_NOT_SET
 */
uint32_t CLKCTR_GetFrequency(CLKCTR_Type * base,
    enum clkctr_extern_freq input);

/*!
 * @brief Извлечение значения указанной частоты
 *
 * @param base  Блок частот
 * @param clk   Частота
 * @param value Значение частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 */
enum clkctr_status CLKCTR_GetClk(CLKCTR_Type * base, enum clkctr_int_freq clk,
    uint32_t *value);

/*!
 * @brief Выбор источника частоты для указанной частоты
 *
 * @param base  Блок частот
 * @param clk   Конфигурируемая частота
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetSwitchClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t value);

/*!
 * @brief Извлечение источника частоты для указанной частоты
 *
 * @param base  Блок частот
 * @param clk   Конфигурируемая частота
 * @param value Индекс источника частоты
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 */
enum clkctr_status CLKCTR_GetSwitchClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t *value);

/*!
 * @brief Установка коэффициента деления для указанной частоты
 *
 * @param base  Блок частот
 * @param clk   Конфигурируемая частота
 * @param value Коэффициент деления
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 * @retval #CLKCTR_Status_ConfigureError
 */
enum clkctr_status CLKCTR_SetDivClk(CLKCTR_Type * base,
   enum clkctr_int_freq clk, uint32_t value);

/*!
 * @brief Извлечение коэффициента деления для указанной частоты
 *
 * @param base  Блок частот
 * @param clk   Конфигурируемая частота
 * @param value Коэффициент деления
 *
 * @retval #CLKCTR_Status_Ok
 * @retval #CLKCTR_Status_InvalidArgument
 */
enum clkctr_status CLKCTR_GetDivClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t *value);

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_CLKCTR_H */

/*!
 * @}
 */
