/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup flash_driver
 * @{
 */

/*!
 * @file hal_flash.c
 *
 * @brief Имплементация драйвера модуля FLASH.
 */

#include "hal_clkctr.h"
#include "hal_flash.h"

#define FLASH_TIMING_MAX_FIELD  (0xf)     /*!< Маска для поля таймингов флеш-памяти. */
#define FLASH_TIMING_FIRST_FREQ (4250000) /*!< Первая частота для расчета таймингов флеш-памяти. */
#define FLASH_TIMING_DELTA_FREQ (3250000) /*!< Шаг частоты для расчета таймингов флеш-памяти. */

#define FLASH_20NS              (20 * 10e-9)
#define FLASH_1US               (10e-6)

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wtype-limits"

/*!
 * @name Блоб-массивы скомпилированных функций драйвера Flash.
 * @{
 */

/*!
 * @brief Блоб-массив функции @ref FLASH_WriteWord.
 *
 */
static uint8_t __attribute__((used)) flash_write_word_blob[] = {
    0xc1, 0xf3, 0x15, 0x01, 0x02, 0x23, 0xc1, 0x61,
    0x02, 0x62, 0x43, 0x61, 0x83, 0x68, 0x13, 0xf0,
    0x1e, 0x0f, 0xfb, 0xd0, 0x9b, 0x07, 0x43, 0xbf,
    0x03, 0x23, 0xc3, 0x60, 0x00, 0x20, 0x02, 0x20,
    0x70, 0x47,
};

/*!
 * @brief Блоб-массив функции @ref FLASH_Erase.
 *
 */
static uint8_t __attribute__((used)) flash_erase_blob[] = {
    0x30, 0xb5, 0xa2, 0xb1, 0x22, 0xf0, 0x03, 0x02,
    0x04, 0x24, 0x03, 0x25, 0x0a, 0x44, 0x8a, 0x42,
    0x01, 0xd8, 0x00, 0x20, 0x30, 0xbd, 0xc1, 0x61,
    0x44, 0x61, 0x83, 0x68, 0x13, 0xf0, 0x1e, 0x0f,
    0xfb, 0xd0, 0x9b, 0x07, 0x05, 0xd5, 0x01, 0xf5,
    0x00, 0x51, 0xc5, 0x60, 0xef, 0xe7, 0x01, 0x20,
    0xf0, 0xe7, 0x02, 0x20, 0xee, 0xe7,
};

/*!
 * @}
 */

#ifdef ELIOT1M
static uint32_t FLASH_CalculateTimings(float time)
{
    uint32_t result;
    uint32_t sys_freq = CLKCTR_GetSysClk(CLOCK_BASE);
    float tmp = time * (float) sys_freq;

    if (tmp > (float)((uint32_t) tmp)) {
        result = (uint32_t)(tmp + 1);
    } else {
        result = (uint32_t) tmp;
    }

    return result;
}
#endif


enum flash_status FLASH_Init(FCTR_Type *base)
{
    uint32_t freq = CLKCTR_GetSysClk(CLOCK_BASE); /* Частота работы FLASH-памяти. */

#ifndef ELIOT1M
    uint32_t tick_index; /* Индекс для таймингов работы FLASH-памяти. */

    /* В документации задано таблично, но зависимость линейная. */
    if (freq <= FLASH_TIMING_FIRST_FREQ) {
        tick_index = 0;
    } else {
        tick_index = (freq - FLASH_TIMING_FIRST_FREQ - 1)
            / FLASH_TIMING_DELTA_FREQ + 1;
    }

    if (tick_index > FLASH_TIMING_MAX_FIELD) {
        return FLASH_Status_ConfigureError;
    }

    base->PSP_TM1 = (tick_index << FCTR_PSP_TM1_TICK_TPROG_Pos)
        | (tick_index << FCTR_PSP_TM1_TICK_TPGS_Pos)
        | (tick_index << FCTR_PSP_TM1_TICK_TNVS_Pos)
        | (tick_index << FCTR_PSP_TM1_TICK_TNVH_Pos)
        | (tick_index << FCTR_PSP_TM1_TICK_TRCV_Pos);
    base->PSP_TM2 = (tick_index << FCTR_PSP_TM2_TICK_TERASE_Pos)
        | (tick_index << FCTR_PSP_TM2_TICK_RLATENCY_LVE_Pos)
        | (tick_index << FCTR_PSP_TM2_TICK_THV_Pos)
        | (tick_index << FCTR_PSP_TM2_TICK_TNVH1_Pos);
#else
    uint32_t cycles_20ns = FLASH_CalculateTimings(FLASH_20NS);
    uint32_t cycles_1us  = FLASH_CalculateTimings(FLASH_1US);

    SET_VAL_MSK(base->PSP_CTRL, FCTR_PSP_CTRL_CYCLES_20NS_Msk,
        FCTR_PSP_CTRL_CYCLES_20NS_Pos, cycles_20ns - 1);
    SET_VAL_MSK(base->PSP_CTRL, FCTR_PSP_CTRL_CYCLES_1US_Msk,
        FCTR_PSP_CTRL_CYCLES_1US_Pos, cycles_1us - 1);

    SET_VAL_MSK(base->IRQ_ENABLE_SET, FCTR_IRQ_ENABLE_SET_CMD_SUCCESS_IRQ_EN_SET_Msk,
        FCTR_IRQ_ENABLE_SET_CMD_SUCCESS_IRQ_EN_SET_Pos, 1UL);
    SET_VAL_MSK(base->IRQ_ENABLE_SET, FCTR_IRQ_ENABLE_SET_CMD_FAIL_IRQ_EN_SET_Msk,
        FCTR_IRQ_ENABLE_SET_CMD_FAIL_IRQ_EN_SET_Pos, 1UL);

    uint32_t vsel = GET_VAL_MSK(PWRCTR->RUNCFG, PWRCTR_RUNCFG_DCDC_VSEL_Msk,
            PWRCTR_RUNCFG_DCDC_VSEL_Pos);

    switch (vsel) {
        case 0:
            if (freq <= FLASH_LVE_MAX_FREQ) {
                SET_VAL_MSK(base->PSP_CTRL, FCTR_PSP_CTRL_READ_LATENCY_Msk,
                    FCTR_PSP_CTRL_READ_LATENCY_Pos, 0);
            } else {
                return FLASH_Status_ConfigureError;
            }
            break;
        case 1:
        case 2:
            if (freq <= FLASH_NO_READ_LATENCY_MAX_FREQ) {
                SET_VAL_MSK(base->PSP_CTRL, FCTR_PSP_CTRL_READ_LATENCY_Msk,
                    FCTR_PSP_CTRL_READ_LATENCY_Pos, 0);
            } else if (freq <= CLKCTR_SYSCLK_MAX) {
                SET_VAL_MSK(base->PSP_CTRL, FCTR_PSP_CTRL_READ_LATENCY_Msk,
                    FCTR_PSP_CTRL_READ_LATENCY_Pos, 1);
            } else {
                return FLASH_Status_ConfigureError;
            }
            break;

        default:
            return FLASH_Status_ConfigureError;
    }
#endif

    return FLASH_Status_Ok;
}

enum flash_status FLASH_WriteWord(FCTR_Type *base, uint32_t addr, uint32_t data)
{
    enum flash_status ret_value; /* Возвращаемое значение функции драйвера. */

    uint32_t priority_mask; /* Маска активных прерываний. */
    uint32_t IRQ_STAT_IC_value; /* Значение поля IC регистра IRQ_STAT. */

    bool IRQs_enable = false; /* Статус обработки прерываний. */

    uint32_t flash_masked_addr = addr & FLASH_ADDR_Msk;

    /* Проверка валидности адреса-смещения и размера записываемых данных. */
    if ((ret_value = FLASH_TEST_ADDRESSES(flash_masked_addr, 0x4U))
        != FLASH_Status_Ok) {
        return ret_value;
    }

    /* Сохранение маски активных прерываний и проверка их наличия. */
    if ((priority_mask = __get_PRIMASK()) == 0) {
        IRQs_enable = true;
    }

    __disable_irq();

    ret_value = ((enum flash_status(*)(FCTR_Type *, uint32_t, uint32_t))
            ((uint32_t) flash_write_word_blob | 0x1))(base, addr, data);

    /* Инвалидация кэша инструкций. */
    ICACHE_CTRL_FINV_SET(0x1U);

    /* Ожидание завершения процесса инвалидации. */
    do {
        IRQ_STAT_IC_value = ICACHE_IRQSTAT_IC_GET;
    } while (IRQ_STAT_IC_value == 0x0U);

    /* Активация прерываний, если они были выключены. */
    if (IRQs_enable) {
        __set_PRIMASK((int) priority_mask);
        __enable_irq();
    }

    return ret_value;
}

enum flash_status FLASH_Program(FCTR_Type *base, uint32_t *addr, uint32_t *src,
    uint32_t length)
{
    enum flash_status result; /* Результат выполнения функций драйвера. */
    uint32_t *pointer_src; /* Записываемые данные. */
    uint32_t iter_address; /* Адрес текущего записываемого слова. */

    uint32_t flash_masked_addr = ((uint32_t)addr) & FLASH_ADDR_Msk;

    if ((result = FLASH_TEST_ADDRESSES(flash_masked_addr, length))
        != FLASH_Status_Ok) {
        return result;
    }

    if ((length == 0) || (src == NULL)) {
        return FLASH_Status_InvalidArgument;
    }

    if (((uint32_t)src) & FLASH_WORD_ALIGN_Msk) {
        return FLASH_Status_AddressAlignmentError;
    }

    for (
        iter_address = (uint32_t)addr,
        pointer_src = src;
        iter_address < (uint32_t)addr + length;
        iter_address+=FLASH_WORD_SIZE_IN_BYTE, pointer_src++) {
        if ((result
                = FLASH_WriteWord(base, iter_address, *pointer_src))
            != FLASH_Status_Ok) {
            return result;
        }
    }

    return FLASH_Status_Ok;
}

enum flash_status FLASH_VerifyProgram(uint32_t *addr, uint32_t length,
    uint32_t *expected_data, uint32_t *failed_address, uint32_t *failed_data)
{
    uint32_t *pointer_flash; /* Записанные в FLASH-память данные. */
    uint32_t *pointer_expected; /* Эталонные данные. */
    enum flash_status result; /* Результат выполнения функций драйвера. */

    uint32_t flash_masked_addr = ((uint32_t)addr) & FLASH_ADDR_Msk;

    if ((result = FLASH_TEST_ADDRESSES(flash_masked_addr, length))
        != FLASH_Status_Ok) {
        return result;
    }

    if (length == 0) {
        return FLASH_Status_InvalidArgument;
    }

    if (((uint32_t)expected_data) & FLASH_WORD_ALIGN_Msk) {
        return FLASH_Status_AddressAlignmentError;
    }

    for(
        pointer_flash = (uint32_t *)addr,
        pointer_expected = (uint32_t *)expected_data;
        pointer_flash < (uint32_t *)(addr + (length >> 2));
        pointer_flash++, pointer_expected++
    ) {
        if (*pointer_flash != *pointer_expected) {
            *failed_data = *pointer_flash;
            *failed_address = (uint32_t)pointer_flash;
            return FLASH_Status_VerifyError;
        }
    }

    return FLASH_Status_Ok;
}

enum flash_status FLASH_Erase(FCTR_Type *base, uint32_t *addr, uint32_t length)
{
    enum flash_status ret_value; /* Возвращаемое значение функции драйвера. */

    uint32_t priority_mask; /* Маска активных прерываний. */
    uint32_t IRQ_STAT_IC_value; /* Значение поля IC регистра IRQ_STAT. */

    bool IRQs_enable = false; /* Статус обработки прерываний. */

    uint32_t flash_masked_addr = ((uint32_t)addr) & FLASH_ADDR_Msk;

    /* Проверка валидности адреса-смещения и размера записываемых данных. */
    if ((ret_value = FLASH_TEST_ADDRESSES(flash_masked_addr, 0x4U))
        != FLASH_Status_Ok) {
        return ret_value;
    }

    /* Сохранение маски активных прерываний и проверка их наличия. */
    if ((priority_mask = __get_PRIMASK()) == 0) {
        IRQs_enable = true;
    }

    __disable_irq();

    ret_value = ((enum flash_status(*)(FCTR_Type *, uint32_t, uint32_t))
            ((uint32_t) flash_erase_blob | 0x1))(base, (uint32_t)addr, length);

    /* Инвалидация кэша инструкций. */
    ICACHE_CTRL_FINV_SET(0x1U);

    /* Ожидание завершения процесса инвалидации. */
    do {
        IRQ_STAT_IC_value = ICACHE_IRQSTAT_IC_GET;
    } while (IRQ_STAT_IC_value == 0x0U);

    /* Активация прерываний, если они были выключены. */
    if (IRQs_enable) {
        __set_PRIMASK((int) priority_mask);
        __enable_irq();
    }

    return ret_value;
}

enum flash_status FLASH_MassErase(FCTR_Type *base, enum flash_region region)
{
    uint32_t IRQ_status; /* Значение регистра статуса прерываний. */
    uint32_t IRQ_STAT_IC_value; /* Значение поля IC регистра IRQ_STAT. */

    base->ADDR = region & FLASH_ADDR_Msk;
    base->CTRL = FCTR_CMD_MASS_ERASE;

    do {
        IRQ_status = base->IRQ_STATUS_SET;
    } while (!(IRQ_status & FCTR_IRQ_STS_SET_RESULT_FLAGS));

    if (!(IRQ_status & FCTR_IRQ_STATUS_SET_CMD_SUCCESS_IRQ_STS_SET_Msk)) {
        return FLASH_Status_CheckError;
    }

    base->IRQ_STATUS_CLR = FCTR_IRQ_STS_CLR_SUCCESS_FLAGS;

    /* Инвалидация кэша инструкций. */
    ICACHE_CTRL_FINV_SET(0x1U);

    /* Ожидание завершения процесса инвалидации. */
    do {
        IRQ_STAT_IC_value = ICACHE_IRQSTAT_IC_GET;
    } while (IRQ_STAT_IC_value == 0x0U);

    return FLASH_Status_Ok;
}

enum flash_status FLASH_VerifyErase(uint32_t *addr, uint32_t length)
{
    uint32_t *pointer_flash; /* Текущее проверяемое слово. */
    enum flash_status result; /* Рeзультат выполнения функции. */

    uint32_t flash_masked_addr = ((uint32_t)addr) & FLASH_ADDR_Msk;

    if ((result = FLASH_TEST_ADDRESSES(flash_masked_addr, length))
        != FLASH_Status_Ok) {
        return result;
    }

    if (length == 0) {
        return FLASH_Status_InvalidArgument;
    }

    for (pointer_flash = addr;
        pointer_flash < (uint32_t *)(addr + (length >> 2));
        pointer_flash++) {
        if (*pointer_flash != FLASH_ERASE_DATA) {
            return FLASH_Status_VerifyError;
        }
    }

    return FLASH_Status_Ok;
}

enum flash_status FLASH_Read(uint32_t *addr, uint32_t *dest, uint32_t length)
{
    uint32_t *pointer_rd; /* Читаемые данные. */
    uint32_t *pointer_wr; /* Записываемые данные. */
    enum flash_status result; /* Рeзультат выполнения функции. */

    uint32_t flash_masked_addr = ((uint32_t)addr) & FLASH_ADDR_Msk;

    if ((result = FLASH_TEST_ADDRESSES(flash_masked_addr, length))
        != FLASH_Status_Ok) {
        return result;
    }

    if (length == 0) {
        return FLASH_Status_InvalidArgument;
    }

    if (((uint32_t)dest) & FLASH_WORD_ALIGN_Msk) {
        return FLASH_Status_AddressAlignmentError;
    }

    pointer_wr = dest;

    for (pointer_rd = (uint32_t *) addr;
        pointer_rd < (uint32_t *)(addr + (length >> 2));
        pointer_rd++) {
        *pointer_wr++ = *pointer_rd;
    }

    return FLASH_Status_Ok;
}

#pragma GCC diagnostic pop

/*!
 * @}
 */
