/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup i2s_driver
 * @{
 */

/*!
 * @file hal_i2s.c
 *
 * @brief Имплементация драйвера модуля I2S
 */

#include "hal_i2s.h"
#include "hal_clkctr.h"
#include "hal_ioim.h"

#define NB_STREAMS_IN_CHANNEL (2U) /*!< Количество потоков в канале (всегда два: левый и правый) */

/*!
 * @brief Возврат количества синхроимпульсов на одно значение, заданное
 *        перечислимым типом
 *
 * @param cycles_enum Количество синхроимпульсов
 *
 * @return Количество синхроимпульсов
 */
static uint32_t I2S_GetCyclesFromEnum(i2s_sclk_per_sample_t cycles_enum)
{
    switch (cycles_enum) {
        case I2S_SclkCycles_16:
            return 16;
        case I2S_SclkCycles_24:
            return 24;
        case I2S_SclkCycles_32:
            return 32;
        default:
            return 32;
    }
}

i2s_status_t I2S_Init(I2S_Type *base, const i2s_config_t *config,
    uint32_t source_clock_hz)
{
    assert(base != NULL);
    assert(config != NULL);
    assert(config->sample_rate != 0U);
    assert((config->sclk_per_sample >= I2S_SclkCycles_16)
        && (config->sclk_per_sample <= I2S_SclkCycles_32));
    assert((config->resolution >= I2S_ResolutionDefault)
        && (config->resolution <= I2S_Resolution_32));
    assert((config->sclk_gating >= I2S_NoSclkGating)
        && (config->sclk_gating <= I2S_SclkGatingCycles_24));
    assert((config->interrupt_level >= I2S_InterruptTriggerLevel_1)
        && (config->interrupt_level <= I2S_InterruptTriggerLevel_16));

    if (CLKCTR_SetI2SClkDiv(CLOCK_BASE, source_clock_hz / config->sample_rate /
            I2S_GetCyclesFromEnum(config->sclk_per_sample) /
            NB_STREAMS_IN_CHANNEL) != CLKCTR_Status_Ok)
        return I2S_Status_UnsupportedBitRate;

    /* Разрешение работы контроллера I2S. */
    SET_VAL_MSK(base->IER, I2S_IER_IEN_Msk, I2S_IER_IEN_Pos, 1U);

    SET_VAL_MSK(base->CCR, I2S_CCR_WSS_Msk, I2S_CCR_WSS_Pos,
        config->sclk_per_sample);
    SET_VAL_MSK(base->CCR, I2S_CCR_SCLKG_Msk, I2S_CCR_SCLKG_Pos,
        config->sclk_gating);
    SET_VAL_MSK(base->TCR0, I2S_TCR0_WLEN_Msk, I2S_TCR0_WLEN_Pos,
        config->resolution);
    SET_VAL_MSK(base->TCR0, I2S_TFCR0_TXCHET_Msk, I2S_TFCR0_TXCHET_Pos,
        config->interrupt_level);

    return I2S_Status_Ok;
}

void I2S_Deinit(I2S_Type *base)
{
    assert(base != NULL);

    I2S_DisableTx(base);

    /* Запрет работы контроллера I2S. */
    SET_VAL_MSK(base->IER, I2S_IER_IEN_Msk, I2S_IER_IEN_Pos, 0U);
}

void I2S_GetDefaultConfig(i2s_config_t *config)
{
    assert(config != NULL);

    (void) memset(config, 0, sizeof(i2s_config_t));

    config->sample_rate = 256000;
}

static void I2S_WriteFifo(__OM uint32_t *fifo, uint32_t sample,
    i2s_resolution_t resolution)
{
    switch (resolution) {
        case I2S_ResolutionDefault:
        case I2S_Resolution_32:
            *fifo = sample;
            break;
        case I2S_Resolution_12:
            *fifo = sample << 20;
            break;
        case I2S_Resolution_16:
            *fifo = sample << 16;
            break;
        case I2S_Resolution_20:
            *fifo = sample << 12;
            break;
        case I2S_Resolution_24:
            *fifo = sample << 8;
            break;
        default:
            break;
    }
}

void I2S_WriteLeftFifo(I2S_Type *base, uint32_t sample)
{
    I2S_WriteFifo(&base->LTHR0, sample,
        (i2s_resolution_t) GET_VAL_MSK(base->TCR0, I2S_TCR0_WLEN_Msk,
            I2S_TCR0_WLEN_Pos));
}

void I2S_WriteRightFifo(I2S_Type *base, uint32_t sample)
{
    I2S_WriteFifo(&base->RTHR0, sample,
        (i2s_resolution_t) GET_VAL_MSK(base->TCR0, I2S_TCR0_WLEN_Msk,
            I2S_TCR0_WLEN_Pos));
}

i2s_status_t I2S_TransferCreateHandle(I2S_Type *base, i2s_handle_t *handle,
    i2s_transfer_callback_t callback, void *user_data)
{
    assert(base != NULL);
    assert(handle != NULL);

    (void) memset(handle, 0, sizeof(i2s_handle_t));

    /* Установка обратного вызова и пользовательских данных. */
    handle->callback  = callback;
    handle->user_data = user_data;

    /* Установка обработчика прерывания. */
    if (IOIM_SetIRQHandler(base, I2S_TransferHandleIRQ, handle)
        != IOIM_Status_Ok)
        return I2S_Status_Fail;

    return I2S_Status_Ok;
}

i2s_status_t I2S_TransferNonBlocking(I2S_Type *base,
    i2s_handle_t *handle, i2s_transfer_t *xfer)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(xfer != NULL);
    assert(xfer->left_samples != NULL);
    assert(xfer->right_samples != NULL);

    /* Проверка структуры передачи I2S. */
    if (xfer->nb_samples == 0U) {
        return I2S_Status_InvalidArgument;
    }

    /* Возврат ошибки, если уже в процессе передачи. */
    if (handle->nb_samples != 0U) {
        return I2S_Status_TxBusy;
    } else {
        /*
         * Отключение IRQ при настройке дескриптора передачи, если произойдет
         * прерывание во время процесса настройки, то значение дескриптора
         * может быть испорчено.
         */
        uint32_t interrupt_mask = I2S_GetEnabledInterruptMask(base);
        I2S_DisableInterruptMask(base, interrupt_mask);

        handle->left_samples  = xfer->left_samples;
        handle->right_samples = xfer->right_samples;
        handle->nb_samples    = xfer->nb_samples;

        /* Разрешение прерывания по выдаче из высокоприоритетного буфера. */
        interrupt_mask |= (1 << I2S_FlagTxFifoEmpty);

        I2S_EnableInterruptMask(base, interrupt_mask);

        /* Здесь возникнет прерывание по опустошению очереди. */

        I2S_EnableTx(base);
    }

    return I2S_Status_Ok;
}

void I2S_TransferAbort(I2S_Type *base, i2s_handle_t *handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    I2S_DisableTx(base);
    I2S_DisableInterrupt(base, I2S_FlagTxFifoEmpty);

    handle->nb_samples = 0U;
}

void I2S_TransferHandleIRQ(I2S_Type *base, i2s_handle_t *handle)
{
    /* Признак включенной передачи */
    bool tx_enabled = (handle->nb_samples != 0U);

    if (I2S_GetInterruptStatus(base, I2S_FlagTxFifoOverrun)) {
        handle->callback(base, handle, I2S_FlagTxFifoOverrun,
            handle->user_data);
        I2S_ClearDataOverrunFlag(base);
    }

    if (I2S_GetInterruptStatus(base, I2S_FlagTxFifoEmpty)) {
        if (tx_enabled) {
            /* Сброс флага переполнения очереди. */
            I2S_ClearDataOverrunFlag(base);

            /* Заполнение очереди до получения флага переполнения. */
            while (handle->nb_samples > 0) {
                I2S_WriteLeftFifo(base, *(handle->left_samples));
                if (I2S_GetInterruptStatus(base, I2S_FlagTxFifoOverrun))
                    break;

                I2S_WriteRightFifo(base, *(handle->right_samples));
                if (I2S_GetInterruptStatus(base, I2S_FlagTxFifoOverrun))
                    break;

                --handle->nb_samples;
                ++handle->left_samples;
                ++handle->right_samples;
            }

            /* Сброс флага переполнения. */
            I2S_ClearDataOverrunFlag(base);

            if (handle->nb_samples == 0) {
                I2S_DisableInterrupt(base, I2S_FlagTxFifoEmpty);
                handle->callback(base, handle, I2S_FlagTxFifoEmpty,
                    handle->user_data);
            }
        } else {
            handle->callback(base, handle, I2S_FlagTxFifoEmpty,
                handle->user_data);
        }
    }
}

/*!
 * @}
 */
