/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "hal_power.h"
#include "hal_clkctr.h"
#include "hal_ppu.h"
#include "hal_ioim.h"
#include <string.h>

/*!
 * @brief Режимы низкого потребления
 */
enum power_lp_mode {
    POWER_LPModeStandby  = 0, /*!< Режим STANDBY */
    POWER_LPModeShutdown = 1, /*!< Режим SHUTDOWN */
};

/*!
 * @brief Получение текущих значений параметров режима RUN или STANDBY
 * 
 * Выбор режима осуществляется передачей в параметре cfg_reg значения регистра
 * RUNCFG для режима RUN, либо STDBYCFG для режима STANDBY
 *
 * @param cfg_reg Текущее значение регистра RUNCFG или STDBYCFG
 * @param config  Структура с параметрами режима, в которую они будут прочитаны
 */
static void POWER_GetModeConfig(uint32_t cfg_reg,
    struct power_mode_config *config)
{
    config->dcdc_level = (enum power_dcdc_vlevel) GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_DCDC_VSEL_Msk, PWRCTR_RUNCFG_DCDC_VSEL_Pos);
    config->dcdc_mode = (enum power_dcdc_mode) GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_DCDC_MODE_Msk, PWRCTR_RUNCFG_DCDC_MODE_Pos);
    config->dcdc_swdrv = GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_DCDC_SWDRV_Msk, PWRCTR_RUNCFG_DCDC_SWDRV_Pos);
    config->dcdc_sink_enable = (GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_DCDC_DISSINK_Msk, PWRCTR_RUNCFG_DCDC_DISSINK_Pos)
        == 0U);
    config->dcdc_ccm_enable = (GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_DCDC_ENCCMTRAN_Msk,
        PWRCTR_RUNCFG_DCDC_ENCCMTRAN_Pos) == 1U);
    config->dcdc_low_consumption_enable = (GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_DCDC_ENSTDBY_Msk, PWRCTR_RUNCFG_DCDC_ENSTDBY_Pos)
        == 1U);
    config->eco_mode = (enum power_eco_mode) GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_APC_ECO_Msk, PWRCTR_RUNCFG_APC_ECO_Pos);
    config->apc_low_clk_enable = (GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_APC_ENLPCLK_Msk, PWRCTR_RUNCFG_APC_ENLPCLK_Pos)
        == 1U);
    config->apc_eco_threshold = (enum power_dcdc_threshold) GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_APC_ECOPROG_Msk, PWRCTR_RUNCFG_APC_ECOPROG_Pos);
    config->flash_power_mode = (enum power_flash_mode) GET_VAL_MSK(
        cfg_reg, PWRCTR_RUNCFG_FLASH_PWR_CTR_Msk,
        PWRCTR_RUNCFG_FLASH_PWR_CTR_Pos);
}

/*!
 * @brief Установка значений параметров режима RUN или STANDBY
 * 
 * Выбор режима осуществляется передачей в параметре cfg_reg указателя 
 * на регистр RUNCFG для режима RUN, либо на регистр STDBYCFG для режима STANDBY
 *
 * @param config Структура с параметрами режима
 * @return       Значение регистра, соответствующее структуре
 */
static uint32_t POWER_SetModeConfig(struct power_mode_config *config)
{
    uint32_t cfg_reg = 0;

    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_DCDC_VSEL_Msk,
        PWRCTR_RUNCFG_DCDC_VSEL_Pos, config->dcdc_level);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_DCDC_MODE_Msk,
        PWRCTR_RUNCFG_DCDC_MODE_Pos, config->dcdc_mode);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_DCDC_SWDRV_Msk,
        PWRCTR_RUNCFG_DCDC_SWDRV_Pos, config->dcdc_swdrv);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_DCDC_DISSINK_Msk,
        PWRCTR_RUNCFG_DCDC_DISSINK_Pos, !config->dcdc_sink_enable);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_DCDC_ENCCMTRAN_Msk,
        PWRCTR_RUNCFG_DCDC_ENCCMTRAN_Pos, config->dcdc_ccm_enable);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_DCDC_ENSTDBY_Msk,
        PWRCTR_RUNCFG_DCDC_ENSTDBY_Pos, config->dcdc_low_consumption_enable);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_APC_ECO_Msk, PWRCTR_RUNCFG_APC_ECO_Pos,
        config->eco_mode);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_APC_ENLPCLK_Msk,
        PWRCTR_RUNCFG_APC_ENLPCLK_Pos, config->apc_low_clk_enable);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_APC_ECOPROG_Msk,
        PWRCTR_RUNCFG_APC_ECOPROG_Pos, config->apc_eco_threshold);
    SET_VAL_MSK(cfg_reg, PWRCTR_RUNCFG_FLASH_PWR_CTR_Msk,
        PWRCTR_RUNCFG_FLASH_PWR_CTR_Pos, config->flash_power_mode);

    return cfg_reg;
}

void POWER_GetCurrentConfig(PWRCTR_Type *base, struct power_config *config)
{
    assert(base != NULL);
    assert(config != NULL);

    POWER_GetModeConfig(base->RUNCFG, &config->run_configuration);
    POWER_GetModeConfig(base->STDBYCFG, &config->standby_configuration);

    config->trim_configuration.apc_vref_it = GET_VAL_MSK(
        base->TRIM, PWRCTR_TRIM_APC_VREF_IT_Msk, PWRCTR_TRIM_APC_VREF_IT_Pos);
    config->trim_configuration.apc_vref_tt = GET_VAL_MSK(
        base->TRIM, PWRCTR_TRIM_APC_VREF_TT_Msk, PWRCTR_TRIM_APC_VREF_TT_Pos);
    config->trim_configuration.apc_vref_vt = GET_VAL_MSK(
        base->TRIM, PWRCTR_TRIM_APC_VREF_VT_Msk, PWRCTR_TRIM_APC_VREF_VT_Pos);
    config->trim_configuration.apc_force_trim = (GET_VAL_MSK(
        base->TRIM, PWRCTR_TRIM_APC_FORCETRIM_Msk,
        PWRCTR_TRIM_APC_FORCETRIM_Pos) == 1U);
    config->trim_configuration.dcdc_imax = GET_VAL_MSK(
        base->TRIM, PWRCTR_TRIM_DCDC_IMAX_Msk, PWRCTR_TRIM_DCDC_IMAX_Pos);
    config->trim_configuration.dcdc_imin = GET_VAL_MSK(
        base->TRIM, PWRCTR_TRIM_DCDC_IMIN_Msk, PWRCTR_TRIM_DCDC_IMIN_Pos);
    config->trim_configuration.dcdc_trimlc = GET_VAL_MSK(
        base->TRIM, PWRCTR_TRIM_DCDC_TRIMLC_Msk, PWRCTR_TRIM_DCDC_TRIMLC_Pos);

    config->flash_low_voltage_read_enable = (GET_VAL_MSK(
        base->CFG, PWRCTR_CFG_FLASH_LVE_Msk, PWRCTR_CFG_FLASH_LVE_Pos) == 1U);
    config->dcdc_enable = (GET_VAL_MSK(
        base->CFG, PWRCTR_CFG_DCDC_DIS_Msk, PWRCTR_CFG_DCDC_DIS_Pos) == 0U);

    config->vlevel0 = (enum power_dcdc_vlevel_value) GET_VAL_MSK(
        base->VLEVEL, PWRCTR_VLEVEL_VLEVEL0_Msk, PWRCTR_VLEVEL_VLEVEL0_Pos);
    config->vlevel1 = (enum power_dcdc_vlevel_value) GET_VAL_MSK(
        base->VLEVEL, PWRCTR_VLEVEL_VLEVEL1_Msk, PWRCTR_VLEVEL_VLEVEL1_Pos);
    config->vlevel2 = (enum power_dcdc_vlevel_value) GET_VAL_MSK(
        base->VLEVEL, PWRCTR_VLEVEL_VLEVEL2_Msk, PWRCTR_VLEVEL_VLEVEL2_Pos);
}

void POWER_SetConfig(PWRCTR_Type *base, struct power_config *config)
{
    assert(base != NULL);
    assert(config != NULL);

    uint32_t reg;

    reg = POWER_SetModeConfig(&config->run_configuration);
    base->RUNCFG = reg;

    reg = POWER_SetModeConfig(&config->standby_configuration);
    base->STDBYCFG = reg;

    SET_VAL_MSK(base->TRIM, PWRCTR_TRIM_APC_VREF_IT_Msk,
        PWRCTR_TRIM_APC_VREF_IT_Pos, config->trim_configuration.apc_vref_it);
    SET_VAL_MSK(base->TRIM, PWRCTR_TRIM_APC_VREF_TT_Msk,
        PWRCTR_TRIM_APC_VREF_TT_Pos, config->trim_configuration.apc_vref_tt);
    SET_VAL_MSK(base->TRIM, PWRCTR_TRIM_APC_VREF_VT_Msk,
        PWRCTR_TRIM_APC_VREF_VT_Pos, config->trim_configuration.apc_vref_vt);
    SET_VAL_MSK(base->TRIM, PWRCTR_TRIM_APC_FORCETRIM_Msk,
        PWRCTR_TRIM_APC_FORCETRIM_Pos, 
        config->trim_configuration.apc_force_trim);
    SET_VAL_MSK(base->TRIM, PWRCTR_TRIM_DCDC_IMAX_Msk,
        PWRCTR_TRIM_DCDC_IMAX_Pos, config->trim_configuration.dcdc_imax);
    SET_VAL_MSK(base->TRIM, PWRCTR_TRIM_DCDC_IMIN_Msk,
        PWRCTR_TRIM_DCDC_IMIN_Pos, config->trim_configuration.dcdc_imin);
    SET_VAL_MSK(base->TRIM, PWRCTR_TRIM_DCDC_TRIMLC_Msk,
        PWRCTR_TRIM_DCDC_TRIMLC_Pos, config->trim_configuration.dcdc_trimlc);

    SET_VAL_MSK(base->CFG, PWRCTR_CFG_FLASH_LVE_Msk, PWRCTR_CFG_FLASH_LVE_Pos,
        config->flash_low_voltage_read_enable);
    SET_VAL_MSK(base->CFG, PWRCTR_CFG_DCDC_DIS_Msk, PWRCTR_CFG_DCDC_DIS_Pos,
        !config->dcdc_enable);

    SET_VAL_MSK(base->VLEVEL, PWRCTR_VLEVEL_VLEVEL0_Msk,
        PWRCTR_VLEVEL_VLEVEL0_Pos, config->vlevel0);
    SET_VAL_MSK(base->VLEVEL, PWRCTR_VLEVEL_VLEVEL1_Msk,
        PWRCTR_VLEVEL_VLEVEL1_Pos, config->vlevel1);
    SET_VAL_MSK(base->VLEVEL, PWRCTR_VLEVEL_VLEVEL2_Msk,
        PWRCTR_VLEVEL_VLEVEL2_Pos, config->vlevel2);
}

void POWER_GetStatus(PWRCTR_Type *base, struct power_state *status)
{
    assert(base != NULL);
    assert(status != NULL);
    
    status->vdda_is_lower_threshold = (GET_VAL_MSK(
        base->STAT, PWRCTR_STAT_VMON_STAT_Msk, PWRCTR_STAT_VMON_STAT_Pos)
        == 1U);
    status->dcdc_is_ready = (GET_VAL_MSK(
        base->STAT, PWRCTR_STAT_DCDC_PG_Msk, PWRCTR_STAT_DCDC_PG_Pos) == 1U);
    status->flash_power_mode = (enum power_flash_mode) GET_VAL_MSK(
        base->STAT, PWRCTR_STAT_FLASH_PWR_STAT_Msk,
        PWRCTR_STAT_FLASH_PWR_STAT_Pos);
    status->flash_low_voltage_read_enabled = (GET_VAL_MSK(
        base->STAT, PWRCTR_STAT_FLASH_LVE_STAT_Msk,
        PWRCTR_STAT_FLASH_LVE_STAT_Pos) == 1U);
}

void POWER_EnableInterrupt(PWRCTR_Type *base, enum power_interrupt idx)
{
    assert(base != NULL);
    assert(idx <= POWER_VmonFalling);

    switch (idx) {
        case POWER_VmonRising:
            SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_RTI_EN_Msk,
                PWRCTR_CFG_VMON_RTI_EN_Pos, 1U);
            break;

        case POWER_VmonFalling:
            SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_FTI_EN_Msk,
                PWRCTR_CFG_VMON_FTI_EN_Pos, 1U);
            break;
    }
}

void POWER_EnableInterruptMask(PWRCTR_Type *base, uint8_t mask)
{
    assert(base != NULL);

    if (mask & (1 << POWER_VmonRising))
        SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_RTI_EN_Msk,
            PWRCTR_CFG_VMON_RTI_EN_Pos, 1U);

    if (mask & (1 << POWER_VmonFalling))
        SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_FTI_EN_Msk,
            PWRCTR_CFG_VMON_FTI_EN_Pos, 1U);
}

bool POWER_IsInterruptEnabled(PWRCTR_Type *base, enum power_interrupt idx)
{
    assert(base != NULL);
    assert(idx <= POWER_VmonFalling);

    switch (idx) {
        case POWER_VmonRising:
            return (GET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_RTI_EN_Msk,
                PWRCTR_CFG_VMON_RTI_EN_Pos) == 1U);
        case POWER_VmonFalling:
            return (GET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_FTI_EN_Msk,
                PWRCTR_CFG_VMON_FTI_EN_Pos) == 1U);
    }

    return false;
}

uint8_t POWER_GetEnabledInterruptMask(PWRCTR_Type *base)
{
    assert(base != NULL);

    uint8_t mask = 0;

    if (GET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_RTI_EN_Msk,
        PWRCTR_CFG_VMON_RTI_EN_Pos) == 1U)
    {
        mask |= (1 << POWER_VmonRising);
    }

    if (GET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_FTI_EN_Msk,
        PWRCTR_CFG_VMON_FTI_EN_Pos) == 1U)
    {
        mask |= (1 << POWER_VmonFalling);
    }

    return mask;
}

void POWER_DisableInterrupt(PWRCTR_Type *base, enum power_interrupt idx)
{
    assert(base != NULL);
    assert(idx <= POWER_VmonFalling);

    switch (idx) {
        case POWER_VmonRising:
            SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_RTI_EN_Msk,
                PWRCTR_CFG_VMON_RTI_EN_Pos, 0U);
            break;

        case POWER_VmonFalling:
            SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_FTI_EN_Msk,
                PWRCTR_CFG_VMON_FTI_EN_Pos, 0U);
            break;
    }
}

void POWER_DisableInterruptMask(PWRCTR_Type *base, uint8_t mask)
{
    assert(base != NULL);

    if (mask & (1 << POWER_VmonRising))
        SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_RTI_EN_Msk,
            PWRCTR_CFG_VMON_RTI_EN_Pos, 0U);

    if (mask & (1 << POWER_VmonFalling))
        SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_FTI_EN_Msk,
            PWRCTR_CFG_VMON_FTI_EN_Pos, 0U);
}

bool POWER_GetInterruptStatus(PWRCTR_Type *base, enum power_interrupt idx)
{
    assert(base != NULL);
    assert(idx <= POWER_VmonFalling);

    switch (idx) {
        case POWER_VmonRising:
            return (GET_VAL_MSK(base->STAT, PWRCTR_STAT_VMON_RTI_STAT_Msk,
                PWRCTR_STAT_VMON_RTI_STAT_Pos) == 1U);
        case POWER_VmonFalling:
            return (GET_VAL_MSK(base->STAT, PWRCTR_STAT_VMON_FTI_STAT_Msk,
                PWRCTR_STAT_VMON_FTI_STAT_Pos) == 1U);
    }

    return false;
}

uint8_t POWER_GetInterruptStatusMask(PWRCTR_Type *base)
{
    assert(base != NULL);

    uint8_t mask = 0;

    if (GET_VAL_MSK(base->STAT, PWRCTR_STAT_VMON_RTI_STAT_Msk,
        PWRCTR_STAT_VMON_RTI_STAT_Pos) == 1U)
    {
        mask |= (1 << POWER_VmonRising);
    }

    if (GET_VAL_MSK(base->STAT, PWRCTR_STAT_VMON_FTI_STAT_Msk,
        PWRCTR_STAT_VMON_FTI_STAT_Pos) == 1U)
    {
        mask |= (1 << POWER_VmonFalling);
    }
    return mask;
}

void POWER_ClearInterrupts(PWRCTR_Type *base)
{
    assert(base != NULL);

    SET_VAL_MSK(base->CFG, PWRCTR_CFG_VMON_INT_CLR_Msk,
        PWRCTR_CFG_VMON_INT_CLR_Pos, 1U);
}

/*!
 * @brief Обработчик прерываний POWER
 * 
 * Вызывает пользовательскую функцию обратного вызова и сбрасывает признаки
 * активных прерываний
 *
 * @param base      Базовый адрес блока POWER
 * @param handle    Структура пользовательского обработчика прерывания
 */
static void POWER_HandleIRQ(PWRCTR_Type *base, struct power_handle *handle)
{
    handle->callback(base, handle, POWER_GetInterruptStatusMask(base),
        handle->user_data);

    POWER_ClearInterrupts(base);
}

enum power_status POWER_CreateHandle(PWRCTR_Type *base,
    struct power_handle *handle, power_callback_t callback, void *user_data)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(callback != NULL);

    /* Установка обратного вызова и пользовательских данных. */
    handle->callback  = callback;
    handle->user_data = user_data;

    /* Установка обработчика прерывания. */
    if (IOIM_SetIRQHandler(base, POWER_HandleIRQ, handle) != IOIM_Status_Ok)
        return POWER_Status_Fail;

    return POWER_Status_Ok;
}

// void POWER_StartTestMode(PWRCTR_Type *base, enum power_test_block test_block)
// {
//     assert(base != NULL);
//     assert(test_block <= POWER_TestBlockRwc);

//     SET_VAL_MSK(base->TEST, PWRCTR_TEST_TEST_SEL_Msk, PWRCTR_TEST_TEST_SEL_Pos,
//         test_block);
//     SET_VAL_MSK(base->TEST, PWRCTR_TEST_TEST_MODE_Msk,
//         PWRCTR_TEST_TEST_MODE_Pos, 1U);
// }

// void POWER_StopTestMode(PWRCTR_Type *base)
// {
//     assert(base != NULL);

//     SET_VAL_MSK(base->TEST, PWRCTR_TEST_TEST_MODE_Msk,
//         PWRCTR_TEST_TEST_MODE_Pos, 0U);
// }

void POWER_DeepSleepThisCpu()
{
    SET_VAL_MSK(SCB->SCR, SCB_SCR_SLEEPDEEP_Msk, SCB_SCR_SLEEPDEEP_Pos, 1U);
    SET_VAL_MSK(SCB->SCR, SCB_SCR_SLEEPONEXIT_Msk, SCB_SCR_SLEEPONEXIT_Pos, 1U);
    __WFI();
}

enum power_status POWER_Standby(PWRCTR_Type * base)
{
    assert(base != NULL);

    struct clkctr_pll_cfg config; /* Для остановки PLL */

    /* 1 установить LP_MODE = 0 в регистре PWRCTR_CFG */
    SET_VAL_MSK(base->CFG, PWRCTR_CFG_LP_MODE_Msk, PWRCTR_CFG_LP_MODE_Pos,
        POWER_LPModeStandby);
    /* 2 настроить параметры DC-DC, APC и Flash памяти в регистре PWRCTR_STDBYCFG */
    
    /* Эти параметры настраиваются пользователем с помощью функции POWER_SetConfig. */
    
    /* 3 убедиться, что все блоки GPR отладочной подсистемы выключены */
    
    /* 4 при необходимости, выбрать HFI или XTI в качестве источника тактирования и выключить PLL */
    CLKCTR_SetSwitchMainClk(CLKCTR_Secure, CLKCTR_MainClkTypeHFIClk);
    CLKCTR_GetPLLConfig(CLKCTR_Secure, &config);
    config.sel = 0;
    CLKCTR_SetPLLConfig(CLKCTR_Secure, config);
    
    /* 5 включаем WIC+EWC для обоих процессоров, выход из сна будет определяться настройками NVIC 
     *   и наличием соответствующего прерывания. */
    SET_VAL_MSK(SYSCTR_Secure->WICCTRL, SYSCTR_WICCTRL_CPU0WICEN_SET_Msk,
        SYSCTR_WICCTRL_CPU0WICEN_SET_Pos, 1U);
    SET_VAL_MSK(SYSCTR_Secure->EWCCTRL, SYSCTR_EWCCTRL_EWC0EN_SET_Msk,
        SYSCTR_EWCCTRL_EWC0EN_SET_Pos, 1U);
    SET_VAL_MSK(SYSCTR_Secure->WICCTRL, SYSCTR_WICCTRL_CPU1WICEN_SET_Msk,
        SYSCTR_WICCTRL_CPU1WICEN_SET_Pos, 1U);
    SET_VAL_MSK(SYSCTR_Secure->EWCCTRL, SYSCTR_EWCCTRL_EWC1EN_SET_Msk,
        SYSCTR_EWCCTRL_EWC1EN_SET_Pos, 1U);
    /* 6 выключить статические домены PD_GMS, PD_GNSS, PD_CRYPTO */
    if (PPU_SetState(GMS_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    if (PPU_SetState(GNSS_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    if (PPU_SetState(CRYPTO_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    /* 7 включить динамический переход доменов памяти PD_SRAMn в состояние OFF или MEM_RET */
    if (PPU_SetStateDynamic(SRAM0_PPU_Secure, PPU_PowerModeMemRet)!=
        PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(SRAM1_PPU_Secure, PPU_PowerModeMemRet) !=
        PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(SRAM2_PPU_Secure, PPU_PowerModeMemRet) !=
        PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(SRAM3_PPU_Secure, PPU_PowerModeMemRet) !=
        PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    /* 8 включить динамический переход доменов PD_DEBUG, PD_CPUn, PD_SYS в OFF */
    if (PPU_SetStateDynamic(DEBUG_PPU_Secure, PPU_PowerModeOff) !=
        PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(CPU0_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    if (PPU_SetStateDynamic(CPU1_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    if (PPU_SetStateDynamic(SYS_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    /* 9 выполнить инструкцию WFI */
    POWER_DeepSleepThisCpu();

    return POWER_Status_Ok;
}

enum power_status POWER_Shutdown(PWRCTR_Type * base)
{
    /* 1 установить LP_MODE = 1 в регистре PWRCTR_CFG */
    SET_VAL_MSK(base->CFG, PWRCTR_CFG_LP_MODE_Msk, PWRCTR_CFG_LP_MODE_Pos,
        POWER_LPModeShutdown);

    /* 2 убедиться, что все блоки GPR отладочной подсистемы выключены */

    /* 3 выключаем WIC+EWC для обоих процессоров */
    SET_VAL_MSK(SYSCTR_Secure->WICCTRL, SYSCTR_WICCTRL_CPU0WICEN_SET_Msk,
        SYSCTR_WICCTRL_CPU0WICEN_SET_Pos, 1U);
    SET_VAL_MSK(SYSCTR_Secure->EWCCTRL, SYSCTR_EWCCTRL_EWC0EN_SET_Msk,
        SYSCTR_EWCCTRL_EWC0EN_SET_Pos, 1U);
    SET_VAL_MSK(SYSCTR_Secure->WICCTRL, SYSCTR_WICCTRL_CPU1WICEN_SET_Msk,
        SYSCTR_WICCTRL_CPU1WICEN_SET_Pos, 1U);
    SET_VAL_MSK(SYSCTR_Secure->EWCCTRL, SYSCTR_EWCCTRL_EWC1EN_SET_Msk,
        SYSCTR_EWCCTRL_EWC1EN_SET_Pos, 1U);

    /* 4 выключить статические домены PD_GMS, PD_GNSS, PD_CRYPTO */
    if (PPU_SetState(GMS_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    if (PPU_SetState(GNSS_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;
    if (PPU_SetState(CRYPTO_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;

    /* 5 включить динамический переход доменов PD_SRAMn, PD_DEBUG, PD_CPUn, PD_SYS в OFF */
    if (PPU_SetStateDynamic(SRAM0_PPU_Secure, PPU_PowerModeOff) !=
        PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(SRAM1_PPU_Secure, PPU_PowerModeOff) !=
            PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(SRAM2_PPU_Secure, PPU_PowerModeOff) !=
            PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(SRAM3_PPU_Secure, PPU_PowerModeOff) !=
            PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(DEBUG_PPU_Secure, PPU_PowerModeOff) !=
            PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(CPU0_PPU_Secure, PPU_PowerModeOff) !=
            PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(CPU1_PPU_Secure, PPU_PowerModeOff) !=
            PPU_Status_Ok)
    {
        return POWER_Status_Fail;
    }
    if (PPU_SetStateDynamic(SYS_PPU_Secure, PPU_PowerModeOff) != PPU_Status_Ok)
        return POWER_Status_Fail;

    /* 6 выполнить инструкцию WFI */
    POWER_DeepSleepThisCpu();
    
    return POWER_Status_Ok;
}
