/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup power_driver Драйвер модуля POWER
 *
 * @brief Драйвер модуля управления питанием и режимами работы
 *
 * Драйвер модуля POWER позволяет управлять режимами питания микросхемы.
 */

/*!
 * @addtogroup power_driver
 * @{
 */

/*!
 * @file hal_power.h
 *
 * @brief Интерфейс драйвера модуля POWER
 */

#ifndef HAL_POWER_H
#define HAL_POWER_H

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#include "hal_common.h"

/*!
 * @brief Коды возврата функций драйвера POWER
 */
enum power_status {
    POWER_Status_Ok   = 0U, /*!< Успешно */
    POWER_Status_Fail = 1U, /*!< Провал */
};

/*!
 * @brief Перечисление выходных напряжений встроенного регулятора DC-DC
 */
enum power_dcdc_vlevel {
    POWER_DcdcVLevel0, /*!< VLEVEL0 */
    POWER_DcdcVLevel1, /*!< VLEVEL1 */
    POWER_DcdcVLevel2  /*!< VLEVEL2 */
};

/*!
 * @brief Перечисление значений для уровней 0-2 выходных напряжений встроенного регулятора DC-DC
 *
 * Параметры VLEVEL0-2 должны выбираться так, чтобы выполнялись условия:
 * - VLEVEL0 < VLEVEL1 < VLEVEL2;
 * - VLEVEL0 < 0,99 В.
 */
enum power_dcdc_vlevel_value {
    POWER_DcdcVLevel_0_85V, /*!< 0,85 В */
    POWER_DcdcVLevel_0_86V, /*!< 0,86 В */
    POWER_DcdcVLevel_0_87V, /*!< 0,87 В */
    POWER_DcdcVLevel_0_88V, /*!< 0,88 В */
    POWER_DcdcVLevel_0_89V, /*!< 0,89 В */
    POWER_DcdcVLevel_0_90V, /*!< 0,90 В */
    POWER_DcdcVLevel_0_91V, /*!< 0,91 В */
    POWER_DcdcVLevel_0_92V, /*!< 0,92 В */
    POWER_DcdcVLevel_0_93V, /*!< 0,93 В */
    POWER_DcdcVLevel_0_94V, /*!< 0,94 В */
    POWER_DcdcVLevel_0_95V, /*!< 0,95 В */
    POWER_DcdcVLevel_0_96V, /*!< 0,96 В */
    POWER_DcdcVLevel_0_97V, /*!< 0,97 В */
    POWER_DcdcVLevel_0_98V, /*!< 0,98 В */
    POWER_DcdcVLevel_0_99V, /*!< 0,99 В */
    POWER_DcdcVLevel_1_00V, /*!< 1,00 В */
    POWER_DcdcVLevel_1_01V, /*!< 1,01 В */
    POWER_DcdcVLevel_1_02V, /*!< 1,02 В */
    POWER_DcdcVLevel_1_03V, /*!< 1,03 В */
    POWER_DcdcVLevel_1_04V, /*!< 1,04 В */
    POWER_DcdcVLevel_1_05V, /*!< 1,05 В */
    POWER_DcdcVLevel_1_06V, /*!< 1,06 В */
    POWER_DcdcVLevel_1_07V, /*!< 1,07 В */
    POWER_DcdcVLevel_1_08V, /*!< 1,08 В */
    POWER_DcdcVLevel_1_09V, /*!< 1,09 В */
    POWER_DcdcVLevel_1_10V, /*!< 1,10 В */
    POWER_DcdcVLevel_1_11V, /*!< 1,11 В */
    POWER_DcdcVLevel_1_12V, /*!< 1,12 В */
    POWER_DcdcVLevel_1_13V, /*!< 1,13 В */
    POWER_DcdcVLevel_1_14V, /*!< 1,14 В */
    POWER_DcdcVLevel_1_15V, /*!< 1,15 В */
    POWER_DcdcVLevel_1_16V  /*!< 1,16 В */
};

/*!
 * @brief Перечисление режимов работы встроенного регулятора DC-DC
 */
enum power_dcdc_mode {
    POWER_DcdcModeAuto,    /*!< Автовыбор PWM/PFM */
    POWER_DcdcModePwm,     /*!< PWM */
    POWER_DcdcModePwmFccm, /*!< PWM FCCM */
    POWER_DcdcModePfm      /*!< PFM (рекомендуется для режима ECO) */
};

/*!
 * @brief Перечисление режимов ECO DC-DC и APC
 */
enum power_eco_mode {
    POWER_EcoOff,       /*!< Режим ECO выключен */
    POWER_EcoDcdc,      /*!< Включен режим ECO DC-DC */
    POWER_EcoReserved,  /*!< Зарезервировано (не используется) */
    POWER_EcoDcdcAndApc /*!< Включен режим ECO DC-DC и APC */
};

/*!
 * @brief Перечисление пороговых напряжений DC-DC
 */
enum power_dcdc_threshold {
    POWER_DcdcThreshold_0_60V, /*!< 0,60 B */
    POWER_DcdcThreshold_0_62V, /*!< 0,62 B */
    POWER_DcdcThreshold_0_64V, /*!< 0,64 B */
    POWER_DcdcThreshold_0_66V, /*!< 0,66 B */
    POWER_DcdcThreshold_0_68V, /*!< 0,68 B */
    POWER_DcdcThreshold_0_70V, /*!< 0,70 B */
    POWER_DcdcThreshold_0_72V, /*!< 0,72 B */
    POWER_DcdcThreshold_0_74V, /*!< 0,74 B */
    POWER_DcdcThreshold_0_76V, /*!< 0,76 B */
    POWER_DcdcThreshold_0_78V, /*!< 0,78 B */
    POWER_DcdcThreshold_0_80V, /*!< 0,80 B */
    POWER_DcdcThreshold_0_82V, /*!< 0,82 B */
    POWER_DcdcThreshold_0_84V, /*!< 0,84 B */
    POWER_DcdcThreshold_0_86V, /*!< 0,86 B */
    POWER_DcdcThreshold_0_88V, /*!< 0,88 B */
    POWER_DcdcThreshold_0_90V  /*!< 0,90 B */
};

/*!
 * @brief Перечисление пороговых напряжений DC-DC
 */
enum power_flash_mode {
    POWER_FlashModeNormal,   /*!< Рабочий режим */
    POWER_FlashModeSleep,    /*!< Сон */
    POWER_FlashModePowerDown /*!< Флеш отключена */
};

/*!
 * @brief Перечисление блоков для тестирования
 */
enum power_test_block {
    POWER_TestBlockApc,  /*!< Тестирование APC */
    POWER_TestBlockDcdc, /*!< Тестирование DC-DC */
    POWER_TestBlockJtm,  /*!< Тестирование JTM */
    POWER_TestBlockRwc   /*!< Тестирование RWC */
};

/*!
 * @brief Перечисление типов фронтов прерываний
 */
enum power_interrupt {
    POWER_VmonRising,  /*!< Прерывание по нарастающему фронту сигнала монитора питания */
    POWER_VmonFalling, /*!< Прерывание по спадающему фронту сигнала монитора питания */
};

/*!
 * @brief Структура параметров режима питания
 */
struct power_mode_config {
    enum power_dcdc_vlevel    dcdc_level;                  /*!< Выходное напряжение встроенного DC-DC */
    enum power_dcdc_mode      dcdc_mode;                   /*!< Режим работы встроенного DC-DC */
    uint8_t                   dcdc_swdrv;                  /*!< Управление мощностью DC-DC */
    bool                      dcdc_sink_enable;            /*!< Включение резистора подтяжки выхода DC-DC к земле */
    bool                      dcdc_ccm_enable;             /*!< Включение генерации 16 CCM-импульсов при снижении нагрузки */
    bool                      dcdc_low_consumption_enable; /*!< Включение низкопотребляющего режима конвертера */
    enum power_eco_mode       eco_mode;                    /*!< Выбор режима ECO DC-DC и APC */
    bool                      apc_low_clk_enable;          /*!< Включение семплирования монитора питания низкочастотным тактом */
    enum power_dcdc_threshold apc_eco_threshold;           /*!< Порог для монитора выходного напряжения DC-DC */
    enum power_flash_mode     flash_power_mode;            /*!< Режим работы флеш-памяти */
};

/*!
 * @brief Структура подстроечных параметров APC и DC-DC
 */
struct power_trim_config {
    uint8_t apc_vref_it;    /*!< Подстройка APC по току */
    uint8_t apc_vref_tt;    /*!< Подстройка APC по температуре */
    uint8_t apc_vref_vt;    /*!< Подстройка APC по напряжению */
    bool    apc_force_trim; /*!< Включение подстройки vref в режиме сверхнизкого потребления */
    uint8_t dcdc_imax;      /*!< Подстройка DC-DC по пику тока в катушке */
    uint8_t dcdc_imin;      /*!< Подстройка синхронной коррекции DC-DC  */
    uint8_t dcdc_trimlc;    /*!< Подстройка выходного фильтра DC-DC  */
};


/*!
 * @brief Структура конфигурации блока POWER
 */
struct power_config {
    struct power_mode_config     run_configuration;             /*!< Конфигурация для режима работы RUN */
    struct power_mode_config     standby_configuration;         /*!< Конфигурация для режима работы STANDBY */
    struct power_trim_config     trim_configuration;            /*!< Подстроечные параметры APC и DC-DC */
    bool                         flash_low_voltage_read_enable; /*!< Включение режима низковольтного чтения флеш-памяти */
    bool                         dcdc_enable;                   /*!< Включение встроенного DC-DC */
    enum power_dcdc_vlevel_value vlevel0;                       /*!< Уровень напряжения VLEVEL0 */
    enum power_dcdc_vlevel_value vlevel1;                       /*!< Уровень напряжения VLEVEL1 */
    enum power_dcdc_vlevel_value vlevel2;                       /*!< Уровень напряжения VLEVEL2 */
};

/*!
 * @brief Структура параметров состояния блока POWER
 */
struct power_state {
    bool                  vdda_is_lower_threshold;        /*!< Признак, VDDA ниже порогового уровня */
    bool                  dcdc_is_ready;                  /*!< Признак готовности DC-DC */
    enum power_flash_mode flash_power_mode;               /*!< Текущий режим работы флеш-памяти */
    bool                  flash_low_voltage_read_enabled; /*!< Признак включения режима низковольтного чтения флеш-памяти */
};

struct power_handle;

/*!
 * @brief Функция обратного вызова для обработки прерывания POWER
 *
 * @param base           Базовый адрес блока POWER
 * @param handle         Указатель на обработчик
 * @param interrupt_mask Причина вызова
 * @param user_data      Данные пользователя
 */
typedef void (*power_callback_t)(PWRCTR_Type *base, struct power_handle *handle,
    uint8_t interrupt_mask, void *user_data);

/*!
 * @brief Структура обработчика драйвера I2S
 */
struct power_handle {
    power_callback_t callback;   /*!< Функция обратного вызова */
    void             *user_data; /*!< Данные пользователя */
};

/*!
 * @name Функции конфигурирования и чтения состояния
 * @{
 */

/*!
 * @brief Получение текущих значений параметров блока POWER
 *
 * @param base   Базовый адрес блока POWER
 * @param config Структура с параметрами конфигурации
 */
void POWER_GetCurrentConfig(PWRCTR_Type *base, struct power_config *config);

/*!
 * @brief Установка параметров блока POWER
 *
 * @param base   Базовый адрес блока POWER
 * @param config Структура с параметрами конфигурации
 */
void POWER_SetConfig(PWRCTR_Type *base, struct power_config *config);

/*!
 * @brief Установка параметров блока POWER
 *
 * @param base   Базовый адрес блока POWER
 * @param status Структура с параметрами состояния блока POWER
 */
void POWER_GetStatus(PWRCTR_Type *base, struct power_state *status);

/*!
 * @}
 */

/*!
 * @name Функции управления прерываниями
 * @{
 */

/*!
 * @brief Разрешение прерывания
 *
 * @param base Базовый адрес блока POWER
 * @param idx  Номер прерывания
 */
void POWER_EnableInterrupt(PWRCTR_Type *base, enum power_interrupt idx);

/*!
 * @brief Разрешение прерываний по маске
 *
 * @param base Базовый адрес блока POWER
 * @param mask Маска флагов прерываний
 */
void POWER_EnableInterruptMask(PWRCTR_Type *base, uint8_t mask);

/*!
 * @brief Запрос - разрешено ли прерывание
 *
 * @param base   Базовый адрес блока POWER
 * @param idx    Номер флага состояния/прерывания
 *
 * @retval true  Прерывание разрешено
 * @retval false Прерывание запрещено
 */
bool POWER_IsInterruptEnabled(PWRCTR_Type *base, enum power_interrupt idx);

/*!
 * @brief Запрос маски разрешенных прерываний
 *
 * Запрос маски разрешенных прерываний, единицы в соответствующих
 * разрядах соответствуют включенным прерываниям.
 *
 * @param base Базовый адрес блока POWER
 *
 * @return Маска разрешенных прерываний
 *
 */
uint8_t POWER_GetEnabledInterruptMask(PWRCTR_Type *base);

/*!
 * @brief Запрет прерывания
 *
 * @param base Базовый адрес блока POWER
 * @param idx  Номер флага состояния/прерывания
 */
void POWER_DisableInterrupt(PWRCTR_Type *base, enum power_interrupt idx);

/*!
 * @brief Запрет прерываний по маске
 *
 * @param base Базовый адрес блока POWER
 * @param mask Маска флагов прерываний
 */
void POWER_DisableInterruptMask(PWRCTR_Type *base, uint8_t mask);

/*!
 * @brief Получение состояния флага прерывания
 *
 * @param base   Базовый адрес блока POWER
 * @param idx    Номер флага состояния/прерывания
 *
 * @retval true  Флаг прерывания установлен
 * @retval false Флаг прерывания сброшен
 */
bool POWER_GetInterruptStatus(PWRCTR_Type *base, enum power_interrupt idx);

/*!
 * @brief Получение маски активных прерываний
 *
 * @param base Базовый адрес блока POWER
 *
 * @return Маска активных прерываний
 */
uint8_t POWER_GetInterruptStatusMask(PWRCTR_Type *base);

/*!
 * @brief Сброс признаков активных прерываний
 *
 * @param base Базовый адрес блока POWER
 */
void POWER_ClearInterrupts(PWRCTR_Type *base);

/*!
 * @brief Инициализация обработчика прерываний блока POWER
 *
 * @param base      Базовый адрес блока POWER
 * @param handle    Обработчик
 * @param callback  Функция обратного вызова
 * @param user_data Данные пользователя
 *
 * @retval POWER_Status_Ok
 * @retval POWER_Status_Fail
 */
enum power_status POWER_CreateHandle(PWRCTR_Type *base,
    struct power_handle *handle, power_callback_t callback, void *user_data);

/*!
 * @}
 */

/*!
 * @name Функции управления тестовыми режимами
 * @{
 */

/*!
 * @brief Запуск тестового режима
 *
 * @param base       Базовый адрес блока POWER
 * @param test_block Тестируемый блок
 */
void POWER_StartTestMode(PWRCTR_Type *base, enum power_test_block test_block);

/*!
 * @brief Останов тестового режима
 *
 * @param base Базовый адрес блока POWER
 */
void POWER_StopTestMode(PWRCTR_Type *base);

/*!
 * @}
 */

/*!
 * @name Функции переключения режимов
 * @{
 */

/*!
 * @brief Функция погружения ядра процессора в сон
 * Погружается в сон то ядро, на котором выполнится данная функция.
 */
void POWER_DeepSleepThisCpu();

/*!
 * @brief Функция погружения процессора в сон
 *
 * @param base Базовый адрес блока POWER
 *
 * @retval #POWER_Status_Ok
 * @retval #POWER_Status_Fail
 */
enum power_status POWER_Standby(PWRCTR_Type * base);

/*!
 * @brief Функция погружения процессора в глубокий сон
 *
 * @param base Базовый адрес блока POWER
 *
 * @retval #POWER_Status_Ok
 * @retval #POWER_Status_Fail
 */
enum power_status POWER_Shutdown(PWRCTR_Type * base);

/*!
 * @}
 */


#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_POWER_H */

/*!
 * @}
 */
