/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup qspi_driver
 * @{
 */

/*!
 * @file hal_qspi_dma.c
 *
 * @brief Имплементация дополнения драйвера QSPI для обмена данными с помощью DMA
 */

#include "hal_qspi_dma.h"

void QSPI_TransferCreateHandleDMA(QSPI_Type *base, qspi_dma_handle_t *handle,
    dma_handle_t *tx_handle, dma_handle_t *rx_handle)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(tx_handle != NULL);
    assert(rx_handle != NULL);

    memset(handle, 0, sizeof(qspi_dma_handle_t));

    handle->base = base;
    handle->tx_handle = tx_handle;
    handle->rx_handle = rx_handle;
}

qspi_dma_status_t QSPI_WriteDataDMA(qspi_dma_handle_t *handle, void *addr,
    uint8_t incr, uint8_t transfer_width, uint32_t size)
{
    assert(handle != NULL);
    assert(addr != NULL);
    assert(incr <= DMA_NoChange);
    assert(transfer_width <= DMA_Transfer256BitWidth);

    if (size == 0) {
        return QSPI_DMA_Status_Fail;
    }

    if (size > DMA_AHB_MAX_BLOCK_SIZE) {
        if (handle->tx_desc == NULL) {
            return QSPI_DMA_Status_InvalidArgument;
        }
        uint32_t block_count = DMA_GetDescriptorCount(size, transfer_width);
        QSPI_DMADescriptorInitTX(handle->base, handle->tx_desc,
            block_count, size, transfer_width, addr, incr);
        DMA_SubmitChannelDescriptor(handle->tx_handle, handle->tx_desc);
    } else {
        uint32_t dst_addr = (uint32_t) & (handle->base->TX_DATA);
        struct dma_channel_ctl_cfg ctlxcfg = {
            .block_size = size,
            .dst_incr = DMA_NoChange,
            .src_incr = incr,
            .dst_tr_width = transfer_width,
            .src_tr_width = transfer_width,
            .transfer_type = DMA_MemoryToPeripheral_DMA
        };
        uint64_t xfercfg = DMA_GetCTLCfgMask(&ctlxcfg);
        DMA_SubmitChannelTransferParameter(handle->tx_handle, xfercfg, addr,
            (void *) dst_addr);
    }

    DMA_HardwareHandshakeEnable(handle->tx_handle, true,
        DMA_HANDSHAKE_QSPI_TX);
    QSPI_EnableDMA(handle->base);

    DMA_StartTransfer(handle->tx_handle);

    return QSPI_DMA_Status_Success;
}

qspi_dma_status_t QSPI_ReadDataDMA(qspi_dma_handle_t *handle, void *addr,
    uint8_t incr, uint8_t transfer_width, uint32_t size)
{
    assert(handle != NULL);
    assert(addr != NULL);

    if (size == 0) {
        return QSPI_DMA_Status_Fail;
    }

    if ((size >> transfer_width) > DMA_AHB_MAX_BLOCK_SIZE) {
        if (handle->rx_desc == NULL) {
            return QSPI_DMA_Status_InvalidArgument;
        }
        QSPI_DMAReadDescriptorInitRX(handle->base, handle->rx_desc,
            size, addr, incr);
        DMA_SubmitChannelDescriptor(handle->rx_handle, handle->rx_desc);
    } else {
        uint32_t dst_addr = (uint32_t) & (handle->base->RX_DATA);
        struct dma_channel_ctl_cfg ctlxcfg = {
            .block_size = size,
            .dst_incr = incr,
            .src_incr = DMA_NoChange,
            .dst_tr_width = transfer_width,
            .src_tr_width = transfer_width,
            .transfer_type = DMA_PeripheralToMemory_DMA
        };
        uint64_t xfercfg = DMA_GetCTLCfgMask(&ctlxcfg);
        DMA_SubmitChannelTransferParameter(handle->rx_handle, xfercfg,
            (void *) dst_addr, addr);
    }

    DMA_HardwareHandshakeEnable(handle->rx_handle, true,
        DMA_HANDSHAKE_QSPI_RX);
    QSPI_EnableDMA(handle->base);

    DMA_StartTransfer(handle->rx_handle);

    return QSPI_DMA_Status_Success;
}

void QSPI_DMAReadDescriptorInitRX(QSPI_Type *base, dma_descriptor_t *desc,
    uint32_t data_size, void *dst_addr, uint8_t dst_addr_incr)
{
    uint32_t desc_count = QSPI_GetReadDMADescriptorsCount(data_size);
    uint32_t src_addr_value = (uint32_t) & (base->RX_DATA);
    uint32_t dst_addr_value = (uint32_t) dst_addr;

    struct dma_channel_ctl_cfg ctlxcfg;
    memset(&ctlxcfg, 0, sizeof(struct dma_channel_ctl_cfg));
    ctlxcfg.src_incr = DMA_NoChange;
    ctlxcfg.dst_incr = dst_addr_incr;
    ctlxcfg.transfer_type = DMA_PeripheralToMemory_DMA;
    ctlxcfg.llp_dst_en = 1;
    ctlxcfg.llp_src_en = 1;

    uint64_t xfercfg = 0;

    if (desc_count == 1) {
        ctlxcfg.block_size = (data_size >> DMA_Transfer32BitWidth);
        xfercfg = DMA_GetCTLCfgMask(&ctlxcfg);
        DMA_SetupDescriptor(&desc[desc_count - 1], xfercfg,
            (void *) (src_addr_value), (void *) (dst_addr_value), NULL);
        return;
    }

    /* Проверка на остаток от деления на 4. */
    uint32_t rest = data_size & 3;
    if (rest) {
        /* Последний дескриптор многоблочной передачи будет передавать 1-3 байт. */
        ctlxcfg.block_size = rest;
        ctlxcfg.dst_tr_width = DMA_Transfer8BitWidth;
        ctlxcfg.src_tr_width = DMA_Transfer8BitWidth;
        xfercfg = DMA_GetCTLCfgMask(&ctlxcfg);
        DMA_SetupDescriptor(&desc[desc_count - 1], xfercfg, (void *) src_addr_value,
            (void *) (dst_addr_value + data_size - rest), NULL);
    }

    ctlxcfg.dst_tr_width = DMA_Transfer32BitWidth;
    ctlxcfg.src_tr_width = DMA_Transfer32BitWidth;

    data_size -= rest;
    uint32_t current_data_size = (data_size >> DMA_Transfer32BitWidth);

    /* Расчет объема каждого блока и адреса Приемника. */
    uint32_t end_index = rest ? desc_count - 1 : desc_count;
    for (uint32_t i = 0; i < end_index; i++) {
        if (current_data_size < DMA_AHB_MAX_BLOCK_SIZE) {
            ctlxcfg.block_size = current_data_size;
        } else {
            ctlxcfg.block_size = DMA_AHB_MAX_BLOCK_SIZE;
        }
        xfercfg = DMA_GetCTLCfgMask(&ctlxcfg);
        DMA_SetupDescriptor(&desc[i], xfercfg, (void *) src_addr_value,
            (void *) dst_addr_value, (i + 1 == desc_count) ? NULL : &desc[i + 1]);
        switch (dst_addr_incr) {
            case DMA_Incr:
                dst_addr_value += (ctlxcfg.block_size << DMA_Transfer32BitWidth);
                break;
            case DMA_Decr:
                dst_addr_value -= (ctlxcfg.block_size << DMA_Transfer32BitWidth);
                break;
            case DMA_NoChange:
                break;
            default:
                return;
        }
        current_data_size -= ctlxcfg.block_size;
    }
}

uint32_t QSPI_GetReadDMADescriptorsCount(uint32_t size_in_bytes)
{
    uint32_t block_count = 0;

    if (size_in_bytes % 4) {
        block_count += 1;
        size_in_bytes -= (size_in_bytes % 4);
    }

    uint32_t size = size_in_bytes >> 2;
    
    while (1) {
        if (size > DMA_AHB_MAX_BLOCK_SIZE) {
            size -= DMA_AHB_MAX_BLOCK_SIZE;
        } else {
            size = 0;
        }
        block_count += 1;
        if (size == 0) {
            break;
        }
    }

    return block_count;
}

/*!
 * @}
 */
