/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup qspi_driver
 * @{
 */

/*!
 * @file hal_qspi_nor_flash.c
 *
 * @brief Имплементация драйвера модуля QSPI. Взаимодействие с флеш-памятью NOR.
 */

#include "hal_common.h"
#include "hal_qspi_nor_flash.h"
#include "hal_nor_flash.h"
#include "hal_qspi_nor_jesd.h"

#if QSPI_USE_DMAC
#include "hal_qspi_dma.h"
#endif

#ifndef QSPI_UDELAY
#define QSPI_UDELAY(x)
#warning "WARNING! QSPI_UDELAY is not defined!"
#endif

#define QSPI_MAX_24BIT_ADDRESSING_SIZE (16UL * 1024 * 1024) /*!< Максимальный размер пространства с адресацией 24 бита */
#define QSPI_256K_SECTOR_SIZE_OFFSET   (18U)                /*!< Смещение сектора размером 256КБ */
#define NOR_SFDP_SIGNATURE             0x50444653           /*!< ASCII: "SFDP" */

enum {
    SerialFlash_ReadSFDP           = 0x5A, /*!< Стандартная команда чтения SFDP */
    SerialFlash_ReadManufacturerId = 0x9F, /*!< Стандартная команда чтения JEDEC ID */
};

/*!
 * @brief Контекст контроллера QSPI
 */
qspi_nor_handle_t qspi_handle;

#if QSPI_USE_DMAC
/*!
 * @brief Контекст передачи между QSPI и DMA
 */
qspi_dma_handle_t *qspi_dma_handle;
#endif

/*!
 * @brief Признак инициализации контроллера QSPI
 */
volatile bool is_qspi_initialized = false;

/*!
 * @brief Установка сигнала Chip Select в ноль
 *
 * @param base Базовый адрес контроллера QSPI
 */
static void QSPI_CSLow(QSPI_Type *base);

/*!
 * @brief Установка сигнала Chip Select в единицу
 *
 * @param base Базовый адрес контроллера QSPI
 */
static void QSPI_CSHigh(QSPI_Type *base);

/*!
 * @brief Очистка буфера Rx FIFO
 *
 * @param base Базовый адрес контроллера QSPI
 */
static void QSPI_FlushRxFIFO(QSPI_Type *base);

/*!
 * @brief Отправка данных в Tx FIFO
 *
 * @param base     Базовый адрес контроллера QSPI
 * @param tx_buff  Буфер памяти с данными для записи в Tx FIFO
 * @param tx_count Количество байт данных для записи в Tx FIFO
 */
static void QSPI_Write(QSPI_Type *base, void *tx_buff, uint32_t tx_count);

/*!
 * @brief Отправка Dummy Bytes в Tx FIFO
 *
 * @param base     Базовый адрес контроллера QSPI
 * @param tx_buff  Количество dummy байт на запись в Tx FIFO
 */
static void QSPI_Write_Dummy(QSPI_Type *base, uint32_t tx_count);

/*!
 * @brief Чтение данных из Rx FIFO
 *
 * @param base     Базовый адрес контроллера QSPI
 * @param rx_buff  Буфер памяти для чтения из Rx FIFO
 * @param rx_count Количество байт данных для чтения из Rx FIFO
 *
 * @return Количество считанных байт
 */
static uint32_t QSPI_Read(QSPI_Type *base, void *rx_buff, uint32_t rx_count);

/*!
 * @brief Отправка команды на чтение статусного регистра
 *
 * @param base     Базовый адрес контроллера QSPI
 * @param readCmd  Команда чтения статуса микросхемы флеш-памяти
 *
 * @return Значение статуса микросхемы флеш-памяти
 */
static uint8_t QSPI_NorReadStatus(QSPI_Type *base, uint8_t readCmd);

/*!
 * @brief Отправка команды Write Enable
 *
 * @param base           Базовый адрес контроллера QSPI
 * @param mem_nor_handle Контекст драйвера NOR Flash
 */
static void QSPI_NorWriteEnable(QSPI_Type *base,
    qspi_nor_handle_t *mem_nor_handle);

/*!
 * @brief Установка бита Quad Enable для режима QUAD через статусный регистр
 *
 * @param base           Базовый адрес контроллера QSPI
 * @param nor_config     Конфигурационный блок для режима Quad
 * @param mem_nor_handle Контекст драйвера NOR Flash
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_InvalidArgument
 */
static nor_status_t QSPI_NorQuadModeEnable(QSPI_Type *base,
    qspi_nor_config_t *nor_config, qspi_nor_handle_t *mem_nor_handle);

/*!
 * @brief Разбор параметров SFDP и заполнение конфигурационных структур qspi_nor
 *
 * @param nor_config      Конфигурационный блок для режима Quad
 * @param tbl             Таблица параметров стандарта JESD216
 * @param nor_init_config Первоначальная конфигурация QSPI
 * @param handle          Контекст драйвера NOR Flash
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_Fail
 */
static nor_status_t QSPI_NorParseSfdp(qspi_nor_config_t *nor_config,
    jedec_info_table_t *tbl, qspi_nor_init_config_t *nor_init_config,
    nor_handle_t *handle);

/*!
 * @brief Чтение параметров SFDP
 *
 * @param base Базовый адрес контроллера QSPI
 * @param tbl  Таблица параметров стандарта JESD216
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_InvalidArgument
 * @retval #NOR_Status_Fail
 */
static nor_status_t QSPI_NorReadSfdpInfo(QSPI_Type *base,
    jedec_info_table_t *tbl);

/*!
 * @brief Отправка команды чтения параметров SFDP
 *
 * @param base    Базовый адрес контроллера QSPI
 * @param address Смещение в таблице SFDP
 * @param buffer  Буфер для чтения таблицы SFDP
 * @param bytes   Количество байт вычитываемой таблицы SFDP
 *
 * @retval #NOR_Status_Success
 */
static nor_status_t QSPI_NorReadSfdp(QSPI_Type *base, uint32_t address,
    uint8_t *buffer, uint32_t bytes);

/*!
 * @brief Получение размера страницы/сектора из параметров SFDP
 *
 * @param handle Контекст драйвера NOR Flash
 * @param tbl    Таблица параметров стандарта JESD216
 *
 * @retval #NOR_Status_Success
 */
static nor_status_t QSPI_GetPageSectorSizeFromSfdp(nor_handle_t *handle,
    jedec_info_table_t *tbl);

/*!
 * @brief Получение максимального времени на стирание/программирование памяти из
 *        параметров SFDP
 *
 * @param handle Контекст драйвера NOR Flash
 * @param tbl    Таблица параметров стандарта JESD216
 *
 * @retval #NOR_Status_Success
 */
static nor_status_t QSPI_GetEraseProgTimingsFromSfdp(nor_handle_t *handle,
    jedec_info_table_t *tbl);

/*!
 * @brief Очистка сектора
 *
 * @param handle  Контекст драйвера NOR Flash
 * @param address Адрес сектора для очистки
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_Fail
 * @retval #NOR_Status_Timeout
 */
static nor_status_t QSPI_NorEraseSector(nor_handle_t *handle, uint32_t address);


static void QSPI_CSLow(QSPI_Type *base)
{
    /* Ожидание завершения всех передач. */
    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;

    /* Установка бита continuous transfer extend. */
    base->CTRL_AUX |= QSPI_CTRL_AUX_CONTXFEREXTEND_Msk;
    while (QSPI_CTRL_AUX_CONTXFEREXTEND_PTYPE_GET(base) == 0)
        ;
}

static void QSPI_CSHigh(QSPI_Type *base)
{
    /* Ожидание завершения всех передач. */
    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;

    /* Сброс бита continuous transfer extend. */
    base->CTRL_AUX &= ~(QSPI_CTRL_AUX_CONTXFEREXTEND_Msk);
    while (QSPI_CTRL_AUX_CONTXFEREXTEND_PTYPE_GET(base) != 0)
        ;
}

static void QSPI_FlushRxFIFO(QSPI_Type *base)
{
    while (QSPI_GetRXLVL(base) != 0)
        QSPI_ReadData(base);
}

static void QSPI_Write(QSPI_Type *base, void *tx_buff, uint32_t tx_count)
{
    uint8_t *p_tx_buff = (uint8_t *)tx_buff;
    QSPI_SetInhibitDin(base, 1);
    for (uint32_t i = 0; i < tx_count; i++) {
        QSPI_WriteDataByte(base, *p_tx_buff++);

        /* Ожидание завершения передачи. */
        while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
            || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
            ;
    }
    QSPI_SetInhibitDin(base, 0);
}

static void QSPI_Write_Dummy(QSPI_Type *base, uint32_t tx_count)
{
    QSPI_SetInhibitDin(base, 1);
    for (uint32_t i = 0; i < tx_count; i++) {
        QSPI_WriteDataByte(base, DUMMY_BYTE);
        /* Ожидание завершения всех передач. */
        while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
            || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
            ;
    }
    QSPI_SetInhibitDin(base, 0);
}

void QSPI_GetDefaultConfigXIP(qspi_xip_config_t *qspi_xip_config)
{
    assert(qspi_xip_config != NULL);

    (void) memset(qspi_xip_config, 0, sizeof(qspi_xip_config_t));

    qspi_xip_config->cmd          = NOR_CmdReadMemory;
    qspi_xip_config->hpen         = 0x0;
    qspi_xip_config->cpha         = 0x0;
    qspi_xip_config->cpol         = 0x0;
    qspi_xip_config->addr4        = 0x0;
    qspi_xip_config->le32         = 0x1;
    qspi_xip_config->hp_mode      = 0x0;
    qspi_xip_config->dummy_cycles = 0x0;
    qspi_xip_config->hp_end_dummy = 0x0;
}

void QSPI_GetDefaultCommandSet(nor_command_set_t *command_set)
{
    assert(command_set != NULL);

    (void) memset(command_set, 0, sizeof(nor_command_set_t));

    command_set->read_status_cmd       = NOR_CmdReadStatus;
    command_set->erase_sector_cmd      = NOR_CmdEraseSector;
    command_set->page_write_memory_cmd = NOR_CmdWriteMemory;
    command_set->write_disable_cmd     = NOR_CmdWriteDisable;
    command_set->write_enable_cmd      = NOR_CmdWriteEnable;
    command_set->write_status_cmd      = NOR_CmdWriteStatus;
    command_set->erase_chip_cmd        = NOR_CmdEraseChip;
    command_set->read_memory_command   = NOR_CmdReadMemory;
}

#if defined(QSPI_DRIVER_VERSION_1_6_0)
nor_status_t QSPI_ConfigureXIP(QSPI_XIP_Type *base,
    qspi_xip_config_t *qspi_xip_config)
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
nor_status_t QSPI_ConfigureXIP(QSPI_Type *base,
    qspi_xip_config_t *qspi_xip_config)
#else
    #error "Please define QSPI controller version"
#endif
{
    assert(qspi_xip_config != NULL);

    if (qspi_xip_config->hpen != 0) {
        if ((qspi_xip_config->cmd != NOR_CmdReadMemorySDR_1_2_2
                && qspi_xip_config->cmd != NOR_CmdReadMemorySDR_1_4_4)) {
            return NOR_Status_InvalidArgument;
        }
    }

#if defined(QSPI_DRIVER_VERSION_1_6_0)
    QSPI_XIP_XIP_CFG_CMD_PTYPE_SET(base, qspi_xip_config->cmd);
    QSPI_XIP_XIP_CFG_HPEN_PTYPE_SET(base, qspi_xip_config->hpen);
    QSPI_XIP_XIP_CFG_CPHA_PTYPE_SET(base, qspi_xip_config->cpha);
    QSPI_XIP_XIP_CFG_CPOL_PTYPE_SET(base, qspi_xip_config->cpol);
    QSPI_XIP_XIP_CFG_ADDR4_PTYPE_SET(base, qspi_xip_config->addr4);
    QSPI_XIP_XIP_CFG_LE32_PTYPE_SET(base, qspi_xip_config->le32);
    QSPI_XIP_XIP_CFG_HP_MODE_PTYPE_SET(base, qspi_xip_config->hp_mode);
    QSPI_XIP_XIP_CFG_DUMMY_CYCLES_PTYPE_SET(base, qspi_xip_config->dummy_cycles);
    QSPI_XIP_XIP_CFG_HP_END_DUMMY_PTYPE_SET(base, qspi_xip_config->hp_end_dummy);
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
    QSPI_XIPCFG_CMD_PTYPE_SET(base, qspi_xip_config->cmd);
    QSPI_XIPCFG_HPEN_PTYPE_SET(base, qspi_xip_config->hpen);
    QSPI_XIPCFG_CPHA_PTYPE_SET(base, qspi_xip_config->cpha);
    QSPI_XIPCFG_CPOL_PTYPE_SET(base, qspi_xip_config->cpol);
    QSPI_XIPCFG_ADDR4_PTYPE_SET(base, qspi_xip_config->addr4);
    QSPI_XIPCFG_LE32_PTYPE_SET(base, qspi_xip_config->le32);
    QSPI_XIPCFG_HPMODE_PTYPE_SET(base, qspi_xip_config->hp_mode);
    QSPI_XIPCFG_DUMMY_CYC_PTYPE_SET(base, qspi_xip_config->dummy_cycles);
    QSPI_XIPCFG_HP_END_DUMMY_PTYPE_SET(base, qspi_xip_config->hp_end_dummy);
#else
    #error "Please define QSPI controller version"
#endif

    return NOR_Status_Success;
}

nor_status_t QSPI_EnableXIP(nor_handle_t *handle)
{
    QSPI_Type *base = (QSPI_Type *) handle->driver_base_addr;
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    QSPI_XIP_Type *base_xip = handle->driver_xip_base_addr;
#endif

    /* Ожидание завершения всех передач. */
    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;

#if defined(QSPI_DRIVER_VERSION_1_6_0)
    if (QSPI_XIP_XIP_CFG_XIP_EN_PTYPE_GET(base_xip)) {
        return NOR_Status_Success;
    }
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
    if (QSPI_XIPCFG_XIP_EN_OUT_PTYPE_GET(base)
        && QSPI_XIPCFG_XIP_EN_REQ_PTYPE_GET(base)) {
        return NOR_Status_Success;
    }
#else
    #error "Please define QSPI controller version"
#endif

    QSPI_FlushRxFIFO(base);

    if (is_qspi_initialized == false) {
        QSPI_GetDefaultConfigXIP(&(handle->xip_config));
    }

#if defined(QSPI_DRIVER_VERSION_1_6_0)
    QSPI_ConfigureXIP(base_xip, &(handle->xip_config));

    QSPI_XIP_XIP_CFG_XIP_EN_PTYPE_SET(base_xip, 1);

    while (QSPI_XIP_XIP_CFG_XIP_EN_PTYPE_GET(base_xip) == 0)
        ;
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
    QSPI_ConfigureXIP(base, &(handle->xip_config));

    QSPI_XIPCFG_XIP_EN_REQ_PTYPE_SET(base, 1);

    while (QSPI_XIPCFG_XIP_EN_OUT_PTYPE_GET(base) == 0)
        ;

    /*
     * Два последовательных обращения к памяти для заполнения буфера.
     * Это необходимо для чтения памяти в XIP с нулевого адреса.
     */
    volatile uint8_t *xip_output8 = (volatile uint8_t *)(QSPI_XIP_MEM);
    (void) *xip_output8;
    (void) *(xip_output8 + 4);
#else
    #error "Please define QSPI controller version"
#endif

    return NOR_Status_Success;
}

void QSPI_DisableXIP(nor_handle_t *handle)
{
    QSPI_Type *base = (QSPI_Type *) handle->driver_base_addr;
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    QSPI_XIP_Type *base_xip = handle->driver_xip_base_addr;
#endif

#if defined(QSPI_DRIVER_VERSION_1_6_0)
    if (QSPI_XIP_XIP_CFG_XIP_EN_PTYPE_GET(base_xip) == 0) {
        return;
    }

    QSPI_XIP_XIP_CFG_XIP_EN_PTYPE_SET(base_xip, 0);

    while (QSPI_XIP_XIP_CFG_XIP_EN_PTYPE_GET(base_xip) != 0)
        ;
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
    if (QSPI_XIPCFG_XIP_EN_OUT_PTYPE_GET(base) == 0
        && QSPI_XIPCFG_XIP_EN_REQ_PTYPE_GET(base) == 0) {
        return;
    }

    QSPI_XIPCFG_XIP_EN_REQ_PTYPE_SET(base, 0);

    while (QSPI_XIPCFG_XIP_EN_OUT_PTYPE_GET(base) != 0)
        ;
#else
    #error "Please define QSPI controller version"
#endif

    QSPI_FlushRxFIFO(base);
}

static uint32_t QSPI_Read(QSPI_Type *base, void *rx_buff, uint32_t rx_count)
{
    uint8_t *p_rx_buff = (uint8_t *) rx_buff;
    uint32_t bytes_read = 0;

    while (QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        ;

    while (bytes_read < rx_count) {
        if (QSPI_GetStatusFlag(base) & QSPI_STAT_RXEMPTY_Msk) {
            QSPI_WriteData(base, DUMMY_BYTE);
            /* Ожидание завершения всех передач. */
            while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
                || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
                ;
        } else {
            /* Ожидание приема. */
            while (QSPI_GetRXLVL(base) == 0)
                ;
            p_rx_buff[bytes_read] = QSPI_ReadDataByte(base);
            bytes_read++;
        }
    }

    return bytes_read;
}

static uint8_t QSPI_NorReadStatus(QSPI_Type *base, uint8_t readCmd)
{
    uint8_t rdsr_cmd = readCmd;
    uint8_t flash_status = 0;

    QSPI_SetBitSize(base, QSPI_FRAME_BITS_8);

    QSPI_FlushRxFIFO(base);

    QSPI_CSLow(base);
    QSPI_Write(base, &rdsr_cmd, 1);
    QSPI_Read(base, &flash_status, 1);
    QSPI_CSHigh(base);

    return flash_status;
}

static void QSPI_NorWriteEnable(QSPI_Type *base, qspi_nor_handle_t *nor_handle)
{
    uint8_t wren_cmd = nor_handle->command_set.write_enable_cmd;

    QSPI_CSLow(base);
    QSPI_Write(base, &wren_cmd, 1);
    QSPI_CSHigh(base);

    while ((QSPI_NorReadStatus(base, nor_handle->command_set.read_status_cmd)
            & FLASH_STAT_WEL) == 0)
        ;
}

static nor_status_t QSPI_NorQuadModeEnable(QSPI_Type *base,
    qspi_nor_config_t *config, qspi_nor_handle_t *nor_handle)
{
    uint8_t quadValue;
    uint8_t valueAfter;

    if (config->quad_read_command == NOR_CmdInvalid) {
        return NOR_Status_InvalidArgument;
    }

    /* Чтение статусного регистра, который содержит бит Quad Enable. */
    quadValue = QSPI_NorReadStatus(base, config->quad_read_command);
    /* Проверка нужно ли записывать бит Quad Enable. */
    if (quadValue & (0x1U << config->quad_enable_bit_shift)) {
        return NOR_Status_Success;
    }

    do {
        QSPI_NorWriteEnable(base, nor_handle);

        if (config->write_two_status_bytes) {
            uint8_t valueStatus;
            uint16_t valueSwitch;

            valueStatus = QSPI_NorReadStatus(base,
                    nor_handle->command_set.read_status_cmd);
            valueSwitch = valueStatus
                | ((quadValue | (0x1U << config->quad_enable_bit_shift)) << 8);

            const uint32_t command_bytes = 3;
            uint8_t tx_buff[command_bytes];

            tx_buff[0] = nor_handle->command_set.write_status_cmd;
            tx_buff[1] = (uint8_t) valueSwitch & 0xFFU;
            tx_buff[2] = (uint8_t)((valueSwitch >> 8U) & 0xFFU);

            QSPI_CSLow(base);
            QSPI_Write(base, tx_buff, command_bytes);
            QSPI_CSHigh(base);

            while ((QSPI_NorReadStatus(base,
                        nor_handle->command_set.read_status_cmd) & valueSwitch)
                == 0)
                ;
        } else {
            const uint32_t command_bytes = 2;
            uint8_t tx_buff[command_bytes];

            tx_buff[0] = nor_handle->command_set.write_status_cmd;
            tx_buff[1] = quadValue | (0x1U << config->quad_enable_bit_shift);

            QSPI_CSLow(base);
            QSPI_Write(base, tx_buff, command_bytes);
            QSPI_CSHigh(base);

            while ((QSPI_NorReadStatus(base,
                        nor_handle->command_set.read_status_cmd)
                    & (0x1U << config->quad_enable_bit_shift)) == 0)
                ;
        }

        valueAfter = QSPI_NorReadStatus(base, config->quad_read_command);
    } while ((valueAfter & (0x1U << config->quad_enable_bit_shift)) == 0);

    return NOR_Status_Success;
}

static nor_status_t QSPI_NorEraseSector(nor_handle_t *handle, uint32_t address)
{
    if ((!handle) || (!handle->device_specific)) {
        return NOR_Status_InvalidArgument;
    }

    qspi_nor_handle_t *qspi_nor_handle =
        (qspi_nor_handle_t *) handle->device_specific;
    QSPI_Type *base = (QSPI_Type *) handle->driver_base_addr;
    uint8_t opcode = qspi_nor_handle->command_set.erase_sector_cmd;
    address &= ~(handle->sector_size_bytes - 1);
    qspi_command_type_t cmd_type = qspi_nor_handle->command_type;
    const uint32_t max_address_size = 4;
    uint8_t address_data[max_address_size];

    QSPI_NorWriteEnable(base, qspi_nor_handle);

    address_data[0] = (address >> 24U) & 0xFFU;
    address_data[1] = (address >> 16U) & 0xFFU;
    address_data[2] = (address >> 8U)  & 0xFFU;
    address_data[3] = (address)        & 0xFFU;

    /* Выполнение операции очистки сектора. */
    QSPI_CSLow(base);
    QSPI_Write(base, &opcode, 1);
    if (cmd_type == QSPI_CommandOpcodeAddrThreeBytes) {
        QSPI_Write(base, &address_data[1], 3);
    } else if (cmd_type == QSPI_CommandOpcodeAddrFourBytes) {
        QSPI_Write(base, address_data, 4);
    } else {
        QSPI_CSHigh(base);
        return NOR_Status_Fail;
    }
    QSPI_CSHigh(base);

    /* Ожидание завершения стирания. */
    uint8_t read_cmd = qspi_nor_handle->command_set.read_status_cmd;
    uint32_t timeout = handle->max_sector_erase_time;
    while ((QSPI_NorReadStatus(base, read_cmd) & FLASH_STAT_BUSY)
        && timeout != 0) {
        QSPI_UDELAY(1000);
        if (timeout >= 1U) {
            timeout -= 1;
        } else {
            timeout = 0;
        }
    }

    return (timeout) ? NOR_Status_Success : NOR_Status_Timeout;
}

nor_status_t QSPI_NorFlashInit(nor_config_t *config, nor_handle_t *handle)
{
    assert(config);
    assert(handle);

    QSPI_Type *base = (QSPI_Type *) config->driver_base_addr;
    nor_status_t status;
    jedec_info_table_t jedec_info_tbl;
    qspi_nor_init_config_t *nor_init_config =
        (qspi_nor_init_config_t *) config->mem_control_config;
    qspi_nor_config_t qspi_nor_config, *nor_config = 
        (qspi_nor_config_t *) config->quad_control_config;

    QSPI_GetDefaultConfig(&(handle->qspi_config));
    QSPI_GetDefaultConfigXIP(&(handle->xip_config));

    /* Установка параметров QSPI NOR по умолчанию. */
    handle->driver_base_addr  = config->driver_base_addr;
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    handle->driver_xip_base_addr  = config->driver_xip_base_addr;
#endif
    handle->memory_size_bytes = QSPI_MAX_24BIT_ADDRESSING_SIZE;

    if (handle->device_specific == NULL) {
        memset(&qspi_handle, 0, sizeof(qspi_handle));
        handle->device_specific = (void *) &qspi_handle;
        QSPI_GetDefaultCommandSet(&qspi_handle.command_set);
    }

    QSPI_Init(base, &(handle->qspi_config));

    status = QSPI_NorReadSfdpInfo(base, &jedec_info_tbl);
    if (status == NOR_Status_Success) {
        if (nor_config == NULL) {
            memset(&qspi_nor_config, 0, sizeof(qspi_nor_config_t));
            nor_config = &qspi_nor_config;
        }
        status = QSPI_NorParseSfdp(nor_config, &jedec_info_tbl,
            nor_init_config, handle);
        if (status != NOR_Status_Success) {
            return status;
        }
    } else if (nor_config == NULL) {
        return NOR_Status_Fail;
    }

    if (nor_config->is_quad_need_enable) {
        status = QSPI_NorQuadModeEnable(base, nor_config, handle->device_specific);
        if (status != NOR_Status_Success) {
            return status;
        }
    }

    is_qspi_initialized = true;
    return NOR_Status_Success;
}

static nor_status_t QSPI_NorReadSfdp(QSPI_Type *base, uint32_t address,
    uint8_t *buffer, uint32_t bytes)
{
    uint32_t opcode = SerialFlash_ReadSFDP;
    const uint32_t adress_bytes_num = 3;
    uint8_t address_data[adress_bytes_num];

    address_data[0] = address >> 16U;
    address_data[1] = address >> 8U;
    address_data[2] = address;

    QSPI_FlushRxFIFO(base);

    QSPI_CSLow(base);

    QSPI_Write(base, &opcode, 1);
    QSPI_Write(base, address_data, adress_bytes_num);
    QSPI_Write_Dummy(base, 1);

    QSPI_Read(base, buffer, bytes);
    QSPI_CSHigh(base);

    return NOR_Status_Success;
}

static nor_status_t QSPI_NorReadSfdpInfo(QSPI_Type *base,
    jedec_info_table_t *tbl)
{
    nor_status_t status = NOR_Status_Fail;
    do {
        if (tbl == NULL) {
            status = NOR_Status_InvalidArgument;
            break;
        }

        sfdp_header_t sfdp_header;
        uint32_t address;

        status = QSPI_NorReadSfdp(base, 0, (uint8_t *) &sfdp_header,
                sizeof(sfdp_header));
        if (status != NOR_Status_Success) {
            break;
        }

        if (sfdp_header.signature != NOR_SFDP_SIGNATURE) {
            status = NOR_Status_Fail;
            break;
        }

        tbl->standard_version = sfdp_header.minor_rev;

        uint32_t parameter_header_number = sfdp_header.param_hdr_num + 1;
        sfdp_parameter_header_t sfdp_param_hdrs[10];
        uint32_t max_hdr_count = parameter_header_number > 10 ? 10
            : parameter_header_number;
        address = 0x08;

        status = QSPI_NorReadSfdp(base, address,
            (uint8_t *) &sfdp_param_hdrs[0],
            max_hdr_count * sizeof(sfdp_parameter_header_t));
        if (status != NOR_Status_Success) {
            break;
        }
        memset(tbl, 0, sizeof(*tbl));

        for (uint32_t i = 0; i < max_hdr_count; i++) {
            uint32_t parameter_id =
                sfdp_param_hdrs[i].parameter_id_lsb
                + ((uint32_t) sfdp_param_hdrs[i].parameter_id_msb << 8);

            if ((parameter_id == ParameterID_BasicSpiProtocol) ||
                (parameter_id == ParameterID_4ByteAddressInstructionTable)) {
                address = 0;
                for (int32_t index = 2; index >= 0; index--) {
                    address <<= 8;
                    address |= sfdp_param_hdrs[i].parameter_table_pointer[index];
                }
                uint32_t table_size = sfdp_param_hdrs[i].table_length_in_32bit
                    * sizeof(uint32_t);

                if (parameter_id == ParameterID_BasicSpiProtocol) {
                    /* Ограничение таблицы поддерживаемым стандартом. */
                    if (table_size > sizeof(jedec_flash_param_table_t)) {
                        table_size = sizeof(jedec_flash_param_table_t);
                    }
                    /* Чтение базовой таблицы протокола SPI. */
                    status = QSPI_NorReadSfdp(base, address,
                            (uint8_t *) &tbl->flash_param_tbl, table_size);
                    if (status != NOR_Status_Success) {
                        break;
                    }
                    tbl->flash_param_tbl_size = table_size;
                } else {
                    /* Чтение 4-байтовой адресной таблицы инструкций. */
                    status = QSPI_NorReadSfdp(base, address,
                            (uint8_t *) &tbl->flash_4b_inst_tbl, table_size);
                    if (status != NOR_Status_Success) {
                        break;
                    }
                    tbl->has_4b_addressing_inst_table = true;
                }
            } else {
                /* Неподдерживаемый тип параметра */
            }
        }

    } while (0);

    return status;
}

static nor_status_t QSPI_GetEraseProgTimingsFromSfdp(nor_handle_t *handle,
    jedec_info_table_t *tbl)
{
    jedec_flash_param_table_t *param_tbl = &tbl->flash_param_tbl;
    const uint32_t sector_types_num = 4;
    uint32_t typ_sector_erase_time = 0;
    uint32_t typ_chip_erase_time = 0;
    uint32_t typ_page_prog_time = 0;
    uint32_t units, count, mult, index;
    uint32_t sector_erase_timings[sector_types_num];
    sector_erase_timings[0] = param_tbl->sector_erase_timing.sector_type_1_typ;
    sector_erase_timings[1] = param_tbl->sector_erase_timing.sector_type_2_typ;
    sector_erase_timings[2] = param_tbl->sector_erase_timing.sector_type_3_typ;
    sector_erase_timings[3] = param_tbl->sector_erase_timing.sector_type_4_typ;

    /* Вычисление времени на очистку сектора. */
    for (index = 0; index < sector_types_num; index++) {
        if (param_tbl->erase_info[index].size == 0) {
            continue;
        }

        units = (sector_erase_timings[index] >> 5U) & 0x3U;
        count = sector_erase_timings[index] & 0x1F;

        switch (units) {
            case ParameterID_SectorErase_1ms:
                sector_erase_timings[index] = (count + 1U) * 1U;
                break;
            case ParameterID_SectorErase_16ms:
                sector_erase_timings[index] = (count + 1U) * 16U;
                break;
            case ParameterID_SectorErase_128ms:
                sector_erase_timings[index] = (count + 1U) * 128U;
                break;
            case ParameterID_SectorErase_1s:
                sector_erase_timings[index] = (count + 1U) * 1000U;
                break;
        }
    }

    /* Выбор максимального значения из всех типичных. */
    for (index = 0; index < sector_types_num; index++) {
        typ_sector_erase_time = MAX(typ_sector_erase_time,
                sector_erase_timings[index]);
    }

    /* Вычисление максимального времени стирания сектора. */
    mult = param_tbl->sector_erase_timing.mult_typ_to_max_erase;
    handle->max_sector_erase_time = 2 * (mult + 1) * typ_sector_erase_time;

    /* Вычисление максимального времени стирания флеш-памяти. */
    typ_chip_erase_time = param_tbl->chip_erase_progrm_info.chip_erase_typ;
    units = (typ_chip_erase_time >> 5U) & 0x1U;
    count = typ_chip_erase_time & 0x1F;
    switch (units) {
        case ParameterID_ChipErase_16ms:
            typ_chip_erase_time = (count + 1U) * 16U;
            break;
        case ParameterID_ChipErase_256ms:
            typ_chip_erase_time = (count + 1U) * 256U;
            break;
        case ParameterID_ChipErase_4s:
            typ_chip_erase_time = (count + 1U) * 4000U;
            break;
        case ParameterID_ChipErase_64s:
            typ_chip_erase_time = (count + 1U) * 64000U;
            break;
    }
    handle->max_chip_erase_time = 2 * (mult + 1) * typ_chip_erase_time;

    /* Вычисление максимального времени программирования страницы. */
    typ_page_prog_time = param_tbl->chip_erase_progrm_info.page_prog_typ;
    units = (typ_page_prog_time >> 5U) & 0x1U;
    count = typ_page_prog_time & 0x1F;

    switch (units) {
        case ParameterID_PageProg_8us:
            typ_page_prog_time = (count + 1U) * 8U;
            break;
        case ParameterID_PageProg_64us:
            typ_page_prog_time = (count + 1U) * 64U;
            break;
    }

    mult = param_tbl->chip_erase_progrm_info.mult_typ_to_max_prog;
    handle->max_page_program_time = 2 * (mult + 1) * typ_page_prog_time;

    return NOR_Status_Success;
}

static nor_status_t QSPI_GetPageSectorSizeFromSfdp(nor_handle_t *handle,
    jedec_info_table_t *tbl)
{
    jedec_flash_param_table_t *param_tbl = &tbl->flash_param_tbl;
    jedec_4byte_addressing_inst_table_t *flash_4b_tbl = &tbl->flash_4b_inst_tbl;
    qspi_nor_handle_t *nor_handle =
        (qspi_nor_handle_t *) handle->device_specific;

    uint32_t flash_size;
    uint32_t flash_density = tbl->flash_param_tbl.flash_density;

    if (flash_density & (1U << 0x1F)) {
        /* Размер флеш-памяти >= 4Гбит. */
        flash_size = 1U << ((flash_density & ~(1U << 0x1F)) - 3);
    } else {
        /* Размер флеш-памяти < 4Гбит. */
        flash_size = (flash_density + 1) >> 3;
    }
    handle->memory_size_bytes = flash_size;

    /* Вычисление размера страницы. */
    uint32_t page_size;
    if (tbl->flash_param_tbl_size < 64U) {
        handle->page_size_bytes = 256U;
    } else {
        page_size = 1 << (param_tbl->chip_erase_progrm_info.page_size);
        handle->page_size_bytes = page_size == (1 << 15) ? 256U : page_size;
    }

    /* Вычисление размера сектора. */
    uint32_t sector_size       = 1U << QSPI_256K_SECTOR_SIZE_OFFSET;
    uint32_t sector_erase_type = 0;

    for (uint32_t index = 0; index < 4; index++) {
        if (param_tbl->erase_info[index].size != 0) {
            uint32_t current_erase_size = 1U
                << param_tbl->erase_info[index].size;
            if ((current_erase_size < sector_size)
                && (current_erase_size < (1024U * 1024U))) {
                sector_size = current_erase_size;
                sector_erase_type = index;
            }
        }
    }

    handle->sector_size_bytes = sector_size;

    if (handle->memory_size_bytes > QSPI_MAX_24BIT_ADDRESSING_SIZE) {
        if (tbl->has_4b_addressing_inst_table) {
            nor_handle->command_set.erase_sector_cmd
                = flash_4b_tbl->erase_inst_info.erase_inst[sector_erase_type];
        } else {
            switch (param_tbl->erase_info[sector_erase_type].inst) {
                case NOR_CmdEraseSector4KB:
                    nor_handle->command_set.erase_sector_cmd
                        = NOR_CmdEraseSector4KBA32;
                    break;
                case NOR_CmdEraseSector:
                    nor_handle->command_set.erase_sector_cmd
                        = NOR_CmdEraseSectorA32;
                    break;
            }
        }
    } else {
        nor_handle->command_set.erase_sector_cmd
            = param_tbl->erase_info[sector_erase_type].inst;
    }

    return NOR_Status_Success;
}

static nor_status_t QSPI_NorParseSfdp(qspi_nor_config_t *config,
    jedec_info_table_t *tbl, qspi_nor_init_config_t *nor_init_config,
    nor_handle_t *handle)
{
    qspi_nor_handle_t *nor_handle
        = (qspi_nor_handle_t *) handle->device_specific;

    jedec_flash_param_table_t *param_tbl = &tbl->flash_param_tbl;
    jedec_4byte_addressing_inst_table_t *flash_4b_tbl
        = &tbl->flash_4b_inst_tbl;
    uint8_t quadModeSetting = NOR_QuadModeNotConfig;
    uint32_t dummy_cycles = 0;
    uint8_t mode_cycles = 0;
    uint32_t address_bits = 24U;

    QSPI_GetPageSectorSizeFromSfdp(handle, tbl);
    QSPI_GetEraseProgTimingsFromSfdp(handle, tbl);

    if (param_tbl->misc.address_bytes == 2U) {
        nor_handle->command_type = QSPI_CommandOpcodeAddrFourBytes;
        handle->xip_config.addr4 = 1;
    } else {
        nor_handle->command_type = QSPI_CommandOpcodeAddrThreeBytes;
        handle->xip_config.addr4 = 0;
    }

    if (handle->memory_size_bytes > QSPI_MAX_24BIT_ADDRESSING_SIZE) {
        address_bits = 32U;
        nor_handle->command_set.page_write_memory_cmd
            = NOR_CmdWriteMemoryA32;
        nor_handle->command_set.read_memory_command = handle->xip_config.cmd
            = NOR_CmdReadMemoryA32;
        nor_handle->command_type = QSPI_CommandOpcodeAddrFourBytes;
        handle->xip_config.addr4 = 1;
    }

    nor_handle->read_cmd_format
        = (qspi_command_format_t) nor_init_config->cmd_format;

    if (nor_handle->read_cmd_format != QSPI_CommandAllSerial) {
        if (nor_handle->read_cmd_format == QSPI_CommandOpcodeSerial
                && param_tbl->misc.supports_1_4_4_fast_read) {
            mode_cycles  = param_tbl->read_1_4_info.mode_clocks_1_4_4_read;
            dummy_cycles = param_tbl->read_1_4_info.dummy_clocks_1_4_4_read;
            /* Вычисление промежуточных байтов. */
            nor_handle->intermediate_len = (mode_cycles + dummy_cycles) / 2U;
            handle->xip_config.dummy_cycles
                = (mode_cycles + dummy_cycles - 2U) / 2U;
            nor_handle->command_set.read_memory_command = handle->xip_config.cmd
                = param_tbl->read_1_4_info.inst_1_4_4_read;
            if ((address_bits == 32U)
                && flash_4b_tbl->cmd_4byte_support_info.support_1_4_4_fast_read) {
                nor_handle->command_set.read_memory_command
                    = NOR_CmdReadMemorySDR_1_4_4_A32;
            }
        } else if (nor_handle->read_cmd_format == QSPI_CommandDataQuad
            && param_tbl->misc.support_1_1_4_fast_read) {
            mode_cycles  = param_tbl->read_1_4_info.mode_clocks_1_1_4_read;
            dummy_cycles = param_tbl->read_1_4_info.dummy_clocks_1_1_4_read;
            /* Вычисление промежуточных байтов. */
            nor_handle->intermediate_len = handle->xip_config.dummy_cycles
                = (mode_cycles + dummy_cycles) / 8U;
            nor_handle->command_set.read_memory_command = handle->xip_config.cmd
                = param_tbl->read_1_4_info.inst_1_1_4_read;
            if ((address_bits == 32U)
                && flash_4b_tbl->cmd_4byte_support_info.support_1_1_4_fast_read)
            {
                nor_handle->command_set.read_memory_command
                    = handle->xip_config.cmd
                    = NOR_CmdReadMemorySDR_1_1_4_A32;
            }
        } else {
            return NOR_Status_Fail;
        }

        if ((tbl->standard_version >= Sfdp_Version_Minor_A) ||
            (tbl->flash_param_tbl_size >= Sfdp_BasicProtocolTableSize_RevA)) {
            switch (param_tbl->mode_4_4_info.quad_enable_requirement) {
                case ParameterID_QER_Bit1_SR2:
                case ParameterID_QER_Bit1_SR2_NoMod:
                case ParameterID_QER_Bit1_SR2_0x01:
                    quadModeSetting = NOR_QuadModeStatusReg2_Bit1;
                    break;
                case ParameterID_QER_Bit6_SR1:
                    quadModeSetting = NOR_QuadModeStatusReg1_Bit6;
                    break;
                case ParameterID_QER_Bit7_SR2:
                    quadModeSetting = NOR_QuadModeStatusReg2_Bit7;
                    break;
                case ParameterID_QER_Bit1_SR2_0x31:
                    quadModeSetting = NOR_QuadModeStatusReg2_Bit1_0x31;
                    break;
                case ParameterID_QER_None:
                default:
                    quadModeSetting = NOR_QuadModeNotConfig;
                    break;
            }
        } else {
            /*
             * Использование настроек Quad режима из конфигурационного блока.
             * Этот параметр используется для микросхем флеш-памяти, которые
             * поддерживают только начальные версии стадарта JESD216.
             */
            quadModeSetting = nor_init_config->quad_mode_setting;
        }
        switch (quadModeSetting) {
            /*
             * Необходима запись в статусные регистры 1 и 2
             * с помощью команды 0x01. Дополнительные сведения
             * см. в документации JESD216B.
             */
            case NOR_QuadModeStatusReg2_Bit1:
                config->is_quad_need_enable    = true;
                config->write_two_status_bytes = true;
                config->quad_read_command      = NOR_CmdReadSecStatus_35;
                config->quad_enable_bit_shift  = 1;
                break;
            /*
             * Необходима запись в статусный регистр 1 с помощью команды 0x01.
             */
            case NOR_QuadModeStatusReg1_Bit6:
                config->is_quad_need_enable    = true;
                config->write_two_status_bytes = false;
                config->quad_read_command      = NOR_CmdReadStatus;
                config->quad_enable_command    = NOR_CmdWriteStatus;
                config->quad_enable_bit_shift  = 6;
                break;
            /*
             * Необходима запись в статусный регистр 2 с помощью команды 0x3E.
             */
            case NOR_QuadModeStatusReg2_Bit7:
                config->is_quad_need_enable    = true;
                config->write_two_status_bytes = false;
                config->quad_read_command      = NOR_CmdReadSecStatus_3F;
                config->quad_enable_command    = NOR_CmdWriteSecStatus_3E;
                config->quad_enable_bit_shift  = 7;
                break;
            /*
             * Необходима запись в статусный регистр 2 с помощью команды 0x31.
             */
            case NOR_QuadModeStatusReg2_Bit1_0x31:
                config->is_quad_need_enable    = true;
                config->write_two_status_bytes = false;
                config->quad_read_command      = NOR_CmdReadSecStatus_35;
                config->quad_enable_command    = NOR_CmdWriteSecStatus_31;
                config->quad_enable_bit_shift  = 1;
                break;
            case NOR_QuadModeNotConfig:
            default:
                config->is_quad_need_enable = false;
                break;
        }
    }

    return NOR_Status_Success;
}

nor_status_t NOR_FlashPageProgram(nor_handle_t *handle, uint32_t address,
    uint8_t *buffer, uint32_t length)
{
    if (!buffer) {
        return NOR_Status_InvalidArgument;
    }

    uint32_t data, i;
    qspi_nor_handle_t *qspi_nor_handle
        = (qspi_nor_handle_t *) handle->device_specific;
    uint32_t pageSize = handle->page_size_bytes;
    QSPI_Type *base = (QSPI_Type *) handle->driver_base_addr;
    uint32_t opcode = qspi_nor_handle->command_set.page_write_memory_cmd;
    qspi_command_type_t cmd_type = qspi_nor_handle->command_type;
    const uint32_t max_address_size = 4;
    uint8_t address_data[max_address_size];

    if ((length == 0) || (length > pageSize)) {
        return NOR_Status_InvalidArgument;
    }

    QSPI_NorWriteEnable(base, qspi_nor_handle);

    /* Выравнивание адреса. */
    address_data[0] = (address >> 24U) & 0xFFU;
    address_data[1] = (address >> 16U) & 0xFFU;
    address_data[2] = (address >> 8U)  & 0xFFU;
    address_data[3] = (address)        & 0xFFU;

    QSPI_SetQMode(base, QSPI_NormalSPI);

    QSPI_CSLow(base);

    QSPI_Write(base, &opcode, 1);
    if (cmd_type == QSPI_CommandOpcodeAddrThreeBytes) {
        QSPI_Write(base, &address_data[1], 3);
    } else if (cmd_type == QSPI_CommandOpcodeAddrFourBytes) {
        QSPI_Write(base, address_data, 4);
    } else {
        QSPI_CSHigh(base);
        return NOR_Status_Fail;
    }

    QSPI_SetBitSize(base, QSPI_FRAME_BITS_32);
    for (i = 0; i < length; i += sizeof(uint32_t)) {
        data = *(uint32_t *) buffer;
        data = BE_TO_LE32(data);
        buffer += sizeof(uint32_t);
        QSPI_WriteData(base, data);
        /* Ожидание завершения всех передач. */
        while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
            || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
            ;
        /* Ожидание невостребованного слова от ведомого. */
        while (QSPI_GetRXLVL(base) == 0)
            ;
        QSPI_ReadData(base);
    }

    QSPI_CSHigh(base);

    /* Ожидание завершения записи. */
    QSPI_SetBitSize(base, QSPI_FRAME_BITS_8);
    uint8_t read_cmd = qspi_nor_handle->command_set.read_status_cmd;
    uint32_t timeout = handle->max_page_program_time;
    uint8_t flash_status = QSPI_NorReadStatus(base, read_cmd);
    while ((flash_status & FLASH_STAT_BUSY)
        && timeout != 0) {
        QSPI_UDELAY(10);
        if (timeout >= 10U) {
            timeout -= 10;
        } else {
            timeout = 0;
        }
        flash_status = QSPI_NorReadStatus(base, read_cmd);
    }

    return (timeout) ? NOR_Status_Success : NOR_Status_Timeout;
}

nor_status_t NOR_FlashEraseBlock(nor_handle_t *handle, uint32_t address,
    uint32_t size)
{
    uint32_t endAddress = address + size;
    nor_status_t status = NOR_Status_Fail;

    if (endAddress > handle->memory_size_bytes) {
        return NOR_Status_InvalidArgument;
    }

    address &= ~(handle->sector_size_bytes - 1);

    while (address < endAddress) {
        status = QSPI_NorEraseSector(handle, address);
        if (status != NOR_Status_Success) {
            return status;
        }
        address += handle->sector_size_bytes;
    }

    return status;
}

nor_status_t NOR_FlashEraseChip(nor_handle_t *handle)
{
    if ((!handle) || (!handle->device_specific)) {
        return NOR_Status_InvalidArgument;
    }

    qspi_nor_handle_t *qspi_nor_handle =
        (qspi_nor_handle_t *) handle->device_specific;
    QSPI_Type *base = (QSPI_Type *) handle->driver_base_addr;
    uint32_t opcode = qspi_nor_handle->command_set.erase_chip_cmd;

    QSPI_NorWriteEnable(base, qspi_nor_handle);

    QSPI_CSLow(base);
    QSPI_Write(base, &opcode, 1);
    QSPI_CSHigh(base);

    while (QSPI_NorReadStatus(base,
            qspi_nor_handle->command_set.read_status_cmd) & FLASH_STAT_BUSY)
        ;
    /* Ожидание завершения стирания. */
    uint8_t read_cmd = qspi_nor_handle->command_set.read_status_cmd;
    uint32_t timeout = handle->max_chip_erase_time;
    while ((QSPI_NorReadStatus(base, read_cmd) & FLASH_STAT_BUSY)
        && timeout != 0) {
        QSPI_UDELAY(100000);
        if (timeout >= 100U) {
            timeout -= 100;
        } else {
            timeout = 0;
        }
    }

    return (timeout) ? NOR_Status_Success : NOR_Status_Timeout;
}

nor_status_t NOR_FlashRead(nor_handle_t *handle, uint32_t address,
    uint8_t *buffer, uint32_t length)
{
    if ((!handle) || (!buffer)) {
        return NOR_Status_InvalidArgument;
    }

    qspi_nor_handle_t *qspi_nor_handle
        = (qspi_nor_handle_t *) handle->device_specific;
    QSPI_Type *base = (QSPI_Type *) handle->driver_base_addr;
    uint32_t opcode = qspi_nor_handle->command_set.read_memory_command;
    qspi_command_type_t cmd_type = qspi_nor_handle->command_type;
    const uint32_t max_address_size = 4;
    uint8_t address_data[max_address_size];

    address_data[0] = (address >> 24U) & 0xFFU;
    address_data[1] = (address >> 16U) & 0xFFU;
    address_data[2] = (address >> 8U)  & 0xFFU;
    address_data[3] = (address)        & 0xFFU;

    QSPI_FlushRxFIFO(base);

    QSPI_CSLow(base);

    QSPI_SetQMode(base, QSPI_NormalSPI);
    QSPI_Write(base, &opcode, 1);

    /* Ожидание завершения всех передач. */
    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;

    /* Отправка адреса, Mode bits и Dummy bytes. */
    if (qspi_nor_handle->read_cmd_format == QSPI_CommandOpcodeSerial) {
        QSPI_SetQMode(base, QSPI_QuadSPI);
    }

    /* Отправка адреса. */
    if (cmd_type == QSPI_CommandOpcodeAddrThreeBytes) {
        QSPI_Write(base, &address_data[1], 3);
    } else if (cmd_type == QSPI_CommandOpcodeAddrFourBytes) {
        QSPI_Write(base, address_data, 4);
    } else {
        QSPI_CSHigh(base);
        return NOR_Status_Fail;
    }
    /* Отправка промежуточных байтов. */
    QSPI_Write_Dummy(base, qspi_nor_handle->intermediate_len);

    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;

    /* Чтение данных. */
    if (qspi_nor_handle->read_cmd_format == QSPI_CommandDataQuad) {
        QSPI_SetQMode(base, QSPI_QuadSPI);
    }

#if QSPI_USE_DMAC
    if (qspi_dma_handle != NULL && qspi_dma_handle->rx_handle != NULL
        && qspi_dma_handle->tx_handle != NULL) {
        while (QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk);
        QSPI_SetBitSize(base, QSPI_FRAME_BITS_32);
        QSPI_ReadDataDMA(qspi_dma_handle, buffer, DMA_Incr,
            DMA_Transfer8BitWidth, length);
        QSPI_WriteDataDMA(qspi_dma_handle, qspi_dma_handle->dummy_data,
            DMA_NoChange, DMA_Transfer8BitWidth, length);
        while (DMA_ChannelIsActive(qspi_dma_handle->rx_handle->base,
                qspi_dma_handle->rx_handle->channel));
    } else {
        QSPI_Read(base, buffer, length);
    }
#else
    QSPI_Read(base, buffer, length);
#endif

    QSPI_CSHigh(base);

    QSPI_SetQMode(base, QSPI_NormalSPI);

#if QSPI_USE_DMAC
    QSPI_FlushRxFIFO(base);
#endif

    return NOR_Status_Success;
}

void NOR_FlashReadXIP(uint32_t address, uint8_t *buffer, uint32_t length)
{
    uint32_t *p_buffer32 = (uint32_t *) buffer;
    volatile uint32_t *xip_output32
        = (volatile uint32_t *)(QSPI_XIP_MEM + address);
    for (uint32_t i = 0; i < length / sizeof(uint32_t); i++) {
        *p_buffer32++ = *xip_output32++;
    }
}

nor_status_t NOR_FlashProgramBlock(nor_handle_t *handle,
    uint32_t address, uint8_t *page_pointer, uint32_t length)
{
    uint32_t curr_length, page_end_addr;
    uint32_t page_size = handle->page_size_bytes;
    uint32_t end_address = address + length;
    nor_status_t status = NOR_Status_Success;

    while (address < end_address) {
        curr_length = (page_size < length) ? page_size : length;
        page_end_addr = (address & ~(page_size - 1)) + page_size;
        if ((address + curr_length) > page_end_addr) {
            curr_length = page_end_addr - address;
        }
        status = NOR_FlashPageProgram(handle, address, page_pointer, curr_length);
        if (status != NOR_Status_Success) {
            return status;
        }
        address += curr_length;
        page_pointer += curr_length;
        length -= curr_length;
    }
    return status;
}
#if QSPI_USE_DMAC
void QSPI_SetDMAHandle(qspi_dma_handle_t *handle)
{
    assert(handle != NULL);

    qspi_dma_handle = handle;
}
#endif

/*!
 * @}
 */
