/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup rwc_driver Драйвер модуля RWC
 *
 * @brief Драйвер счетчика реального времени и Wake-контроллера
 *
 * @note Драйвер модуля RWC управляет счетчиком реального времени и
 *       Wake-контроллером.
 */

/*!
 * @addtogroup rwc_driver
 * @{
 */

/*!
 * @file hal_rwc.h
 *
 * @brief Интерфейс драйвера модуля RWC
 */

#ifndef HAL_RWC_H
#define HAL_RWC_H

#include "hal_common.h"

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Количество циклов ожидания
 * @note 0 - максимальное ожидание.
 *       Если необходимо бесконечное ожидание,
 *       закомментируйте этот макрос
 */
#ifndef RWC_SYNC_RETRY_TIMES
#define RWC_SYNC_RETRY_TIMES 0U
#endif /* RWC_SYNC_RETRY_TIMES */

/*!
 * @brief Количество циклов установки времени
 */
#define RWC_SET_RETRY_TIMES 10U

/*!
 * @name Макросы для операций перевода времени
 * @{
 */
#define SECONDS_IN_A_DAY    (86400U) /*!< Секунд в одном дне */
#define SECONDS_IN_A_HOUR   (3600U)  /*!< Секунд в одном часе */
#define SECONDS_IN_A_MINUTE (60U)    /*!< Секунд в одной минуте */
#define DAYS_IN_A_YEAR      (365U)   /*!< Дней в одном году */
#define YEAR_RANGE_START    (1970U)  /*!< Начальный год */
#define YEAR_RANGE_END      (2099U)  /*!< Максимально допустимый год */

/*!
 * @}
 */

/*!
 * @name Описание полей внутренних регистров
 * @{
 */

/*!
 * @brief Регистр записи значения подстройки из регистра TRIM
 *
 * @note Запись в этот регистр применяет значение подстройки в регистре TRIM
 */
struct rwc_trimload_reg {
uint32_t trimload :
    FIELD_BIT(0, 0);  /*!< Поле для инициации записи */
uint32_t          :
    FIELD_BIT(31, 1); /*!< Резерв */
};

/*!
 * @brief Регистр текущего значения счетчика времени
 *
 * @note Запись устанавливает значение счетчика времени.
 *       Чтение возвращает текущее значение счетчика.
 */
struct rwc_time_reg {
uint32_t time :
    FIELD_BIT(31, 0); /*!< Значение времени в тиках */
};

/*!
 * @brief Регистр времени пробуждения
 *
 * Значение времени пробуждения, сравниваемое с регистром TIME.
 */
struct rwc_alarm_reg {
uint32_t alarm :
    FIELD_BIT(31, 0); /*!< Значение времени в тиках */
};

/*!
 * @brief Регистр подстройки осцилляторов
 *
 * @note Регистр поля подстройки осцилляторов ( @ref trim_lfe и @ref trim_lfi),
 *       поле режима работы LFE ( @ref lfe_bypass) и поля-признаки режима SHUTDOWN
 *       ( @ref wake_stat2 и @ref wake_stat3).
 *       Значения полей @ref trim_lfe и @ref trim_lfi применяются только после записи
 *       регистра @ref rwc_trimload_reg.
 */
struct rwc_trim_reg {
uint32_t trim_lfe     :
    FIELD_BIT(10, 0);  /*!< Подстройка частоты 1 Гц для оциллятора LFE */
uint32_t trim_lfi     :
    FIELD_BIT(21, 11); /*!< Подстройка частоты 1 Гц для оциллятора LFI */
uint32_t lfe_bypass   :
    FIELD_BIT(22, 22); /*!< Режим работы осциллятора LFE ( @ref rwc_lfe_bypass ) */
uint32_t              :
    FIELD_BIT(24, 23); /*!< Поле зарезервировано */

    /*!
     * Бит устанавливается при выходе из режима SHUTDOWN
     * по внешнему событию WKUP, либо по сбросу SRSTn.
     * Бит сбрасывается при переходе в режим SHUTDOWN.
     */
uint32_t wake_stat2   :
    FIELD_BIT(25, 25);

    /*!
     * Бит устанавливается после первичной подачи питания на RWC.
     * Бит сбрасывается при переходе в режим SHUTDOWN.
     */
uint32_t wake_stat3   :
    FIELD_BIT(26, 26);
uint32_t              :
    FIELD_BIT(31, 27); /*!< Поле зарезервировано. Исходное состояние 0x10 */
};

/*!
 * @brief Конфигурационный регистр
 *
 * @note Управляет выбором тактирования счетчика времени, выбором используемого
 *       осциллятора, установкой делителя частоты RTCCLK, сбросов внутренних
 *       регистров, разрешением прерывания ALARM, работой входа WKUP,
 *       принудительным переходом в режим SHUTDOWN, а также содержит признак
 *       режима SHUTDOWN.
 */
struct rwc_config_reg {
    /*!
     * Выбор сигнала для тактирования счетчика времени
     * ( @ref rwc_time_clk_sel )
     */
uint32_t time_clk_sel           :
    FIELD_BIT(0, 0);

uint32_t                        :
    FIELD_BIT(3, 1);   /*!< Поле зарезервировано */
uint32_t osc_sel                :
    FIELD_BIT(4, 4);   /*!< Выбор осциллятора ( @ref rwc_rtcclk_type ) */
uint32_t                        :
    FIELD_BIT(5, 5);   /*!< Поле зарезервировано. Исходное состояние 1 */
uint32_t clk_div                :
    FIELD_BIT(10, 6);  /*!< Поле для деления тактового сигнала (@ref rwc_rtcclk_divisor) */

    /*!
     * Поле влияние сброса SRSTn на состояние внутренних
     * регистров RWC ( @ref rwc_reset_type )
     */
uint32_t reset_en               :
    FIELD_BIT(11, 11);

    /*!
     * Разрешение прерывания RWC_ALARM по совпадению
     * значений регистров TIME и ALARM
     */
uint32_t alarm_en               :
    FIELD_BIT(12, 12);

    /*!
     * Бит устанавливается при первом включении питания.
     * Сбрасывать нельзя.
     */
uint32_t pz                     :
    FIELD_BIT(13, 13);

    /*!
     * Бит устанавливается при первом включении питания.
     * Сбрасывать нельзя.
     */
uint32_t pl                     :
    FIELD_BIT(14, 14);
uint32_t             wake_in_en :
    FIELD_BIT(15, 15); /*!< Разрешение работы входа WKUP */
uint32_t                        :
    FIELD_BIT(29, 16); /*!< Поле зарезервировано */

    /*!
     * Установка этого бита приводит к принудительному
     * переходу системы в режим SHUTDOWN.
     * Не рекомендуется использовать.
     */
uint32_t shutdown_force         :
    FIELD_BIT(30, 30);

    /*!
     * Бит устанавливается при выходе из режима SHUTDOWN,
     * сбрасывается при переходе в режим SHUTDOWN.
     */
uint32_t wake_stat1             :
    FIELD_BIT(31, 31);
};

/*!
 * @brief Регистр общего назначения
 *
 * @note В документации назван GPR. Сохраняет свое состояние в любом режиме работы
 *       микросхемы при наличии питания VBAT. Сброс регистра выполняется только при
 *       первичном включении питания VBAT.
 */
struct rwc_general_reg {
uint32_t time :
    FIELD_BIT(31, 0); /*!< Поле для хранения информации */
};

/*!
 * @brief Регистр настройки контроллера пробуждения
 *
 * @note В документации назван wakecfg. Позволяет задать полярность сигнала и
 *       управлять разрешением прерывания RWC_WKUP.
 */
struct rwc_wake_config_reg {
uint32_t wake_en  :
    FIELD_BIT(0, 0);   /*!< Разрешение прерывания RWC_WKUP. */
uint32_t          :
    FIELD_BIT(15, 1);  /*!< Резерв */
uint32_t wake_pol :
    FIELD_BIT(16, 16); /*!< Полярность сигнала WKUP для генерирования прерывания ( @ref rwc_wake_up_polarity ) */
uint32_t          :
    FIELD_BIT(31, 17); /*!< Резерв */
};

/*!
 * @brief Объединение для доступа к регистрам
 *
 * @note Объединение для доступа к регистрам через функции
 * @ref RWC_GetInternalRegister и @ref RWC_SetInternalRegister.
 */
union rwc_union_reg {
    uint32_t                   reg_value;   /*!< Представление регистра как uint32_t */
    struct rwc_trimload_reg    trimload;    /*!< Регистр записи значения подстройки из регистра TRIM */
    struct rwc_time_reg        time;        /*!< Регистр текущего значения счетчика времени */
    struct rwc_alarm_reg       alarm;       /*!< Регистр времени пробуждения */
    struct rwc_trim_reg        trim;        /*!< Регистр подстройки осцилляторов */
    struct rwc_config_reg      config;      /*!< Конфигурационный регистр */
    struct rwc_general_reg     general;     /*!< Регистр общего назначения */
    struct rwc_wake_config_reg wake_config; /*!< Регистр настройки контроллера пробуждения */
};

/*!
 * @}
 */

/*!
 * @brief Выбор сигнала для тактирования счетчика времени
 *
 */
enum rwc_time_clk_sel {
    RWC_TimeClock32kHz = 0, /*!< 32 кГц (RTCCLK) */
    RWC_TimeClock1Hz   = 1, /*!< 1 Гц*/
};

/*!
 * @brief Уровень активного сигнала WKUP для генерирования прерывания
 *
 */
enum rwc_wake_up_polarity {
    RWC_WkUpPolarityHigh = 0, /*!< Высокий уровень */
    RWC_WkUpPolarityLow  = 1, /*!< Низкий уровень */
};

/*!
 * @brief Разрешение прерывания RWC_WKUP
 *
 */
enum rwc_wake_up_irq_enable {
    RWC_IRQWkUpDisable = 0, /*!< Запрещёно */
    RWC_IRQWkUpEnable  = 1, /*!< Разрешено */
};

/*!
 * @brief Режимы работы осциллятора LFE
 *
 */
enum rwc_lfe_bypass {
    RWC_QuartzResonator = 0, /*!< Режим осциллятора соместно с внешним кварцевым резонатором */
    RWC_CMOSSignal      = 1, /*!< Режим буфера для внешнего КМОП сигнала на выводе XTI32 */
};

/*!
 * @brief Внутренние регистры с батарейным питанием
 *
 */
enum rwc_internal_register {
    RWC_TrimLoad    = 2, /*!< Регистр записи значения подстройки из регистра TRIM */
    RWC_Time        = 3, /*!< Регистр текущего значения счетчика времени */
    RWC_Alarm       = 4, /*!< Регистр времени пробуждения */
    RWC_Trim        = 5, /*!< Регистр подстройки осцилляторов */
    RWC_Config      = 6, /*!< Конфигурационный регистр */
    RWC_GeneralReg  = 7, /*!< Регистр общего назначения */
    RWC_WakeConfig  = 8, /*!< Регистр настройки контроллера пробуждения */
};

/*!
 * @brief Источник частоты, подаваемой на RTCCLK
 *
 */
enum rwc_rtcclk_type {
    RWC_RTCClkTypeLFI = 0, /*!< RTCClk переключена на LFI */
    RWC_RTCClkTypeLFE = 1, /*!< RTCClk переключена на LFE */
};


/*!
 * @brief Виды сбросов внутренних регистров при сигнале SRSTn
 *
 */
enum rwc_reset_type {
    RWC_ResetOnlyWakeConfigAndConfig = 0, /*!< Cбрасываются только регистры WakeConfig и Config */
    RWC_ResetAllExpectedGeneralReg   = 1, /*!< Сбрасываются все регистры кроме GeneralReg */
};

/*!
 * @brief Делители частоты RTCCLK
 *
 */
enum rwc_rtcclk_divisor {
    RWC_Div1     = 0,  /*!< Делитель LFI на 1 */
    RWC_Div2     = 1,  /*!< Делитель LFI на 2 */
    RWC_Div4     = 2,  /*!< Делитель LFI на 4 */
    RWC_Div8     = 3,  /*!< Делитель LFI на 8 */
    RWC_Div16    = 4,  /*!< Делитель LFI на 16 */
    RWC_Div32    = 5,  /*!< Делитель LFI на 32 */
    RWC_Div64    = 6,  /*!< Делитель LFI на 64 */
    RWC_Div128   = 7,  /*!< Делитель LFI на 128 */
    RWC_Div256   = 8,  /*!< Делитель LFI на 256 */
    RWC_Div512   = 9,  /*!< Делитель LFI на 512 */
    RWC_Div1024  = 10, /*!< Делитель LFI на 1024 */
    RWC_Div2048  = 11, /*!< Делитель LFI на 2048 */
    RWC_Div4096  = 12, /*!< Делитель LFI на 4096 */
    RWC_Div8192  = 13, /*!< Делитель LFI на 8192 */
    RWC_Div16384 = 14, /*!< Делитель LFI на 16384 */
    RWC_Div32768 = 15, /*!< Делитель LFI на 32768 */

    RWC_DivMax   = RWC_Div32768, /*!< Максимальный делитель */
};

/*!
 * @brief Команды доступа к внутренним регистрам
 *
 */
enum rwc_cmd {
    RWC_CmdWait  = 0, /*!< Признак завершения выполнения команды, записывать нельзя */
    RWC_CmdWrite = 1, /*!< Команда записи регистра */
    RWC_CmdRead  = 2, /*!< Команда чтения регистра */
};

/*!
 * @brief Статусы драйвера CLKCTR
 *
 */
enum rwc_status {
    RWC_Status_Ok              = 0, /*!< Нет ошибок */
    RWC_Status_InvalidArgument = 1, /*!< Недопустимый параметр */
    RWC_Status_CheckError      = 2, /*!< Получена ошибка от оборудования */
    RWC_Status_VerifyError     = 3, /*!< Верификация не прошла */
    RWC_Status_ConfigureError  = 4, /*!< Недопустимая конфигурация или ошибка в описании оборудования */
    RWC_Status_HardwareBusy    = 5, /*!< Оборудование не готово */
    RWC_Status_Timeout         = 6, /*!< Тайм-аут */
};

/*!
 * @brief Статусы таймера
 *
 */
enum rwc_timer_status {
    /*!
     * Бит устанавливается после первичной подачи питания на RWC.
     * Бит сбрасывается при переходе в режим SHUTDOWN.
     */
    RWC_WakeStat3 = 1,

    /*!
     * Бит устанавливается при выходе из режима SHUTDOWN по внешнему
     * событию WKUP либо по сбросу SRSTn.
     * Бит сбрасывается при переходе в режим SHUTDOWN.
     */
    RWC_WakeStat2 = 2,

    /*!
     * Бит устанавливается при выходе из режима SHUTDOWN.
     * Бит сбрасывается при переходе в режим SHUTDOWN.
     */
    RWC_WakeStat1 = 4,
};

/*!
 * @brief Разрешение прерывания ALARM
 *
 */
enum rwc_alarm_enable {
    RWC_AlarmDisable = 0, /*!< Запретить прерывание ALARM */
    RWC_AlarmEnable  = 1, /*!< Разрешить прерывание ALARM */
};

/*!
 * @brief Разрешение работы входа WKUP
 *
 */
enum rwc_wkup_enable {
    RWC_WkUpDisable = 0, /*!< Прерывание WKUP запрещено */
    RWC_WkUpEnable  = 1, /*!< Прерывание WKUP разрешено */
};

/*!
 * @brief Принудительный переход в SHUTDOWN
 *
 */
enum rwc_shutdown_force {
    RWC_ShutdownNoSet = 0, /*!< Не переходить в SHUTDOWN */
    RWC_ShutdownSet   = 1, /*!< Переходить в SHUTDOWN */
};

/*!
 * @brief Значения делителей частоты внутреннего интерфейса
 *
 */
enum rwc_freq_serial {
    RWC_FS8MHz   = 0, /*!< 8 МГц */
    RWC_FS4MHz   = 1, /*!< 4 МГц */
    RWC_FS2MHz   = 2, /*!< 2 МГц */
    RWC_FS1MHz   = 3, /*!< 1 МГц */
    RWC_FS500kHz = 4, /*!< 500 кГц */
    RWC_FS250kHz = 5, /*!< 250 кГц*/
    RWC_FS125kHz = 6, /*!< 125 кГц*/
    RWC_FS625hHz = 7, /*!< 62,5 кГц */
};

/*!
 * @brief Структура используемая для хранения даты и времени
 *
 */
typedef struct _rtc_datetime {
    uint16_t year;  /*!< Диапазон от 1970 до 2099.*/
    uint8_t month;  /*!< Диапазон от 1 до 12.*/
    uint8_t day;    /*!< Диапазон от 1 до 31 (зависит от месяца).*/
    uint8_t hour;   /*!< Диапазон от 0 до 23.*/
    uint8_t minute; /*!< Диапазон от 0 до 59.*/
    uint8_t second; /*!< Диапазон от 0 до 59.*/
} rtc_datetime_t;

/*!
 * @brief Структура используемая для конфигурирования RWC
 *
 */
struct rwc_config {
    bool                        trimload;       /*!< Признак применения значения trim_lfi и trim_lfe */
    uint32_t                    time;           /*!< Значение счетчика времени */
    uint32_t                    alarm_time;     /*!< Значение времени пробуждения - будильника */
    enum rwc_lfe_bypass         lfe_bypass;     /*!< Режим работы осциллятора LFE */
    uint32_t                    trim_lfi;       /*!< Подстройка частоты 1 Гц для оциллятора LFI */
    uint32_t                    trim_lfe;       /*!< Подстройка частоты 1 Гц для оциллятора LFE */

    /*!
     * Бит устанавливается при выходе из режима SHUTDOWN.
     * Бит сбрасывается при переходе в режим SHUTDOWN.
     */
    uint32_t                    wake_stat1;

    /*!
     * Принудительный переход системы в режим SHUTDOWN.
     * Не рекомендуется использовать.
     */
    enum rwc_shutdown_force     shutdown_force;
    enum rwc_wkup_enable        wake_in_en;     /*!< Разрешение работы входа WKUP */
    uint32_t                    pl;             /*!< Бит устанавливается при первом включении питания. Сбрасывать нельзя */
    uint32_t                    pz;             /*!< Бит устанавливается при первом включении питания. Сбрасывать нельзя */

    /*!
     * Разрешение прерывания RWC_ALARM по совпадению
     * значений регистров TIME и ALARM.
     */
    enum rwc_alarm_enable       alarm_en;
    enum rwc_reset_type         reset_en;       /*!< Влияние сброса SRSTn на состояние внутренних регистров RWC */
    enum rwc_rtcclk_divisor     clkdiv;         /*!< Значение делителя тактового сигнала clk_32kHz и соответствующего выхода RTCCLK */
    enum rwc_rtcclk_type        osc_sel;        /*!< Выбор осциллятора */
    enum rwc_time_clk_sel       time_clk_sel;   /*!< Выбор сигнала для тактирования счетчика времени */
    uint32_t                    general;        /*!< Регистр общего назначения */
    enum rwc_wake_up_polarity   wake_pol;       /*!< Полярность сигнала WKUP для генерирования прерывания */
    enum rwc_wake_up_irq_enable wake_en;        /*!< Разрешение прерывания RWC_WKUP */
    enum rwc_freq_serial        presc;          /*!< Делитель для формирования тактовой частоты интерфейса.
                                                 * Рекомендуемое значение частоты - RWC_FS1MHz */
};

/*!
 * @name Функции чтения/записи внутренних регистров
 *
 * @{
 */

/*!
 * @brief Чтение значения из внутреннего регистра RWC
 *
 * @note Функцию не рекомендуется использовать без
 *       понимания функционирования блока
 *
 * @param base  Таймер
 * @param reg   Внутренний регистр
 * @param value Значение регистра
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_Timeout
 */
enum rwc_status RWC_GetInternalRegister(RWC_Type *base,
    enum rwc_internal_register reg, union rwc_union_reg *value);

/*!
 * @brief Запись значения во внутренний регистр RWC
 *
 * @note Функцию не рекомендуется использовать без
 *       понимания функционирования блока
 *
 * @param base  Таймер
 * @param reg   Внутренний регистр
 * @param value Записываемое значение
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_Timeout
 */
enum rwc_status RWC_SetInternalRegister(RWC_Type *base,
    enum rwc_internal_register reg, union rwc_union_reg value);
/*!
 * @}
 */

/*!
 * @name Функции для работы с частотой тактирования
 * @{
 */

/*!
 * @brief Чтение делителя и источника частоты RTCCLK
 *
 * @note Функцию не рекомендуется использовать без
 *       понимания функционирования блока
 *
 * @param div Делитель частоты
 * @param src Источник частоты
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_GetRTCClkParam(uint32_t *div, enum rwc_rtcclk_type *src);

/*!
 * @brief Запись делителя и источника частоты RTCCLK
 *
 * @note Допустимые значения для делителя частоты - степени 2 от 2^0 до 2^RWC_DivMax.
 *
 * @note Функцию не рекомендуется использовать без
 *       понимания функционирования блока
 *
 * @param div Делитель частоты
 * @param src Источник частоты
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_SetRTCClkParam(uint32_t div, enum rwc_rtcclk_type src);

/*!
 * @brief Чтение значения счетчика реального времени
 *
 * @param base  Таймер
 *
 * @retval Значение регистра времени TIME
 */
uint32_t RWC_GetTime(RWC_Type *base);
/*!
 * @}
 */

/*!
 * @name Интерфейс таймера реального времени
 * @{
 */

/*!
 * @name Инициализация и деинициализация
 * @{
 */

/*!
 * @brief Инициализирует таймер реального времени
 *
 * @note Эту функцию следует вызывать перед использованием драйвера
 *       таймера реального времени.
 *
 * @param base Таймер
 * @param cfg  Конфигурация
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_Init(RWC_Type *base, struct rwc_config cfg);

/*!
 * @brief Деинициализирует таймер реального времени
 *
 * @note Сбрасывает регистры RWC_Config и RWC_WakeConfig таймера
 *       к значениям по умолчанию
 *
 * @param base Таймер
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_Deinit(RWC_Type *base);
/*!
 * @}
 */

/*!
 * @name Текущее время и предупреждение
 * @{
 */

/*!
 * @brief Устанавливает текущее время и дату согласно заданной структуре
 *
 * @note Процедура может приводить к зависанию.
 *
 * @param base     Таймер
 * @param datetime Структура дата/время
 *
 * @return #RWC_Status_Ok
 * @return #RWC_Status_InvalidArgument
 * @return #RWC_Status_Timeout
 */
enum rwc_status RWC_SetDatetime(RWC_Type *base, const rtc_datetime_t *datetime);

/*!
 * @brief Получает текущее дату/время и сохраняет в указанную структуру
 *
 * @param base     Таймер
 * @param datetime Структура дата/время
 *
 * @return RWC_Status_Ok
 * @return RWC_Status_InvalidArgument
 */
enum rwc_status RWC_GetDatetime(RWC_Type *base, rtc_datetime_t *datetime);

/*!
 * @brief Устанавливает время будильника
 *
 * @note Функция проверяет, превышает ли указанное время срабатывания
 *       будильника текущее время. Если нет, то функция не устанавливает
 *       сигнал будильника и возвращает сообщение об ошибке.
 *
 * @param base      Таймер
 * @param alarmTime Структура дата/время
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_ConfigureError
 */
enum rwc_status RWC_SetAlarm(RWC_Type *base, const rtc_datetime_t *alarmTime);

/*!
 * @brief Возвращает время будильника
 *
 * @param base     Таймер
 * @param datetime Структура дата/время
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_GetAlarm(RWC_Type *base, rtc_datetime_t *datetime);

/*! @}*/


/*!
 * @brief Устанавливает время будильника в секундах
 *
 * @param base        Таймер
 * @param matchValue  Время будильника в секундах
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_SetSecondsTimerMatch(RWC_Type *base, uint32_t match_value);

/*!
 * @brief Получает актуальное время срабатывания будильника в секундах
 *
 * @param base Таймер
 *
 * @return Врема будильника в секундах
 */
uint32_t RWC_GetSecondsTimerMatch(RWC_Type *base);

/*!
 * @brief Устанавливает текущее время в секундах
 *
 * @note  Время устанавливается до тех пор, пока не будет установлено
 *        либо не будет превышено максимальное число попыток записи.
 *
 * @param base       Таймер
 * @param countValue Время в секундах
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_Timeout
 */
enum rwc_status RWC_SetSecondsTimerCount(RWC_Type *base, uint32_t count_value);

/*!
 * @brief Получение текущее время в секундах
 *
 * @param base Таймер
 * @param sec  Считанное значение секунд
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 */
enum rwc_status RWC_GetSecondsTimerCount(RWC_Type *base, uint32_t *sec);

/*!
 * @name Interrupt Interface
 * @{
 */

/*!
 * @brief Разрешение прерывания по сигналу wake-up из режима
 * глубокого отключения питания
 *
 * @param base   Таймер
 * @param enable Управление разрешением.
 *               - true: разрешено.
 *               - false: запрешено.
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_EnableWakeUpTimerInterruptFromDPD(RWC_Type *base,
    bool enable);

/*!
 * @brief Разрешение прерывания по сигналу будильника из режима
 * глубокого отключения питания
 *
 * @param base   Таймер
 * @param enable Управление разрешением.
 *               - true: разрешено.
 *               - false: запрешено.
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_EnableAlarmTimerInterruptFromDPD(RWC_Type *base,
    bool enable);

/*!
 * @brief Сброс прерывания RWC_ALARM
 *
 * @param base Таймер
 *
 * @retval RWC_Status_Ok
 * @retval RWC_Status_InvalidArgument
 */
enum rwc_status RWC_InterruptClear(RWC_Type *base);
/*!
 * @}
 */

/*!
 * @name Status Interface
 * @{
 */

/*!
 * @brief Получение статусов таймера реального времени
 *
 * @param base Таймер
 *
 * @return Объединение статусов rwc_timer_status
 */
uint32_t RWC_GetStatusFlags(RWC_Type *base);

/*!
 @}
 */

/*!
 * @name Функции работы с внешним сигналом сигналом пробуждения (wake-up)
 * @{
 */

/*!
 * @brief Разрешение работы входа wake_up
 *
 * @param base   Базовый адрес
 * @param enable Разрешение
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_SetWakeUpEnable(RWC_Type *base, bool enable);

/*!
 * @brief Установка полярности сигнала wake_up
 *
 * @param base  Базовый адрес
 * @param value Полярность сигнала
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_SetWakeUpActiveLewel(RWC_Type *base,
    enum rwc_wake_up_polarity value);

/*!
 * @}
 */

/*!
 * @brief Получение конфигурации таймера по умолчанию
 *
 * @param config Конфигурация
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 */
enum rwc_status RWC_GetDefaultConfig(struct rwc_config *config);

/*!
 * @brief Выбор режима работы осциллятора LFE
 *
 * @param base  Базовый адрес
 * @param value Режим работы осциллятора LFE
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_SetLFEBypass(RWC_Type *base,
    enum rwc_lfe_bypass value);

/*!
 * @brief Выбор генератора LFE или LFI
 *
 * @param base  Базовый адрес
 * @param value Генератор
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_SetLFx(RWC_Type *base,
    enum rwc_rtcclk_type value);

/*!
 * @brief Получение статуса выполнения функции,
 * тип результата которой отличен от enum rwc_status
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_GetLastAPIStatus();

/*!
 * @brief Управление сбросом регистров при сигнале на входе SRSTn
 *
 * @param base  Базовый адрес
 * @param value Вид сброса
 *
 * @retval #RWC_Status_Ok
 * @retval #RWC_Status_InvalidArgument
 * @retval #RWC_Status_CheckError
 */
enum rwc_status RWC_SetResetCtrl(RWC_Type *base,
    enum rwc_reset_type value);

#if defined(__cplusplus)
}
#endif

/*!
 * @}
 */

/*!
 * @name Функции для удобного взаимодействия с типом данных datetime
 * @{
 */

/*!
 * @brief Преобразование переменной типа данных datetime в секунды
 *
 * @param datetime Преобразуемая дата
 *
 * @return Количество секунд, прошедших с 1 января 1970 года по указанную дату
 */
uint32_t RWC_ConvertDatetimeToSeconds(const rtc_datetime_t *datetime);

/*!
 * @}
 */

#endif /* HAL_RWC_H */

/*!
 * @}
 */
