/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup uart_driver Драйвер модуля UART
 *
 * @brief Интерфейс предназначен для организации связи с другими цифровыми
 *        устройствами
 *
 * Драйвер позволяет производить приемо/передачу по последовательному
 * асинхронному интерфейсу в режиме ожидания (polling) и режиме без
 * ожидания (interrupt). Поддерживает некоторые расширенные возможности:
 * IR-режим, режим петли, режим RS485. Использует аппаратный Rx и Тх
 * FIFO.
 */

/*!
 * @addtogroup uart_driver
 * @{
 */

/*!
 * @file hal_uart.h
 *
 * @brief Интерфейс драйвера UART
 */

#ifndef HAL_UART_H
#define HAL_UART_H

#include "hal_common.h"

/*!
 * @brief Версия драйвера UART
 */
#define HAL_UART_DRIVER_VERSION (MAKE_VERSION(0, 1, 0))

/*!
 * @brief Количество циклов ожидания
 */
#ifndef UART_RETRY_TIMES
#define UART_RETRY_TIMES 0U /* 0 - ожидание до получения значения */
#endif /* UART_RETRY_TIMES */

/*!
 * @brief Статусы драйвера UART
 */
enum uart_status {
    UART_Status_Ok                   =  0U, /*!< Успешно */
    UART_Status_Fail                 =  1U, /*!< Провал */
    UART_Status_ReadOnly             =  2U, /*!< Только чтение */
    UART_Status_InvalidArgument      =  3U, /*!< Неверный аргумент */
    UART_Status_Timeout              =  4U, /*!< Отказ по таймауту */
    UART_Status_NoTransferInProgress =  5U, /*!< Нет текущей передачи данных */
    UART_Status_TxBusy               =  6U, /*!< Передатчик занят */
    UART_Status_RxBusy               =  7U, /*!< Ресивер занят */
    UART_Status_TxIdle               =  8U, /*!< Передатчик простаивает */
    UART_Status_RxIdle               =  9U, /*!< Приемник простаивает */
    UART_Status_TxError              = 10U, /*!< Ошибка в TxFIFO */
    UART_Status_RxError              = 11U, /*!< Ошибка в RxFIFO */
    UART_Status_RxRingBufferOverrun  = 12U, /*!< Ошибка в кольцевом буфере Rx */
    UART_Status_RxFifoBufferOverrun  = 13U, /*!< Ошибка переполнения hw RxFIFO буферa */
    UART_Status_BreakLineError       = 14U, /*!< Ошибка обрыва линии */
    UART_Status_FramingError         = 15U, /*!< Ошибка кадра */
    UART_Status_ParityError          = 16U, /*!< Ошибка четности */
    UART_Status_BaudrateNotSupport   = 17U, /*!< Скорость передачи не поддерживается для текущего источника синхронизации */
    UART_Status_TxRingBufferNull     = 18U, /*!< Кольцевой буфер передатчика не инициализирован */
};

/*!
 * @brief Конфигурация прерываний для UART
 */
enum uart_interrupt_enable {
    UART_ThresoldInterruptEnable = (UART_IER_PTIME_Msk),        /*!< 0x80 По порогу */
    UART_ModemInterruptEnable    = (UART_IER_EDSSI_Msk),        /*!< 0x08 По статусу модема */
    UART_RxLineInterruptEnable   = (UART_IER_ELSI_Msk),         /*!< 0x04 По состоянию линии приема */
    UART_TxInterruptEnable       = (UART_IER_ETBEI_Msk),        /*!< 0x02 По опустошении регистра передатчика */
    UART_RxInterruptEnable       = (UART_IER_ERBFI_Msk),        /*!< 0x01 По доступности полученных данных или при включенном FIFO, прерывания по таймауту входных данных */
    UART_AllInterruptsEnable     = (UART_ThresoldInterruptEnable
            | UART_ModemInterruptEnable
            | UART_RxLineInterruptEnable
            | UART_TxInterruptEnable
            | UART_RxInterruptEnable),     /*!< 0x8f Все прерывания */
};

/*!
 * @brief Флаги состояния UART LSR
 */
enum uart_lsr_flags {
    UART_LSR_FlagRxError          = (UART_LSR_RFE_Msk),  /*!< Суммарный бит ошибки приемника, cбрасывается при чтении LSR */
    UART_LSR_FlagTxHwEmpty        = (UART_LSR_TEMT_Msk), /*!< Бит отсутствия передаваемых данных в FIFO буфере и сдвиговом регистре передатчика */
    UART_LSR_FlagTxSwEmpty        = (UART_LSR_THRE_Msk), /*!< Бит отсутствия данных в буфере передатчика */
    UART_LSR_FlagRxLinebreakError = (UART_LSR_BI_Msk),   /*!< Ошибка обрыв линии,сбрасывается при чтении LSR */
    UART_LSR_FlagRxFrameError     = (UART_LSR_FE_Msk),   /*!< Ошибка кадрирования LSR */
    UART_LSR_FlagRxParityError    = (UART_LSR_PE_Msk),   /*!< Ошибка четности */
    UART_LSR_FlagRxOverflowError  = (UART_LSR_OE_Msk),   /*!< Ошибка переполнения, бит сбрасывается при чтении содержимого регистра LSR */
    UART_LSR_FlagRxReady          = (UART_LSR_DR_Msk),   /*!< Есть данные в приемнике, которые еще не были прочитаны */
};

/*!
 * @brief Режимы четности UART
 */
enum uart_parity_mode {
    UART_ParityOdd  = 0U, /*!< Тип - нечетная */
    UART_ParityEven = 1U, /*!< Тип - четная */
};

/*!
 * @brief Количество стоп-битов для UART
 */
enum uart_stop_bit_count {
    UART_OneStopBit             = 0U, /*!< 1 стоп бит */
    UART_TwoOrOneAndHalfStopBit = 1U, /*!< 1.5 или 2 стоп бит, зависит от количества бит данных в передаваемом символе */
};

/*!
 * @brief Количество бит данных в передаваемом символе
 */
enum uart_data_len {
    UART_5BitsPerChar = 0U, /*!< 5 бит на символ */
    UART_6BitsPerChar = 1U, /*!< 6 бит на символ */
    UART_7BitsPerChar = 2U, /*!< 7 бит на символ */
    UART_8BitsPerChar = 3U, /*!< 8 бит на символ */
};

/*!
 * @brief Триггер уровня заполненности TxFIFO
 */
enum uart_txfifo_watermark {
    UART_TxFifoEmpty       = 0U, /*!< TxFIFO пуст */
    UART_TxFifoTwoChars    = 1U, /*!< В TxFIFO - 2 символа */
    UART_TxFifoQuarterFull = 2U, /*!< TxFIFO заполнен на четверть, 1/4 */
    UART_TxFifoHalfFull    = 3U, /*!< TxFIFO заполнен на половину, 1/2 */
};

/*!
 * @brief Триггер уровня заполненности RxFIFO
 */
enum uart_rxfifo_watermark {
    UART_RxFifoOneChar     = 0U, /*!< В RxFIFO - 1 символ */
    UART_RxFifoQuarterFull = 1U, /*!< RxFIFO заполнен на четверть, 1/4 */
    UART_RxFifoHalfFull    = 2U, /*!< RxFIFO заполнен на половину, 1/2 */
    UART_RxFifoTwoToFull   = 3U, /*!< RxFIFO на 2 меньше чем полный */
};

/*!
 * @brief Режим работы RS485
 */
enum uart_rs485_mode {
    UART_RS485_ModeFullDuplex       = 0U, /*!< Дуплекс */
    UART_RS485_ModeHalfDuplexManual = 1U, /*!< Полудуплекс: переключение направления передачи вручную */
    UART_RS485_ModeHalfDuplexAuto   = 2U, /*!< Полудуплекс: автопереключение направления передачи */
};

/*!
 * @brief Активное сотояние линии для RS485
 */
enum uart_rs485_active_state {
    UART_RS485_ActiveStateHigh = 0U, /*!< Активный уровень для линии высокий */
    UART_RS485_ActiveStateLow  = 1U, /*!< Активный уровень для линии низкий */
};

/*!
 * @brief Конфигурация UART
 */
struct uart_config {
    uint32_t                   baudrate_bps;                 /*!< Скорость интерфейса UART */
    bool                       enable_parity;                /*!< Включена ли четность (по умолчанию - выключена) */
    enum uart_parity_mode      parity_mode;                  /*!< Режим четности - чет или нечет */
    bool                       parity_manual;                /*!< Ручное управление битом четности */
    enum uart_stop_bit_count   stop_bit_count;               /*!< Количество стоп-битов */
    enum uart_data_len         bit_count_per_char;           /*!< Количество бит данных в передаваемом символе: от 5 до 8 бит */
    bool                       enable_rxfifo;                /*!< Включена ли RxFIFO */
    bool                       enable_txfifo;                /*!< Включена ли TxFIFO */
    bool                       enable_loopback;              /*!< Включена ли петля */
    bool                       enable_infrared;              /*!< Включен ли инфракрасный режим интерфейса */
    bool                       enable_hardware_flow_control; /*!< Включено ли аппаратное управление потоком RTS/CTS */
    bool                       break_line;                   /*!< Бит обрыва линии */
    /*  enum uart_txfifo_watermark tx_watermark;          */ /*!< Метка-триггер уровня заполенности TxFIFO */
    /*  enum uart_rxfifo_watermark rx_watermark;          */ /*!< Метка-триггер уровня заполенности RxFIFO */
    /*  bool                       rs485_enable;          */ /*!< Режим RS485: включен ли режим */
    /*  enum uart_rs485_mode       rs485_mode;            */ /*!< Режим RS485: тип обмена */
    /*  bool                       rs485_de_active_state; */ /*!< Режим RS485: DE активное состояние (true - высокий, false - низкий) */
    /*  bool                       rs485_re_active_state; */ /*!< Режим RS485: RE активное состояние (true - высокий, false - низкий) */
                                                             /*!< Режим RS485: задержка переключения из RE в DE */
                                                             /*!< Режим RS485: задержка переключения из DE в RE */
                                                             /*!< Режим RS485: DE_De-assertion_Time */
                                                             /*!< Режим RS485: DE-Assertion_Time */
};

/*!
 * @brief Указатель на буфер приема или передачи
 *
 * Раздельные указатели rx_data и tx_data, потому что tx_data const.
 */
struct uart_transfer {
    union {
        uint8_t       *rx_data; /*!< Буфер для приема */
        uint8_t const *tx_data; /*!< Буфер на передачу */
    };
    size_t data_size;           /*!< Счетчик байтов */
};

/* Объявление, описание ниже. */
struct uart_handle;

/*!
 * @brief Сallback-функция
 */
typedef void (*uart_transfer_callback_t)(UART_Type *base,
    struct uart_handle *handle, enum uart_status status, void *user_data);

/*!
 * @brief Дескриптор состояния приема/передачи для неблокирующих функций
 *        обмена
 */
struct uart_handle {
    volatile const uint8_t   *tx_data;            /*!< Адрес оставшихся данных для отправки */
    volatile size_t          tx_data_size;        /*!< Размер оставшихся данных для отправки */
    size_t                   tx_data_size_all;    /*!< Размер данных для отправки */

    uint8_t                  *tx_ring_buffer;     /*!< Начальный адрес кольцевого буфера передатчика */
    size_t                   tx_ring_buffer_size; /*!< Размер кольцевого буфера передатчика */
    volatile uint16_t        tx_ring_buffer_head; /*!< Индекс, позволяющий пользователю записывать данные в кольцевой буфер Tx */
    volatile uint16_t        tx_ring_buffer_tail; /*!< Индекс для драйвера для отправки данных из кольцевого буфера Tx */

    volatile uint8_t         *rx_data;            /*!< Адрес оставшихся данных для получения */
    volatile size_t          rx_data_size;        /*!< Размер оставшихся данных для получения */
    size_t                   rx_data_size_all;    /*!< Размер получаемых данных */

    uint8_t                  *rx_ring_buffer;     /*!< Начальный адрес кольцевого буфера приемника */
    size_t                   rx_ring_buffer_size; /*!< Размер кольцевого буфера */
    volatile uint16_t        rx_ring_buffer_head; /*!< Индекс для драйвера для сохранения полученных данных в кольцевом буфере */
    volatile uint16_t        rx_ring_buffer_tail; /*!< Индекс, позволяющий пользователю получать данные из кольцевого буфера */

    uart_transfer_callback_t callback;            /*!< Функция обратного вызова */
    void                     *user_data;          /*!< UART-параметр функции обратного вызова */

    volatile uint8_t         tx_state;            /*!< Состояние передачи */
    volatile uint8_t         rx_state;            /*!< Состояние приема */
};

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

/*!
 * @name Инициализация и деинициализация
 * @{
 */

/*!
 * @brief Инициализирует модуль UART структурой конфигурации пользователя и
 *        частотой периферии
 *
 * Эта функция конфигурирует модуль UART с пользовательскими настройками.
 * Пользователь может настроить конфигурацию структуры, а также получить
 * конфигурацию по умолчанию с помощью функции @ref UART_GetDefaultConfig.
 *
 * Пример ниже показывает, как использовать эту функцию для настройки UART:
 * @code{.c}
 *   config->baudrate_bps                 = 115200U;
 *   config->enable_parity                = false;
 *   config->stop_bit_count               = UART_OneStopBit;
 *   config->bit_count_per_char           = UART_8BitPerChar;
 *   config->enable_rxfifo                = true;
 *   config->enable_txfifo                = true;
 *   config->enable_loopback              = false;
 *   config->enable_infrared              = false;
 *   config->enable_hardware_flow_control = false;
 *   config->break_line                   = false;
 *   UART_Init(UART1, &uart_config, 20000000U);
 * @endcode
 *
 * @param base         Указатель на базовый адрес UART
 * @param config       Указатель на определяемую пользователем структуру
 *                     конфигурации
 * @param src_clock_hz Тактовая частота источника в Гц
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_Init(UART_Type *base, const struct uart_config *config,
    uint32_t src_clock_hz);

/*!
 * @brief Деинициализирует модуль UART
 *
 * Функция ожидает завершения передачи, отключает передачу и прием и
 * отключает синхронизацию модуля UART.
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_Deinit(UART_Type *base);

/*!
 * @brief Получает структуру конфигурации по умолчанию
 *
 * Эта функция инициализирует структуру конфигурации UART значением по
 * умолчанию:
 * @code{.c}
 *   config->baudrate_bps                 = 115200U;
 *   config->enable_parity                = false;
 *   config->stop_bit_count               = UART_OneStopBit;
 *   config->bit_count_per_char           = UART_8BitPerChar;
 *   config->enable_rxfifo                = true;
 *   config->enable_txfifo                = true;
 *   config->enable_loopback              = false;
 *   config->enable_infrared              = false;
 *   config->enable_hardware_flow_control = false;
 *   config->break_line                   = false;
 *   UART_Init(UART1, &uart_config, 20000000U);
 * @endcode
 *
 * @param config Указатель на структуру конфигурации
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_GetDefaultConfig(struct uart_config *config);

/*!
 * @brief Устанавливает скорость модуля UART
 *
 * Эта функция настраивает скорость передачи модуля UART. Используется для
 * обновления скорости модуля после его инициализации с помощью @ref UART_Init :
 * @code{.c}
 *   UART_SetBaudRate(UART1, 115200U, 20000000U);
 * @endcode
 *
 * @param base         Указатель на базовый адрес UART
 * @param baudrate_bps Устанавливаемая скорость передачи данных
 * @param src_clock_hz Тактовая частота источника в Гц
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_SetBaudRate(UART_Type *base, uint32_t baudrate_bps,
    uint32_t src_clock_hz);

/*!
 * @}
 */

/*!
 * @name Состояние
 * @{
 */

/*!
 * @brief Извлекает флаги состояния UART
 *
 * Функция получает все флаги состояния UART, флаги возвращаются как логические
 * ИЛИ значение @ref uart_lsr_flags. Чтобы проверить конкретный статус,
 * сравните возвращаемое значение с перечислителями в
 * @ref uart_lsr_flags. Например, чтобы проверить, пуст ли TX:
 * @code{.c}
 *   if (UART_LSR_FlagTxHwEmpty & UART_GetStatusFlags(UART1))
 *   {
 *       ...
 *   }
 * @endcode
 *
 * @param base Указатель на базовый адрес UART
 *
 * @return Маска флагов состояния UART
 */
static inline uint32_t UART_GetStatusFlags(UART_Type *base)
{
    return (base->LSR & 0xFFUL);
}

/*!
 * @}
 */

/*!
 * @name Включение/выключение и настройка прерываний
 * @{
 */

/*!
 * @brief Разрешает прерывания UART в соответствии с предоставленной маской
 *
 * Эта функция разрешает прерывания UART в соответствии с предоставленной
 * маской. Маска - это логическое ИЛИ членов перечисления
 * (@ref uart_interrupt_enable).
 * Например, чтобы разрешить прерывание TX и прерывание RX:
 * @code{.c}
 *   UART_EnableInterrupts(UART1, UART_TxInterruptEnable | UART_RxInterruptEnable);
 * @endcode
 *
 * @param base Указатель на базовый адрес UART
 * @param mask Маска разрешаемых прерываний
 */
static inline void UART_EnableInterrupts(UART_Type *base, uint32_t mask)
{
    /*
     * Работаем только с прерываниями, зарегистрированными в
     * @ref uart_interrupt_enable.
     */
    base->IER |= mask & UART_AllInterruptsEnable;
}

/*!
 * @brief Отключает прерывания UART в соответствии с предоставленной маской
 *
 * Эта функция отключает прерывания UART в соответствии с предоставленной
 * маской. Маска - это логическое ИЛИ членов перечисления
 * (@ref uart_interrupt_enable).
 * В этом примере показано, как отключить пустое прерывание TX и полное
 * прерывание RX:
 * @code{.c}
 *   UART_DisableInterrupts(UART1, UART_TxInterruptEnable | UART_RxInterruptEnable);
 * @endcode
 *
 * @param base Указатель на базовый адрес UART
 * @param mask Маска запрещаемых прерываний
 */
static inline void UART_DisableInterrupts(UART_Type *base, uint32_t mask)
{
    mask &= UART_AllInterruptsEnable;
    base->IER &= ~mask;
}

/*!
 * @brief Запрос маски включенных прерываний в UART
 *
 * Запрос маски включенных прерываний в UART; единицы в соответствующих
 * разрядах соответствуют включенным прерываниям.
 *
 * @param base Указатель на базовый адрес UART
 *
 * @return Маска включенных прерываний
 */
static inline uint32_t UART_GetEnabledInterrupts(UART_Type *base)
{
    return base->IER & UART_AllInterruptsEnable;
}

/*!
 * @brief Устанавливает триггер уровня заполненности RxFIFO
 *
 * @param base  Указатель на базовый адрес UART
 * @param water Триггер уровня заполненности RxFIFO
 *
 */
static inline void UART_SetRxFifoWatermark(UART_Type *base,
    enum uart_rxfifo_watermark water)
{
    SET_VAL_MSK(base->FCR, UART_FCR_RT_Msk, UART_FCR_RT_Pos, water);
}

/*!
 * @brief Устанавливает триггер уровня заполненности TxFIFO
 *
 * @param base  Указатель на базовый адрес UART
 * @param water Триггер уровня заполненности TxFIFO
 *
 */
static inline void UART_SetTxFifoWatermark(UART_Type *base,
    enum uart_txfifo_watermark water)
{
    SET_VAL_MSK(base->FCR, UART_FCR_TET_Msk, UART_FCR_TET_Pos, water);
}

/*!
 * @}
 */

/*!
 * @name Включение/выключение и настройка расширенных режимов работы UART
 * @{
 */

/*!
 * @brief Включение/выключение режима петли
 *
 * @param base   Указатель на базовый адрес UART
 * @param enable Включение/выключение режима
 *
 */
static inline void UART_SetLoopback(UART_Type *base, bool enable)
{
    SET_VAL_MSK(base->MCR, UART_MCR_LOOPBACK_Msk, UART_MCR_LOOPBACK_Pos,
        enable);
}

/*!
 * @brief Включение/выключение инфракрасного режима работы
 *
 * @param base   Указатель на базовый адрес UART
 * @param enable Включение/выключение режима
 *
 */
static inline void UART_SetIr(UART_Type *base, bool enable)
{
    SET_VAL_MSK(base->MCR, UART_MCR_SIRE_Msk, UART_MCR_SIRE_Pos, enable);
}

/*!
 * @brief Включение/выключение RS485 режима работы
 *
 * @param base   Указатель на базовый адрес UART
 * @param enable Включение/выключение режима
 *
 */
static inline void UART_SetRs485(UART_Type *base, bool enable)
{
    SET_VAL_MSK(base->TCR, UART_TCR_RS485_EN_Msk, UART_TCR_RS485_EN_Pos,
        enable);
}

/*!
 * @brief Установка режима работы RS485
 *
 * @param base Указатель на базовый адрес UART
 * @param mode Режим работы RS485
 * @param de   #uart_rs485_active_state
 * @param re   #uart_rs485_active_state
 *
 */
static inline void UART_Rs485Mode(UART_Type *base, enum uart_rs485_mode mode,
    enum uart_rs485_active_state de, enum uart_rs485_active_state re)
{
    SET_VAL_MSK(base->TCR, UART_TCR_XFER_MODE_Msk, UART_TCR_XFER_MODE_Pos,
        mode);
    SET_VAL_MSK(base->TCR, UART_TCR_DE_POL_Msk, UART_TCR_DE_POL_Pos, de);
    SET_VAL_MSK(base->TCR, UART_TCR_RE_POL_Msk, UART_TCR_RE_POL_Pos, re);
}

/*!
 * @}
 */

/*!
 * @name Прием и передача без использования прерываний
 * @{
 */

/*!
 * @brief Записывает данные на передачу в регистр передачи
 *
 * Эта функция записывает данные на передачу в регистр передачи THR.
 * Верхний уровень должен убедиться, что в нем есть место для записи байта.
 *
 * @param base Указатель на базовый адрес UART
 * @param data Байт для записи
 *
 */
static inline void UART_WriteByte(UART_Type *base, uint8_t data)
{
    /* Доступен, если LCR[7](DLAB) == 0x0. */
    base->THR = data;
}

/*!
 * @brief Записывает данные на передачу в регистр передачи c ожиданием
 *        освобождения места
 *
 * Эта функция пишет в регистр передачи THR. Будет ожидать освобождения
 * места для записи байта.
 *
 * @param base Указатель на базовый адрес UART
 * @param data Байт для записи
 *
 */
static inline void UART_WriteByteWait(UART_Type *base, uint8_t data)
{
    /* Пока есть данные в буфере передатчика. */
    while (GET_VAL_MSK(base->LSR, UART_LSR_THRE_Msk, UART_LSR_THRE_Pos) == 0U)
        ;

    /* Доступен, если LCR[7](DLAB) == 0x0. */
    base->THR = data;
}

/*!
 * @brief Вычитывает байт из регистра приема
 *
 * Читает байт из регистра приема RBR. Верхний уровень должен проверить,
 * что в регистре RBR что-то есть перед вызовом.
 *
 * @param base Указатель на базовый адрес UART
 *
 * @return Байт, считанный из регистра
 *
 */
static inline uint8_t UART_ReadByte(UART_Type *base)
{
    /* Доступен, если LCR[7](DLAB) == 0x0. */
    return (uint8_t) base->RBR;
}

/*!
 * @brief Вычитывает байт из регистра приема с ожиданием получения
 *
 * Читает из регистра приема RBR. Если в регистре нет данных, функция будет
 * ждать получения хотя бы одного байта.
 *
 * @param base Указатель на базовый адрес UART
 *
 * @return Байт, считанный из регистра
 *
 */
static inline uint8_t UART_ReadByteWait(UART_Type *base)
{
    /* Пока нет данных в приемнике. */
    while (GET_VAL_MSK(base->LSR, UART_LSR_DR_Msk, UART_LSR_DR_Pos) == 0U);

    /* Доступен, если LCR[7](DLAB) == 0x0. */
    return (uint8_t) base->RBR;
}

/*!
 * @brief Получить количество байтов в RxFIFO
 *
 * @param base Указатель на базовый адрес UART
 *
 * @return Количество байт в RxFIFO
 *
 */
static inline uint8_t UART_GetRxFifoCount(UART_Type *base)
{
    return (uint8_t) GET_VAL_MSK(base->TFL, UART_RFL_RFL_Msk, UART_RFL_RFL_Pos);
}

/*!
 * @brief Получить количество байтов в TxFIFO
 *
 * @param base Указатель на базовый адрес UART
 *
 * @return Количество байтов в TxFIFO
 *
 */
static inline uint8_t UART_GetTxFifoCount(UART_Type *base)
{
    return (uint8_t) GET_VAL_MSK(base->TFL, UART_TFL_TFL_Msk, UART_TFL_TFL_Pos);
}

/*!
 * @brief Записывает в регистр TX с использованием метода блокировки
 *
 * Эта функция опрашивает регистр TX, ожидает, пока регистр TX не станет
 * пустым, или пока не появится место в TX FIFO.
 *
 * @param base   Указатель на базовый адрес UART
 * @param data   Указатель на начальный адрес данных для записи
 * @param length Размер записываемых данных
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 * @retval #UART_Status_Timeout
 */
enum uart_status UART_WriteBlocking(UART_Type *base, const uint8_t *data,
    size_t length);

/*!
 * @brief Чтение регистра данных RX с использованием метода блокировки
 *
 * Эта функция опрашивает регистр RX, ожидает, пока регистр RX будет
 * заполнен или пока RX FIFO не будет иметь данные и читать данные из
 * регистра TX.
 *
 * @param base   Указатель на базовый адрес UART
 * @param data   Указатель на начальный адрес буфера для хранения полученных
 *               данных
 * @param length Размер буфера
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_Timeout
 * @retval #UART_Status_RxError
 * @retval #UART_Status_FramingError
 * @retval #UART_Status_ParityError
 */
enum uart_status UART_ReadBlocking(UART_Type *base, uint8_t *data,
    size_t length);

/*!
 * @}
 */

/*!
 * @name Прием данных через прерывания с использованием буферов
 * @{
 */

/*!
 * @brief Инициализация колцевого буфера на прием
 *
 * Эта функция устанавливает кольцевой буфер для заданного дескриптора
 * UART.
 *
 * Когда RX кольцевой буфер используется, полученные данные сохраняются в
 * кольцевом буфере даже если пользователь не вызывает
 * @ref UART_TransferReceiveNonBlocking API. Если в кольцевом буфере уже есть
 * данные, пользователь может получить полученные данные из кольцевого
 * буфера напрямую.
 *
 * При использовании кольцевого буфера RX один байт зарезервирован для
 * внутреннего использования, т.е. если ring_buffer_size равно 32, то для
 * сохранения данных используется только 31 байт.
 *
 * @param base             Указатель на базовый адрес UART
 * @param handle           Указатель на дескриптор
 * @param ring_buffer      Указатель на начальный адрес кольцевого буфера для
 *                         фонового приема (NULL - отключение буфера)
 * @param ring_buffer_size Размер кольцевого буфера
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_TransferStartRingBuffer(UART_Type *base,
    struct uart_handle *handle, uint8_t *ring_buffer, size_t ring_buffer_size);

/*!
 * @brief Прерывает фоновую передачу и удаляет кольцевой буфер
 *
 * Эта функция прерывает фоновую передачу и удаляет кольцевой буфер.
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_TransferStopRingBuffer(UART_Type *base,
    struct uart_handle *handle);

/*!
 * @brief Получить длину данных, принятых в кольцевом RX буфере
 *
 * @param handle Указатель на дескриптор
 *
 * @return Длина данных, принятых в кольцевом RX буфере
 */
size_t UART_TransferGetRxRingBufferLength(struct uart_handle *handle);

/*!
 * @brief Прием данных в асинхронном режиме (без ожидания) по прерыванию
 *
 * Эта функция получает данные по прерыванию. Неблокирующая функция, которая
 * возвращается, не дожидаясь получения всех ожидаемых данных.
 *
 * Если кольцевой буфер RX используется и не пуст, данные из кольцевого буфера
 * копируются в xfer->rx_datа, а параметр received_bytes показывает, сколько
 * байтов скопировано из кольцевого буфера. После копирования, если данных в
 * кольцевом буфере недостаточно для чтения, прием запрос сохраняется драйвером
 * UART. Когда поступают новые данные, запрос на получение обслуживается в
 * первую очередь. Когда все данные получены, драйвер UART уведомляет верхний
 * уровень через функцию обратного вызова c параметром состояния
 * #UART_Status_RxIdle.
 *
 * Например: верхнему уровню требуется 10 байтов, но в кольцевом буфере всего 5
 * байтов. 5 байтов копируются в данные xfer->rx_datа, и эта функция
 * возвращается с параметром received_bytes установлен в 5. Для оставшихся 5
 * байтов вновь поступившие данные сохраняются в xfer->data[5..10]. При
 * получении 5 байтов драйвер UART уведомляет верхний уровень. Если кольцевой
 * буфер RX не включен, эта функция разрешает прерывание и RX для приема данные
 * в xfer->data[]. Когда все данные получены, уведомляется верхний уровень.
 *
 * @param base           Указатель на базовый адрес UART
 * @param handle         Указатель на дескриптор
 * @param xfer           Указатель на структуру с указателем на линейный буфер
 *                       приема или передачи
 * @param received_bytes Указатель на количество байтов полученных напрямую из
 *                       кольцевого буфера
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 * @retval #UART_Status_RxBusy
 */
enum uart_status UART_TransferReceiveNonBlocking(UART_Type *base,
    struct uart_handle *handle, struct uart_transfer *xfer,
    size_t *received_bytes);

/*!
 * @brief Отмена приема данных по прерыванию через линейный буфер в
 *        дескрипторе
 *
 * Не оказывает никакого влияния на работу кольцевого буфера. Для отмены приема
 * через кольцевой буффер используется @ref UART_TransferStopRingBuffer.
 * Пользователь может взять handle->rx_data_size, чтобы узнать сколько еще не
 * принято.
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_TransferAbortReceive(UART_Type *base,
    struct uart_handle *handle);

/*!
 * @brief Возвращает количество принятых байтов
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 * @param count  Указатель, возвращает количество принятых байтов
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 * @retval #UART_Status_NoTransferInProgress
 */
enum uart_status UART_TransferGetReceiveCount(UART_Type *base,
    struct uart_handle *handle, uint32_t *count);

/*!
 * @}
 */

/*!
 * @name Отправка данных через прерывания с использованием буферов
 * @{
 */

/*!
 * @brief Передает буфер данных по прерыванию
 *
 * Отправляет данные по прерыванию. Это неблокирующая функция, которая
 * возвращается напрямую, не дожидаясь записи всех данных в регистр TX. Когда
 * все данные записываются в регистр TX в обработчике IRQ, в прерывании
 * происходит callback с передачей в него #UART_Status_TxIdle в качестве
 * параметра статуса.
 *
 * @note #UART_Status_TxIdle передается на верхний уровень, когда все данные
 *       записаны в регистр TX. Однако это не гарантирует, что все данные будут
 *       отправлены. Перед отключением TX проверьте:
 *       @code{.c}
 *         if (UART_LSR_FlagTxHwEmpty & UART_GetStatusFlags(UART1))
 *       @endcode
 *       чтобы убедиться, что передача завершена.
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 * @param xfer   Указатель на структуру с указателем на линейный буфер приема
 *               или передачи
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 * @retval #UART_Status_TxBusy
 */
enum uart_status UART_TransferSendNonBlocking(UART_Type *base,
    struct uart_handle *handle, struct uart_transfer *xfer);

/*!
 * @brief Останавливает передачу данных, управляемую прерыванием
 *
 * Эта функция останавливает отправку данных, управляемую прерыванием.
 * Пользователь может получить остаток, чтобы узнать сколько байтов еще не
 * отправлено.
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_TransferAbortSend(UART_Type *base,
    struct uart_handle *handle);

/*!
 * @brief Возвращает количество байтов, отправленных в шину
 *
 * Эта функция возвращает количество байтов, отправленных в шину по прерыванию.
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 * @param count  Указатель, возвращает количество байтов отправленных в шину
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 * @retval #UART_Status_NoTransferInProgress
 */
enum uart_status UART_TransferGetSendCount(UART_Type *base,
    struct uart_handle *handle, uint32_t *count);

/*!
 * @brief Инициализация колцевого буфера на передачу
 *
 * Эта функция устанавливает кольцевой буфер Tx для заданного дескриптора
 * UART.
 *
 * @param base             Указатель на базовый адрес UART
 * @param handle           Указатель на дескриптор
 * @param tx_ring_buffer   Указатель на начальный адрес кольцевого буфера
 * @param ring_buffer_size Размер кольцевого буфера
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_TransferStartTxRingBuffer(UART_Type *base,
    struct uart_handle *handle,
    uint8_t *tx_ring_buffer, size_t ring_buffer_size);

/*!
 * @brief Получить количество байт данных для отправки в кольцевом Tx буфере
 *
 * @param handle Указатель на дескриптор
 *
 * @return Количество байт
 */
size_t UART_TransferGetTxRingBufferLength(struct uart_handle *handle);

/*!
 * @brief Передать линейный буфер на передачу через кольцевой буфер
 *
 * @param base    Указатель на базовый адрес UART 
 * @param handle  Указатель на дескриптор
 * @param data    Указатель на буфер, который требуется передать
 * @param length  Обьем байт на передачу , если не удалось поместить в кольцевой
 *                буфер весь обем данных length будет содержать непереданное кол-во
 *                байт  
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 * @retval #UART_Status_TxRingBufferNull
 */
enum uart_status UART_WriteTxRing(UART_Type *base, struct uart_handle *handle,
    const uint8_t *data, size_t *length);

/*!
 * @brief Отключение кольцевого буфера передатчика
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_TransferStopTxRingBuffer(UART_Type *base,
    struct uart_handle *handle);

/*!
 * @}
 */

/*!
 * @name Прием и передача через прерывания с использованием буферов
 * @{
 */

/*!
 * @brief Инициализирует дескриптор UART
 *
 * Для указанного экземпляра UART требуется вызвать эту функцию единожды,
 * чтобы получить инициализированный дескриптор.
 *
 * @param base      Указатель на базовый адрес UART
 * @param handle    Указатель на дескриптор
 * @param callback  Указатель на функцию обратного вызова
 * @param user_data Указатель на параметр функции обратного вызова
 *
 * @retval #UART_Status_Ok
 * @retval #UART_Status_InvalidArgument
 */
enum uart_status UART_TransferCreateHandle(UART_Type *base,
    struct uart_handle *handle, uart_transfer_callback_t callback,
    void *user_data);

/*!
 * @brief Функция-обработчик UART IRQ
 *
 * Эта функция обрабатывает запросы на передачу и прием UART IRQ.
 *
 * @param base   Указатель на базовый адрес UART
 * @param handle Указатель на дескриптор
 */
void UART_TransferHandleIRQ(UART_Type *base, struct uart_handle *handle);

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif /* __cplusplus */

/*!
 * @}
 */

#endif /* HAL_UART_H */
