/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup uart_driver
 * @{
 */

/*!
 * @file hal_uart_dma.c
 *
 * @brief Имплементация дополнения драйвера UART с пересылкой данных через DMA
 */

#include "hal_uart_dma.h"

/* Число UART модулей */
#define UART_UNIT_COUNT (4U)
/* Маска для формирования secure адресов */
#define SECURE_BIT_MSK (0x10000000)

/*!
 * @brief Дескриптор для UART-DMA передач
 */
typedef struct _uart_dma_private_handle {
    UART_Type *base;           /*!< Базовый адрес UART */
    uart_dma_handle_t *handle; /*!< Контекст данных прерывания UART-DMA */
} uart_dma_private_handle_t;

/*!
 * @brief Состояния шины UART
 */
enum {
    UART_TxIdle = 0U, /*!< TX свободен */
    UART_TxBusy = 1U, /*!< TX занят */
    UART_RxIdle = 2U, /*!< RX свободен */
    UART_RxBusy = 3U  /*!< RX занят */
};

/*! @brief Массив базовых адресов UART. */
static UART_Type *uart_bases[UART_UNIT_COUNT] = {UART0, UART1, UART2, UART3};
/*! @brief Массив дескрипторов UART. */
static uart_dma_private_handle_t s_dma_private_handle[UART_UNIT_COUNT];

/*!
 * @brief Функция, включающая TX запрос на передачу данных с DMA
 *
 * @param base   Базовый адрес UART
 * @param enable Разрешение на TX запрос
 */
static void UART_EnableTxDMA(UART_Type *base, bool enable);

/*!
 * @brief Функция, включающая RX запрос на передачу данных с DMA
 *
 * @param base   Базовый адрес UART
 * @param enable Разрешение на RX запрос
 */
static void UART_EnableRxDMA(UART_Type *base, bool enable);

/*!
 * @brief Функция, осуществляющая поиск номера экземпляра UART контроллера
 *
 * @param base Базовый адрес UART
 *
 * @return Номер экземпляра UART
 */
static uint32_t UART_GetInstance(UART_Type *base);

/*!
 * @brief Функция обратного вызова для DMA
          канала, осуществляющего передачу данных в TX буфер
 *
 * @param handle        Дескриптор DMA
 * @param user_data     Указатель на дескриптор UART-DMA передачи
 * @param intmode       Тип прерывания DMA
 */
static void UART_TransferSendDMACallback(dma_handle_t *handle, void *user_data,
    dma_irq_t intmode);

/*!
 * @brief Функция обратного вызова для DMA
          канала, осуществляющего прием данных из RX буфера
 *
 * @param handle        Дескриптор DMA
 * @param user_data     Указатель на дескриптор UART-DMA передачи
 * @param intmode       Тип прерывания DMA
 */
static void UART_TransferReceiveDMACallback(dma_handle_t *handle,
    void *user_data, dma_irq_t intmode);

void UART_TransferCreateHandleDMA(UART_Type *base,
    uart_dma_handle_t *handle,
    uart_dma_transfer_callback_t callback,
    void *user_data, dma_handle_t *tx_dma_handle,
    dma_handle_t *rx_dma_handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    uint32_t instance = UART_GetInstance(base);

    (void) memset(handle, 0, sizeof(uart_dma_handle_t));

    s_dma_private_handle[instance].base = base;
    s_dma_private_handle[instance].handle = handle;

    handle->rx_state = UART_RxIdle;
    handle->tx_state = UART_TxIdle;

    handle->callback = callback;
    handle->user_data = user_data;

    handle->rx_dma_handle = rx_dma_handle;
    handle->tx_dma_handle = tx_dma_handle;

    if (tx_dma_handle != NULL) {
        DMA_SetCallback(tx_dma_handle, UART_TransferSendDMACallback,
            &s_dma_private_handle[instance]);
    }

    if (rx_dma_handle != NULL) {
        DMA_SetCallback(rx_dma_handle, UART_TransferReceiveDMACallback,
            &s_dma_private_handle[instance]);
    }
}

static void UART_TransferSendDMACallback(dma_handle_t *handle, void *user_data,
    dma_irq_t intmode)
{
    assert(handle != NULL);
    assert(user_data != NULL);

    UNUSED(intmode)

    uart_dma_private_handle_t *uart_private_handle =
        (uart_dma_private_handle_t *) user_data;

    /* Ожидание завершения транзакции. */
    UART_WaitWhileActive(uart_private_handle->base);

    /* Выкл TX handshake DMA. */
    UART_EnableTxDMA(uart_private_handle->base, false);

    /* Установка Idle состояния TX .*/
    uart_private_handle->handle->tx_state = UART_TxIdle;

    /* Если установлена функция обратного вызова, запуск функции. */
    if (uart_private_handle->handle->callback != NULL) {
        uart_private_handle->handle->callback(uart_private_handle->base,
            uart_private_handle->handle, UART_Status_TxIdle,
            uart_private_handle->handle->user_data);
    }
}

static void UART_TransferReceiveDMACallback(dma_handle_t *handle,
    void *user_data, dma_irq_t intmode)
{
    assert(handle != NULL);
    assert(user_data != NULL);

    UNUSED(intmode);

    uart_dma_private_handle_t *uart_private_handle =
        (uart_dma_private_handle_t *) user_data;

    /* Ожидание завершения транзакции. */
    UART_WaitWhileActive(uart_private_handle->base);

    /* Выкл RX handshake DMA. */
    UART_EnableRxDMA(uart_private_handle->base, false);

    /* Установка Idle состояния RX. */
    uart_private_handle->handle->rx_state = UART_RxIdle;

    /* Если установлена функция обратного вызова, запуск функции. */
    if (uart_private_handle->handle->callback != NULL) {
        uart_private_handle->handle->callback(uart_private_handle->base,
            uart_private_handle->handle, UART_Status_RxIdle,
            uart_private_handle->handle->user_data);
    }
}

enum uart_status UART_TransferSendDMA(UART_Type *base,
    uart_dma_handle_t *handle, struct uart_transfer *xfer)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(handle->tx_dma_handle != NULL);
    assert(xfer != NULL);
    assert(xfer->tx_data != NULL);
    assert(xfer->data_size != 0U);

    /* Получение номера экземпляра UART. */
    uint32_t instance = UART_GetInstance(base);
    uint32_t dma_base_addr = (uint32_t) handle->tx_dma_handle->base;
    uint32_t dma_base_addr_ns = dma_base_addr & ~SECURE_BIT_MSK;
    uint32_t dma0_base = (uint32_t) DMA0;
    uint32_t dma1_base = (uint32_t) DMA1;

    uint32_t desc_count = 0;

    if ((instance < 2) && (dma_base_addr_ns != dma0_base)) {
        /* DMA0 поддерживает handshake запрос только с UART0 и UART1. */
        return UART_Status_InvalidArgument;
    } else if ((instance > 1) && (dma_base_addr_ns != dma1_base)) {
        /* DMA1 поддерживает handshake запрос только с UART2 и UART3. */
        return UART_Status_InvalidArgument;
    }

    if (instance > 1) {
        instance -= 2;
    }

    /* Адрес TX буфера UART. */
    uint32_t address = (uint32_t) &base->THR;

    /* Если шина занята, выход из функции. */
    if (handle->tx_state == (uint8_t) UART_TxBusy) {
        return UART_Status_TxBusy;
    }

    /* Установка Busy состояния TX. */
    handle->tx_state = (uint8_t) UART_TxBusy;
    /* Установка размера передачи. */
    handle->tx_data_size_all = xfer->data_size;

    /* Включение сигнала dma_req_tx_n. */
    UART_EnableTxDMA(base, true);

    /* Включение handshake TX DMA. */
    DMA_HardwareHandshakeEnable(handle->tx_dma_handle, true,
        DMA_HANDSHAKE_UART0_TX + instance * 2);

    if (xfer->data_size > DMA_AHB_MAX_BLOCK_SIZE) {
        desc_count = DMA_GetDescriptorCount(xfer->data_size,
                DMA_Transfer8BitWidth);
        UART_DMADescriptorInitTX(base, handle->tx_desc, desc_count,
            xfer->data_size, DMA_Transfer8BitWidth, (void *) xfer->tx_data);
        DMA_SubmitChannelDescriptor(handle->tx_dma_handle,
            handle->tx_desc);
    } else {
        /* Установка конфигурации DMA канала. */
        uint64_t xfercfg = DMA_CHANNEL_CTL(xfer->data_size,
                false, false,
                DMA_MemoryToPeripheral_DMA,
                false, false,
                DMA_BurstSize1, DMA_BurstSize1,
                DMA_Incr, DMA_NoChange,
                DMA_Transfer8BitWidth, DMA_Transfer8BitWidth,
                true
            );

        /* Отправка конфигурации DMA. */
        (void) DMA_SubmitChannelTransferParameter(handle->tx_dma_handle, xfercfg,
            (void *) xfer->tx_data, (void *) address);
    }
    /* Включение прерывания по завершении DMA передачи. */
    DMA_EnableChannelInterrupt(handle->tx_dma_handle->base,
        handle->tx_dma_handle->channel, DMA_IntTfr | DMA_IntBlock | DMA_IntError);
    /* Начало передачи данных через DMA в TX буфер. */
    dma_status_t dma_status = DMA_StartTransfer(handle->tx_dma_handle);
    if (dma_status != DMA_Status_Success) {
        return UART_Status_Fail;
    }

    return UART_Status_Ok;
}

enum uart_status UART_TransferReceiveDMA(UART_Type *base,
    uart_dma_handle_t *handle, struct uart_transfer *xfer)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(handle->rx_dma_handle != NULL);
    assert(xfer != NULL);
    assert(xfer->rx_data != NULL);
    assert(xfer->data_size != 0U);

    uint32_t instance = UART_GetInstance(base);
    uint32_t dma_base_addr = (uint32_t) handle->rx_dma_handle->base;
    uint32_t dma_base_addr_ns = dma_base_addr & ~SECURE_BIT_MSK;
    uint32_t dma0_base = (uint32_t) DMA0;
    uint32_t dma1_base = (uint32_t) DMA1;

    uint32_t desc_count = 0;

    if ((instance < 2) && (dma_base_addr_ns != dma0_base)) {
        /* DMA0 поддерживает handshake запрос только с UART0 и UART1. */
        return UART_Status_InvalidArgument;
    } else if ((instance > 1) && (dma_base_addr_ns != dma1_base)) {
        /* DMA1 поддерживает handshake запрос только с UART2 и UART3. */
        return UART_Status_InvalidArgument;
    }

    if (instance > 1) {
        instance -= 2;
    }

    /* Адрес RX буфера UART */
    uint32_t address = (uint32_t) &base->RBR;

    /* Если шина занята, выход из функции. */
    if (handle->rx_state == (uint8_t) UART_RxBusy) {
        return UART_Status_RxBusy;
    }

    /* Установка Busy состояния RX. */
    handle->rx_state = (uint8_t) UART_RxBusy;
    /* Установка размера передачи. */
    handle->rx_data_size_all = xfer->data_size;

    /* Включение сигнала dma_req_rx_n. */
    UART_EnableRxDMA(base, true);
    /* Включение handshake RX DMA. */
    DMA_HardwareHandshakeEnable(handle->rx_dma_handle, true,
        DMA_HANDSHAKE_UART0_RX + instance * 2);

    if (handle->rx_desc != NULL) {
        desc_count = DMA_GetDescriptorCount(xfer->data_size,
                DMA_Transfer8BitWidth);
        UART_DMADescriptorInitRX(base, handle->rx_desc, desc_count,
            xfer->data_size, DMA_Transfer8BitWidth, (void *) xfer->rx_data);
        DMA_SubmitChannelDescriptor(handle->rx_dma_handle,
            handle->rx_desc);
    } else {
        /* Установка конфигурации DMA канала. */
        uint64_t xfercfg = DMA_CHANNEL_CTL(xfer->data_size,
                false, false,
                DMA_PeripheralToMemory_DMA,
                false, false,
                DMA_BurstSize1, DMA_BurstSize1,
                DMA_NoChange, DMA_Incr,
                DMA_Transfer8BitWidth, DMA_Transfer8BitWidth,
                true
            );

        /* Отправка конфигурации DMA. */
        DMA_SubmitChannelTransferParameter(handle->rx_dma_handle, xfercfg,
            (void *) address, (void *) xfer->rx_data);
    }
    /* Включение прерывания по завершении DMA передачи. */
    DMA_EnableChannelInterrupt(handle->rx_dma_handle->base,
        handle->rx_dma_handle->channel, DMA_IntTfr | DMA_IntBlock | DMA_IntError);
    /* Начало передачи данных через DMA в RX буфер. */
    dma_status_t dma_status = DMA_StartTransfer(handle->rx_dma_handle);
    if (dma_status != DMA_Status_Success) {
        return UART_Status_Fail;
    }

    return UART_Status_Ok;
}

void UART_TransferAbortSendDMA(UART_Type *base, uart_dma_handle_t *handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    if (handle->tx_state != UART_TxIdle) {
        /* Остановка DMA. */
        DMA_AbortTransfer(handle->tx_dma_handle);
        /* Установка Idle состояния шины. */
        handle->tx_state = UART_TxIdle;
    }
}

void UART_TransferAbortReceiveDMA(UART_Type *base, uart_dma_handle_t *handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    if (handle->tx_state != UART_RxIdle) {
        /* Остановка DMA. */
        DMA_AbortTransfer(handle->rx_dma_handle);
        /* Установка Idle состояния шины. */
        handle->rx_state = UART_RxIdle;
    }
}

static void UART_EnableTxDMA(UART_Type *base, bool enable)
{
    /* Включение первого режима для DMA. */
    base->FCR |= UART_FCR_DMAM_Msk;
    if (enable) {
        /* Включить FIFO. */
        SET_VAL_MSK(base->FCR, UART_FCR_FIFOE_Msk, UART_FCR_FIFOE_Pos, 1);
        /* Сигнал dma_tx_req_n генерируется, когда FIFO пуст. */
        SET_VAL_MSK(base->FCR, UART_FCR_TET_Msk, UART_FCR_TET_Pos, 0);
    } else {
        /* Выключить FIFO. */
        SET_VAL_MSK(base->FCR, UART_FCR_FIFOE_Msk, UART_FCR_FIFOE_Pos, 0);
    }
}

static void UART_EnableRxDMA(UART_Type *base, bool enable)
{
    /* Включение первого режима для DMA. */
    base->FCR |= UART_FCR_DMAM_Msk;
    if (enable) {
        /* Включить FIFO. */
        SET_VAL_MSK(base->FCR, UART_FCR_FIFOE_Msk, UART_FCR_FIFOE_Pos, 1);
        /* Сигнал dma_rx_req_n генерируется, когда в FIFO есть хотя бы 1 символ. */
        SET_VAL_MSK(base->FCR, UART_FCR_RT_Msk, UART_FCR_RT_Pos, 0);
    } else {
        /* Выключить FIFO. */
        SET_VAL_MSK(base->FCR, UART_FCR_FIFOE_Msk, UART_FCR_FIFOE_Pos, 0);
    }
}

static uint32_t UART_GetInstance(UART_Type *base)
{
    assert(base != NULL);

    uint32_t i;
    uint32_t base_addr = ((uint32_t) base) & ~SECURE_BIT_MSK;
    uint32_t instance_addr;

    for (i = 0; i < UART_UNIT_COUNT; i++) {
        instance_addr = (uint32_t) uart_bases[i];
        if (base_addr == instance_addr) {
            break;
        }
    }
    assert(i < UART_UNIT_COUNT);

    return i;
}

/*!
 * @}
 */
