/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup watchdog_driver Драйвер модуля WDT
 *
 * @brief Драйвер сторожевого таймера
 *
 * Драйвер модуля сторожевого таймера управляет сторожевым таймером.
 */

/*!
 * @addtogroup watchdog_driver
 * @{
 */

/*!
 * @file hal_wdt.h
 *
 * @brief Интерфейс драйвера сторожевого таймера
 */

#ifndef HAL_WDT_H
#define HAL_WDT_H

#include "hal_common.h"

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#define WDT_NUMBER_OF_TIMERS 3 /*!< Количество сдвоенных таймеров */

/*!
 * @brief Статусы драйвера сторожевого таймера
 */
enum wdt_status {
    WDT_Status_Ok              = 0, /*!< Нет ошибок */
    WDT_Status_InvalidArgument = 1, /*!< Недопустимый аргумент */
    WDT_Status_TimerBusy       = 2, /*!< Таймер уже занят */
    WDT_Status_BadConfigure    = 3, /*!< Недопустимая конфигурация */
};

/*!
 * @brief Управление сбросом при таймауте сторожевого таймера
 */
enum wdt_resen_type {
    WDT_ResenDisable = 0, /*!< Сброс запрещён */
    WDT_ResenEnable  = 1, /*!< Сброс разрешен */
};

/*!
 * @brief Управление прерыванием предупреждения от сторожевого таймера\
 * и разрешением работы таймера
 */
enum wdt_inten_type {
    WDT_IntenDisable = 0, /*!< Прерывание запрещёно, таймер не работает */
    WDT_IntenEnable  = 1, /*!< Прерывание разрешено, таймер работает */
};

/*!
 * @brief Структура инициализации сторожевого таймера
 */
struct wdt_config {
    uint32_t load;             /*!< Время срабатывания предупреждения или половина времени таймаута */
    enum wdt_resen_type resen; /*!< Разрещение сброса по таймауту */
    enum wdt_inten_type inten; /*!< Разрешение прерывания и работы сторожевого таймера */
};

/*!
 * @name Интерфейс драйвера
 * @{
 */

/*!
 * @name Инициализация и деинициализации таймера
 * @{
 */

/*!
 * @brief Создание конфигурации по умолчанию
 *
 * Функция инициализации структуры с настройками таймера "по умолчанию":
 * @code
 *  config->enableWwdt = true; ...
 * @endcode
 *
 * @param config Конфигурация таймера
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_GetDefaultConfig(struct wdt_config *config);

/*!
 * @brief Инициализация таймера
 *
 * Функция инициализации таймера с указанными настройками
 *
 * @param base   Таймер
 * @param config Конфигурация таймера
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_Init(WDT_Type *base,
    const struct wdt_config *config);

/*!
 * @brief Деинициализация таймера
 *
 * Функция деинициализации таймера
 *
 * @param base   Таймер
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_Deinit(WDT_Type *base);

/*!
 * @}
 */

/*!
 * @name Функции управления WDT
 * @{
 */

/*!
 * @brief Разрешение работы таймера
 *
 * @param base   Таймер
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_Enable(WDT_Type *base);

/*!
 * @brief Запрещение работы таймера
 *
 * @param base   Таймер
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_Disable(WDT_Type *base);

/*!
 * @brief Получение немаскированных статусов таймера
 *
 * @param base   Таймер
 */
uint32_t WDT_GetStatusFlagsRaw(WDT_Type *base);

/*!
 * @brief Получение маскированных статусов таймера
 *
 * @param base   Таймер
 */
uint32_t WDT_GetStatusFlagsMsk(WDT_Type *base);

/*!
 * @brief Очищение статусов таймера
 *
 * @param base   Таймер
 * @param mask   Маска статусов
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_ClearStatusFlags(WDT_Type *base, uint32_t mask);

/*!
 * @brief Установка времени срабатывания предупреждения
 *
 * Установка времени срабатывания предупреждения
 * в значениях периода частоты тактирования.
 * Значение таймаута таймера равно удвоенному значению времени
 * срабатывания предупреждения
 *
 * @param base          Таймер
 * @param warning_value Время срабатывания,
 * @note минимальное значение warning_value равно 1
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_SetWarningValue(WDT_Type *base, uint32_t warning_value);

/*!
 * @brief Установка времени таймаута таймера
 *
 * Установка времени таймаута таймера в значениях периода частоты тактирования.
 * Значение времени срабатывания предупреждения равно половине значения времени
 * таймаута таймера
 *
 * @param base          Таймер
 * @param timeout_count Время таймаута таймера
 *
 * @note timeout_count должно быть четное, минимальное значение равно 2
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_SetTimeoutValue(WDT_Type *base, uint32_t timeout_count);

/*!
 * @brief Получение значения счетчика
 *
 * @param base Таймер
 *
 * @return Значение счетчика
 */
uint32_t WDT_GetWarningValue(WDT_Type *base);

/*!
 * @brief Обновление времени сторожевого таймера
 *
 * @param base         Таймер
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 */
enum wdt_status WDT_Refresh(WDT_Type *base);

/*!
 * @brief Получение статуса выполнения функции,
 * тип результата которой отличен от enum wdt_status
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_InvalidArgument
 * @retval #WDT_Status_TimerBusy
 * @retval #WDT_Status_BadConfigure
 */
enum wdt_status WDT_GetLastAPIStatus();

/*!
 * @}
 */

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_WDT_H */

/*!
 * @}
 */
