python


import gdb
import os
import struct
import sys


# Script constants
SCRIPT_NAME = "load_image_to_otp.sh"
SCRIPT_PATH = os.path.realpath(sys.argv[0]) + "/"
TCL_UTILS_FILE = SCRIPT_PATH + "utils.tcl"
DUMP_DIR = SCRIPT_PATH + "memory_dump/"

WORD_SIZE                            = 4                               # word size for ELIOT1
OTPCTR_ADDRESS                       = 0x50033000                      # OTPCTR start address
OTPCTR_SIZE                          = 0x1000                          # 4096 bytes (1024 words)
OTP_ADDRESS                          = 0x5008A000                      # OTP start address (NVM view)
OTP_SIZE                             = 0x400                           # 1024 bytes (256 words)
OTP_USER_AREA_CODE_ADDRESS           = 0x1e000000                      # OTP User Area start address (Code view)
OTP_USER_AREA_NVM_ADDRESS            = 0x5008A0B4                      # OTP User Area start address (NVM view)
OTP_USER_AREA_SIZE                   = 0x298                           # 664 bytes (166 words)
OTP_USER_AREA_WORD_COUNT             = OTP_USER_AREA_SIZE // WORD_SIZE # Number of words in OTP User Area (166 words)


# Function: print error and exit
def error_exit(errstr):
    print >> sys.stderr, "{}: {}".format(SCRIPT_NAME, errstr)
    exit(1)


# Function: read binary file, return byte array
def read_binary_file(fname):
    if not os.path.isfile(fname):
        error_exit("No such image file: \'{}\'".format(fname))

    f = open(fname, "rb")
    fdata = bytearray(f.read())
    f.close()
    return fdata


# Function: convert byte array to int array
def bytearr_to_intarr(arr):
    count = len(arr) // WORD_SIZE
    return struct.unpack("{}I".format(count), arr)


# Function: dump memory
def dump_memory(addr, size):
    if (size % WORD_SIZE) != 0:
        error_exit("Invalid memory size to read: {} (must be a divisor of a word size - {})".format(size, WORD_SIZE))

    tmpfile = DUMP_DIR + "tmp.bin"
    gdb.execute("dump memory {} 0x{:08x} 0x{:08x}"
        .format(tmpfile, addr, addr + size))
    dump = read_binary_file(tmpfile)
    os.remove(tmpfile)

    return bytearr_to_intarr(dump)


# Function: dump memory to file
def dump_memory_to_file(fname, addr, size):
    gdb.execute("dump memory {} 0x{:08x} 0x{:08x}"
        .format(fname, addr, addr + size)
    )


# Function: write words to OTP User Area
def otp_user_area_write(rel_addr, words):
    gdb.execute("monitor otp_write_init 1 1 16")
    dump_memory_to_file(DUMP_DIR + "otp-dump-after-write-initialization.bin",
        OTP_ADDRESS, OTP_SIZE)
    dump_memory_to_file(DUMP_DIR + "otpctr-dump-after-write-initialization.bin",
        OTPCTR_ADDRESS, OTPCTR_SIZE)

    for word in words:
        gdb.execute("monitor otp_user_area_write_word {} {}".format(rel_addr, word))
        rel_addr += WORD_SIZE

    dump_memory_to_file(DUMP_DIR + "otp-dump-after-image-burning.bin",
        OTP_ADDRESS, OTP_SIZE)
    dump_memory_to_file(DUMP_DIR + "otpctr-dump-after-image-burning.bin",
        OTPCTR_ADDRESS, OTPCTR_SIZE)


# Main
if not os.path.exists(TCL_UTILS_FILE):
    error_exit("No such utility file: {}".format(TCL_UTILS_FILE))

if fname == "":
    error_exit("Missing file operand")

## create a directory for memory dump files if not exist
try:
    os.mkdir(DUMP_DIR)
except OSError:
    if not os.path.isdir(DUMP_DIR):
        raise

## read image
image = read_binary_file(fname)
if len(image) == 0:
    error_exit("Empty image: '{}'".format(fname))
if (len(image) % WORD_SIZE) != 0:
    error_exit("Not aligned image (word size - {} bytes): '{}', size - {} bytes".format(WORD_SIZE, fname, image_size))

image = bytearr_to_intarr(image)
image_size = len(image)
if offset.isdigit():
    image_offset = int(offset)
else:
    image_offset = 0

if (image_size + image_offset) > OTP_USER_AREA_WORD_COUNT:
    error_exit("Invalid image size ({} words) and/or image offset ({} words): {} + {} > {} (OTP User Area word count)".format(image_size, image_offset, image_size, image_offset, OTP_USER_AREA_WORD_COUNT))

## connect to the board
gdb.execute("set pagination off")
gdb.execute("set confirm off")
gdb.execute("target extended-remote :3333")
gdb.execute("monitor source {}".format(TCL_UTILS_FILE))

## dump otp user area memory current state and print memory map
otp_current = dump_memory(OTP_USER_AREA_NVM_ADDRESS, OTP_USER_AREA_SIZE)
aligned_image = (( 0, ) * image_offset) + (image) + (( 0, ) * (OTP_USER_AREA_WORD_COUNT - image_offset - image_size))
otp_expected = tuple(map(lambda x, y: x | y, otp_current, aligned_image))
if otp_current == otp_expected:
    error_exit("No need to burn the image (current state of OTP User Area memory is identical with the expected one")

print("\nThe {} script is ready to program the OTP memory.".format(SCRIPT_NAME))
print( "\nWarning: The programming process of OTP is irreversible.")
print( "Please check carefully the OTP memory current state, image dump to be burnt and OTP memory expected state; confirm the operation only after checking.")

nvm_addresses = range(OTP_USER_AREA_NVM_ADDRESS, OTP_USER_AREA_NVM_ADDRESS + OTP_USER_AREA_SIZE, WORD_SIZE)
code_addresses = range(OTP_USER_AREA_CODE_ADDRESS, OTP_USER_AREA_CODE_ADDRESS + OTP_USER_AREA_SIZE, WORD_SIZE)
print ("\nRead image: size - {} words, offset - {} words\n".format(image_size, image_offset))
print ("|{}|{}|{}|{}|".format('-' * 7, '-' * 16, '-' * 16, '-' * 16))
print ("|{:^7s}|{:^16s}|{:^16s}|{:^16s}|".format("Word", "OTP (current)", "Image", "OTP (expected)"))
print ("|{}|{}|{}|{}|".format('-' * 7, '-' * 16, '-' * 16, '-' * 16))
for i, (current_word, image_word, expected_word) in enumerate(zip(otp_current, aligned_image, otp_expected)):
    print ("|  {:3d}  |    {:08x}    |    {:08x}    |    {:08x}    |".format(i + 1, current_word, image_word, expected_word))
print ("|{}|{}|{}|{}|".format('-' * 7, '-' * 16, '-' * 16, '-' * 16))
dump_memory_to_file(DUMP_DIR + "otp-dump-after-connection.bin",
        OTP_ADDRESS, OTP_SIZE)
dump_memory_to_file(DUMP_DIR + "otpctr-dump-after-connection.bin",
        OTPCTR_ADDRESS, OTPCTR_SIZE)

## burn otp user area memory with image
print ('\nBurn the image? [y / n (any other symbol)]')
answer = sys.stdin.readline()
if (answer != "y\n") and (answer != "y\r\n"):
    error_exit("Image burn cancelled")

otp_user_area_write(image_offset * WORD_SIZE, image)
print ("\nBurn image: {} words have been written to ELIOT1 OTP User Area memory".format(image_size, image_size * WORD_SIZE, image_size * WORD_SIZE))

## check written image
otp_result = dump_memory(OTP_USER_AREA_NVM_ADDRESS, OTP_USER_AREA_SIZE)
print ("\nCheck burnt image: {}\n".format("correct" if (otp_result == otp_expected) else "incorrect"))
if otp_result != otp_expected:
    print ("|{}|{}|{}|{}|".format('-' * 7, '-' * 16, '-' * 16, '-' * 10))
    print ("|{:^7s}|{:^16s}|{:^16s}|{:^10s}|".format("Word", "OTP (expected)", "OTP (result)", "Equality"))
    print ("|{}|{}|{}|{}|".format('-' * 7, '-' * 16, '-' * 16, '-' * 10))
    for i, (expected_word, result_word) in enumerate(zip(otp_expected, otp_result)):
        print ("|  {:3d}  |    {:08x}    |    {:08x}    | {:^8s} |".format(i + 1, expected_word, result_word, "X" if (expected_word != result_word) else ""))
    print ("|{}|{}|{}|{}|".format('-' * 7, '-' * 16, '-' * 16, '-' * 10))


end


quit
