# -----------------------------------------------------------------------------
# Memory map
# -----------------------------------------------------------------------------

set IOCTR_BASE_ADDR         0x50030000
set CLKCTR_BASE_ADDR        0x50031000
set PWRCTR_BASE_ADDR        0x50032000
set OTPCTR_BASE_ADDR        0x50033000
set RWC_BASE_ADDR           0x50403000
set JTM_BASE_ADDR           0x50404000
set CRYPTO_BASE_ADDR        0x50088000
set FCTR_BASE_ADDR          0x50400000
set SMC_BASE_ADDR           0x50282000
set GPIO0_BASE_ADDR         0x5010A000
set GPIO1_BASE_ADDR         0x5010B000
set GPIO2_BASE_ADDR         0x5010C000
set GPIO3_BASE_ADDR         0x5010D000

# IOCTR
set reg_addr_ar(IOCTR_PA_MODE)      [expr $IOCTR_BASE_ADDR + 0x000 + 0x000]
set reg_addr_ar(IOCTR_PA_AFL)       [expr $IOCTR_BASE_ADDR + 0x000 + 0x004]
set reg_addr_ar(IOCTR_PA_AFH)       [expr $IOCTR_BASE_ADDR + 0x000 + 0x008]
set reg_addr_ar(IOCTR_PA_PUPD)      [expr $IOCTR_BASE_ADDR + 0x000 + 0x00C]
set reg_addr_ar(IOCTR_PA_DS)        [expr $IOCTR_BASE_ADDR + 0x000 + 0x010]
set reg_addr_ar(IOCTR_PA_SR)        [expr $IOCTR_BASE_ADDR + 0x000 + 0x014]
set reg_addr_ar(IOCTR_PA_OTYPE)     [expr $IOCTR_BASE_ADDR + 0x000 + 0x018]
set reg_addr_ar(IOCTR_PA_ITYPE)     [expr $IOCTR_BASE_ADDR + 0x000 + 0x01C]
set reg_addr_ar(IOCTR_PA_I2CMODE)   [expr $IOCTR_BASE_ADDR + 0x000 + 0x020]

set reg_addr_ar(IOCTR_PB_MODE)      [expr $IOCTR_BASE_ADDR + 0x100 + 0x000]
set reg_addr_ar(IOCTR_PB_AFL)       [expr $IOCTR_BASE_ADDR + 0x100 + 0x004]
set reg_addr_ar(IOCTR_PB_AFH)       [expr $IOCTR_BASE_ADDR + 0x100 + 0x008]
set reg_addr_ar(IOCTR_PB_PUPD)      [expr $IOCTR_BASE_ADDR + 0x100 + 0x00C]
set reg_addr_ar(IOCTR_PB_DS)        [expr $IOCTR_BASE_ADDR + 0x100 + 0x010]
set reg_addr_ar(IOCTR_PB_SR)        [expr $IOCTR_BASE_ADDR + 0x100 + 0x014]
set reg_addr_ar(IOCTR_PB_OTYPE)     [expr $IOCTR_BASE_ADDR + 0x100 + 0x018]
set reg_addr_ar(IOCTR_PB_ITYPE)     [expr $IOCTR_BASE_ADDR + 0x100 + 0x01C]
set reg_addr_ar(IOCTR_PB_I2CMODE)   [expr $IOCTR_BASE_ADDR + 0x100 + 0x020]

set reg_addr_ar(IOCTR_PC_MODE)      [expr $IOCTR_BASE_ADDR + 0x200 + 0x000]
set reg_addr_ar(IOCTR_PC_AFL)       [expr $IOCTR_BASE_ADDR + 0x200 + 0x004]
set reg_addr_ar(IOCTR_PC_AFH)       [expr $IOCTR_BASE_ADDR + 0x200 + 0x008]
set reg_addr_ar(IOCTR_PC_PUPD)      [expr $IOCTR_BASE_ADDR + 0x200 + 0x00C]
set reg_addr_ar(IOCTR_PC_DS)        [expr $IOCTR_BASE_ADDR + 0x200 + 0x010]
set reg_addr_ar(IOCTR_PC_SR)        [expr $IOCTR_BASE_ADDR + 0x200 + 0x014]
set reg_addr_ar(IOCTR_PC_OTYPE)     [expr $IOCTR_BASE_ADDR + 0x200 + 0x018]
set reg_addr_ar(IOCTR_PC_ITYPE)     [expr $IOCTR_BASE_ADDR + 0x200 + 0x01C]

set reg_addr_ar(IOCTR_PD_MODE)      [expr $IOCTR_BASE_ADDR + 0x300 + 0x000]
set reg_addr_ar(IOCTR_PD_AFL)       [expr $IOCTR_BASE_ADDR + 0x300 + 0x004]
set reg_addr_ar(IOCTR_PD_AFH)       [expr $IOCTR_BASE_ADDR + 0x300 + 0x008]
set reg_addr_ar(IOCTR_PD_PUPD)      [expr $IOCTR_BASE_ADDR + 0x300 + 0x00C]
set reg_addr_ar(IOCTR_PD_DS)        [expr $IOCTR_BASE_ADDR + 0x300 + 0x010]
set reg_addr_ar(IOCTR_PD_SR)        [expr $IOCTR_BASE_ADDR + 0x300 + 0x014]
set reg_addr_ar(IOCTR_PD_OTYPE)     [expr $IOCTR_BASE_ADDR + 0x300 + 0x018]
set reg_addr_ar(IOCTR_PD_ITYPE)     [expr $IOCTR_BASE_ADDR + 0x300 + 0x01C]

# CLKCTR
set reg_addr_ar(CLKCTR_PLLCFG)      [expr $CLKCTR_BASE_ADDR + 0x000]
set reg_addr_ar(CLKCTR_PLLDIAG)     [expr $CLKCTR_BASE_ADDR + 0x004]
set reg_addr_ar(CLKCTR_CFG)         [expr $CLKCTR_BASE_ADDR + 0x008]
set reg_addr_ar(CLKCTR_CLKFORCE)    [expr $CLKCTR_BASE_ADDR + 0x00C]
set reg_addr_ar(CLKCTR_FCLKDIV)     [expr $CLKCTR_BASE_ADDR + 0x010]
set reg_addr_ar(CLKCTR_SYSCLKDIV)   [expr $CLKCTR_BASE_ADDR + 0x014]
set reg_addr_ar(CLKCTR_QSPICLKDIV)  [expr $CLKCTR_BASE_ADDR + 0x018]
set reg_addr_ar(CLKCTR_I2SCLKDIV)   [expr $CLKCTR_BASE_ADDR + 0x01C]
set reg_addr_ar(CLKCTR_MCODIV)      [expr $CLKCTR_BASE_ADDR + 0x020]
set reg_addr_ar(CLKCTR_GNSSCLKIV)   [expr $CLKCTR_BASE_ADDR + 0x024]
set reg_addr_ar(CLKCTR_HFITRIM)     [expr $CLKCTR_BASE_ADDR + 0x030]

# PWRCTR
set reg_addr_ar(PWRCTR_CFG)         [expr $PWRCTR_BASE_ADDR + 0x000]
set reg_addr_ar(PWRCTR_STAT)        [expr $PWRCTR_BASE_ADDR + 0x004]
set reg_addr_ar(PWRCTR_RUNCFG)      [expr $PWRCTR_BASE_ADDR + 0x008]
set reg_addr_ar(PWRCTR_STDBYCFG)    [expr $PWRCTR_BASE_ADDR + 0x00C]
set reg_addr_ar(PWRCTR_VLEVEL)      [expr $PWRCTR_BASE_ADDR + 0x010]
set reg_addr_ar(PWRCTR_TRIM)        [expr $PWRCTR_BASE_ADDR + 0x014]

# RWC
set reg_addr_ar(RWC_STR)            [expr $RWC_BASE_ADDR + 0x000]
set reg_addr_ar(RWC_RWDATA)         [expr $RWC_BASE_ADDR + 0x004]
set reg_addr_ar(RWC_CTRL)           [expr $RWC_BASE_ADDR + 0x008]
set reg_addr_ar(RWC_INTCLR)         [expr $RWC_BASE_ADDR + 0x00C]

# JTM
set reg_addr_ar(JTM_CFG0)           [expr $JTM_BASE_ADDR + 0x000]
set reg_addr_ar(JTM_CFG1)           [expr $JTM_BASE_ADDR + 0x004]
set reg_addr_ar(JTM_CTR)            [expr $JTM_BASE_ADDR + 0x008]
set reg_addr_ar(JTM_STAT)           [expr $JTM_BASE_ADDR + 0x00C]
set reg_addr_ar(JTM_INTEN)          [expr $JTM_BASE_ADDR + 0x010]
set reg_addr_ar(JTM_INTSTAT)        [expr $JTM_BASE_ADDR + 0x014]
set reg_addr_ar(JTM_INTCLR)         [expr $JTM_BASE_ADDR + 0x018]

# CRYPTO
set reg_addr_ar(CRYPTO_HOST_DCU_EN0) [expr $CRYPTO_BASE_ADDR + 0x1E00]
set reg_addr_ar(CRYPTO_PROG_COMP)    [expr $CRYPTO_BASE_ADDR + 0x1F04]

# OTPCTR
set reg_addr_ar(OTPCTR_DAP_DR)      [expr $OTPCTR_BASE_ADDR + 0x000]
set reg_addr_ar(OTPCTR_DAP_ER)      [expr $OTPCTR_BASE_ADDR + 0x020]
set reg_addr_ar(OTPCTR_DAP_RQ0)     [expr $OTPCTR_BASE_ADDR + 0x030]
set reg_addr_ar(OTPCTR_DAP_RQ1)     [expr $OTPCTR_BASE_ADDR + 0x034]
set reg_addr_ar(OTPCTR_DAP_RQ2)     [expr $OTPCTR_BASE_ADDR + 0x038]
set reg_addr_ar(OTPCTR_DAP_CQCTRL)  [expr $OTPCTR_BASE_ADDR + 0x03C]
set reg_addr_ar(OTPCTR_PMC_RQ0)     [expr $OTPCTR_BASE_ADDR + 0x0B0]
set reg_addr_ar(OTPCTR_PMC_RQ1)     [expr $OTPCTR_BASE_ADDR + 0x0B4]
set reg_addr_ar(OTPCTR_PMC_RQ2)     [expr $OTPCTR_BASE_ADDR + 0x0B8]
set reg_addr_ar(OTPCTR_PMC_CQCTRL)  [expr $OTPCTR_BASE_ADDR + 0x0BC]
set reg_addr_ar(OTPCTR_DAP_CMD)     [expr $OTPCTR_BASE_ADDR + 0x100]
set reg_addr_ar(OTPCTR_PMC_CMD)     [expr $OTPCTR_BASE_ADDR + 0x180]

# FCTR
set reg_addr_ar(FCTR_IRQ_ENABLE_SET)    [expr $FCTR_BASE_ADDR + 0x0000]
set reg_addr_ar(FCTR_IRQ_ENABLE_CLR)    [expr $FCTR_BASE_ADDR + 0x0004]
set reg_addr_ar(FCTR_IRQ_STATUS_SET)    [expr $FCTR_BASE_ADDR + 0x0008]
set reg_addr_ar(FCTR_IRQ_STATUS_CLR)    [expr $FCTR_BASE_ADDR + 0x000C]
set reg_addr_ar(FCTR_IRQ_MASKED_STATUS) [expr $FCTR_BASE_ADDR + 0x0010]
set reg_addr_ar(FCTR_CTRL)              [expr $FCTR_BASE_ADDR + 0x0014]
set reg_addr_ar(FCTR_STATUS)            [expr $FCTR_BASE_ADDR + 0x0018]
set reg_addr_ar(FCTR_ADDR)              [expr $FCTR_BASE_ADDR + 0x001C]
set reg_addr_ar(FCTR_DATA0)             [expr $FCTR_BASE_ADDR + 0x0020]
set reg_addr_ar(FCTR_PSP_CTRL)          [expr $FCTR_BASE_ADDR + 0x1000]
set reg_addr_ar(FCTR_PSP_TM1)           [expr $FCTR_BASE_ADDR + 0x1004]
set reg_addr_ar(FCTR_PSP_TM2)           [expr $FCTR_BASE_ADDR + 0x1008]
set reg_addr_ar(FCTR_PSP_STATUS)        [expr $FCTR_BASE_ADDR + 0x100C]

# SMC
set reg_addr_ar(SMC_SET_CYCLES)     [expr $SMC_BASE_ADDR + 0x1014]
set reg_addr_ar(SMC_SET_OPMODE)     [expr $SMC_BASE_ADDR + 0x1018]
set reg_addr_ar(SMC_DIRECT_CMD)     [expr $SMC_BASE_ADDR + 0x1010]
set reg_addr_ar(SMC_USER_CONFIG)    [expr $SMC_BASE_ADDR + 0x1204]

# GPIO
set reg_addr_ar(GPIO0_DATA_IN)      [expr $GPIO0_BASE_ADDR + 0x000]
set reg_addr_ar(GPIO0_DATA_OUT)     [expr $GPIO0_BASE_ADDR + 0x004]
set reg_addr_ar(GPIO0_OUT_EN_SET)   [expr $GPIO0_BASE_ADDR + 0x010]
set reg_addr_ar(GPIO0_OUT_EN_CLR)   [expr $GPIO0_BASE_ADDR + 0x014]

# -----------------------------------------------------------------------------
# Common
# -----------------------------------------------------------------------------

proc read_reg_by_name {reg_name} {
    global reg_addr_ar

    if {[info exists reg_addr_ar($reg_name)] == 0} {
        error [format "Register %s doesn't exist" $reg_name]
    }
    mem2array val 32 $reg_addr_ar($reg_name) 1
    return $val(0)
}

proc write_reg_by_name {reg_name val} {
    global reg_addr_ar

    if {[info exists reg_addr_ar($reg_name)] == 0} {
        error [format "Register %s doesn't exist" $reg_name]
    }
    mww $reg_addr_ar($reg_name) $val
}

# -----------------------------------------------------------------------------
# IO control
# -----------------------------------------------------------------------------

proc set_port_mode {port pin mode {af "-1"}} {
    array set mode_values {HIZ 0 GPIO 1 AF 2}
    set port [string toupper $port]
    set mode [string toupper $mode]

    if {$port != "PA" && $port != "PB" && $port != "PC" && $port != "PD"} {
        error [format "Wrong port name %s" $port]
    }

    if {$pin < 0 || $pin > 15} {
        error [format "Wrong pin %s" $pin]
    }

    if {$mode != "HIZ" && $mode != "GPIO" && $mode != "AF"} {
        error [format "Wrong port mode %s" $mode]
    }

    if {$mode == "AF" && ($af < 0 || $af > 7)} {
        error "Port function is not correctly specified"
    }

    echo -n [format "Setting port %s\[%d\] mode to %s" $port $pin $mode]
    if {$mode == "AF"} {
        echo -n [format ", function AF%d" $af]
    }
    echo ""

    set mode_val $mode_values($mode)
    set reg_name [format "IOCTR_%s_MODE" $port]

    set val [read_reg_by_name $reg_name]
    set val [expr ($val & ~(3 << $pin*2)) | ($mode_val << $pin*2)]
    write_reg_by_name $reg_name $val

    if {$mode == "AF"} {
        if {$pin < 8} {
            set reg_name [format "IOCTR_%s_AFL" $port]
        } else {
            set reg_name [format "IOCTR_%s_AFH" $port]
        }
        set val [read_reg_by_name $reg_name]
        set val [expr ($val & ~(7 << ($pin & 7)*4)) | ($af << ($pin & 7)*4)]
        write_reg_by_name $reg_name $val
    }
}

# -----------------------------------------------------------------------------
# Clock control
# -----------------------------------------------------------------------------

proc set_pll_freq {mult} {
    set val [expr (($mult - 1) & 0x1FF)]
    echo [format "Setting PLL Multiplier = %d" $mult]
    write_reg_by_name CLKCTR_PLLCFG $val
}

proc set_pll_freq_man {od nf nr} {
    echo [format "Setting PLL: OD = %d, NF = %d, NR = %d" $od $nf $nr]
    # SEL = 1; MAN = 1
    set val [expr 0x1 | (0x1 << 9)]
    set val [expr $val | (($od & 0xF) << 10)]
    set val [expr $val | (($nf & 0x1FFF) << 14)]
    set val [expr $val | (($nr & 0xF) << 27)]
    write_reg_by_name CLKCTR_PLLCFG $val
}

proc set_pllref_sel {sel} {
    if {$sel == 0} {
        echo "Setting PLLREF_SEL to HFICLK"
    } elseif {$sel == 1} {
        echo "Setting PLLREF_SEL to XTICLK"
    } else {
        error [format "Wrong PLLREF_SEL value $s" $sel]
    }

    set val [expr [read_reg_by_name CLKCTR_CFG] & ~(1 << 6)]
    set val [expr $val | ($sel << 6)]
    write_reg_by_name CLKCTR_CFG $val
}

proc get_pll_lock {} {
    set lock [expr [read_reg_by_name CLKCTR_PLLCFG] >> 31]
    set lock_macro [expr ([read_reg_by_name CLKCTR_PLLDIAG] >> 4) & 1]
    echo [format "PLL LOCK       = %d" $lock]
    echo [format "PLL MACRO_LOCK = %d" $lock_macro]
}

proc set_mainclk_sel {sel} {
    if {$sel == 0 || $sel == "HFICLK"} {
        echo "Setting MAINCLK_SEL to HFICLK"
    } elseif {$sel == 1 || $sel == "XTICLK"} {
        echo "Setting MAINCLK_SEL to XTICLK"
    } elseif {$sel == 2 || $sel == "PLLCLK"} {
        echo "Setting MAINCLK_SEL to PLLCLK"
    } else {
        error [format "Wrong MAINCLK_SEL value %s" $sel]
    }

    set val [expr [read_reg_by_name CLKCTR_CFG] & ~(3 << 4)]
    set val [expr $val | ($sel << 4)]
    write_reg_by_name CLKCTR_CFG $val
}

proc set_clkdiv {clk_name divider} {
    set clk_name [string toupper $clk_name]
    switch $clk_name {
        FCLK     {set reg_name CLKCTR_FCLKDIV}
        SYSCLK   {set reg_name CLKCTR_SYSCLKDIV}
        QSPICLK  {set reg_name CLKCTR_QSPICLKDIV}
        I2SCLK   {set reg_name CLKCTR_I2SCLKDIV}
        MCO      {set reg_name CLKCTR_MCODIV}
        GNSSCLK  {set reg_name CLKCTR_GNSSCLKIV}
        default  {
            error [format "Wrong clock name %s" $clk_name]
        }
    }

    # if {$reg_name == CLKCTR_GNSSCLKIV} {
    #     set divider_max 7
    # } else {
    #     set divider_max 31
    # }

    # if {$divider < 0 || $divider > $divider_max} {
    #     error [format "Wrong %s divider value: %d" $clk_name $divider]
    # }

    write_reg_by_name $reg_name [expr $divider - 1]

    for {set i 0} {$i < 100} {incr i} {
        set val [expr [read_reg_by_name $reg_name] >> 16]
        # GNSSCLKDIV_CUR is not readable in r2p2
        if {($val == [expr $divider - 1]) || ($clk_name == "GNSSCLK")} {
            echo [format "%s divider has been set to %d" $clk_name $divider]
            return 0
        }
    }
    error [format "%s divider hasn't been set" $clk_name]
}

proc mco_enable {en} {
    if {$en == 1} {
        echo "Enabling MCO"

        # Configure IOCTR
        set_port_mode PA 15 AF 0

        # Enable MCO output
        set val [read_reg_by_name CLKCTR_CFG]
        set val [expr $val | (1 << 13)]
        write_reg_by_name CLKCTR_CFG $val
    } else {
        echo "Disabling MCO"
        set val [read_reg_by_name CLKCTR_CFG]
        set val [expr $val & ~(1 << 13)]
        write_reg_by_name CLKCTR_CFG $val
    }
}

proc mco_config {clk_name divider} {
    set clk_name [string toupper $clk_name]

    switch $clk_name {
        HFICLK   {set sel 0}
        RTCCLK   {set sel 1}
        LPCLK    {set sel 2}
        MAINCLK  {set sel 3}
        PLLCLK   {set sel 4}
        SYSCLK   {set sel 5}
        FCLK_INT {set sel 6}
        FCLK     {set sel 7}
        default  {
            error [format "Wrong clock name %s" $clk_name]
        }
    }

    echo [format "Setting MCO: Clock = %s, Divider = %d" $clk_name $divider]

    set val [read_reg_by_name CLKCTR_CFG]
    set val [expr ($val & ~(0x7 << 9)) | ($sel << 9)]
    write_reg_by_name CLKCTR_CFG $val

    set_clkdiv MCO $divider
}

# -----------------------------------------------------------------------------
# Power control
# -----------------------------------------------------------------------------

proc set_vlevel {vlevel} {
    if {$vlevel < 0 || $vlevel > 2} {
        error [format "Wrong VLEVEL value %s" $vlevel]
    }

    set val [read_reg_by_name PWRCTR_RUNCFG]
    set cur_vlevel [expr $val & 3]

    # if {($cur_vlevel == 0 && $vlevel == 2) || ($cur_vlevel == 2 && $vlevel == 0)} {
    #     # Make smooth change
    #     write_reg_by_name PWRCTR_RUNCFG $val
    # }
    set val [expr ($val & ~3) | $vlevel]
    write_reg_by_name PWRCTR_RUNCFG $val
}

# -----------------------------------------------------------------------------
# RWC
# -----------------------------------------------------------------------------

proc read_rwc_reg {reg_name} {
    array set reg_addrs {TICKGEN 1 TRIMLOAD 2 TIME 3 ALARM 4 TRIM 5 CONFIG 6 GPR 7 WAKECFG 8}

    set reg_name [string toupper $reg_name]

    if {[info exists reg_addrs($reg_name)] == 0} {
        error [format "Wrong RWC register %s" $reg_name]
    }
    set reg_addr $reg_addrs($reg_name)

    set val [read_reg_by_name RWC_CTRL]
    # Clear RSEL field
    set val [expr $val & ~0xF0]
    # Set RSEL field
    set val [expr $val | (($reg_addr & 0xF) << 4)]
    # Set CMD = read
    set val [expr $val | 0x2]
    write_reg_by_name RWC_CTRL $val

    # Wait command completion
    for {set i 0} {$i < 100} {incr i} {
        set val [expr [read_reg_by_name RWC_CTRL] & 0x3]
        if {$val == 0} {
            set val [read_reg_by_name RWC_RWDATA]
            # echo [format "RWC register %s has been read 0x%x" $reg_name $val]
            return [format "0x%x" $val]
        }
    }
    error [format "Unable to read RWC register %s" $reg_name]
}

proc write_rwc_reg {reg_name data} {
    array set reg_addrs {TICKGEN 1 TRIMLOAD 2 TIME 3 ALARM 4 TRIM 5 CONFIG 6 GPR 7 WAKECFG 8}

    set reg_name [string toupper $reg_name]

    if {[info exists reg_addrs($reg_name)] == 0} {
        error [format "Wrong RWC register %s" $reg_name]
    }
    set reg_addr $reg_addrs($reg_name)

    # Set write data
    write_reg_by_name RWC_RWDATA $data
    set val [read_reg_by_name RWC_CTRL]
    # Clear RSEL field
    set val [expr $val & ~0xF0]
    # Set RSEL field
    set val [expr $val | (($reg_addr & 0xF) << 4)]
    # Set CMD = write
    set val [expr $val | 0x1]
    write_reg_by_name RWC_CTRL $val

    # Wait command completion
    for {set i 0} {$i < 100} {incr i} {
        set val [expr [read_reg_by_name RWC_CTRL] & 0x3]
        if {$val == 0} {
            # echo [format "RWC register %s has been written" $reg_name]
            return 0
        }
    }
    error [format "RWC register %s hasn't been written" $reg_name]
}

proc set_trim_lfi {trim} {
    set val [read_rwc_reg TRIM]
    set val [expr $val & ~(0x7FF << 11)]
    set val [expr $val | (($trim & 0x7FF) << 11)]
    write_rwc_reg TRIM $val
}

proc get_trim_lfi {} {
    set val [read_rwc_reg TRIM]
    return [format "0x%x" [expr ($val >> 11) & 0x7FF]]
}

proc set_trim_lfe {trim} {
    set val [read_rwc_reg TRIM]
    set val [expr $val & ~(0x7FF << 0)]
    set val [expr $val | (($trim & 0x7FF) << 0)]
    write_rwc_reg TRIM $val
}

proc get_trim_lfe {} {
    set val [read_rwc_reg TRIM]
    return [format "0x%x" [expr ($val >> 0) & 0x7FF]]
}

proc set_rwc_osc {osc} {
    set config [read_rwc_reg CONFIG]
    if {[string toupper $osc] == "LFE"} {
        set config [expr $config | (0x1 << 4)]
        write_rwc_reg CONFIG $config
    } elseif {[string toupper $osc] == "LFI"} {
        set config [expr $config & ~(0x1 << 4)]
        write_rwc_reg CONFIG $config
    } else {
        error [format "Wrong RWC oscillator specified: %s. Valid values are: LFE, LFI" $osc]
    }
}


# -----------------------------------------------------------------------------
# OTP
# -----------------------------------------------------------------------------

proc otp_write_init {{ecc_gen 0} {brp_gen 0} {freq 16}} {
    if {$freq != 16} {
        error "SYSCLK must be 16 MHz"
    }

    # Change FSM's entry state to configure ECC generation
    set pmc_cq [expr [read_reg_by_name OTPCTR_PMC_CQCTRL] & 0xFFFF]
    if {$ecc_gen == 1 && $brp_gen == 1} {
        # ENTRY = PROG_START (0x00)
        write_reg_by_name OTPCTR_PMC_CQCTRL [expr ($pmc_cq & ~0x1F) | 0x00]
    } elseif {$ecc_gen == 0 && $brp_gen == 1} {
        # ENTRY = PROG_BRP (0x04)
        write_reg_by_name OTPCTR_PMC_CQCTRL [expr ($pmc_cq & ~0x1F) | 0x04]
    } elseif {$ecc_gen == 0 && $brp_gen == 0} {
        # ENTRY = PROG_PROGRAM (0x0A)
        write_reg_by_name OTPCTR_PMC_CQCTRL [expr ($pmc_cq & ~0x1F) | 0x0A]
    } else {
        error "It's not possible to disable BRP generation with ECC enabled"
    }

    # Configure OTP controller for write operation
    # IPS_VQQ = 0x5
    write_reg_by_name OTPCTR_DAP_RQ0 [expr ([read_reg_by_name OTPCTR_DAP_RQ0] & ~(0x7 << 16)) | (0x5 << 16)]
    # IPS_VPP = 0x5
    write_reg_by_name OTPCTR_DAP_RQ0 [expr ([read_reg_by_name OTPCTR_DAP_RQ0] & ~(0x7 << 19)) | (0x5 << 19)]
    # SHF_CKDEL
    write_reg_by_name OTPCTR_DAP_RQ1 [expr ([read_reg_by_name OTPCTR_DAP_RQ1] & ~(0x1F << 16)) | (0x0 << 16)]


    # PROG_MODEx
    write_reg_by_name OTPCTR_PMC_RQ0 0xE8164802
    write_reg_by_name OTPCTR_PMC_RQ1 0x4C017D12
    # Timing
    #   0:  SHORT_READ_PW
    #   4:  LONG_READ_PW
    #   8:  BRP_CHECK_READ_PW
    #   9:  PRE_PROG_READ_PW
    #   10: POST_PROG_READ_PW
    #   11: POST_SOAK_READ_PW
    #   12: COMPARE_READ_PW
    #   13: READ_RECOVERY
    #   14: PROG_RECOVERY
    #   16: PROG_PW (30 us)
    #   19: SOAK_PW (120 us)
    #   22: MODE_SETTING_TIME (657 ns)
    #   24: DAP_ADDR
    write_reg_by_name OTPCTR_PMC_RQ2 [expr (0x0 <<  0) | (0x2 <<  4) | (0x1 <<  8) | (0x0 <<  9) | \
                                           (0x0 << 10) | (0x0 << 11) | (0x0 << 12) | (0x0 << 13) | \
                                           (0x0 << 14) | (0x3 << 16) | (0x2 << 19) | (0x0 << 22) | \
                                           (0x2 << 24)]
}

proc otp_write {addr data} {
    global CRYPTO_BASE_ADDR

    mww [expr $CRYPTO_BASE_ADDR + 0x2000 + $addr] $data
    while {[read_reg_by_name CRYPTO_PROG_COMP] == 0} {
        echo "Waiting for programming to complete ..."
    }
    echo "OTP Programming completed"
}

proc otp_user_area_write_word {rel_addr data} {
    global CRYPTO_BASE_ADDR

    mww [expr $CRYPTO_BASE_ADDR + 0x2000 + 0x00B4 + $rel_addr] $data
    while {[read_reg_by_name CRYPTO_PROG_COMP] == 0} {
    }
}

proc otp_read_word_sys {addr} {
    # Clear REDUND
    write_reg_by_name OTPCTR_DAP_RQ0 [expr [read_reg_by_name OTPCTR_DAP_RQ0] & ~(1 << 5)]
    # Set PASS, disable ECC and BRP
    write_reg_by_name OTPCTR_DAP_RQ2 [expr [read_reg_by_name OTPCTR_DAP_RQ2] | (1 << 21) | (1 << 19) | (1 << 16)]
    # Set address
    write_reg_by_name OTPCTR_DAP_CQCTRL [expr $addr & 0xFF]
    # Send command
    write_reg_by_name OTPCTR_DAP_CMD 0x40
    # Read data reg
    set val [read_reg_by_name OTPCTR_DAP_DR]
    set val [expr $val | ([read_reg_by_name OTPCTR_DAP_ER] << 32)]

    return $val
}

proc otp_dump_sys {} {
    poll off

    echo "Reading OTP through the system interface"
    echo " \[address\]: {ecc, data}"
    for {set i 0} {$i < 256} {incr i} {
        set data [otp_read_word_sys $i]
        echo -n [format "    \[0x%02x\]: 0x%010x" $i $data]
        if {$data != 0} {
            echo " !"
        } else {
            echo ""
        }
    }

    poll on
}

proc otp_bist_run_single {start_addr size prog_en} {
    write_reg_by_name OTPCTR_DAP_RQ0 0x042d4802
    write_reg_by_name OTPCTR_DAP_RQ1 0x01000000
    write_reg_by_name OTPCTR_DAP_RQ2 0x00000008
    write_reg_by_name OTPCTR_DAP_CQCTRL $start_addr
    write_reg_by_name OTPCTR_PMC_RQ0 0xf8014802
    write_reg_by_name OTPCTR_PMC_RQ1 [expr 0x00004e01 | ($size << 16)]
    write_reg_by_name OTPCTR_PMC_RQ2 0x02004d00
    write_reg_by_name OTPCTR_PMC_CQCTRL [expr (0x9 << 24) | (($prog_en & 1) << 14)]

    # echo [format "Starting BIST from address 0x%x" $start_addr]
    # Send START instruction
    write_reg_by_name OTPCTR_PMC_CMD 0x1

    # Wait completion
    while {1} {
        # echo "Waiting BIST to complete"

        set cqctrl [expr [read_reg_by_name OTPCTR_PMC_CQCTRL] >> 24]
        set status [expr $cqctrl >> 6]
        set error [expr ($cqctrl >> 4) & 0x3]
        set cntr [expr ($cqctrl & 0xF)]

        if {$status == 0x1} {
            # Send STOP instruction
            write_reg_by_name OTPCTR_PMC_CMD 0x2

            if {$error == 0x0} {
                # echo "Test passed"
                return 0
            } elseif {$error == 0x1} {
                set addr [expr [read_reg_by_name OTPCTR_DAP_CQCTRL] & 0xFF]
                set data [otp_read_word_sys $addr]
                echo [format "error at address 0x%02x, data = 0x%010x" $addr $data]
                return 1
            } else {
                error [format "CQCTRL.ERROR = %d should not occur" $error]
            }
        } elseif {$status == 0x2} {
            error "PMC not performing an operation"
        } elseif {$status == 0x3} {
            echo "PMC is busy"
            echo "PMC_CTRL_STATUS:"
            echo [format "    DONE/BUSY : %d" $status]
            echo [format "    ERROR     : %d" $error]
            echo [format "    CNTR      : %d" $cntr]
        } else {
            error "CACTRL.STATUS = 0x0 should not occur"
        }
    }

    return 1
}

proc otp_bist_run {{prog_en 0}} {
    set errors 0

    echo "\n***** Running OTP BIST test *******************************"
    if {$prog_en == 1} {
        echo "Bit programming is enabled"
    } else {
        echo "Bit programming is disabled"
    }

    set otp_size 256
    set cur_addr 0x00

    while {1} {
        set remainig_words [expr $otp_size - $cur_addr]
        set res [otp_bist_run_single $cur_addr $remainig_words $prog_en]
        # echo [format "cur_addr = 0x%x, res = %0d" $cur_addr $res]

        if {$res == 1} {
            incr errors
            set cur_addr [expr [read_reg_by_name OTPCTR_DAP_CQCTRL] & 0xFF]
            incr cur_addr
        } else {
            break
        }
    }

    echo [format "BIST completed with %d errors" $errors]
}

proc otp_boot {} {
    echo "Starting OTP BOOT"
    write_reg_by_name OTPCTR_DAP_RQ0 0x04094802
    write_reg_by_name OTPCTR_DAP_RQ1 0x01000000
    write_reg_by_name OTPCTR_DAP_RQ2 0x00000008
    write_reg_by_name OTPCTR_DAP_CQCTRL 0x00
    write_reg_by_name OTPCTR_PMC_RQ0 0x480F4802
    write_reg_by_name OTPCTR_PMC_RQ1 0x00004821
    write_reg_by_name OTPCTR_PMC_RQ2 0x02000422
    write_reg_by_name OTPCTR_PMC_CQCTRL [expr 0x8 << 24]

    # Send START instruction
    write_reg_by_name OTPCTR_PMC_CMD 0x1

    # Wait completion
    while {1} {
        echo "Waiting BOOT to complete"

        set cqctrl [expr [read_reg_by_name OTPCTR_PMC_CQCTRL] >> 24]
        set status [expr $cqctrl >> 6]
        set error [expr ($cqctrl >> 4) & 0x3]
        set cntr [expr ($cqctrl & 0xF)]

        if {$status == 0x1} {
            # Send STOP instruction
            write_reg_by_name OTPCTR_PMC_CMD 0x2

            if {$error == 0x0} {
                echo "BOOT test passed"
                return
            } elseif {$error == 0x1} {
                echo "BOOT test failed"
                return
            } else {
                error [format "CQCTRL.ERROR = %d should not occur" $error]
            }
        } elseif {$status == 0x2} {
            error "PMC not performing an operation"
        } elseif {$status == 0x3} {
            echo "PMC is busy"
            echo "PMC_CTRL_STATUS:"
            echo [format "    DONE/BUSY : %d" $status]
            echo [format "    ERROR     : %d" $error]
            echo [format "    CNTR      : %d" $cntr]
        } else {
            error "CACTRL.STATUS = 0x0 should not occur"
        }
    }
}

proc otp_test_row_sel {{row_sel 0}} {
    switch $row_sel {
        0 {echo "Test row disabled"}
        1 {echo "Boot ROM Test - Test Row 0 (ROM) enabled"}
        3 {echo "Test Row 0 (OTP) enabled"}
        5 {echo "Test Row 0 (OTP) enabled"}
        7 {echo "Bit Line (BL) Test - Test Row 1 (ROM) enabled"}
        default {error [format "Wrong test mode 0x%x" $row_sel]}
    }

    set val [read_reg_by_name OTPCTR_DAP_RQ1]
    set val [expr ($val & ~(0x7 << 8)) | (($row_sel & 0x7) << 8)]
    write_reg_by_name OTPCTR_DAP_RQ1 $val
}

# -----------------------------------------------------------------------------
# FLASH
# -----------------------------------------------------------------------------

proc flash_wait_irq_status {} {
    # Wait IRQ
    while {[expr [read_reg_by_name FCTR_IRQ_STATUS_SET] & 0x1E] == 0} {
        echo "Waiting for IRQ"
    }

    set status [read_reg_by_name FCTR_IRQ_STATUS_SET]

    if {[expr $status & (1 << 1)] != 0} {
        write_reg_by_name FCTR_IRQ_STATUS_CLR 0x3
        # echo "FLASH CMD_SUCCESS IRQ"
    } elseif {[expr $status & (1 << 2)] != 0} {
        error "FLASH CMD_FAIL IRQ"
    } elseif {[expr $status & (1 << 3)] != 0} {
        error "FLASH CMD_REJECT IRQ"
    } elseif {[expr $status & (1 << 4)] != 0} {
        error "FLASH READ_OVERFLOW IRQ"
    }
}

proc flash_write_word {addr data} {
    # TODO check SYSCLK frequency

    # Check address
    if {![is_flash_address $addr]} {
        error [format "Address 0x%x is not in FLASH memory" $addr]
    }

    write_reg_by_name FCTR_ADDR $addr
    write_reg_by_name FCTR_DATA0 $data
    # Send CMD_WRITE
    write_reg_by_name FCTR_CTRL 0x2

    flash_wait_irq_status
}

proc flash_erase {addr} {
    # TODO check SYSCLK frequency

    write_reg_by_name FCTR_ADDR $addr
    # Send CMD_ERASE
    write_reg_by_name FCTR_CTRL 0x4

    flash_wait_irq_status
}

proc flash_mass_erase {addr} {
    echo "Erasing FLASH"
    # TODO check SYSCLK frequency
    write_reg_by_name FCTR_ADDR $addr
    # Send CMD_MASS_ERASE
    write_reg_by_name FCTR_CTRL 0x7

    flash_wait_irq_status
    echo "FLASH Mass Erase completed"
}

proc is_flash_address {addr} {
    set addr [expr $addr & ~(1 << 28)]
    if {($addr >= 0x00000000 && $addr <= 0x9FFFF) || ($addr >= 0x200000 && $addr <= 0x207FFF)} {
        return 1
    } else {
        return 0
    }
}


proc flash_program_hex {filename {erase 1}} {
    set byte_addr 0

    set fp [open $filename r]
    set file_data [read $fp]
    close $fp

    poll off

    if {$erase == 1} {
        flash_mass_erase 0x10000000
        flash_mass_erase 0x10200000
    }

    set data [split $file_data "\n"]
    set num_lines [llength $data]
    set line_cnt 0

    foreach line $data {
        # echo [format "*** %s length = %d" $line [string length $line]]
        if {[string index $line 0] == "@"} {
            # Get starting byte address
            set word_addr [format "0x%s" [string range $line 1 end]]
            set byte_addr [expr $word_addr*4]
            # echo [format "byte_addr = 0x%x" $byte_addr]
        } elseif {[string length $line] > 0} {
            set word_data [format "0x%s" [string range $line 0 end]]
            # echo [format "\[0x%08x\]: 0x%08x" $byte_addr $word_data]
            flash_write_word $byte_addr $word_data
            incr byte_addr 4
        }

        incr line_cnt
        echo -n [format "%.2f %% done\r" [expr $line_cnt*100.0/$num_lines]]
    }
    echo ""

    poll on
}

# -----------------------------------------------------------------------------
# GPIO
# -----------------------------------------------------------------------------

proc gpio_get_data_in {port} {
    global reg_addr_ar

    set addr $reg_addr_ar(GPIO0_DATA_OUT)
    if {$port == 1} {
        set addr [expr $addr + 0x1000]
    } elseif {$port == 2} {
        set addr [expr $addr + 0x2000]
    } elseif {$port == 3} {
        set addr [expr $addr + 0x3000]
    }

    mem2array val 32 $addr 1
    return [format "0x%x" $val(0)]
}

proc gpio_set_data_out {port data} {
    global reg_addr_ar

    set addr $reg_addr_ar(GPIO0_DATA_OUT)
    if {$port == 1} {
        set addr [expr $addr + 0x1000]
    } elseif {$port == 2} {
        set addr [expr $addr + 0x2000]
    } elseif {$port == 3} {
        set addr [expr $addr + 0x3000]
    }

    mww $addr $data
}

proc gpio_set_out_en {port pin} {
    global reg_addr_ar

    set addr $reg_addr_ar(GPIO0_OUT_EN_SET)
    if {$port == 1} {
        set addr [expr $addr + 0x1000]
    } elseif {$port == 2} {
        set addr [expr $addr + 0x2000]
    } elseif {$port == 3} {
        set addr [expr $addr + 0x3000]
    }

    mww $addr [expr 1 << $pin]
}

proc gpio_clr_out_en {port pin} {
    global reg_addr_ar

    set addr $reg_addr_ar(GPIO0_OUT_EN_CLR)
    if {$port == 1} {
        set addr [expr $addr + 0x1000]
    } elseif {$port == 2} {
        set addr [expr $addr + 0x2000]
    } elseif {$port == 3} {
        set addr [expr $addr + 0x3000]
    }

    mww $addr [expr 1 << $pin]
}

# -----------------------------------------------------------------------------
# JTM
# -----------------------------------------------------------------------------

proc jtm_set_cfg {{wcal 1255} {tcal 129} {wtconf 45} {wtcalconf 5}} {
    set value [expr ($wcal << 0) | ($tcal << 16)]
    write_reg_by_name JTM_CFG0 $value
    set value [expr ($wtconf << 0) | ($wtcalconf << 8)]
    write_reg_by_name JTM_CFG1 $value
    echo [format "JTM config:"]
    echo [format "\tWCAL      = %d" $wcal]
    echo [format "\tTCAL      = %d" $tcal]
    echo [format "\tWTCONF    = %d" $wtconf]
    echo [format "\tWTCALCONF = %d" $wtcalconf]
}

proc jtm_conv {ch {mode 0}} {
    set value [expr 1 | (($mode & 3) << 4) | (($ch & 7) << 8)]
    write_reg_by_name JTM_CTR $value

    set value 0
    while {($value & 1) == 0} {
        set value [read_reg_by_name JTM_STAT]
    }

    set code [expr ($value >> 20) & 0xFFF]
    set temp [expr (($value >> 8) & 0xFFF)*0.25]

    if {$mode == 1} {
        set volt [expr $code*1.0/4096]
    } elseif {$mode == 2} {
        set volt [expr $code*0.5/4096]
    } elseif {$mode == 3} {
        set volt [expr $code*2.0/4096]
    }

    echo [format "JTM channel %0d mode %d:" $ch $mode]
    echo [format "\t CODE = %d" $code]
    if {$mode == 0} {
        echo [format "\t TEMP = %.2f C" $temp]
    } else {
        echo [format "\t VOLT = %.2f V" $volt]
    }
}

proc jtm_temp {} {
    jtm_conv 0 0
}

# -----------------------------------------------------------------------------
# SMC
# -----------------------------------------------------------------------------

proc smc_init_pins {} {
    # set_port_mode PD  2 AF 7; # NBLS0
    # set_port_mode PD  3 AF 7; # NBLS1
    set_port_mode PD  2 GPIO; # NBLS0
    set_port_mode PD  3 GPIO; # NBLS1
    # Assert NBLS0
    gpio_set_data_out 3 [expr [gpio_get_data_in 3] & ~(1 << 2)]
    # Dessert NBLS1
    gpio_set_data_out 3 [expr [gpio_get_data_in 3] |  (1 << 3)]
    gpio_set_out_en 3 2
    gpio_set_out_en 3 3

    # DA bus
    set_port_mode PB 11 AF 7; # DA0
    set_port_mode PB 12 AF 7; # DA1
    set_port_mode PB 13 AF 7; # DA2
    set_port_mode PB 14 AF 7; # DA3
    set_port_mode PB 15 AF 7; # DA4
    set_port_mode PA  9 AF 7; # DA5
    set_port_mode PA 10 AF 7; # DA6
    set_port_mode PA 11 AF 7; # DA7
    set_port_mode PA 12 AF 7; # DA8
    # set_port_mode PA 13 AF 7; # DA9
    # set_port_mode PA 14 AF 7; # DA10
    set_port_mode PC  6 AF 7; # DA11
    set_port_mode PC  7 AF 7; # DA12
    set_port_mode PC  8 AF 7; # DA13
    set_port_mode PC  9 AF 7; # DA14
    set_port_mode PC 10 AF 7; # DA15
    # Addr Hi
    set_port_mode PB  3 AF 7; # A16
    set_port_mode PB  4 AF 7; # A17
    set_port_mode PB  5 AF 7; # A18
    set_port_mode PB  6 AF 7; # A19
    # Control
    set_port_mode PC 11 AF 7; # NWE
    set_port_mode PC 13 AF 7; # NOE

    set_port_mode PC 14 AF 7; # NCS0
    # Assert NCS0
    # set_port_mode PC 14 GPIO; # NCS0
    # gpio_set_out_en 2 14
    # gpio_set_data_out 2 [expr [gpio_get_data_in 2] & ~(1 << 14)]

    set_port_mode PD  1 AF 7; # NADV
    # write_reg_by_name IOCTR_PD_PUPD [expr (3 << 2*1)]
    # write_reg_by_name IOCTR_PD_DS [expr (1 << 2*1)]
    write_reg_by_name IOCTR_PD_OTYPE [expr (1 << 1)]
    # gpio_set_out_en 3 1
    # gpio_set_data_out 3 [expr [gpio_get_data_in 3] | (1 << 1)]
}

proc smc_init {} {
    set val 0
    set val [expr $val | (2 <<  0)]; # tRC
    set val [expr $val | (3 <<  4)]; # tWC
    set val [expr $val | (1 <<  8)]; # tCEOE
    set val [expr $val | (1 << 11)]; # tWP
    set val [expr $val | (0 << 14)]; # tPC
    set val [expr $val | (1 << 17)]; # tTR
    write_reg_by_name SMC_SET_CYCLES $val

    set val 0
    set val [expr $val | (1 <<  0)]; # mw
    set val [expr $val | (0 <<  2)]; # rd_sync
    set val [expr $val | (0 <<  3)]; # rd_bl
    set val [expr $val | (0 <<  6)]; # wr_sync
    set val [expr $val | (0 <<  7)]; # wr_bl
    set val [expr $val | (0 << 10)]; # baa
    set val [expr $val | (1 << 11)]; # adv
    set val [expr $val | (1 << 12)]; # bls
    set val [expr $val | (0 << 13)]; # burst_align
    write_reg_by_name SMC_SET_OPMODE $val

    write_reg_by_name SMC_DIRECT_CMD 0x400000

    # CONV_INCR_DIS0
    write_reg_by_name SMC_USER_CONFIG [expr 1 << 6]
}

proc smc_test {} {
    set base_addr 0x70000000
    set num_words 256
    set errors 0

    echo "Fill $num_words half-words with incr pattern"
    for {set i 0} {$i < $num_words} {incr i} {
        set mem_wr($i) $i
    }
    array2mem mem_wr 16 $base_addr $num_words

    mem2array mem_rd 16 $base_addr $num_words
    for {set i 0} {$i < $num_words} {incr i} {
        if {[expr $mem_rd($i) & 0xFF] != $mem_wr($i) } {
            set addr [expr $base_addr + $i*2]
            echo [format "Error at address 0x%x: expected 0x%x, read 0x%x" $addr $mem_wr($i) $mem_rd($i)]
            incr errors
        }
    }

    echo "Fill $num_words half-words with inverted incr pattern"
    # Invert pattern
    for {set i 0} {$i < $num_words} {incr i} {
        set mem_wr($i) [expr ~$i & 0xFF]
    }
    array2mem mem_wr 16 $base_addr $num_words

    mem2array mem_rd 16 $base_addr $num_words
    for {set i 0} {$i < $num_words} {incr i} {
        if {[expr $mem_rd($i) & 0xFF] != $mem_wr($i)} {
            set addr [expr $base_addr + $i*2]
            echo [format "Error at address 0x%x: expected 0x%x, read 0x%x" $addr $mem_wr($i) $mem_rd($i)]
            incr errors
        }
    }

    echo [format "\nnumber of errors: %d" $errors]
}
