# HAL for ELIoT-01

## Требования и зависимости

- OS Linux или Windows (в зависимости от архива)
- Embedded GCC ARM 10.3.1
- CMake minimum required ver. 3.20
- Git for Windows v2.49.0 или выше (для архива для Windows)
- Python 3.x

Убедитесь, что пути к файлам и директориям проекта не содержат символы кириллицы. Используйте только латинские символы, цифры и допустимые символы (например, подчеркивание, дефис).

## Структура проекта

Каталоги в корне:

- **CMSIS** - заголовочные файлы от ARM.
- **boards** - примеры использования и тесты драйверов под каждую из плат. Все
  unit-тесты драйверов размещаются в одном каталоге, а примеры использования
  драйверов - каждый в отдельном каталоге со своими скриптами сборки и другими
  вспомогательными файлами. В каталогах `<board_name\>_cfg` располагаются BSP
  библиотека и файлы конфигурации платы.
- **devices** - драйверы для каждого из представленных чипов.
- **docs** - документация.
- **tools** - CMake toolchain файлы.

Следующие заголовочные файлы являются устаревшими и в следующих версиях будут
удалены, а драйверы будут переписаны с использованием новых заголовочников:
- ELIOT1\_unions.h

## Начальная инициализация платы перед первым запуском

Перед первым запуском примера на плате необходимо выполнить вспомогательные
действия:

- запустить OpenOCD на компьютере, к которому подключена плата. Подробнее в
  документации к OpenOCD в Device Family Pack Eliot1 (DFP Eliot1);
- однократно прошить загрузчик в системный раздел flash, который находится в
  каталоге `devices/eliot1/gcc/simple_bootloader/`, согласно инструкции
  `README.md`;

## Сборка и запуск тестов и примеров

Для демонстрации процедуры сборки и запуска рассмотрим пример программы
с использованием двух ядер Core0 и Core1.

Перейдем в каталог с примером:

```bash
cd boards/eliot1m_mo/multicore_examples/core1_startup/
```

Программа для ядра Core0 находится в каталоге:

```bash
cd cm33_core0
```

Для сборки примера необходимо добавить инструменты ARM GCC в пути поиска
системной переменной PATH, перейти в каталог `armgcc` и запустить скрипт сборки
примера:

```bash
export PATH=${path_to_tools}/bin:${PATH}
cd armgcc
bash build.sh
```

- указать имя компьютера, к которому подключена плата, в файле
  `eliot1.gdbinit` в каталоге `armgcc` (localhost, remote-pc и т.д., порт 3333);
- запустить Minicom на компьютере, к которому подключен UART, для вывода
  информации с UART запустить программы minicom или putty:
  ```bash
  minicom -D ${path_to_com} -b 115200
  putty -serial ${path_to_com} -sercfg 115200,8,n,1,N
  ```

Для запуска программы в режиме отладки необходимо запустить GDB:

```bash
arm-none-eabi-gdb-py -x eliot1.gdbinit
```

Ожидаемый вывод UART0 при работе примера:

```bash
CORE_0: Started (50 MHz)
CORE_0: GLOBAL var 0xaabb, BSS var 0x0, CONST var [0x0000ac28 : 0x12345678]
CORE_0: All sections are OK
CORE_0: This is RAMFunc print. My address 0x200009c1
CORE_0: Init NVIC
CORE_0: Hello from SysTick
CORE_0: Test BOARD API
    BOARD_GetTime() = 39522
    BOARD_Udelay(1000000)
    BOARD_GetTime() = 1046468
    BOARD API is OK!
CORE_0: Start Core1 (CPUWAIT 0x00000002)
CORE_1: Started (150 MHz)
CORE_1: GLOBAL var 0xaabb, BSS var 0x0, CONST var [0x0008aa14 : 0x12345678]
CORE_1: All sections are OK
CORE_1: This is RAMFunc print. My address 0x200409c1
CORE_1: Init NVIC
CORE_1: Hello from SysTick
CORE_1: Test BOARD API
    BOARD_GetTime() = 1075679
    BOARD_Udelay(1000000)
    BOARD_GetTime() = 2082794
    BOARD API is OK!
CORE_1: Send MHU0 0x1 to CPU0
CORE_0: Recieved MHU0 0x1 from CPU1
CORE_0: Message from Core1: Hello from CPU1
```

## Создание нового проекта

Для создания нового проекта на CMake, необходимо выполнить следующие действия:

- Определить название платы (например: BUB, MO или JC4) и перейти в
  соответствующий каталог в папке boards, например, плата MO - переходим в
  каталог `boards/eliot1m_mo_cfg/`. В нем находятся исходные файлы и файлы
  конфигурации платы BSP части (Board Support Package). Она включает в себя:
  - необходимые API-функции для настройки частот процессора ELIOT1;
  - настройку отладочной печати через UART или Semihosting;
  - настройку необходимых GPIO выводов, а также карту GPIO всех устройств;

  Для вызова этих функций необходимо включить в проект заголовочный файл
  `eliot1_board.h`. Если программа не предполагает специфичную настройку
  устройств, то достаточно вызвать в программе функцию `BOARD_InitAll()`, чтобы
  выполнить все необходимые действия по настройке платы.

- Определить сколько ядер будет использовано в программе. Если необходимо
  использовать Core1, то сборка программы будет состоять из двух частей -
  программа для Core0, программа для Core1. BSP библиотека также собирается
  отдельно для каждого из ядер.

- Собрать BSP библиотеку и добавить в проект сборки. Для этого в каталоге
  `boards/eliot1m_mo_cfg/armgcc/bsp_core0/` находится файл `CMakeLists.txt` для
  сборки статичной библиотеки `libbsp_core0.a`. Библиотеку отдельно можно не
  собирать, а включить все файлы с исходным кодом BSP-библиотеки в свой проект.
  В каталоге `boards/eliot1m_mo_cfg/armgcc/bsp_core1/` соответственно
  располагается сборка BSP-библиотеки для Core1. Подробнее со сборкой библиотеки
  и включением ее в свой проект можно ознакомиться в документе
  `boards/eliot1m_mo_cfg/armgcc/README.md`.

- Если в проекте используются какие-либо устройства из ELIOT1, то нужно добавить
  в проект необходимые драйвера этих устройств из каталога
  `devices/eliot1/drivers/`. Драйвера устройств CLKCTR, UART, GPIO, IOIM, RWC,
  TIM уже включены в BSP библиотеку.

- Startup файл для настройки векторов прерываний и начальной инициализации
  процессора и программы уже содержится в сборке BSP-библиотеки, он
  располагается по пути `devices/eliot1/gcc/startup_eliot1_cm33.S` и подходит
  для обоих ядер Core0 и Core1. По умолчанию все вектора прерываний
  инициализированы weak функцией `Default_Handler`, которая является пустым
  бесконечным циклом. Если драйвер устройства имеет обработчик прерывания в
  драйвере, то данный обработчик вызывается в weak функции. Чтобы добавить свой
  обработчик прерывания, необходимо создать функцию-обработчик с таким же
  названием, как у соответствующего вектора прерывания в файле
  `startup_eliot1_cm33.S`. При этом функция-обработчик заменит weak функцию при
  сборке проекта. Например, создадим обработчик прерывания `SysTick_Handler`:
  ```c
  void SysTick_Handler()
  {
      printf("Hello from SysTick\r\n");
      global_var = 1;
      __DSB();
  }
  ```
  Теперь при срабатывании прерывания таймера SysTick будет вызываться эта
  функция-обработчик. Для некоторых I/O устройств и таймеров создание своего
  обработчика не нужно. Например, для классов устройств UART, SPI, I2C, I2S и
  TIM. Они имеют функции регистрирования обработчика прерывания и callback
  функции, например, в UART это функция `UART_TransferCreateHandle`.

- Создать файл с функцией `int main()` и вызвать инициализацию платы
  `BOARD_InitAll()`:
  ```c
  #include <stdio.h>
  #include "eliot1_board.h"

  int main()
  {
      BOARD_InitAll();

      printf("Hello World!\r\n");

      return 0;
  }
  ```

- Выбрать подходящий скрипт линковки программы. В каталоге `devices/eliot1/gcc/`
  лежат базовые скрипты линковки для всех ядер Core0 и Core1. Скрипты с
  суффиксом `_flash` предназначены для сборки программы по адресам внутренней
  Flash, данные программы располагаются в памяти SRAM. Этот вариант сборки
  подходит, если необходимо, чтобы программа работала с отладчиком и без
  отладчика при включении питании платы. Скрипты с суффиксом `_ram` собирают
  программу по адресам SRAM, данные программы располагаются также в SRAM, этот
  вариант подходит, если необходим только запуск программы через отладчик GDB.

- Выбрать файл описания инструментов сборки. В каталоге
  `tools/cmake_toolchain_files/` расположены два файла описания:
  - `armgcc.cmake` - инструменты ARM GCC, библиотека `nosys.specs` и печать
    `printf` в UART;
  - `armgcc_semihosting.cmake` - инструменты ARM GCC, библиотека `rdimon.specs`
    и печать `printf` в Semihosting;

- Составить файл `CMakeLists.txt`:
  - Указываем минимальную версию CMake 3.20 и пути до основных компонентов
    ```cmake
    cmake_minimum_required(VERSION 3.20)

    set(ROOT_DIR ${CMAKE_CURRENT_SOURCE_DIR}/../../../../../..) # каталог расположения eliot1-hal
    set(SYSTEM_DIR ${ROOT_DIR}/devices/eliot1)
    set(ARM_GCC_DIR ${ROOT_DIR}/devices/eliot1/gcc)
    set(DRIVERS_DIR ${ROOT_DIR}/devices/eliot1/drivers)
    set(BOARD_CFG_DIR ${ROOT_DIR}/boards/eliot1m_mo_cfg) # каталог выбранной конфигурации платы
    set(BOARD_BSP_DIR ${BOARD_CFG_DIR}/armgcc/bsp_core0/build)
    ```
  - Указываем название проекта:
    ```cmake
    project(my_project)
    ```
  - Включаем язык ASM, если программа содержит ассемблерные исходные файлы:
    ```cmake
    enable_language(ASM)
    ```
  - Добавляем исходные файлы \*.c и \*.S:
    ```cmake
    add_executable(${PROJECT_NAME}.elf
        ${CMAKE_CURRENT_SOURCE_DIR}/main.c
        ${DRIVERS_DIR}/hal_spi.c
        # можно добавить файлы BSP части, если необходимо встроить библиотеку в проект в исходных кодах
    )
    ```
  - Добавляем каталоги с заголовочными файлами \*.h:
    ```cmake
    include_directories(
        ${ROOT_DIR}/CMSIS/Include
        ${ROOT_DIR}/devices/eliot1
        ${DRIVERS_DIR}
        ${BOARD_CFG_DIR}
    )
    ```
  - Подключаем BSP библиотеку и другие необходимые:
    ```cmake
    target_link_directories(${PROJECT_NAME}.elf PUBLIC ${BOARD_BSP_DIR})
    target_link_libraries(${PROJECT_NAME}.elf bsp_core0)
    ```
  - Прописываем ключи сборки компилятора и линковщика:
    ```cmake
    set(CMAKE_ASM_FLAGS "${CMAKE_ASM_FLAGS} \
     -DBOARD -DELIOT1M -DCPU_ELIOT1M_cm33_core0 \
     -mfloat-abi=soft -g")

    set(CMAKE_EXE_LINKER_FLAGS "${CMAKE_C_FLAGS} \
    -mfloat-abi=soft \
    -fdata-sections -ffunction-sections -Wl,--gc-sections \
    -T${ARM_GCC_DIR}/eliot1_cm33_core0_flash.ld")

    set(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} \
    -DBOARD \
    -DELIOT1M \
    -DCPU_ELIOT1M_cm33_core0 \
    -mfloat-abi=soft -MMD -MP \
    -O0 -g -fdata-sections -ffunction-sections")
    ```
    - Ключ `-DBOARD` указывает, что программа использует BSP бибилиотеку. Можно
      явно указать название платы `-DBOARD=BOARD_MO`, чтобы различать платы в
      коде.
    - Ключи `-DELIOT1M` и `-DCPU_ELIOT1M_cm33_core0` указывают архитектуру и номер ядра, для
      сборки программы для Core1 необходимо использовать ключ
      `-DCPU_ELIOT1M_cm33_core1`.
    - Ключ `-T${ARM_GCC_DIR}/eliot1_cm33_core0_flash.ld` указывает путь до
      скрипта линковки, можно указать свой скрипт линковки.
    - Для поддержки hard-float у программы для Core1 необходимо поменять ключ
      `-mfloat-abi=soft` на `-mfloat-abi=hard -mfpu=fpv5-sp-d16`. Если в
      программе не используются вычисления с двойной точностью (тип данных
      double), то рекомендуется добавить ключ `-fsingle-precision-constant`,
      тогда компилятор не будет применять функции вычисления с двойной
      точностью, что значительно ускорит работу программы.

- Собираем библиотеку BSP, переходим в каталог
  `boards/eliot1m_mo_cfg/armgcc/bsp_core0/`, создаем каталог `build`, вызываем
  CMake и make:
  ```bash
  mkdir build
  cd build

  cmake -G "Unix Makefiles" \
      -DCMAKE_TOOLCHAIN_FILE="${toolchain_file}" \
      ".."
  make
  ```
  Где `${toolchain_file}` - путь до выбранного файла описания инструментов
  сборки в зависимости от нужного способа печати UART или Semihosting, см. выше.
  Далее переходим в каталог проекта и собираем его аналогичным образом. В итоге
  должен появиться файл в `build/my_project.elf`.

## Запуск программы

- Перед первым запуском программы необходимо запустить программу Openocd и
  однократно прошить загрузчик из каталога
  `devices/eliot1/gcc/simple_bootloader/` (см. "Начальная инициализация платы
  перед первым запуском"). Создаем файл конфигурации GDB:
  ```python
  python
  class RegisterRunCommand (gdb.Command):
  def __init__ (self):
      command_name ="run"
      super (RegisterRunCommand, self).__init__ (command_name, gdb.COMMAND_USER)
  def invoke (self, arg, from_tty):
      gdb.execute('c')
      self.dont_repeat()

  RegisterRunCommand ()
  end

  target extended-remote localhost:3333
  file build/my_project.elf
  load
  ```

- Если проект собран с поддержкой печати Semihosting, то необходимо добавить
  команды:
  ```python
  mon arm semihosting enable
  mon arm semihosting_fileio enable
  ```

- Вызываем GDB командой:
  ```bash
  arm-none-eabi-gdb-py -x eliot1.gdbinit
  ```
  В консоли GDB вводим команду "run" или "c". В этом скрипте GDB реализована
  команда "run", которая часто используется при отладке в различных IDE. Если
  отладочная плата расположена на другом компьютере в сети, то `localhost`
  необходимо сменить на название или ip адрес другого компьютера.

## Генерация документации

Для генерации документации необходимо запустить скрипт:

```bash
bash tools/post_build/build_docs.sh
```

Генерируется три варианта документации:

- HTML: `docs/hal_html/hal_documentation.html`
- PDF: `docs/hal_pdf/hal_documentation.pdf`
- RTF: `docs/hal_rtf/hal_documentation.rtf`
