/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "eliot1_board.h"
#include "hal_dma.h"

/* Маска для формирования secure адресов. */
#define SECURE_BIT_MSK (0x10000000)
/* DMA, используемый в примере */
#define DMA_EXAMPLE DMA0
/* Канал DMA, используемый для передачи данных. */
#define CHANNEL_EXAMPLE 0

#define LEN 4095U

#define ARRAY_COUNT 6U

uint32_t src_data[ARRAY_COUNT][LEN];

uint32_t dst_data[ARRAY_COUNT][LEN];

/* Флаг разрешения выхода из программы. */
volatile bool flag = false;
/* Счетчик прерываний типа IntBlock */
volatile int counter = 0;

/* Инициализация массива числом start. */
void init_array(uint32_t *array, uint32_t size, uint32_t start)
{
    for (uint32_t i = 0; i < size; i++) {
        array[i] = start;
    }
}

/* Функция сравнения массивов. */
int check_copy(uint32_t *src_array, uint32_t *dst_array, uint32_t size)
{
    for (uint32_t i = 0; i < size; i++) {
        if (src_array[i] != dst_array[i]) {
            return 0;
        }
    }
    /* Если массивы полностью идентичны, возврат 1. */
    return 1;
}

/* Функция обратного вызова. */
void DMA_Callback(dma_handle_t *handle, void *param, dma_irq_t type)
{
    UNUSED(handle);
    UNUSED(param)
    UNUSED(type);

    if (type == DMA_IntTfr) {
        /* Если конец всей передачи, разрешаем выход из программы. */
        flag = true;
    } else if (type == DMA_IntBlock) {
        /* Если конец передачи одного блока, увеличиваем counter. */
        counter += 1;
    }
}

int main(void)
{
    dma_handle_t handle;
    /* Объявление массива дескрипторов и выравнивание адресов. */
    DMA_ALLOCATE_HEAD_DESCRIPTORS(desc, ARRAY_COUNT);

    BOARD_InitAll();

    printf("Test started...\r\n\n");

    /* Инициализация src массивов. */
    for (uint32_t i = 0; i < ARRAY_COUNT; i++) {
        init_array(src_data[i], LEN, i + 1);
    }

    void *src_data_s = (void *)((uint32_t)src_data | SECURE_BIT_MSK);
    void *dst_data_s = (void *)((uint32_t)dst_data | SECURE_BIT_MSK);

    void *desc_s = (void *)((uint32_t) desc | SECURE_BIT_MSK);

    dma_multiblock_config_t config = {
        .count          = ARRAY_COUNT,
        .data_size      = LEN * ARRAY_COUNT,
        .transfer_type  = DMA_MemoryToMemory_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize256,
        .dst_burst_size = DMA_BurstSize256,
        .src_incr       = DMA_Incr,
        .dst_incr       = DMA_Incr,
        .src_data_width = DMA_Transfer32BitWidth,
        .dst_data_width = DMA_Transfer32BitWidth,
        .src_addr       = src_data_s,
        .dst_addr       = dst_data_s,
        .int_en         = true
    };

    /* Установка конфигурации передачи каждого блока. */
    DMA_InitMultiblockDescriptor((dma_descriptor_t *) desc_s, &config);

    /* Инициализация DMA. */
    DMA_Init(DMA_EXAMPLE);
    DMA_CreateHandle(&handle, DMA_EXAMPLE, CHANNEL_EXAMPLE);

    /* Установка функции обратного вызова. */
    DMA_SetCallback(&handle, DMA_Callback, NULL);
    /* Разрешение прерываний типа IntBlock. */
    DMA_EnableChannelInterrupt(DMA_EXAMPLE, CHANNEL_EXAMPLE, DMA_IntBlock);
    /* Разрешение прерываний типа IntTfr. */
    DMA_EnableChannelInterrupt(DMA_EXAMPLE, CHANNEL_EXAMPLE, DMA_IntTfr);
    /* Отправка значений конфигурации. */
    DMA_SubmitChannelDescriptor(&handle, desc_s);
    volatile uint64_t time_dma = BOARD_GetTime();
    /* Начало передачи. */
    DMA_StartTransfer(&handle);
    /* Ожидание окончания передачи цепочки DMA. */
    while (!flag)
        ;
    time_dma = BOARD_GetTime() - time_dma;

    bool successed = false;
    for (uint32_t i = 0; i < ARRAY_COUNT; i++) {
        successed = check_copy(src_data[i], dst_data[i], LEN);
        if (!successed) {
            break;
        }
    }
    if (successed) {
        printf("DMA multiblock transfer was successful\r\n");
    } else {
        printf("DMA multiblock transfer was failed\r\n");
    }
    printf("Time of DMA transfer: %lld us\r\n", time_dma);
    double dma_speed = (double) (ARRAY_COUNT * LEN * sizeof(uint32_t)) / time_dma;
    printf("DMA speed: %lf MB/s\r\n", dma_speed);

    /* Деинициализация DMA. */
    DMA_Deinit(DMA_EXAMPLE);

    volatile uint64_t time_cpu = BOARD_GetTime();
    memcpy(dst_data, src_data, ARRAY_COUNT * sizeof(uint32_t) * LEN);
    time_cpu = BOARD_GetTime() - time_cpu;
    printf("Time of CPU transfer: %lld us\r\n", time_cpu);
    double cpu_speed = (double) (ARRAY_COUNT * LEN * sizeof(uint32_t)) / time_cpu;
    printf("CPU speed: %lf MB/s\r\n\n", cpu_speed);

    printf("Test finished\r\n");

    asm volatile("bkpt");

    while (1) {
        ;
    }

    return 0;
}
