/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include <stdio.h>
#include <string.h>
#include "eliot1_board.h"
#include "hal_jtm.h"

#define TEST_TCAL      129  /*!< Значение калибровочного параметра TCAL для партии микросхем MPW */
#define TEST_WCAL      1311 /*!< Значение калибровочного параметра WCAL для партии микросхем MPW */
#define TEST_WTCONF    0x3D /*!< Значение калибровочного параметра WTCONF для партии микросхем MPW */
#define TEST_WTCALCONF 0    /*!< Значение калибровочного параметра WTCALCONF для партии микросхем MPW */

static jtm_config_t jtm_config; /*!< Параметры JTM */
static jtm_handle_t jtm_handle; /*!< Дескриптор обработчика прерываний от JTM */
static volatile bool reading_complete = false; /*!< Признак окончания процесса чтения параметра */

/*!
 * @brief Функция обратного вызова, печатающая результат.
 *
 * @param handle    Дескриптор драйвера JTM
 * @param parameter Параметр, прочитанный с помощью драйвера JTM
 * @param value     Значение параметра
 * @param user_data Указатель на произвольные пользовательские данные (не используется)
 */
void JTM_Callback(jtm_handle_t *handle, jtm_parameter_t parameter,
        int32_t value, void *user_data)
{
    /* Устранение предупреждений компилятора о неиспользуемых переменных. */
    handle = handle;
    user_data = user_data;

    switch (parameter) {
        case JTM_Temperature:
            printf("Temperature:   %f\r\n", value / 1000.);
            break;
        case JTM_Vcasn:
            printf("V CASN:        %f\r\n", value / 1000.);
            break;
        case JTM_Vcore:
            printf("V CORE (VDDC): %f\r\n", value / 1000.);
            break;
        default:
            printf("Unknown parameter in callback!\r\n");
            break;
    }

    reading_complete = true;
}

/*!
 * @brief Тест драйвера JTM в части неблокирующих функций чтения
 */
int main(void)
{
    jtm_status_t status;

    BOARD_InitAll();

    printf("JTM nonblocking reading example\r\n");
    printf("===============================\r\n\n");

    jtm_config.tcal = TEST_TCAL;
    jtm_config.wcal = TEST_WCAL;
    jtm_config.wtconf = TEST_WTCONF;
    jtm_config.wtcalconf = TEST_WTCALCONF;

    JTM_Init(JTM_Secure, &jtm_config);

    status = JTM_CreateHandle(JTM_Secure, &jtm_handle, JTM_Callback, NULL);
    if (status != JTM_Status_Ok) {
        printf("Creating JTM handle failed with code: %d\r\n", status);
        while (1);
    }

    reading_complete = false;

    status = JTM_GetParameterValueNonBlocking(JTM_Secure, &jtm_handle,
                JTM_Temperature);
    if (status != JTM_Status_Ok) {
        printf("Failed to start reading temperature, code: %d\r\n", status);
        while (1);
    }

    while (!reading_complete);

    reading_complete = false;

    status = JTM_GetParameterValueNonBlocking(JTM_Secure, &jtm_handle,
                JTM_Vcasn);
    if (status != JTM_Status_Ok) {
        printf("Failed to start reading V CASN, code: %d\r\n", status);
        while (1);
    }

    while (!reading_complete);

    reading_complete = false;

    status = JTM_GetParameterValueNonBlocking(JTM_Secure, &jtm_handle,
                JTM_Vcore);
    if (status != JTM_Status_Ok) {
        printf("Failed to start reading V CORE (VDDC), code: %d\r\n", status);
        while (1);
    }

    while (1)
        ;
}
