/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "eliot1_board.h"
#include "hal_qspi_nor_flash.h"
#include "hal_nor_flash.h"
#include "hal_qspi_dma.h"

/* Маска для формирования Secure адресов */
#define SECURE_BIT_MSK 0x10000000
/* DMA, используемый в примере */
#define DMA_EXAMPLE DMA0_Secure
/* Канал, используемый для доставки данных в TX буфер QSPI */
#define CHANNEL_TX 0U
/* Канал, используемый для передачи данных из RX буфера QSPI */
#define CHANNEL_RX 1U
/* Количество секунд в микросекунде */
#define US_IN_SECOND     1000000
/* Количество байт в килобайте */
#define BYTES_IN_KB      1024
/* Количество байт в мегабайте */
#define BYTES_IN_MB      (1024 * 1024)
/* Размер тестового буфера для замера скорости чтения */
#define TEST_SIZE        (80 * BYTES_IN_KB)
#define PAGE_SIZE        256
#define TEST_ADDRESS     0x0000
#define BUFFER_SIZE      256U
#define XIP_FUNC_ADDR    (TEST_ADDRESS + TEST_SIZE)
#define ITER_NUM         10000000

uint8_t rate_buffer[TEST_SIZE];

typedef uint32_t (*sum_func_t)(uint32_t, uint32_t, uint32_t);

extern const uint32_t _sum_func_end;

__attribute__((noinline, section(".sum_func_section")))
uint32_t sum_func(uint32_t a, uint32_t b, uint32_t iter)
{
    uint32_t sum = 0;
    for (uint32_t i = 0; i < iter; i++) {
        sum += (a + b) * i;
    }
    return sum;
}

__attribute__((noinline, section(".sum_func_flash_section")))
uint32_t sum_func_flash(uint32_t a, uint32_t b, uint32_t iter)
{
    uint32_t sum = 0;
    for (uint32_t i = 0; i < iter; i++) {
        sum += (a + b) * i;
    }
    return sum;
}

int main()
{
    uint8_t input_buffer[BUFFER_SIZE];
    uint8_t output_buffer[BUFFER_SIZE];
    void *input_buffer_s = (void *)((uint32_t) input_buffer | SECURE_BIT_MSK);
    void *output_buffer_s = (void *)((uint32_t) output_buffer | SECURE_BIT_MSK);
    nor_status_t status;
    nor_config_t nor_config;
    nor_handle_t nor_handle;
    qspi_nor_init_config_t nor_init_config;

    (void)memset(input_buffer, 0, sizeof(input_buffer));
    (void)memset(output_buffer, 0, sizeof(output_buffer));
    (void)memset(&nor_handle, 0, sizeof(nor_handle));
    (void)memset(&nor_config, 0, sizeof(nor_config));
    (void)memset(&nor_init_config, 0, sizeof(nor_init_config));

    BOARD_InitAll();
    printf("\x1b[2J\x1b[1;1H"); /* Последовательность для очистки экрана */

    /* Настройка выводов для использования QSPI. */
    GPIO_PinMode_Function(GPIO_QSPI_SCK, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_SS,  GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO0, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO1, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO2, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO3, GPIO_ALT_FUNC_QSPI_SPI2);

    /* Установка выходного тока 12мА. */
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_SCK, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_SS,  GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO0, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO1, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO2, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO3, GPIO_MAX_CURRENT_12mA);

    /* Установка Pull-up. Используются внешние подтяжки. */
    GPIO_PinSet_PUPD(GPIO_QSPI_SCK, GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_SS, GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO0, GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO1, GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO2, GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO3, GPIO_PULL_UP);

    qspi_dma_handle_t handle;
    /* Инициализация буфера фиктивных данных. */
    uint32_t dummy_data;
    dma_handle_t dma_handle_tx;
    dma_handle_t dma_handle_rx;

    printf("\r\nTest started...\r\nQSPI OutFreq. : %ldMHz\r\n",
        BOARD_QSPI_FREQ / 1000000UL);

    DMA_Init(DMA_EXAMPLE);

    DMA_CreateHandle(&dma_handle_rx, DMA_EXAMPLE, CHANNEL_RX);
    DMA_CreateHandle(&dma_handle_tx, DMA_EXAMPLE, CHANNEL_TX);

    QSPI_TransferCreateHandleDMA(QSPI_Secure, &handle, &dma_handle_tx,
        &dma_handle_rx);
    handle.dummy_data = (void *)(((uint32_t) &dummy_data) | SECURE_BIT_MSK);

    uint32_t read_desc_count = QSPI_GetReadDMADescriptorsCount(TEST_SIZE);
    uint32_t dummy_desc_count = QSPI_GetDummyDMADescriptorsCount(TEST_SIZE);

    DMA_ALLOCATE_HEAD_DESCRIPTORS(tx_desc, dummy_desc_count);
    DMA_ALLOCATE_HEAD_DESCRIPTORS(rx_desc, read_desc_count);

    void *tx_desc_s = (void *)((uint32_t) tx_desc | SECURE_BIT_MSK);
    void *rx_desc_s = (void *)((uint32_t) rx_desc | SECURE_BIT_MSK);

    /* Инициализация. */
    nor_init_config.cmd_format = QSPI_CommandDataQuad;
    /* Если не удастся считать расположение бита Quad Enable с SFDP. */
    nor_init_config.quad_mode_setting = NOR_QuadModeStatusReg1_Bit6;
    nor_config.driver_base_addr = QSPI_Secure;
    nor_config.mem_control_config = &nor_init_config;
    status = QSPI_NorFlashInit(&nor_config, &nor_handle);
    if (status != NOR_Status_Success) {
        printf("QSPI NOR init error: %d\r\n", status);
        return 1;
    }
    printf("QSPI NOR init completed\r\n");

    /* Очистка памяти. */
    printf("Chip erasing. Wait 1 minute...\r\n");
    status = NOR_FlashEraseChip(&nor_handle);
    if (status != NOR_Status_Success) {
        printf("Chip erase error: %d\r\n", status);
        return 1;
    }
    printf("Chip erase completed\r\n");

    /* Программирование инкрементом. */
    for (uint32_t i = 0; i < BUFFER_SIZE; i++) {
        input_buffer[i] = i;
    }

    status = NOR_FlashPageProgram(&nor_handle, TEST_ADDRESS, input_buffer_s, PAGE_SIZE);
    if (status != NOR_Status_Success) {
        printf("Page program error: %d\r\n", status);
        return 1;
    }
    printf("Page program completed\r\n");

    /* Чтение в XIP. */
    /* Используются настройки из таблицы SFDP, полученные во время исполнения функции QSPI_NorFlashInit. */
    if (QSPI_EnableXIP(&nor_handle) != NOR_Status_Success) {
        printf("Enabling XIP mode error: %d\r\n", status);
        return 1;
    }

    NOR_FlashReadXIP(TEST_ADDRESS, output_buffer, BUFFER_SIZE);

    QSPI_DisableXIP(&nor_handle);

    /* Реинициализация из-за сброса настроек после режима XIP. */
    status = QSPI_NorFlashInit(&nor_config, &nor_handle);
    if (status != NOR_Status_Success) {
        printf("QSPI NOR reinit error: %d\r\n", status);
        return 1;
    }

    /* Верификация. */
    /* Режим XIP поддерживается только для флеш-памятей семейств W25Q, MX25L, S25FL, N25Q, AT25S, GD25V. */
    for (uint32_t i = 0; i < BUFFER_SIZE; i++) {
        if (input_buffer[i] != output_buffer[i]) {
            printf("Data mismatch!\r\n");
            printf("input_buffer[%ld] : 0x%02X!\r\n", i, input_buffer[i]);
            printf("output_buffer[%ld]: 0x%02X!\r\n", i, output_buffer[i]);
            break;
        }
    }
    printf("Page program verification completed\r\n");

    /* Очистка блока. */
    status = NOR_FlashEraseBlock(&nor_handle, TEST_ADDRESS, BUFFER_SIZE);
    if (status != NOR_Status_Success) {
        printf("Erase block error: %d\r\n", status);
        return 1;
    }
    printf("Block erase completed\r\n");

    /* Чтение с DMA. */
    handle.rx_desc = rx_desc_s;
    handle.tx_desc = tx_desc_s;
    status = NOR_FlashRead(&nor_handle, TEST_ADDRESS,
            output_buffer_s, BUFFER_SIZE);
    if (status != NOR_Status_Success) {
        printf("Read error: %d\r\n", status);
        return 1;
    }

    /* Верификация. */
    for (uint32_t i = 0; i < BUFFER_SIZE; i++) {
        if (output_buffer[i] != 0xFF) {
            printf("Flash is not erased!\r\n");
            printf("output_buffer[%ld]: 0x%02X!\r\n", i, output_buffer[i]);
            break;
        }
    }
    printf("Block erase verification completed\r\n");

    /* Замер времени исполнения функции в XIP. */
    volatile uint64_t time_mks = 0;
    time_mks = BOARD_GetTime();
    volatile uint32_t sum = sum_func(1, 2, ITER_NUM);
    time_mks = BOARD_GetTime() - time_mks;
    printf("SUM = %lu Time :: %llu mks (from RAM)\r\n", sum, time_mks);

    time_mks = BOARD_GetTime();
    sum = sum_func_flash(1, 2, ITER_NUM);
    time_mks = BOARD_GetTime() - time_mks;
    printf("SUM = %lu Time :: %llu mks (from FLASH)\r\n", sum, time_mks);

    size_t func_size = (size_t) &_sum_func_end - (size_t) sum_func + 1; /* thumb-инструкция */
    uint8_t *func_buffer = (uint8_t *) malloc(func_size);
    if (func_buffer == NULL) {
        printf("Error memory allocation\r\n");
        asm volatile("bkpt");
    }
    /* Учет особенностей thumb-инструкций при копировании. */
    void *p_sum_func = (void *)((uint32_t) sum_func & ~0x1);
    memcpy((void *) func_buffer, p_sum_func, func_size);
    status = NOR_FlashEraseBlock(&nor_handle, XIP_FUNC_ADDR, func_size);
    if (status != NOR_Status_Success) {
        printf("Chip erase error: %d\r\n", status);
        return 1;
    }
    status = NOR_FlashProgramBlock(&nor_handle, XIP_FUNC_ADDR,
            func_buffer, PAGE_SIZE);
    if (status != NOR_Status_Success) {
        printf("Block program error: %d\r\n", status);
    }
    sum_func_t xip_func_ptr = (sum_func_t)(QSPI_XIP_MEM + XIP_FUNC_ADDR + 0x1);
    if (QSPI_EnableXIP(&nor_handle) != NOR_Status_Success) {
        printf("Enabling XIP mode error: %d\r\n", status);
        return 1;
    }
    time_mks = BOARD_GetTime();
    sum = xip_func_ptr(1, 2, ITER_NUM);
    time_mks = BOARD_GetTime() - time_mks;
    QSPI_DisableXIP(&nor_handle);
    printf("SUM = %lu Time :: %llu mks (from XIP)\r\n", sum, time_mks);

    free(func_buffer);

    /* Замер скорости чтения для QSPI XIP. */
    /* Реинициализация из-за сброса настроек после режима XIP. */
    printf("QSPI XIP speed measuring...\r\n");
    status = QSPI_NorFlashInit(&nor_config, &nor_handle);
    if (status != NOR_Status_Success) {
        printf("QSPI NOR reinit error: %d\r\n", status);
        return 1;
    }
    
    void *rate_buffer_s = (void *)((uint32_t) rate_buffer | SECURE_BIT_MSK);    
    for (uint32_t i = 0; i < TEST_SIZE; i++) {
        rate_buffer[i] = i;
    }
    status = NOR_FlashProgramBlock(&nor_handle, TEST_ADDRESS,
        rate_buffer_s, TEST_SIZE);
    if (status != NOR_Status_Success) {
        printf("Block program error: %d\r\n", status);
    }
    (void)memset(rate_buffer, 0, sizeof(rate_buffer));

    if (QSPI_EnableXIP(&nor_handle) != NOR_Status_Success) {
        printf("Enabling XIP mode error: %d\r\n", status);
        return 1;
    }
    time_mks = BOARD_GetTime();
    NOR_FlashReadXIP(TEST_ADDRESS, rate_buffer_s, TEST_SIZE);
    time_mks = BOARD_GetTime() - time_mks;
    double speed_write = (double)TEST_SIZE / BYTES_IN_MB;
    speed_write = speed_write * US_IN_SECOND / time_mks;
    printf("Data transfer rate: %f MB/s\r\n", speed_write);
    QSPI_DisableXIP(&nor_handle);

    for (uint32_t i = 0; i < TEST_SIZE; i++) {
        if ((uint8_t)i != rate_buffer[i]) {
            printf("Data mismatch!\r\n");
            printf("expected[%ld] : 0x%02X!\r\n", i, (uint8_t)i);
            printf("rate_buffer[%ld]: 0x%02X!\r\n", i, rate_buffer[i]);
            break;
        }
    }
    printf("Speed read verification completed\r\n");

    printf("Test finished\r\n");

    asm volatile("bkpt");

    return 0;
}
