/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup bsp_eliot1m_mo_cfg
 * @{
 */

/*!
 * @file eliot1_board.c
 *
 * @brief Имплементация пакета поддержки платформы ELIOT1m MO
 */

#include "eliot1_board.h"
#include "hal_sdmmc.h"
#include "hal_flash.h"
#include "hal_jtm.h"
#include "hal_qspi_nor_flash.h"
#include "hal_nor_flash.h"

void (*CONSOLE_print_char_fnc)(UART_Type *base, uint8_t) = 0; /*!< Указатель на функцию печати символа */
uint8_t (*CONSOLE_get_char_fnc)(UART_Type *base)         = 0; /*!< Указатель на функцию приема символа */

static const uint8_t uart_rx_pinmap[] =
{
    GPIO_UART0_RXD, GPIO_UART1_RXD, GPIO_UART2_RXD, GPIO_UART3_RXD
};

static const uint8_t uart_tx_pinmap[] =
{
    GPIO_UART0_TXD, GPIO_UART1_TXD, GPIO_UART2_TXD, GPIO_UART3_TXD
};

void BOARD_InitAll()
{
    BOARD_InitCLKCTR();
    BOARD_EnableICache();
#ifdef BOARD_TIMER
    BOARD_InitTimer();
#endif
    BOARD_InitConsole();
    if (CPU_IDENTITY->CPUID == 0) {
        BOARD_Udelay(1000);
        uint32_t fclk = CLKCTR_GetFClk(CLOCK_BASE) / 1000000;
        uint32_t sysclk = CLKCTR_GetSysClk(CLOCK_BASE) / 1000000;
        jtm_config_t jtm_config = {
            .tcal      = 129,
            .wcal      = 1311,
            .wtconf    = 0x3D,
            .wtcalconf = 0
        };
        JTM_Init(JTM_Secure, &jtm_config);
        int32_t reg_value;
        float vcore = 0.f;
        float temp_core = 0.f;
        jtm_status_t js = JTM_GetParameterValue(JTM_Secure, JTM_Vcore, &reg_value);
        if (js == JTM_Status_Ok) {
            vcore = reg_value / 1000.f;
        }
        js = JTM_GetParameterValue(JTM_Secure, JTM_Temperature, &reg_value);
        if (js == JTM_Status_Ok) {
            temp_core = reg_value / 1000.f;
        }
        struct power_config pwr_cfg = {0};
        POWER_GetCurrentConfig(PWRCTR_Secure, &pwr_cfg);
        printf("Board CFG: FCLK : %ldMHz, SysCLK : %ldMHz, VCore %.2fv (%s vlv%d), T = %.2f°C\r\n",
            fclk, sysclk, vcore, pwr_cfg.dcdc_enable? "DC-DC" :"External",
            pwr_cfg.run_configuration.dcdc_level, temp_core);
    }
}

void BOARD_InitCLKCTR()
{
    volatile board_cfg_status_t status = BOARD_Status_OK;

    CLKCTR_SetHFI(CLOCK_BASE, BOARD_HFI_FREQUENCY);
    CLKCTR_SetLFI(CLOCK_BASE, BOARD_LFI_FREQUENCY);
    CLKCTR_SetXTI(CLOCK_BASE, BOARD_XTI_FREQ);

    if (BOARD_FindAndSelectPLLConfig(BOARD_XTI_FREQ) == NULL) {
        goto board_init_error;
    }
    if (BOARD_FindAndSelectPowerConfig(BOARD_PWR_CONFIG) == NULL) {
        goto board_init_error;
    }

    if (CPU_IDENTITY->CPUID == 0) {
        /* Workaround */
        CLKCTR_SetPll(CLOCK_BASE, 0, 0); // default HFI freq
        CLKCTR->CFG &= ~(3 << 16); // disable HFE
        while ((CLKCTR->CFG & (3 << 16)) != 0)
            ;
        CLKCTR->CFG |= (1 << 16); // enable HFE quartz resonator
        while ((CLKCTR->CFG & (3 << 16)) == 0)
            ;
        if ((status = BOARD_ApplyPowerConfig(CLKCTR, PWRCTR, FCTR,
                false)) != BOARD_Status_OK) {
            goto board_init_error;
        }
    }
    SystemCoreClockUpdate();
    return;

board_init_error:
    asm volatile("bkpt");
    while (1)
        ;
}

void BOARD_EnableICache()
{
    SET_VAL_MSK(ICACHE_Secure->CTRL, ICACHE_CTRL_CACHEEN_Msk,
        ICACHE_CTRL_CACHEEN_Pos, 1);
}

void BOARD_InitConsole()
{
#ifdef BOARD_CONSOLE_SEMIHOSTING
    extern void initialise_monitor_handles(void);
    initialise_monitor_handles();
#endif

#ifdef BOARD_CONSOLE_UART
    /* Установка выводов в функцию UART. */
    if (CPU_IDENTITY->CPUID == 0) {
        uint32_t num = IOIM_GetIRQNumber(BOARD_CONSOLE_UART) - UART0_IRQn;
        GPIO_PinMode_Function(uart_tx_pinmap[num], GPIO_ALT_FUNC_UART);
        GPIO_PinMode_Function(uart_rx_pinmap[num], GPIO_ALT_FUNC_UART);

        struct uart_config config;
        UART_GetDefaultConfig(&config);
        UART_Init(BOARD_CONSOLE_UART, &config, CLKCTR_GetSysClk(CLOCK_BASE));
    }
    CONSOLE_print_char_fnc = UART_WriteByteWait;
    CONSOLE_get_char_fnc = UART_ReadByteWait;
#else
    UNUSED(CONSOLE_print_char_fnc);
    UNUSED(CONSOLE_get_char_fnc);
    UNUSED(uart_rx_pinmap);
    UNUSED(uart_tx_pinmap);
#endif
}

void BOARD_EnableXIP()
{
    nor_handle_t nor_handle = {0};
    nor_config_t nor_config = {0};
    qspi_nor_init_config_t nor_init_config = {0};
    nor_init_config.cmd_format = QSPI_CommandDataQuad;
    /* Если не удастся считать расположение бита Quad Enable с SFDP. */
    nor_init_config.quad_mode_setting = NOR_QuadModeStatusReg1_Bit6;
    nor_config.driver_base_addr = QSPI_Secure;
    nor_config.mem_control_config = &nor_init_config;

    QSPI_NorFlashInit(&nor_config, &nor_handle);
    /* Настройка выводов для использования QSPI. */
    GPIO_PinMode_Function(GPIO_QSPI_SCK, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_SS,  GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO0, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO1, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO2, GPIO_ALT_FUNC_QSPI_SPI2);
    GPIO_PinMode_Function(GPIO_QSPI_IO3, GPIO_ALT_FUNC_QSPI_SPI2);

    /* Установка выходного тока 12мА. */
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_SCK, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_SS,  GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO0, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO1, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO2, GPIO_MAX_CURRENT_12mA);
    GPIO_PinSet_MaxCurrent(GPIO_QSPI_IO3, GPIO_MAX_CURRENT_12mA);

    /* Установка Pull-up. */
    GPIO_PinSet_PUPD(GPIO_QSPI_SCK, GPIO_PULL_NONE);
    GPIO_PinSet_PUPD(GPIO_QSPI_SS,  GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO0, GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO1, GPIO_PULL_UP);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO2, GPIO_PULL_NONE);
    GPIO_PinSet_PUPD(GPIO_QSPI_IO3, GPIO_PULL_UP);

    (void)QSPI_EnableXIP(&nor_handle);

    SYSCTR_Secure->INITSVTOR1 = 0x08400000UL;
}

#ifdef BOARD_TIMER

static uint32_t ticks_per_microsecond = 0;

void BOARD_InitTimer()
{
    static const struct timer_hardware_config timer_config = {
        .start_value = 0xFFFFFFFF,
        .reload_value = 0xFFFFFFFF,
        .interrupt_enable = 1,
        .work_type = TIMER_Work,
        .start_enable = 1
    };
    ticks_per_microsecond = CLKCTR_GetSysClk(CLOCK_BASE) / 1000000;

#ifdef CPU_ELIOT1M_cm33_core0
    TIMER_Init(BOARD_TIMER, timer_config, TIMER_Software, NULL, 0xFFFFFFFF);
#endif

#ifdef CPU_ELIOT1M_cm33_core1
    TIMER_SetConfig(BOARD_TIMER, timer_config, TIMER_Software, NULL,
        0xFFFFFFFF);
#endif
}

__WEAK void BOARD_Udelay(uint32_t us)
{
    register uint64_t wait_ticks = TIMER_GetTicks(BOARD_TIMER)
        + us * ticks_per_microsecond;

    if (TIMER_GetAPIStatus() != TIMER_Status_Ok) {
        return;
    }

    while (wait_ticks > TIMER_GetTicks(BOARD_TIMER)) {
        asm volatile("nop");
    }
}

__WEAK uint64_t BOARD_GetTime()
{
    return TIMER_GetTicks(BOARD_TIMER) / ticks_per_microsecond;
}

#endif /* BOARD_TIMER */

#if defined(BOARD_CONSOLE_UART) && !defined(BOARD_NO_UART_STD)

/*!
 * @brief Реализация функции std::_write
 */
int _write(int fd, char *ptr, int len)
{
    UNUSED(fd);

    if (CONSOLE_print_char_fnc == 0) {
        return 0;
    }

    /*
     * Запись 'len' символов, начиная с адреса 'ptr', в файл 'fd'. Функция
     * возвращает количество записанных символов.
     */
    int i = 0;
    for (i = 0; i < len; i += 1) {
        CONSOLE_print_char_fnc(BOARD_CONSOLE_UART, *ptr++);
    }

    return len;
}

/*!
 * @brief Реализация функции std::_read
 */
int _read(int fd, char *ptr, int len)
{
    UNUSED(fd);

    if (CONSOLE_print_char_fnc == 0 || CONSOLE_get_char_fnc == 0) {
        return 0;
    }

    /*
     * Чтение 'len' символов из файла 'fd' по адресу 'ptr'. Функция возвращает
     * количество прочитанных символов.
     */
    int cnt = 0;
    char ch = 0;
    for (; len > 0 && ch != '\r'; len--) {
        ch = CONSOLE_get_char_fnc(BOARD_CONSOLE_UART);
        CONSOLE_print_char_fnc(BOARD_CONSOLE_UART, ch);
        *ptr++ = ch;
        cnt += 1;
    }

    *ptr = '\n';
    CONSOLE_print_char_fnc(BOARD_CONSOLE_UART, '\n');
    cnt++;

    return cnt;
}

/*!
 * @brief Реализация функции std::_ttywrch
 */
void _ttywrch(int ch)
{
    if (CONSOLE_print_char_fnc == 0) {
        return;
    }

    /* Запись одного символа 'ch' в терминал по умолчанию. */
    CONSOLE_print_char_fnc(BOARD_CONSOLE_UART, ch);
}

#endif /* CONSOLE_UART */

/*!
 * @}
 */
