/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#ifndef ELIOT1_CONFIG_H
#define ELIOT1_CONFIG_H

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>
#include "hal_common.h"

#define BOARD_XTI_0MHZ  0UL        /*!< Значение XTI 0МГц */
#define BOARD_XTI_16MHZ 16000000UL /*!< Значение XTI 16МГц */
#define BOARD_XTI_24MHZ 24000000UL /*!< Значение XTI 24МГц */

/*!
 * @brief Статусы установки конфигураций платы
 */
typedef enum {
    BOARD_Status_OK            = 0, /*!< Успех */
    BOARD_Status_UnknownConfig = 1, /*!< Несуществующая конфигурация */
    BOARD_Status_InvalidArgs   = 2, /*!< Неверные аргументы функции */
    BOARD_Status_HWError       = 3  /*!< Аппаратная ошибка при установке конфигурации */
} board_cfg_status_t;

enum {
    BOARD_PowerConfig_0V9 = 0, /*!< Конфигурация для питания ядра 0,9в */
    BOARD_PowerConfig_1V0,     /*!< Конфигурация для питания ядра 1,0в */
    BOARD_PowerConfig_1V1      /*!< Конфигурация для питания ядра 1,1в */
};

typedef struct {
    uint32_t xti_freq;        /*!< Значение внешней частоты в Гц */
    uint32_t pll_main_clk;    /*!< Значение выходной частоты в Гц после модуля PLL (не должно превышать 375МГц) */
    struct {
        uint32_t pll_sel : 10; /*!< Целочисленный множитель частоты PLL */
        uint32_t pll_man : 1;  /*!< Включение ручного режима множителя PLL */
        uint32_t pll_od : 5;   /*!< Предделитель частоты PLL */
        uint32_t pll_nr : 5;   /*!< Предделитель опорной частоты PLL */
        uint32_t : 11;         /*!< Не используется */
        uint32_t pll_nf : 14;  /*!< Множитель опорной частоты PLL */
        uint32_t : 18;         /*!< Не используется */
    };
} board_pll_cfg_t;

typedef struct {
    uint32_t vlevel;       /*!< Уровень напряжения ядра в контроллере PWRCTR (значения 0, 1, 2) */
    uint32_t fclk_freq;    /*!< Частота в Гц подсистемы FCLK */
    uint32_t sysclk_freq;  /*!< Частота в Гц подсистемы SysCLK */
    uint32_t gnssclk_freq; /*!< Частота в Гц подсистемы GNSS */
    uint32_t qspiclk_freq; /*!< Частота в Гц подсистемы QSPI */
    uint32_t i2sclk_freq;  /*!< Частота в Гц подсистемы I2S */
} board_power_cfg_t;

/*!
 * @brief Поиск и выбор конфигурации PLL, соответствующей частоте XTI
 *
 * @param xti_hz Значение внешней частоты генератора/резонатор
 *               (при значении 0 используется внутренний генератор HFI)
 * @retval 0 Конфигурация не найдена
 * @return Указатель на найденную конфигурацию
 */
const board_pll_cfg_t *BOARD_FindAndSelectPLLConfig(uint32_t xti_hz);

/*!
 * @brief Выбор конфигурации PLL
 *
 * @param pll_cfg Конфигурация PLL
 */
void BOARD_SelectPLLConfig(const board_pll_cfg_t *pll_cfg);

/*!
 * @brief Применение выбранной конфигурации PLL
 *
 * @note Перед применением конфигурации PLL необходимо выбрать конфигурацию PLL
 *      через функции BOARD_FindAndSelectPLLConfig() или BOARD_SelectPLLConfig()
 *
 * @param clkctr_base Базовый адрес модуля CLKCTR
 * @retval #BOARD_Status_UnknownConfig
 * @retval #BOARD_Status_HWError
 * @retval #BOARD_Status_OK
 */
board_cfg_status_t BOARD_ApplyPLLConfig(CLKCTR_Type *clkctr_base);

/*!
 * @brief Поиск и выбор конфигурации POWER, соответствующей порядковому номеру
 *
 * @param pwr_cfg_idx Порядковый номер конфигурации питания
 * @retval 0 Конфигурация не найдена
 * @return Указатель на найденную конфигурацию
 */
const board_power_cfg_t *BOARD_FindAndSelectPowerConfig(uint32_t pwr_cfg_idx);

/*!
 * @brief Выбор конфигурации POWER
 *
 * @param pll_cfg Конфигурация POWER
 */
void BOARD_SelectPowerConfig(const board_power_cfg_t *pwr_cfg);

/*!
 * @brief Установка конфигурации питания чипа
 *
 * @note Перед применением конфигурации POWER необходимо выбрать конфигурацию PLL
 *      через функции BOARD_FindAndSelectPLLConfig() или BOARD_SelectPLLConfig(),
 *      а также конфигурацию POWER через функции BOARD_FindAndSelectPowerConfig()
 *      или BOARD_SelectPowerConfig().
 *
 * @param clkctr_base Базовый адрес модуля CLKCTR
 * @param pwrctr_base Базовый адрес модуля PWRCTR
 * @param fctr_base Базовый адрес модуля FCTR
 * @param ext_dcdc Использовать внешний источник питания (1) да, (0) нет
 * @retval #BOARD_Status_UnknownConfig
 * @retval #BOARD_Status_HWError
 * @retval #BOARD_Status_OK
 */
board_cfg_status_t BOARD_ApplyPowerConfig(CLKCTR_Type *clkctr_base,
    PWRCTR_Type *pwrctr_base, FCTR_Type *fctr_base, bool ext_dcdc);


/*!
 * @brief Настройка памяти Flash перед переключением на новую частоту работы.
 *
 * @param base Базовый адрес контроллера Flash
 * @param freq Выставляемая частота
 */
void BOARD_PrepareFlashToFreq(FCTR_Type *base, uint32_t freq);

/*!
 * @brief Настройка питания процессора
 *
 * @param base Базовый адрес контроллера PWRCTR
 * @param vlvl Уровень напряжения ядра
 * @param ext_dcdc Внешний (1) или внутренний (0) DC-DC преобразователь
 */
void BOARD_InitPWR(PWRCTR_Type *base, uint32_t vlvl, bool ext_dcdc);

/*!
 * @brief Получение частоты FCLK из текущей конфигурации платы
 *
 * @return Частота в Гц
 */
uint32_t BOARD_Config_GetFCLKFreq();

/*!
 * @brief Получение частоты SysCLK из текущей конфигурации платы
 *
 * @return Частота в Гц
 */
uint32_t BOARD_Config_GetSysCLKFreq();

/*!
 * @brief Получение частоты GNSSCLK из текущей конфигурации платы
 *
 * @return Частота в Гц
 */
uint32_t BOARD_Config_GetGNSSCLKFreq();

/*!
 * @brief Получение частоты QSPICLK из текущей конфигурации платы
 *
 * @return Частота в Гц
 */
uint32_t BOARD_Config_GetQSPICLKFreq();

#ifdef __cplusplus
}
#endif

#endif /* ELIOT1_CONFIG_H */
