/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup can_driver
 * @{
 */

/*!
 * @file hal_can.c
 *
 * @brief Имплементация драйвера модуля ввода-вывода по интерфейсу CAN
 */

#include "hal_can.h"
#include "hal_ioim.h"

#define MAX_CAN_BAUDRATE       (1000000U) /*!< Максимальная бодовая скорость управления (классического CAN) */
#define MAX_CANFD_BAUDRATE     (8000000U) /*!< Максимальная бодовая скорость данных CAN FD */
#define CAN_DEFAULT_BAUDRATE   (500000U)  /*!< Бодовая скорость управления (классического CAN) по умолчанию */
#define CANFD_DEFAULT_DATARATE (1000000U) /*!< Бодовая скорость данных CAN FD по умолчанию */
#define RX_FIFO_CELL_NUMBER    (16U)      /*!< Количество мест в приемной очереди */

/*!
 * @name Макросы для расчета временных параметров битов (bit timings)
 * @{
 */
#define HI_BAUDRATE_BOUND      (1000000U)                               /*!< Условная граница бодовой скорости, выше которой скорость обмена считается высокой */
#define MID_BAUDRATE_BOUND     (800000U)                                /*!< Условная граница бодовой скорости, выше которой скорость обмена считается средней */
#define IDEAL_SP_HIGH_BAUDRATE (750U)                                   /*!< Идеальное положение точки семплирования бита при высокой скорости обмена */
#define IDEAL_SP_MID_BAUDRATE  (800U)                                   /*!< Идеальное положение точки семплирования бита при средней скорости обмена */
#define IDEAL_SP_LOW_BAUDRATE  (875U)                                   /*!< Идеальное положение точки семплирования бита при низкой скорости обмена */
#define IDEAL_SP_FACTOR        (1000U)                                  /*!< Масштаб величины положения точки семплирования (расчет ведется в целых числах с масштабом) */
#define MAX_DSJW               \
    (CAN_F_SJW_F_SJW_Msk >> CAN_F_SJW_F_SJW_Pos)                        /*!< Максимальное значение поля SJW для данных */
#define MAX_DTSEG2             \
    (CAN_F_SEG_2_F_SEG_2_Msk >> CAN_F_SEG_2_F_SEG_2_Pos)                /*!< Максимальное значение поля SEG2 для данных */
#define MAX_DTSEG1             \
    (CAN_F_SEG_1_F_SEG_1_Msk >> CAN_F_SEG_1_F_SEG_1_Pos)                /*!< Максимальное значение поля SEG1 для данных */
#define MAX_NTSEG2             \
    (CAN_S_SEG_2_S_SEG_2_Msk >> CAN_S_SEG_2_S_SEG_2_Pos)                /*!< Максимальное значение поля SEG2 для управления */
#define MAX_NTSEG1             \
    (CAN_S_SEG_1_S_SEG_1_Msk >> CAN_S_SEG_1_S_SEG_1_Pos)                /*!< Максимальное значение поля SEG1 для управления */
#define MIN_NTSEG2             (3U)                                     /*!< Минимальное разумное значение поля SEG2 для управления */
#define CONVENIENT_SJW         (5U - 1U)                                /*!< Разумное значение поля SJW (и для управления, и для данных) */
#define DBTP_MAX_TIME_QUANTA   (1U + MAX_DTSEG2 + 1U + MAX_DTSEG1 + 1U) /*!< Максимальное количество квантов для бита данных */
#define DBTP_MIN_TIME_QUANTA   (3U)                                     /*!< Минимальное количество квантов для бита данных */
#define NBTP_MAX_TIME_QUANTA   (1U + MAX_NTSEG2 + 1U + MAX_NTSEG1 + 1U) /*!< Максимальное количество квантов для бита управления */
#define NBTP_MIN_TIME_QUANTA   (3U)                                     /*!< Минимальное количество квантов для бита управления */
/*!
 * @}
 */

/*!
 * @name Значения поля TSSTAT регистра TCTRL, когда TTEN==0.
 * @{
 */
#define TCTRL_TSSTAT_NOTT_STB_EMPTY             (0U)    /*!< Очередь низкоприоритетного буфера выдачи пуста */
#define TCTRL_TSSTAT_NOTT_STB_LESS_THAN_HALF    (1U)    /*!< Очередь низкоприоритетного буфера выдачи заполнена наполовину или меньше */
#define TCTRL_TSSTAT_NOTT_STB_MORE_THAN_HALF    (2U)    /*!< Очередь низкоприоритетного буфера выдачи заполнена более, чем наполовину */
#define TCTRL_TSSTAT_NOTT_STB_FULL              (3U)    /*!< Очередь низкоприоритетного буфера выдачи заполнена до конца */
/*!
 * @}
 */

/*!
 * @name Значения поля RSTAT регистра RCTRL.
 * @{
 */
#define RCTRL_RSTAT_EMPTY                       (0U)    /*!< Приёмная очередь пуста */
#define RCTRL_RSTAT_LESS_THAN_ALMOST_FULL       (1U)    /*!< Приёмная очередь заполнена менее границы AFWL */
#define RCTRL_RSTAT_ALMOST_FULL                 (2U)    /*!< Приёмная очередь заполнена до границы AFWL или выше */
#define RCTRL_RSTAT_FULL                        (3U)    /*!< Приёмная очередь заполнена до конца */
/*!
 * @}
 */

/*!
 * @brief Функция, которая непосредственно устанавливает временные параметры
 *        (битовых таймингов) CAN
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с временными параметрами
 */
static void CAN_DoSetArbitrationTimingConfig(CAN_Type *base,
    const can_timing_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    SET_VAL_MSK(base->S_PRESC, CAN_S_PRESC_S_PRESC_Msk, CAN_S_PRESC_S_PRESC_Pos,
        config->prescaler);
    SET_VAL_MSK(base->S_SJW, CAN_S_SJW_S_SJW_Msk, CAN_S_SJW_S_SJW_Pos,
        config->sjw);
    SET_VAL_MSK(base->S_SEG_1, CAN_S_SEG_1_S_SEG_1_Msk, CAN_S_SEG_1_S_SEG_1_Pos,
        config->seg1);
    SET_VAL_MSK(base->S_SEG_2, CAN_S_SEG_2_S_SEG_2_Msk, CAN_S_SEG_2_S_SEG_2_Pos,
        config->seg2);
    SET_VAL_MSK(base->F_PRESC, CAN_F_PRESC_F_PRESC_Msk, CAN_F_PRESC_F_PRESC_Pos,
        config->data_prescaler);
    SET_VAL_MSK(base->F_SJW, CAN_F_SJW_F_SJW_Msk, CAN_F_SJW_F_SJW_Pos,
        config->data_sjw);
    SET_VAL_MSK(base->F_SEG_1, CAN_F_SEG_1_F_SEG_1_Msk, CAN_F_SEG_1_F_SEG_1_Pos,
        config->data_seg1);
    SET_VAL_MSK(base->F_SEG_2, CAN_F_SEG_2_F_SEG_2_Msk, CAN_F_SEG_2_F_SEG_2_Pos,
        config->data_seg2);
    SET_VAL_MSK(base->TDC, CAN_TDC_TDCEN_Msk, CAN_TDC_TDCEN_Pos,
        config->delay_compensation_enable);
    SET_VAL_MSK(base->TDC, CAN_TDC_SSPOFF_Msk, CAN_TDC_SSPOFF_Pos,
        config->secondary_sample_point_offset);
}

void CAN_SetArbitrationTimingConfig(CAN_Type *base,
    const can_timing_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    /*
     * Регистры таймингов доступны, только если контроллер находится в состоянии
     * программного сброса. Но при установке программного сброса можно сбросить
     * не все биты регистров, поэтому необходимо сохранить значения тех
     * регистров, которые потом нужно восстановить.
     */

    uint8_t cfg_stat = base->CFG_STAT;

    SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_RESET_Msk, CAN_CFG_STAT_RESET_Pos,
        1U);

    CAN_DoSetArbitrationTimingConfig(base, config);

    /* Снятие состояния сброса. */
    SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_RESET_Msk, CAN_CFG_STAT_RESET_Pos,
        0U);

    /* Восстанавление сохраненных битов. */
    base->CFG_STAT |= cfg_stat
        & (CAN_CFG_STAT_LBMI_Msk | CAN_CFG_STAT_LBME_Msk | CAN_CFG_STAT_TPSS_Msk
            | CAN_CFG_STAT_TSSS_Msk);
}

/*!
 * @brief Функция, которая непосредственно устанавливает параметры фильтрации
 *        кадров при приеме
 *
 * @param base   Базовый адрес контроллера
 * @param config Структура с параметрами фильтрации кадров
 */
static void CAN_DoSetFilterConfig(CAN_Type *base,
    const can_frame_filter_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    if (config->nb_filters_used != 0) {
        unsigned i;
        for (i = 0; i < CAN_NB_OF_FILTERS; ++i) {
            if (i < config->nb_filters_used) {
                /*
                 * Доступ к регистрам IDCODE и IDMASK только через 32-разрядную
                 * запись. Обращение по 32-разрядному указателю вместо доступа
                 * через поля структуры config.
                 */
                uint32_t *pFilter = (uint32_t *) &config->filter[i];

                /* Выбор фильтра для записи. */
                SET_VAL_MSK(base->ACFCTRL, CAN_ACFCTRL_ACFADR_Msk,
                    CAN_ACFCTRL_ACFADR_Pos, i);

                /* Установка идентификатора кадра. */
                SET_VAL_MSK(base->ACFCTRL, CAN_ACFCTRL_SELMASK_Msk,
                    CAN_ACFCTRL_SELMASK_Pos, 0U);
                base->ACF = pFilter[0];

                /*
                 * Установка маски (в контроллере используется инверсный
                 * смысл битов).
                 */
                SET_VAL_MSK(base->ACFCTRL, CAN_ACFCTRL_SELMASK_Msk,
                    CAN_ACFCTRL_SELMASK_Pos, 1U);

                uint32_t regValue = ~pFilter[1] & CAN_ACF_ACODE_AMASK_Msk;
                regValue |= pFilter[1] & (CAN_ACF_AIDEE_Msk | CAN_ACF_AIDE_Msk);
                base->ACF = regValue;

                /* Разрешение работы фильтра. */
                SET_VAL_MSK(base->ACF_EN, (1U << i), i, 1U);
            } else {
                /* Запрет работы фильтра. */
                SET_VAL_MSK(base->ACF_EN, (1U << i), i, 0U);
            }
        }
    }
}

void CAN_SetFilterConfig(CAN_Type *base,
    const can_frame_filter_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    /*
     * Регистры фильтров доступны, только если контроллер находится в состоянии
     * программного сброса. Но при установке программного сброса можно сбросить
     * не все биты регистров, поэтому необходимо сохранить значения тех
     * регистров, которые потом нужно восстановить.
     */

    uint8_t cfg_stat = base->CFG_STAT;

    SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_RESET_Msk, CAN_CFG_STAT_RESET_Pos,
        1U);

    CAN_DoSetFilterConfig(base, config);

    /* Снятие состояния сброса. */
    SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_RESET_Msk, CAN_CFG_STAT_RESET_Pos,
        0U);

    /* Восстанавление сохраненных битов. */
    base->CFG_STAT |= cfg_stat
        & (CAN_CFG_STAT_LBMI_Msk | CAN_CFG_STAT_LBME_Msk | CAN_CFG_STAT_TPSS_Msk
            | CAN_CFG_STAT_TSSS_Msk);
}

void CAN_SetPrimaryTxBufferConfig(CAN_Type *base,
    const can_ptb_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    if (config->tx_single_shot) {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_TPSS_Msk,
            CAN_CFG_STAT_TPSS_Pos, 1U);
    } else {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_TPSS_Msk,
            CAN_CFG_STAT_TPSS_Pos, 0U);
    }
}

void CAN_SetSecondaryTxBufferConfig(CAN_Type *base,
    const can_stb_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    if (config->tx_single_shot) {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_TSSS_Msk,
            CAN_CFG_STAT_TSSS_Pos, 1U);
    } else {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_TSSS_Msk,
            CAN_CFG_STAT_TSSS_Pos, 0U);
    }

    SET_VAL_MSK(base->TCTRL, CAN_TCTRL_TSMODE_Msk, CAN_TCTRL_TSMODE_Pos,
        config->tx_discipline);
}

void CAN_SetRxBufferConfig(CAN_Type *base, const can_rxb_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    SET_VAL_MSK(base->LIMIT, CAN_LIMIT_AFWL_Msk, CAN_LIMIT_AFWL_Pos,
        config->almost_full_level / RX_FIFO_CELL_NUMBER);
    SET_VAL_MSK(base->RCTRL, CAN_RCTRL_SACK_Msk, CAN_RCTRL_SACK_Pos,
        config->self_acknowledge ? 1U : 0U);
    SET_VAL_MSK(base->RCTRL, CAN_RCTRL_ROM_Msk, CAN_RCTRL_ROM_Pos,
        config->prohibit_overflow ? 1U : 0U);
}

can_status_t CAN_Init(CAN_Type *base, const can_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    /*
     * Установка программного сброса (некоторые регистры доступны только в этом
     * режиме).
     */
    SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_RESET_Msk, CAN_CFG_STAT_RESET_Pos,
        1U);

    /* Запрет по умолчанию всех прерываний. */
    base->RTIE = 0;
    base->TTCFG = 0;

    if (config->enable_listen_only) {
        SET_VAL_MSK(base->TCMD, CAN_TCMD_LOM_Msk, CAN_TCMD_LOM_Pos, 1U);
    }

    SET_VAL_MSK(base->TCTRL, CAN_TCTRL_FD_ISO_Msk, CAN_TCTRL_FD_ISO_Pos,
        config->can_fd_mode);
    SET_VAL_MSK(base->TCTRL, CAN_TCTRL_TSMODE_Msk, CAN_TCTRL_TSMODE_Pos,
        config->stb_config.tx_discipline);

    /* Установка таймингов. */
    CAN_DoSetArbitrationTimingConfig(base, &config->timing_config);

    /* Установка фильтров. */
    CAN_DoSetFilterConfig(base, &config->filter_config);

    /* Снятие состояния сброса. */
    SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_RESET_Msk, CAN_CFG_STAT_RESET_Pos,
        0U);

    if (config->enable_loopback_int) {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_LBMI_Msk,
            CAN_CFG_STAT_LBMI_Pos, 1U);
    } else {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_LBMI_Msk,
            CAN_CFG_STAT_LBMI_Pos, 0U);
    }

    if (config->enable_loopback_ext) {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_LBME_Msk,
            CAN_CFG_STAT_LBME_Pos, 1U);
    } else {
        SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_LBME_Msk,
            CAN_CFG_STAT_LBME_Pos, 0U);
    }

    CAN_SetPrimaryTxBufferConfig(base, &config->ptb_config);
    CAN_SetSecondaryTxBufferConfig(base, &config->stb_config);
    CAN_SetRxBufferConfig(base, &config->rxb_config);

    return CAN_Status_Ok;
}

void CAN_Deinit(CAN_Type *base)
{
    assert(base != NULL);

    SET_VAL_MSK(base->CFG_STAT, CAN_CFG_STAT_RESET_Msk, CAN_CFG_STAT_RESET_Pos,
        1U);
}

void CAN_EnterNormalMode(CAN_Type *base)
{
    assert(base != NULL);

    SET_VAL_MSK(base->TCMD, CAN_TCMD_STBY_Msk, CAN_TCMD_STBY_Pos, 0U);
}

void CAN_EnterStandbyMode(CAN_Type *base)
{
    assert(base != NULL);

    SET_VAL_MSK(base->TCMD, CAN_TCMD_STBY_Msk, CAN_TCMD_STBY_Pos, 1U);
}

/*!
 * @brief Расчет длительности сегментов бита классического CAN
 *
 * @param baudrate Требуемая бодовая скорость
 * @param tq_num   Количество квантов, на которые разбит бит
 * @param pconfig  Структура конфигурации временных параметров
 */
static bool CAN_GetSegments(uint32_t baudrate, uint32_t tq_num,
    can_timing_config_t *pconfig)
{
    assert(pconfig != NULL);
    assert(tq_num != 0);
    assert((baudrate > 0) && (baudrate <= MAX_CAN_BAUDRATE));

    uint32_t ideal_sp;
    uint32_t p1;
    bool fg_ret = false;

    if (baudrate >= HI_BAUDRATE_BOUND) {
        ideal_sp = IDEAL_SP_HIGH_BAUDRATE;
    } else if (baudrate >= MID_BAUDRATE_BOUND) {
        ideal_sp = IDEAL_SP_MID_BAUDRATE;
    } else {
        ideal_sp = IDEAL_SP_LOW_BAUDRATE;
    }

    p1 = tq_num * ideal_sp;
    pconfig->seg1 = (uint8_t)(p1 / (uint32_t) IDEAL_SP_FACTOR - 1U);

    if (pconfig->seg1 <= ((uint8_t) tq_num - MIN_NTSEG2)) {
        pconfig->seg2 = (uint8_t) tq_num - (pconfig->seg1 + MIN_NTSEG2);
        if (pconfig->seg2 <= MAX_NTSEG2) {
            pconfig->sjw = ((uint8_t) tq_num + CONVENIENT_SJW) / CONVENIENT_SJW;
            fg_ret = true;
        }
    }

    return fg_ret;
}

/*!
 * @brief Расчет временных параметров классического CAN
 *
 * @param baudrate        Требуемая бодовая скорость
 * @param source_clock_hz Частота синхросигнала, тактирующего контроллер CAN
 * @param pconfig         Структура конфигурации временных параметров
 */
static bool CAN_CalculateClassicTimingValues(uint32_t baudrate,
    uint32_t source_clock_hz, can_timing_config_t *pconfig)
{
    assert(pconfig != NULL);
    assert(source_clock_hz != 0);
    assert((baudrate > 0) && (baudrate <= MAX_CAN_BAUDRATE));

    uint32_t clk;
    uint32_t tq_num;
    bool fg_ret = false;

    assert(baudrate <= MAX_CAN_BAUDRATE);

    for (tq_num = NBTP_MAX_TIME_QUANTA; tq_num >= NBTP_MIN_TIME_QUANTA;
        tq_num--) {
        clk = baudrate * tq_num;
        if (clk > source_clock_hz) {
            continue;
        }

        if ((source_clock_hz / clk * clk) != source_clock_hz) {
            continue;
        }

        pconfig->prescaler = (uint16_t)(source_clock_hz / clk - 1U);

        if (CAN_GetSegments(baudrate, tq_num, pconfig)) {
            fg_ret = true;
            break;
        }
    }

    return fg_ret;
}

/*!
 * @brief Расчет длительности сегментов бита CAN FD
 *
 * @param baudrate Требуемая бодовая скорость для сегмента данных CAN FD
 * @param tq_num   Количество квантов, на которые разбит бит данных
 * @param pconfig  Структура конфигурации временных параметров
 */
static bool CAN_FdGetSegments(uint32_t baudrate_data, uint32_t tq_num,
    can_timing_config_t *pconfig)
{
    assert(pconfig != NULL);
    assert(tq_num != 0);
    assert((baudrate_data > 0) && (baudrate_data <= MAX_CANFD_BAUDRATE));

    uint32_t ideal_sp;
    uint32_t p1;
    bool fg_ret = false;

    if (baudrate_data >= HI_BAUDRATE_BOUND) {
        ideal_sp = IDEAL_SP_HIGH_BAUDRATE;
    } else if (baudrate_data >= MID_BAUDRATE_BOUND) {
        ideal_sp = IDEAL_SP_MID_BAUDRATE;
    } else {
        ideal_sp = IDEAL_SP_LOW_BAUDRATE;
    }

    p1 = tq_num * (uint32_t) ideal_sp;
    pconfig->data_seg1 = (uint8_t)(p1 / (uint32_t) IDEAL_SP_FACTOR - 1U);
    if (pconfig->data_seg1 <= MAX_DTSEG1) {
        if (pconfig->data_seg1 <= ((uint8_t) tq_num - MIN_NTSEG2)) {
            pconfig->data_seg2 = (uint8_t) tq_num
                - (pconfig->data_seg1 + MIN_NTSEG2);

            if (pconfig->data_seg2 <= MAX_DTSEG2) {
                pconfig->data_sjw = ((uint8_t) tq_num + CONVENIENT_SJW)
                    / CONVENIENT_SJW;

                if (pconfig->data_sjw > MAX_DSJW) {
                    pconfig->data_sjw = MAX_DSJW;
                }

                fg_ret = true;
            }
        }
    }

    return fg_ret;
}

bool CAN_CalculateImprovedTimingValues(uint32_t baudrate,
    uint32_t baudrate_data, uint32_t source_clock_hz,
    can_timing_config_t *pconfig)
{
    assert(pconfig != NULL);
    assert(source_clock_hz != 0);
    assert((baudrate > 0) && (baudrate <= MAX_CAN_BAUDRATE));
    assert((baudrate_data > 0) && (baudrate_data <= MAX_CANFD_BAUDRATE));

    uint32_t clk;
    uint32_t tq_num;
    bool fg_ret = false;

    if (CAN_CalculateClassicTimingValues(baudrate, source_clock_hz, pconfig)) {
        for (tq_num = DBTP_MAX_TIME_QUANTA; tq_num >= DBTP_MIN_TIME_QUANTA;
            tq_num--) {
            clk = baudrate_data * tq_num;

            if (clk > source_clock_hz) {
                continue;
            }

            if ((source_clock_hz / clk * clk) != source_clock_hz) {
                continue;
            }

            pconfig->data_prescaler = (uint16_t)(source_clock_hz / clk - 1U);

            if (CAN_FdGetSegments(baudrate_data, tq_num, pconfig)) {
                fg_ret = true;
                break;
            }
        }
    }

    return fg_ret;
}

void CAN_GetDefaultConfig(can_config_t *config, uint32_t source_clock_hz)
{
    assert(config != NULL);
    assert(source_clock_hz != 0);

    (void) memset(config, 0, sizeof(can_config_t));

    CAN_CalculateImprovedTimingValues(CAN_DEFAULT_BAUDRATE,
        CANFD_DEFAULT_DATARATE, source_clock_hz, &config->timing_config);
    config->can_fd_mode = CAN_IsoFd;

    config->rxb_config.almost_full_level = RX_FIFO_CELL_NUMBER;
    config->rxb_config.prohibit_overflow = true;

    config->filter_config.nb_filters_used = 0;
}

bool CAN_GetStatusFlag(CAN_Type *base, can_flag_t idx, can_status_t *status)
{
    assert(base != NULL);

    if (status) {
        *status = CAN_Status_Ok;
    }

    switch (idx) {
        case CAN_FlagAbortState:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_AIF_Msk, CAN_RTIF_AIF_Pos);
        case CAN_FlagError:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_EIF_Msk, CAN_RTIF_EIF_Pos);
        case CAN_FlagTransmissionSecondary:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_TSIF_Msk,
                    CAN_RTIF_TSIF_Pos);
        case CAN_FlagTransmissionPrimary:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_TPIF_Msk,
                    CAN_RTIF_TPIF_Pos);
        case CAN_FlagTransmitBufferFull:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_TSFF_Msk,
                    CAN_RTIE_TSFF_Pos);
        case CAN_FlagRBAlmostFull:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_RAFIF_Msk,
                    CAN_RTIF_RAFIF_Pos);
        case CAN_FlagRBFull:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_RFIF_Msk,
                    CAN_RTIF_RFIF_Pos);
        case CAN_FlagRBOverrun:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_ROIF_Msk,
                    CAN_RTIF_ROIF_Pos);
        case CAN_FlagReceive:
            return GET_VAL_MSK(base->RTIF, CAN_RTIF_RIF_Msk, CAN_RTIF_RIF_Pos);
        case CAN_FlagBusError:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_BEIF_Msk,
                    CAN_ERRINT_BEIF_Pos);
        case CAN_FlagArbitrationLost:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_ALIF_Msk,
                    CAN_ERRINT_ALIF_Pos);
        case CAN_FlagErrorPassiveInterrupt:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_EPIF_Msk,
                    CAN_ERRINT_EPIF_Pos);
        case CAN_FlagErrorPassiveState:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_EPASS_Msk,
                    CAN_ERRINT_EPASS_Pos);
        case CAN_FlagErrorWarningState:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_EWARN_Msk,
                    CAN_ERRINT_EWARN_Pos);
        case CAN_FlagTimeTriggered:
            return GET_VAL_MSK(base->TTCFG, CAN_TTCFG_TTIF_Msk,
                    CAN_TTCFG_TTIF_Pos);
        case CAN_FlagTriggerError:
            return GET_VAL_MSK(base->TTCFG, CAN_TTCFG_TEIF_Msk,
                    CAN_TTCFG_TEIF_Pos);
        case CAN_FlagWatchTriggerError:
            return GET_VAL_MSK(base->TTCFG, CAN_TTCFG_WTIF_Msk,
                    CAN_TTCFG_WTIF_Pos);
        default:
            if (status) {
                *status = CAN_Status_InvalidArgument;
            }

            return false;
    }
}

uint32_t CAN_GetStatusFlagMask(CAN_Type *base)
{
    assert(base != NULL);

    uint32_t mask = 0;

    for (can_flag_t i = CAN_FlagAbortState; i < CAN_FlagsNumber; ++i) {
        if (CAN_GetStatusFlag(base, i, NULL))
            mask |= (1 << i);
    }

    return mask;
}

can_status_t CAN_ClearStatusFlag(CAN_Type *base, can_flag_t idx)
{
    assert(base != NULL);

    can_status_t status = CAN_Status_Ok;

    switch (idx) {
        case CAN_FlagAbortState:
            base->RTIF = 1U << CAN_RTIF_AIF_Pos;
            break;
        case CAN_FlagError:
            base->RTIF = 1U << CAN_RTIF_EIF_Pos;
            break;
        case CAN_FlagTransmissionSecondary:
            base->RTIF = 1U << CAN_RTIF_TSIF_Pos;
            break;
        case CAN_FlagTransmissionPrimary:
            base->RTIF = 1U << CAN_RTIF_TPIF_Pos;
            break;
        case CAN_FlagTransmitBufferFull:
            base->RTIF = 1U << CAN_RTIE_TSFF_Pos;
            break;
        case CAN_FlagRBAlmostFull:
            base->RTIF = 1U << CAN_RTIF_RAFIF_Pos;
            break;
        case CAN_FlagRBFull:
            base->RTIF = 1U << CAN_RTIF_RFIF_Pos;
            break;
        case CAN_FlagRBOverrun:
            base->RTIF = 1U << CAN_RTIF_ROIF_Pos;
            break;
        case CAN_FlagReceive:
            base->RTIF = 1U << CAN_RTIF_RIF_Pos;
            break;
        case CAN_FlagBusError:
            base->ERRINT = 1U << CAN_ERRINT_BEIF_Pos;
            break;
        case CAN_FlagArbitrationLost:
            base->ERRINT = 1U << CAN_ERRINT_ALIF_Pos;
            break;
        case CAN_FlagErrorPassiveInterrupt:
            base->ERRINT = 1U << CAN_ERRINT_EPIF_Pos;
            break;
        case CAN_FlagErrorPassiveState:
            base->ERRINT = 1U << CAN_ERRINT_EPASS_Pos;
            break;
        case CAN_FlagErrorWarningState:
            base->ERRINT = 1U << CAN_ERRINT_EWARN_Pos;
            break;
        case CAN_FlagTimeTriggered:
            base->TTCFG = 1U << CAN_TTCFG_TTIF_Pos;
            break;
        case CAN_FlagTriggerError:
            base->TTCFG = 1U << CAN_TTCFG_TEIF_Pos;
            break;
        case CAN_FlagWatchTriggerError:
            base->TTCFG = 1U << CAN_TTCFG_WTIF_Pos;
            break;
        default:
            status = CAN_Status_InvalidArgument;
    }

    return status;
}

void CAN_ClearStatusFlagMask(CAN_Type *base, uint32_t mask)
{
    assert(base != NULL);

    for (can_flag_t i = CAN_FlagAbortState; i < CAN_FlagsNumber; ++i) {
        if (mask & (1 << i)) {
            CAN_ClearStatusFlag(base, i);
        }
    }
}

/*!
 * @brief Установка разрешения прерывания в требуемое значение
 *        (разрешено/запрещено)
 *
 * @param base  Базовый адрес контроллера
 * @param idx   Индекс флага прерывания в перечислении #can_flag_t
 * @param value Значение разрешения (0 - запретить, 1 - разрешить)
 */
static can_status_t CAN_SetInterruptEnable(CAN_Type *base, can_flag_t idx,
    uint8_t value)
{
    assert(base != NULL);

    can_status_t status = CAN_Status_Ok;

    switch (idx) {
        case CAN_FlagError:
            SET_VAL_MSK(base->RTIE, CAN_RTIE_EIE_Msk, CAN_RTIE_EIE_Pos, value);
            break;
        case CAN_FlagTransmissionSecondary:
            SET_VAL_MSK(base->RTIE, CAN_RTIE_TSIE_Msk, CAN_RTIE_TSIE_Pos,
                value);
            break;
        case CAN_FlagTransmissionPrimary:
            SET_VAL_MSK(base->RTIE, CAN_RTIE_TPIE_Msk, CAN_RTIE_TPIE_Pos,
                value);
            break;
        case CAN_FlagRBAlmostFull:
            SET_VAL_MSK(base->RTIE, CAN_RTIE_RAFIE_Msk, CAN_RTIE_RAFIE_Pos,
                value);
            break;
        case CAN_FlagRBFull:
            SET_VAL_MSK(base->RTIE, CAN_RTIE_RFIE_Msk, CAN_RTIE_RFIE_Pos,
                value);
            break;
        case CAN_FlagRBOverrun:
            SET_VAL_MSK(base->RTIE, CAN_RTIE_ROIE_Msk, CAN_RTIE_ROIE_Pos,
                value);
            break;
        case CAN_FlagReceive:
            SET_VAL_MSK(base->RTIE, CAN_RTIE_RIE_Msk, CAN_RTIE_RIE_Pos,
                value);
            break;
        case CAN_FlagBusError:
            SET_VAL_MSK(base->ERRINT, CAN_ERRINT_BEIE_Msk, CAN_ERRINT_BEIE_Pos,
                value);
            break;
        case CAN_FlagArbitrationLost:
            SET_VAL_MSK(base->ERRINT, CAN_ERRINT_ALIE_Msk, CAN_ERRINT_ALIE_Pos,
                value);
            break;
        case CAN_FlagErrorPassiveInterrupt:
            SET_VAL_MSK(base->ERRINT, CAN_ERRINT_EPIE_Msk, CAN_ERRINT_EPIE_Pos,
                value);
            break;
        case CAN_FlagTimeTriggered:
            SET_VAL_MSK(base->TTCFG, CAN_TTCFG_TTIE_Msk, CAN_TTCFG_TTIE_Pos,
                value);
            break;
        case CAN_FlagWatchTriggerError:
            SET_VAL_MSK(base->TTCFG, CAN_TTCFG_WTIE_Msk, CAN_TTCFG_WTIE_Pos,
                value);
            break;
        default:
            status = CAN_Status_InvalidArgument;
    }

    return status;
}

can_status_t CAN_EnableInterrupt(CAN_Type *base, can_flag_t idx)
{
    assert(base != NULL);

    return CAN_SetInterruptEnable(base, idx, 1U);
}

void CAN_EnableInterruptMask(CAN_Type *base, uint32_t mask)
{
    assert(base != NULL);

    for (can_flag_t i = CAN_FlagAbortState; i < CAN_FlagsNumber; ++i) {
        if (mask & (1 << i)) {
            CAN_EnableInterrupt(base, i);
        }
    }
}

can_status_t CAN_DisableInterrupt(CAN_Type *base, can_flag_t idx)
{
    assert(base != NULL);

    return CAN_SetInterruptEnable(base, idx, 0U);
}

void CAN_DisableInterruptMask(CAN_Type *base, uint32_t mask)
{
    assert(base != NULL);

    for (can_flag_t i = CAN_FlagAbortState; i < CAN_FlagsNumber; ++i) {
        if (mask & (1 << i)) {
            CAN_DisableInterrupt(base, i);
        }
    }
}

bool CAN_IsInterruptEnabled(CAN_Type *base, can_flag_t idx,
    can_status_t *status)
{
    assert(base != NULL);

    if (status) {
        *status = CAN_Status_Ok;
    }

    switch (idx) {
        case CAN_FlagError:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_EIE_Msk, CAN_RTIE_EIE_Pos);
        case CAN_FlagTransmissionSecondary:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_TSIE_Msk, CAN_RTIE_TSIE_Pos);
        case CAN_FlagTransmissionPrimary:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_TPIE_Msk, CAN_RTIE_TPIE_Pos);
        case CAN_FlagRBAlmostFull:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_RAFIE_Msk, CAN_RTIE_RAFIE_Pos);
        case CAN_FlagRBFull:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_RFIE_Msk, CAN_RTIE_RFIE_Pos);
        case CAN_FlagRBOverrun:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_ROIE_Msk, CAN_RTIE_ROIE_Pos);
        case CAN_FlagReceive:
            return GET_VAL_MSK(base->RTIE, CAN_RTIE_RIE_Msk, CAN_RTIE_RIE_Pos);
        case CAN_FlagBusError:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_BEIE_Msk, CAN_ERRINT_BEIE_Pos);
        case CAN_FlagArbitrationLost:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_ALIE_Msk, CAN_ERRINT_ALIE_Pos);
        case CAN_FlagErrorPassiveInterrupt:
            return GET_VAL_MSK(base->ERRINT, CAN_ERRINT_EPIE_Msk, CAN_ERRINT_EPIE_Pos);
        case CAN_FlagTimeTriggered:
            return GET_VAL_MSK(base->TTCFG, CAN_TTCFG_TTIE_Msk, CAN_TTCFG_TTIE_Pos);
        case CAN_FlagWatchTriggerError:
            return GET_VAL_MSK(base->TTCFG, CAN_TTCFG_WTIE_Msk, CAN_TTCFG_WTIE_Pos);
        default:
            if (status) {
                *status = CAN_Status_InvalidArgument;
            }

            return false;
    }
}

uint32_t CAN_GetEnabledInterruptMask(CAN_Type *base)
{
    assert(base != NULL);

    uint32_t mask = 0;

    for (can_flag_t i = CAN_FlagAbortState; i < CAN_FlagsNumber; ++i) {
        if (CAN_IsInterruptEnabled(base, i, NULL))
            mask |= (1 << i);
    }

    return mask;
}

bool CAN_IsPrimaryTransmitRequestPending(CAN_Type *base)
{
    assert(base != NULL);

    return GET_VAL_MSK(base->TCMD, CAN_TCMD_TPE_Msk, CAN_TCMD_TPE_Pos);
}

bool CAN_IsSecondaryTransmitRequestPending(CAN_Type *base)
{
    assert(base != NULL);

    return (GET_VAL_MSK(base->TCMD, CAN_TCMD_TSONE_Msk, CAN_TCMD_TSONE_Pos)
            || GET_VAL_MSK(base->TCMD, CAN_TCMD_TSALL_Msk, CAN_TCMD_TSALL_Pos));
}

bool CAN_IsSecondaryTxBufferEmpty(CAN_Type *base)
{
    assert(base != NULL);

    return (GET_VAL_MSK(base->TCTRL, CAN_TCTRL_TSSTAT_Msk, CAN_TCTRL_TSSTAT_Pos)
            == TCTRL_TSSTAT_NOTT_STB_EMPTY);
}

bool CAN_IsSecondaryTxBufferMoreThanHalfFull(CAN_Type *base)
{
    assert(base != NULL);

    return (GET_VAL_MSK(base->TCTRL, CAN_TCTRL_TSSTAT_Msk, CAN_TCTRL_TSSTAT_Pos)
            == TCTRL_TSSTAT_NOTT_STB_MORE_THAN_HALF);
}

bool CAN_IsSecondaryTxBufferFull(CAN_Type *base)
{
    return (GET_VAL_MSK(base->TCTRL, CAN_TCTRL_TSSTAT_Msk, CAN_TCTRL_TSSTAT_Pos)
            == TCTRL_TSSTAT_NOTT_STB_FULL);
}

/*!
 * @brief Функция копирования фрейма между буферами контроллера CAN и пользовательскими
 * структурами, которая учитывает, что к памяти CAN разрешён только 32-разрядный доступ.
 *
 * @param base  Базовый адрес контроллера
 * @param frame Указатель на фрейм данных
 * @param size  Размер фрейма в байтах
 */
static void CAN_SendFrame(CAN_Type *base, const void *frame, uint32_t size)
{
    size_t nb_words = size >> 2U;
    size_t word_n;

    assert(size % 4 == 0);
    assert(base != NULL);
    assert(frame != NULL);

    /* Получаем словные указатели на источник и приёмник данных. */
    uint32_t *dst32 = (uint32_t *) &base->TBUF0;
    const uint32_t *src32 = (const uint32_t *) frame;

    for (word_n = 0; word_n < nb_words; ++word_n)
        *dst32++ = *src32++;
}

/*!
 * @brief Функция копирования фрейма между буферами контроллера CAN и пользовательскими
 * структурами, которая учитывает, что к памяти CAN разрешён только 32-разрядный доступ.
 *
 * @param base  Базовый адрес контроллера
 * @param frame Указатель на фрейм данных
 * @param size  Размер фрейма в байтах
 */
static void CAN_RecieveFrame(CAN_Type *base, void *frame, uint32_t size)
{
    size_t nb_words = size >> 2U;
    size_t word_n;

    assert(size % 4 == 0);
    assert(base != NULL);
    assert(frame != NULL);

    /* Получаем словные указатели на источник и приёмник данных. */
    const uint32_t *src32 = (const uint32_t *) &base->RBUF0;
    uint32_t *dst32 = (uint32_t *) frame;

    for (word_n = 0; word_n < nb_words; ++word_n)
        *dst32++ = *src32++;
}

can_status_t CAN_WritePrimaryTxBuffer(CAN_Type *base,
    const can_tx_buffer_frame_t *ptxframe)
{
    assert(base != NULL);
    assert(ptxframe != NULL);

    if (CAN_IsPrimaryTransmitRequestPending(base)) {
        return CAN_Status_TxBusy;
    }

    /* Выбор для записи PTB. */
    SET_VAL_MSK(base->TCMD, CAN_TCMD_TBSEL_Msk, CAN_TCMD_TBSEL_Pos, 0U);

    CAN_SendFrame(base, ptxframe, sizeof(can_tx_buffer_frame_t));

    /* Запуск передачи. */
    SET_VAL_MSK(base->TCMD, CAN_TCMD_TPE_Msk, CAN_TCMD_TPE_Pos, 1U);

    return CAN_Status_Ok;
}

can_status_t CAN_WriteSecondaryTxBuffer(CAN_Type *base,
    const can_tx_buffer_frame_t *ptxframe)
{
    assert(base != NULL);
    assert(ptxframe != NULL);

    if (CAN_IsSecondaryTxBufferFull(base)) {
        return CAN_Status_TxBusy;
    }

    /* Выбор для записи STB. */
    SET_VAL_MSK(base->TCMD, CAN_TCMD_TBSEL_Msk, CAN_TCMD_TBSEL_Pos, 1U);

    CAN_SendFrame(base, ptxframe, sizeof(can_tx_buffer_frame_t));

    /* Выставление признака готовности кадра к размещению в STB. */
    SET_VAL_MSK(base->TCTRL, CAN_TCTRL_TSNEXT_Msk, CAN_TCTRL_TSNEXT_Pos, 1U);

    /* Запуск передачи. */
    SET_VAL_MSK(base->TCMD, CAN_TCMD_TSALL_Msk, CAN_TCMD_TSALL_Pos, 1U);

    return CAN_Status_Ok;
}

void CAN_AbortPrimaryTxBuffer(CAN_Type *base)
{
    assert(base != NULL);

    SET_VAL_MSK(base->TCMD, CAN_TCMD_TPA_Msk, CAN_TCMD_TPA_Pos, 1U);
}

void CAN_AbortSecondaryTxBuffer(CAN_Type *base)
{
    assert(base != NULL);

    SET_VAL_MSK(base->TCMD, CAN_TCMD_TSA_Msk, CAN_TCMD_TSA_Pos, 1U);
}

bool CAN_IsRxBufferEmpty(CAN_Type *base)
{
    assert(base != NULL);

    return (GET_VAL_MSK(base->RCTRL, CAN_RCTRL_RSTAT_Msk, CAN_RCTRL_RSTAT_Pos)
            == RCTRL_RSTAT_EMPTY);
}

bool CAN_IsRxBufferAlmostFull(CAN_Type *base)
{
    assert(base != NULL);

    return (GET_VAL_MSK(base->RCTRL, CAN_RCTRL_RSTAT_Msk, CAN_RCTRL_RSTAT_Pos)
            == TCTRL_TSSTAT_NOTT_STB_MORE_THAN_HALF);
}

bool CAN_IsRxBufferFull(CAN_Type *base)
{
    assert(base != NULL);

    return (GET_VAL_MSK(base->RCTRL, CAN_RCTRL_RSTAT_Msk, CAN_RCTRL_RSTAT_Pos)
            == TCTRL_TSSTAT_NOTT_STB_FULL);
}

can_status_t CAN_ReadRxBuffer(CAN_Type *base, can_rx_buffer_frame_t *prxframe)
{
    assert(base != NULL);
    assert(prxframe != NULL);

    if (CAN_IsRxBufferEmpty(base)) {
        return CAN_Status_RxEmpty;
    }

    CAN_RecieveFrame(base, prxframe, sizeof(can_rx_buffer_frame_t));

    /* Выставление признака, что кадр из приемного буфера прочитан. */
    SET_VAL_MSK(base->RCTRL, CAN_RCTRL_RREL_Msk, CAN_RCTRL_RREL_Pos, 1U);

    return CAN_Status_Ok;
}

can_status_t CAN_TransferSendPrimaryBlocking(CAN_Type *base,
    can_tx_buffer_frame_t *ptxframe, size_t nb_frames)
{
    assert(base != NULL);
    assert(ptxframe != NULL);

    /* Если nb_frames == 0, то возврат ошибки CAN_Status_InvalidArgument. */
    can_status_t status = CAN_Status_InvalidArgument;

    /* Отправка кадров с проверкой занятости передатчика. */
    for (; nb_frames > 0U; --nb_frames, ++ptxframe) {
        for (;;) {
            /*
             * Если передатчик занят, то перезапуск транзакции.
             * Если какая-либо другая ошибка, то завершение функции
             * с этим кодом ошибки.
             */
            status = CAN_WritePrimaryTxBuffer(base, ptxframe);
            if (status == CAN_Status_Ok)
                break;
            else if (status == CAN_Status_TxBusy)
                continue;
            else
                return status;
        }
    }

    return status;
}

can_status_t CAN_TransferSendSecondaryBlocking(CAN_Type *base,
    can_tx_buffer_frame_t *ptxframe, size_t nb_frames)
{
    assert(base != NULL);
    assert(ptxframe != NULL);

    /* Если nb_frames == 0, то возврат ошибки CAN_Status_InvalidArgument. */
    can_status_t status = CAN_Status_InvalidArgument;

    /* Отправка кадров с проверкой занятости передатчика. */
    for (; nb_frames > 0U; --nb_frames, ++ptxframe) {
        for (;;) {
            /*
             * Если передатчик занят, то перезапуск транзакции.
             * Если какая-либо другая ошибка, то завершение функции
             * с этим кодом ошибки.
             */
            status = CAN_WriteSecondaryTxBuffer(base, ptxframe);
            if (status == CAN_Status_Ok)
                break;
            else if (status == CAN_Status_TxBusy)
                continue;
            else
                return status;
        }
    }

    return status;
}

can_status_t CAN_TransferReceiveFifoBlocking(CAN_Type *base,
    can_rx_buffer_frame_t *prxframe, size_t nb_frames)
{
    assert(base != NULL);
    assert(prxframe != NULL);

    /* Если nb_frames == 0, то возврат ошибки CAN_Status_InvalidArgument. */
    can_status_t status = CAN_Status_InvalidArgument;

    /* Прием заданного числа кадров в цикле. */
    for (; nb_frames > 0U; --nb_frames, ++prxframe) {
        for (;;) {
            /*
             * Если передатчик занят, то перезапуск транзакции.
             * Если какая-либо другая ошибка, то завершение функции
             * с этим кодом ошибки.
             */
            status = CAN_ReadRxBuffer(base, prxframe);
            if (status == CAN_Status_Ok)
                break;
            else if (status == CAN_Status_RxEmpty)
                continue;
            else
                return status;
        }
    }

    return status;
}

can_status_t CAN_TransferCreateHandle(CAN_Type *base, can_handle_t *handle,
    can_transfer_callback_t callback, void *user_data)
{
    assert(base != NULL);
    assert(handle != NULL);

    (void) memset(handle, 0, sizeof(*handle));

    /* Установка обратного вызова и пользовательских данных. */
    handle->callback  = callback;
    handle->user_data = user_data;

    /* Установка обработчика прерывания. */
    if (IOIM_SetIRQHandler(base, CAN_TransferHandleIRQ, handle) != IOIM_Status_Ok)
        return CAN_Status_Fail;

    return CAN_Status_Ok;
}

can_status_t CAN_TransferSendPrimaryNonBlocking(CAN_Type *base,
    can_handle_t *handle, can_tx_transfer_t *xfer)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(xfer != NULL);
    assert(xfer->frames != NULL);

    /* Проверка структуры передачи CAN. */
    if (xfer->nb_frames == 0U) {
        return CAN_Status_InvalidArgument;
    }

    /* Возврат ошибки, если уже в процессе передачи. */
    if (handle->tx_nb_frames_rest_prim != 0U) {
        return CAN_Status_TxBusy;
    } else {
        /*
         * Отключение IRQ при настройке дескриптора передачи, если произойдет
         * прерывание во время процесса настройки, то значение дескриптора
         * может быть испорчено.
         */
        uint32_t interrupt_mask = CAN_GetEnabledInterruptMask(base);
        CAN_DisableInterruptMask(base, interrupt_mask);

        handle->tx_frames_prim          = xfer->frames;
        handle->tx_nb_frames_rest_prim  = xfer->nb_frames;
        handle->tx_nb_frames_all_prim   = xfer->nb_frames;

        /* Разрешение прерывания по выдаче из высокоприоритетного буфера. */
        interrupt_mask |= (1 << CAN_FlagTransmissionPrimary);

        CAN_EnableInterruptMask(base, interrupt_mask);

        /*
         * Загрузка высокоприоритетного буфера,
         * в нем может быть только один кадр.
         */
        return CAN_WritePrimaryTxBuffer(base, xfer->frames);
    }

    return CAN_Status_Ok;
}

can_status_t CAN_TransferGetSentPrimaryCount(CAN_Type *base,
    can_handle_t *handle, uint32_t *nb_frames)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(nb_frames != NULL);

    if (handle->tx_nb_frames_rest_prim == 0U)
        return CAN_Status_TxIdle;

    *nb_frames = handle->tx_nb_frames_all_prim - handle->tx_nb_frames_rest_prim;

    return CAN_Status_Ok;
}

void CAN_TransferAbortSendPrimary(CAN_Type *base, can_handle_t *handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    CAN_DisableInterrupt(base, CAN_FlagTransmissionPrimary);

    handle->tx_nb_frames_all_prim = 0U;
    handle->tx_nb_frames_rest_prim = 0U;
}

can_status_t CAN_TransferSendSecondaryNonBlocking(CAN_Type *base,
    can_handle_t *handle, can_tx_transfer_t *xfer)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(xfer != NULL);
    assert(xfer->frames != NULL);

    /* Проверка структуры передачи CAN. */
    if (xfer->nb_frames == 0U) {
        return CAN_Status_InvalidArgument;
    }

    /* Возврат ошибки, если уже в процессе передачи. */
    if (handle->tx_nb_frames_rest_sec != 0U) {
        return CAN_Status_TxBusy;
    } else {
        /*
         * Отключение IRQ при настройке дескриптора передачи, если произойдет
         * прерывание во время процесса настройки, то значение дескриптора
         * может быть испорчено.
         */
        uint32_t interrupt_mask = CAN_GetEnabledInterruptMask(base);
        CAN_DisableInterruptMask(base, interrupt_mask);

        handle->tx_frames_sec          = xfer->frames;
        handle->tx_nb_frames_rest_sec  = xfer->nb_frames;
        handle->tx_nb_frames_all_sec   = xfer->nb_frames;

        /* Разрешение прерывания по выдаче из низкоприоритетного буфера. */
        interrupt_mask |= (1 << CAN_FlagTransmissionSecondary);

        CAN_EnableInterruptMask(base, interrupt_mask);

        /* Загрузка одного кадра в низкоприоритетный буфер. Преимущества
           FIFO не используются, поскольку есть прерывание только при
           отправке одного кадра, и нет прерываний по опустошению FIFO. */
        return CAN_WriteSecondaryTxBuffer(base, xfer->frames);
    }

    return CAN_Status_Ok;
}

can_status_t CAN_TransferGetSentSecondaryCount(CAN_Type *base,
    can_handle_t *handle, uint32_t *nb_frames)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(nb_frames != NULL);

    if (handle->tx_nb_frames_rest_sec == 0U)
        return CAN_Status_TxIdle;

    *nb_frames = handle->tx_nb_frames_all_sec - handle->tx_nb_frames_rest_sec;

    return CAN_Status_Ok;
}

void CAN_TransferAbortSendSecondary(CAN_Type *base, can_handle_t *handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    CAN_DisableInterrupt(base, CAN_FlagTransmissionSecondary);

    handle->tx_nb_frames_all_sec = 0U;
    handle->tx_nb_frames_rest_sec = 0U;
}

can_status_t CAN_TransferReceiveFifoNonBlocking(CAN_Type *base,
    can_handle_t *handle, can_rx_transfer_t *xfer)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(xfer != NULL);
    assert(xfer->frames != NULL);

    /* Проверка структуры передачи CAN. */
    if (xfer->nb_frames == 0U) {
        return CAN_Status_InvalidArgument;
    }

    /* Возврат ошибки, если уже в процессе передачи. */
    if (handle->rx_nb_frames_rest != 0U) {
        return CAN_Status_RxBusy;
    } else {
        uint32_t interrupt_mask = CAN_GetEnabledInterruptMask(base);
        CAN_DisableInterruptMask(base, interrupt_mask);

        handle->rx_frames         = xfer->frames;
        handle->rx_nb_frames_rest = xfer->nb_frames;
        handle->rx_nb_frames_all  = xfer->nb_frames;

        /* Разрешение прерывания по приему очередного кадра. */
        interrupt_mask |= (1 << CAN_FlagReceive);

        CAN_EnableInterruptMask(base, interrupt_mask);
    }

    return CAN_Status_Ok;
}

can_status_t CAN_TransferGetReceivedCount(CAN_Type *base,
    can_handle_t *handle, uint32_t *nb_frames)
{
    assert(base != NULL);
    assert(handle != NULL);
    assert(nb_frames != NULL);

    if (handle->rx_nb_frames_rest == 0U)
        return CAN_Status_RxIdle;

    *nb_frames = handle->rx_nb_frames_all - handle->rx_nb_frames_rest;

    return CAN_Status_Ok;
}

void CAN_TransferAbortReceive(CAN_Type *base, can_handle_t *handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    CAN_DisableInterrupt(base, CAN_FlagReceive);

    handle->rx_nb_frames_rest = 0U;
    handle->rx_nb_frames_all = 0U;
}

void CAN_TransferHandleIRQ(CAN_Type *base, can_handle_t *handle)
{
    /* Признак включенной передачи через высокоприоритетный буфер */
    bool tx_primary_enabled = (handle->tx_nb_frames_rest_prim != 0U);
    /* Признак включенной передачи через низкоприоритетный буфер */
    bool tx_secondary_enabled = (handle->tx_nb_frames_rest_sec != 0U);
    /* Признак включенного приема */
    bool rx_enabled = (handle->rx_nb_frames_rest != 0U);

    /* Обработка активных прерываний. */

    for (can_flag_t i = CAN_FlagAbortState; i < CAN_FlagsNumber; ++i) {
        /*
         * Если передача запущена с помощью CAN_TransferSendPrimaryNonBlocking,
         * то коллбэк-функция для признака CAN_FlagTransmissionPrimary
         * не вызывается.
         */
        if (tx_primary_enabled && (i == CAN_FlagTransmissionPrimary))
            continue;
        /*
         * Если передача запущена с помощью CAN_TransferSendSecondaryNonBlocking,
         * то коллбэк-функция для признака CAN_FlagTransmissionSecondary
         * не вызывается.
         */
        if (tx_secondary_enabled && (i == CAN_FlagTransmissionSecondary))
            continue;
        /*
         * Если передача запущена с помощью CAN_TransferReceiveFifoNonBlocking,
         * то коллбэк-функция для признака CAN_FlagReceive не вызывается.
         */
        if (rx_enabled && (i == CAN_FlagReceive))
            continue;

        /* Вызов коллбэк-функции, если прерывание активно. */
        if (CAN_IsInterruptEnabled(base, i, NULL)
            && CAN_GetStatusFlag(base, i, NULL)) {
            handle->callback(base, handle, CAN_Status_Ok, i, handle->user_data);
            CAN_ClearStatusFlag(base, i);
        }
    }

    /* Обработка активных прерываний */

    if (tx_primary_enabled) {
        /* Передача через высокоприоритетный буфер */
        if (CAN_GetStatusFlag(base, CAN_FlagTransmissionPrimary, NULL)) {
            CAN_ClearStatusFlag(base, CAN_FlagTransmissionPrimary);

            --handle->tx_nb_frames_rest_prim;
            if (handle->tx_nb_frames_rest_prim == 0U) {
                handle->callback(base, handle, CAN_Status_TxIdle,
                    CAN_FlagTransmissionPrimary, handle->user_data);
            } else {
                CAN_WritePrimaryTxBuffer(base,
                    (const can_tx_buffer_frame_t *)(handle->tx_frames_prim +
                        handle->tx_nb_frames_all_prim -
                        handle->tx_nb_frames_rest_prim));
            }
        }
    }

    if (tx_secondary_enabled) {
        /* Передача через низкоприоритетный буфер. */
        if (CAN_GetStatusFlag(base, CAN_FlagTransmissionSecondary, NULL)) {
            CAN_ClearStatusFlag(base, CAN_FlagTransmissionSecondary);

            --handle->tx_nb_frames_rest_sec;
            if (handle->tx_nb_frames_rest_sec == 0U) {
                handle->callback(base, handle, CAN_Status_TxIdle,
                    CAN_FlagTransmissionSecondary, handle->user_data);
            } else {
                CAN_WriteSecondaryTxBuffer(base,
                    (const can_tx_buffer_frame_t *)(handle->tx_frames_sec +
                        handle->tx_nb_frames_all_sec -
                        handle->tx_nb_frames_rest_sec));
            }
        }
    }

    if (rx_enabled) {
        /* Включен прием. */
        if (CAN_GetStatusFlag(base, CAN_FlagReceive, NULL)) {

            can_status_t status;
            while (1) {
                status = CAN_ReadRxBuffer(base,
                        (can_rx_buffer_frame_t *)(handle->rx_frames +
                            handle->rx_nb_frames_all -
                            handle->rx_nb_frames_rest));

                if (status == CAN_Status_Ok) {
                    --handle->rx_nb_frames_rest;
                    if (handle->rx_nb_frames_rest == 0U) {
                        handle->callback(base, handle, CAN_Status_RxIdle,
                            CAN_FlagReceive, handle->user_data);
                        break;
                    }
                } else {
                    break;
                }
            }

            CAN_ClearStatusFlag(base, CAN_FlagReceive);
        }
    }
}

/*!
 * @}
 */
