/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup dma_driver
 * @{
 */

/*!
 * @file hal_dma.c
 *
 * @brief Имплементация драйвера прямого доступа к памяти
 */

#include "hal_dma.h"
#include "hal_ioim.h"

/*! @brief Указатели на базовые адреса DMA */
#define DMA_BASE_PTRS { DMA0, DMA1 }
/*! @brief Маска для формирования secure адресов. */
#define SECURE_BIT_MSK (0x10000000)

/*!
 * @brief Получить номер экземпляра DMA
 *
 * @param base Базовый адрес DMA
 *
 * @retval 0 - DMA0
 * @retval 1 - DMA1
 * @retval 2 - Указанный базовый адрес не существует
 */
static uint32_t DMA_GetInstance(DMA_Type *base);

/*!
 * @brief Получение указателя на регистры канала
 *
 * @param base Базовый адрес контроллера
 * @param channel Номер канала
 *
 * @retval Указатель на регистры выбранного канала
 */
static dma_channel_regs *DMA_GetChannelDescriptor(DMA_Type *base,
    uint32_t channel);

/*! @brief Массив указателей на базовые адреса DMA */
static DMA_Type *const s_dma_bases[] = DMA_BASE_PTRS;

/*! @brief Указатель на контекст драйвера для каждого канала DMA */
static dma_handle_t *s_DMAHandle[HAL_FEATURE_DMA_ALL_CHANNELS];

static uint32_t DMA_GetInstance(DMA_Type *base)
{
    uint32_t instance;
    uint32_t base_addr = (uint32_t) base & ~SECURE_BIT_MSK;
    uint32_t instance_addr;
    /* Поиск номера контроллера DMA по таблице базовых адресов DMA. */
    for (instance = 0; instance < DIM(s_dma_bases); instance++) {
        instance_addr = (uint32_t) s_dma_bases[instance];
        if (base_addr == instance_addr) {
            break;
        }
    }

    return instance;
}

static dma_channel_regs *DMA_GetChannelDescriptor(DMA_Type *base,
    uint32_t channel)
{
    assert(base != NULL);
    assert(channel < HAL_FEATURE_DMA_NUMBER_OF_CHANNELS);

    dma_channel_regs *ch_desc = NULL;
    uint32_t base_addr = (uint32_t) base;

    ch_desc = (dma_channel_regs *)(base_addr + channel * REG_OFFSET);

    return ch_desc;
}

void DMA_Init(DMA_Type *base)
{
    assert(base != NULL);

    base->DMACFGREG_LO |= DMA_DMACFGREG_LO_DMA_EN_Msk;
}

void DMA_Deinit(DMA_Type *base)
{
    assert(base != NULL);

    do {
        /* Установка нуля в поле DMA_EN. */
        SET_VAL_MSK(base->DMACFGREG_LO, DMA_DMACFGREG_LO_DMA_EN_Msk,
            DMA_DMACFGREG_LO_DMA_EN_Pos, 0);
    } while (base->DMACFGREG_LO & DMA_DMACFGREG_LO_DMA_EN_Msk);

    for (uint32_t i = 0; i < HAL_FEATURE_DMA_NUMBER_OF_CHANNELS; i++) {
        IOIM_ClearIRQHandler_DMA(base, i);
    }
}

dma_status_t DMA_CreateHandle(dma_handle_t *handle, DMA_Type *base,
    uint32_t channel
)
{
    assert((handle != NULL)
        && (channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS)
        && (base != NULL)
    );

    uint32_t start_channel = 0;
    /* Проверка на существование базового адреса. */
    uint32_t instance = DMA_GetInstance(base);
    if (instance > 1) {
        return DMA_Status_NoBase;
    } else if (instance) {
        start_channel = 8;
    }

    (void)memset(handle, 0, sizeof(dma_handle_t));

    handle->base = base;
    handle->channel = channel;

    s_DMAHandle[start_channel + channel] = handle;

    /* Разрешение прерываний канала. */
    if (IOIM_ClearIRQHandler(base) != IOIM_Status_Ok) {
        return DMA_Status_Fail;
    }

    if (IOIM_SetIRQHandler_DMA(base, channel, DMA_ChannelIRQHandle,
            handle) != IOIM_Status_Ok) {
        return DMA_Status_Fail;
    }

    return DMA_Status_Success;
}

void DMA_SetChannelPriority(DMA_Type *base, uint32_t channel,
    dma_priority_t priority)
{
    assert((base != NULL) &&
        (channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS));

    dma_channel_regs *ch_desc = DMA_GetChannelDescriptor(base, channel);

    SET_VAL_MSK(ch_desc->CFG_LO, DMA_CFG0_LO_CH_PRIOR_Msk,
        DMA_CFG0_LO_CH_PRIOR_Pos, priority);
}

dma_priority_t DMA_GetChannelPriority(DMA_Type *base, uint32_t channel)
{
    assert((base != NULL)
        && (channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS));

    dma_priority_t priority;

    dma_channel_regs *ch_desc = DMA_GetChannelDescriptor(base, channel);
    priority = GET_VAL_MSK(ch_desc->CFG_LO, DMA_CFG0_LO_CH_PRIOR_Msk,
            DMA_CFG0_LO_CH_PRIOR_Pos);
    return priority;
}

void DMA_DisableChannel(DMA_Type *base, uint32_t channel)
{
    assert((base != NULL)
        && (channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS));
    uint32_t tmp = 0;
    do {
        tmp = base->CHENREG_LO;
        tmp &= ~(1 << channel);
        tmp |= (1 << (channel + 8));
        base->CHENREG_LO = tmp;
    } while (1 & (base->CHENREG_LO >> channel));
}

void DMA_SetCallback(dma_handle_t *handle, dma_callback callback,
    void *user_data)
{
    assert(handle != NULL);

    handle->callback = callback;
    handle->user_data = user_data;
}

uint64_t DMA_GetCTLCfgMask(struct dma_channel_ctl_cfg *ctlxcfg)
{
    uint64_t ctlxcfg_mask = 0ULL;

    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_LLP_SRC_EN_Msk,
        DMA_CTL0_LO_LLP_SRC_EN_Pos, ctlxcfg->llp_src_en);
    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_LLP_DST_EN_Msk,
        DMA_CTL0_LO_LLP_DST_EN_Pos, ctlxcfg->llp_dst_en);

    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_DST_SCATTER_EN_Msk,
        DMA_CTL0_LO_DST_SCATTER_EN_Pos, ctlxcfg->scatter_en);
    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_SRC_GATHER_EN_Msk,
        DMA_CTL0_LO_SRC_GATHER_EN_Pos, ctlxcfg->gather_en);

    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_TT_FC_Msk,
        DMA_CTL0_LO_TT_FC_Pos, ctlxcfg->transfer_type);
    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_SRC_TR_WIDTH_Msk,
        DMA_CTL0_LO_SRC_TR_WIDTH_Pos, ctlxcfg->src_tr_width);
    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_DST_TR_WIDTH_Msk,
        DMA_CTL0_LO_DST_TR_WIDTH_Pos, ctlxcfg->dst_tr_width);

    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_DINC_Msk,
        DMA_CTL0_LO_DINC_Pos, ctlxcfg->dst_incr);
    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_SINC_Msk,
        DMA_CTL0_LO_SINC_Pos, ctlxcfg->src_incr);

    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_DEST_MSIZE_Msk,
        DMA_CTL0_LO_DEST_MSIZE_Pos, ctlxcfg->dst_burst_size);
    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_SRC_MSIZE_Msk,
        DMA_CTL0_LO_SRC_MSIZE_Pos, ctlxcfg->src_burst_size);

    SET_VAL_MSK(ctlxcfg_mask, DMA_CTL0_LO_INT_EN_Msk,
        DMA_CTL0_LO_INT_EN_Pos, ctlxcfg->int_en);

    ctlxcfg_mask |= (uint64_t)((uint64_t)(ctlxcfg->block_size) << 32);

    return ctlxcfg_mask;
}

void DMA_PrepareChannelTransfer(dma_channel_config_t *config,
    void *src_addr, void *dst_addr, uint64_t ctlx_cfg)
{
    assert(config != NULL);
    assert((src_addr != NULL) && (dst_addr != NULL));

    (void) memset(config, 0, sizeof(dma_channel_config_t));

    config->src_addr = src_addr;
    config->dst_addr = dst_addr;
    config->ctlx_cfg = ctlx_cfg;

    dma_transfer_type_t type = GET_VAL_MSK(ctlx_cfg, DMA_CTL0_LO_TT_FC_Msk,
            DMA_CTL0_LO_TT_FC_Pos);
    switch (type) {
        case DMA_PeripheralToMemory_DMA:
        case DMA_PeripheralToMemory_Peripheral:
            config->is_src_periph = true;
            break;
        case DMA_MemoryToPeripheral_DMA:
        case DMA_MemoryToPeripheral_Peripheral:
            config->is_dst_periph = true;
            break;
        case DMA_PeripheralToPeripheral_DMA:
        case DMA_PeripheralToPeripheral_SRC:
        case DMA_PeripheralToPeripheral_DST:
            config->is_src_periph = true;
            config->is_dst_periph = true;
            break;
        default:
            break;
    }
}

dma_status_t DMA_SubmitChannelTransfer(dma_handle_t *handle,
    dma_channel_config_t *config
)
{
    assert((handle != NULL) && (config != NULL));
    assert(handle->channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS);

    /* Проверка на существование базового адреса. */
    uint32_t instance = DMA_GetInstance(handle->base);
    if (instance > 1) {
        return DMA_Status_NoBase;
    }

    uint32_t base_addr = (uint32_t) handle->base;
    uint32_t channel = handle->channel;

    if (DMA_ChannelIsActive(handle->base, channel)) {
        return DMA_Status_Busy;
    }

    dma_channel_regs *regs =
        (dma_channel_regs *)(base_addr + channel * REG_OFFSET);

    regs->SAR_LO = (uint32_t) config->src_addr;
    regs->DAR_LO = (uint32_t) config->dst_addr;
    regs->CTL_LO = (uint32_t) config->ctlx_cfg;
    regs->CTL_HI = (uint32_t)(config->ctlx_cfg >> 32);

    return DMA_Status_Success;
}

void DMA_SubmitChannelDescriptor(dma_handle_t *handle,
    dma_descriptor_t *descriptor)
{
    assert((handle != NULL && handle->base != NULL
            && handle->channel < HAL_FEATURE_DMA_NUMBER_OF_CHANNELS));

    dma_channel_regs *ch_desc = DMA_GetChannelDescriptor(handle->base,
            handle->channel);

    ch_desc->LLP_LO = (uint32_t) descriptor;
}

void DMA_EnableChannelInterrupt(DMA_Type *base, uint32_t channel,
    uint8_t int_mask)
{
    assert((base != NULL)
        && (channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS));

    const uint32_t dma_channel_count = HAL_FEATURE_DMA_NUMBER_OF_CHANNELS;

    uint32_t tmp_mask = 0;

    for (uint8_t int_index = 0; int_index < DMA_AllIRQCount; int_index++) {
        tmp_mask = int_mask & (1 << int_index);
        switch (tmp_mask) {
            case DMA_IntTfr:
                base->MASKTFR_LO |= ((1 << (dma_channel_count + channel))
                        | (1 << channel));
                break;
            case DMA_IntBlock:
                base->MASKBLOCK_LO |= ((1 << (dma_channel_count + channel)
                            | (1 << channel)));
                break;
            case DMA_IntSrcTran:
                base->MASKSRCTRAN_LO |= ((1 << (dma_channel_count + channel)
                            | (1 << channel)));
                break;
            case DMA_IntDstTran:
                base->MASKDSTTRAN_LO |= ((1 << (dma_channel_count + channel)
                            | (1 << channel)));
                break;
            case DMA_IntError:
                base->MASKERR_LO |= ((1 << (dma_channel_count + channel)
                            | (1 << channel)));
                break;
            default:
                break;
        }
    }
}

void DMA_DisableChannelInterrupt(DMA_Type *base, uint32_t channel,
    uint8_t int_mask)
{
    assert((base != NULL)
        && (channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS));

    const uint32_t dma_channel_count = HAL_FEATURE_DMA_NUMBER_OF_CHANNELS;

    uint32_t tmp_mask = 0;
    uint32_t tmp = 0;

    for (uint8_t int_index = 0; int_index < DMA_AllIRQCount; int_index++) {
        tmp_mask = int_mask & (1 << int_index);
        switch (tmp_mask) {
            case DMA_IntTfr:
                tmp = (base->MASKTFR_LO & ~((uint32_t)(1 << channel)));
                tmp |= (1 << (channel + dma_channel_count));
                base->MASKTFR_LO = tmp;
                break;
            case DMA_IntBlock:
                tmp = (base->MASKBLOCK_LO & ~((uint32_t)(1 << channel)));
                tmp |= (1 << (channel + dma_channel_count));
                base->MASKBLOCK_LO = tmp;
                break;
            case DMA_IntDstTran:
                tmp = (base->MASKDSTTRAN_LO & ~((uint32_t)(1 << channel)));
                tmp |= (1 << (channel + dma_channel_count));
                base->MASKDSTTRAN_LO = tmp;
                break;
            case DMA_IntSrcTran:
                tmp = (base->MASKSRCTRAN_LO & ~((uint32_t)(1 << channel)));
                tmp |= (1 << (channel + dma_channel_count));
                base->MASKSRCTRAN_LO = tmp;
                break;
            case DMA_IntError:
                tmp = (base->MASKERR_LO & ~((uint32_t)(1 << channel)));
                tmp |= (1 << (channel + dma_channel_count));
                base->MASKERR_LO = tmp;
                break;
            default:
                break;
        }
    }
}

void DMA_HardwareHandshakeEnable(dma_handle_t *handle, bool enable,
    uint32_t req_num)
{
    assert(handle->base != NULL);
    assert(handle->channel < HAL_FEATURE_DMA_NUMBER_OF_CHANNELS);
    assert(req_num < HAL_FEATURE_DMA_MAX_NUMBER_HW_HANDSHAKES);

    dma_channel_regs *regs = DMA_GetChannelDescriptor(handle->base,
            handle->channel);

    /* Проверка на четность: четные - Приемники (tx), нечётные - Источники (rx). */
    if (req_num & 1) {
        /* Включение/выключение аппаратного запроса Источника. */
        SET_VAL_MSK(regs->CFG_LO, DMA_CFG0_LO_HS_SEL_SRC_Msk,
            DMA_CFG0_LO_HS_SEL_SRC_Pos, !enable);
        /* Установка номера handshake запроса. */
        SET_VAL_MSK(regs->CFG_HI, DMA_CFG0_HI_SRC_PER_Msk,
            DMA_CFG0_HI_SRC_PER_Pos, req_num);
    } else {
        /* Включение/выключение аппаратного запроса Приемника. */
        SET_VAL_MSK(regs->CFG_LO, DMA_CFG0_LO_HS_SEL_DST_Msk,
            DMA_CFG0_LO_HS_SEL_DST_Pos, !enable);
        /* Установка номера handshake запроса. */
        SET_VAL_MSK(regs->CFG_HI, DMA_CFG0_HI_DEST_PER_Msk,
            DMA_CFG0_HI_DEST_PER_Pos, req_num);
    }
}

dma_status_t DMA_SubmitChannelTransferParameter(dma_handle_t *handle,
    uint64_t ctlx_cfg, void *src_addr, void *dst_addr)
{
    assert(handle != NULL);
    assert(handle->base != NULL);
    assert(handle->channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS);

    dma_channel_config_t config;

    memset(&config, 0, sizeof(dma_channel_config_t));

    config.src_addr = src_addr;
    config.dst_addr = dst_addr;
    config.ctlx_cfg = ctlx_cfg;

    return DMA_SubmitChannelTransfer(handle, &config);
}

dma_status_t DMA_StartTransfer(dma_handle_t *handle)
{
    assert(handle != NULL);
    assert(handle->base != NULL);
    assert(handle->channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS);

    /* Проверка базового адреса. */
    uint32_t instance = DMA_GetInstance(handle->base);
    if (instance > 1) {
        return DMA_Status_NoBase;
    }
    uint32_t channel = handle->channel;

    if (DMA_ChannelIsActive(handle->base, channel)) {
        return DMA_Status_Busy;
    }

    dma_channel_regs *ch_desc = DMA_GetChannelDescriptor(handle->base,
            handle->channel);

    /* Проверка на одноблочную передачу. */
    if (ch_desc->LLP_LO == 0) {
        /* Выключение режима цепочки. */
        SET_VAL_MSK(ch_desc->CTL_LO, DMA_CTL0_LO_LLP_DST_EN_Msk,
            DMA_CTL0_LO_LLP_DST_EN_Pos, false);
        SET_VAL_MSK(ch_desc->CTL_LO, DMA_CTL0_LO_LLP_SRC_EN_Msk,
            DMA_CTL0_LO_LLP_SRC_EN_Pos, false);
    } else {
        /* Включение режима цепочки. */
        SET_VAL_MSK(ch_desc->CTL_LO, DMA_CTL0_LO_LLP_DST_EN_Msk,
            DMA_CTL0_LO_LLP_DST_EN_Pos, true);
        SET_VAL_MSK(ch_desc->CTL_LO, DMA_CTL0_LO_LLP_SRC_EN_Msk,
            DMA_CTL0_LO_LLP_SRC_EN_Pos, true);
    }
    /* Включение канала передачи. */
    DMA_EnableChannel(handle->base, handle->channel);

    return DMA_Status_Success;
}

void DMA_AbortTransfer(dma_handle_t *handle)
{
    assert(handle != NULL);

    dma_channel_regs *reg_addr = DMA_GetChannelDescriptor(handle->base,
            handle->channel);

    while (DMA_ChannelIsActive(handle->base, handle->channel)) {
        if (GET_VAL_MSK(reg_addr->CFG_LO, DMA_CFG0_LO_FIFO_EMPTY_Msk,
                DMA_CFG0_LO_FIFO_EMPTY_Pos)) {
            SET_VAL_MSK(reg_addr->CFG_LO, DMA_CFG0_LO_CH_SUSP_Msk,
                DMA_CFG0_LO_CH_SUSP_Pos, 1);
        }
        DMA_DisableChannel(handle->base, handle->channel);
    }
}

void DMA_SetupDescriptor(dma_descriptor_t *desc, uint64_t ctlx_cfg,
    void *src_addr, void *dst_addr, void *next_desc)
{
    assert(desc != NULL);

    memset(desc, 0, sizeof(dma_descriptor_t));

    desc->SAR = (uint32_t) src_addr;
    desc->DAR = (uint32_t) dst_addr;
    desc->CTL_LO = (uint32_t) ctlx_cfg;
    desc->CTL_HI =
        (uint32_t)(ctlx_cfg >> DMA_CHANNEL_CTL_BLOCKSIZE_SHIFT);
    desc->LLP = (uint32_t) next_desc;
}

void DMA_ScatterGatherEnable(struct dma_channel_ctl_cfg *ctlxcfg,
    bool scatter_en, bool gather_en)
{
    assert(ctlxcfg != NULL);

    ctlxcfg->scatter_en = scatter_en;
    ctlxcfg->gather_en = gather_en;
}

void DMA_SetupSrcGather(dma_handle_t *handle,
    uint32_t count, uint32_t interval)
{
    assert(handle != NULL);
    assert(handle->base != NULL &&
        handle->channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS);

    dma_channel_regs *ch_desc = DMA_GetChannelDescriptor(handle->base,
            handle->channel);

    SET_VAL_MSK(ch_desc->SGR_LO, DMA_SGR0_LO_SGC_Msk,
        DMA_SGR0_LO_SGC_Pos, count);
    SET_VAL_MSK(ch_desc->SGR_LO, DMA_SGR0_LO_SGI_Msk,
        DMA_SGR0_LO_SGI_Pos, interval);
}

void DMA_SetupDstScatter(dma_handle_t *handle,
    uint32_t count, uint32_t interval)
{
    assert(handle != NULL);
    assert(handle->base != NULL &&
        handle->channel < (uint32_t) HAL_FEATURE_DMA_NUMBER_OF_CHANNELS);

    dma_channel_regs *ch_desc = DMA_GetChannelDescriptor(handle->base,
            handle->channel);

    SET_VAL_MSK(ch_desc->DSR_LO, DMA_DSR0_LO_DSC_Msk,
        DMA_DSR0_LO_DSC_Pos, count);
    SET_VAL_MSK(ch_desc->DSR_LO, DMA_DSR0_LO_DSI_Msk,
        DMA_DSR0_LO_DSI_Pos, interval);
}

uint32_t DMA_GetDescriptorCount(uint32_t size_bytes, uint8_t transfer_width)
{
    if (size_bytes == 0) {
        return size_bytes;
    }

    if (transfer_width > DMA_Transfer256BitWidth) {
        return 0;
    }

    uint32_t temp = (size_bytes >> transfer_width);
    uint32_t rest = ((temp % DMA_AHB_MAX_BLOCK_SIZE != 0) ? 1 : 0);

    return (temp / DMA_AHB_MAX_BLOCK_SIZE + rest);
}

void DMA_InitMultiblockDescriptor(dma_descriptor_t *desc,
    dma_multiblock_config_t *config)
{
    if (config->data_size == 0U) {
        return;
    }

    uint32_t size = 0;
    uint64_t xfercfg = 0;

    uint32_t src_addr_i = (uint32_t) config->src_addr;
    uint32_t dst_addr_i = (uint32_t) config->dst_addr;

    int32_t src_block_increment = 0;
    int32_t dst_block_increment = 0;

    switch (config->src_incr) {
        case DMA_Incr:
            src_block_increment = DMA_AHB_MAX_BLOCK_SIZE;
            break;
        case DMA_Decr:
            src_block_increment = -DMA_AHB_MAX_BLOCK_SIZE;
            break;
        case DMA_NoChange:
            src_block_increment = 0;
            break;
        default:
            break;
    }

    switch (config->dst_incr) {
        case DMA_Incr:
            dst_block_increment = DMA_AHB_MAX_BLOCK_SIZE;
            break;
        case DMA_Decr:
            dst_block_increment = -DMA_AHB_MAX_BLOCK_SIZE;
            break;
        case DMA_NoChange:
            dst_block_increment = 0;
            break;
        default:
            break;
    }

    for (uint32_t i = 0; i < config->count - 1; i++) {
        size = (config->data_size > DMA_AHB_MAX_BLOCK_SIZE) ?
            DMA_AHB_MAX_BLOCK_SIZE : config->data_size;
        xfercfg = DMA_CHANNEL_CTL(size,
                true, true,
                config->transfer_type,
                config->scatter_en, config->gather_en,
                config->src_burst_size, config->dst_burst_size,
                config->src_incr, config->dst_incr,
                config->src_data_width, config->dst_data_width,
                config->int_en
            );
        config->data_size -= DMA_AHB_MAX_BLOCK_SIZE;
        DMA_SetupDescriptor(&desc[i], xfercfg, (void *) src_addr_i,
            (void *) dst_addr_i, &desc[i + 1]);
        switch (config->src_data_width) {
            case DMA_Transfer8BitWidth:
                src_addr_i += src_block_increment;
                break;
            case DMA_Transfer16BitWidth:
                src_addr_i += 2 * src_block_increment;
                break;
            case DMA_Transfer32BitWidth:
                src_addr_i += 4 * src_block_increment;
                break;
            case DMA_Transfer64BitWidth:
                src_addr_i += 8 * src_block_increment;
                break;
            case DMA_Transfer128BitWidth:
                src_addr_i += 16 * src_block_increment;
                break;
            case DMA_Transfer256BitWidth:
                src_addr_i += 32 * src_block_increment;
                break;
            default:
                break;
        }
        switch (config->dst_data_width) {
            case DMA_Transfer8BitWidth:
                dst_addr_i += dst_block_increment;
                break;
            case DMA_Transfer16BitWidth:
                dst_addr_i += 2 * dst_block_increment;
                break;
            case DMA_Transfer32BitWidth:
                dst_addr_i += 4 * dst_block_increment;
                break;
            case DMA_Transfer64BitWidth:
                dst_addr_i += 8 * dst_block_increment;
                break;
            case DMA_Transfer128BitWidth:
                dst_addr_i += 16 * dst_block_increment;
                break;
            case DMA_Transfer256BitWidth:
                dst_addr_i += 32 * dst_block_increment;
            default:
                break;
        }
    }
    xfercfg = DMA_CHANNEL_CTL(config->data_size,
            true, true,
            config->transfer_type,
            config->scatter_en, config->gather_en,
            config->src_burst_size, config->dst_burst_size,
            config->src_incr, config->dst_incr,
            config->src_data_width, config->src_data_width,
            config->int_en
        );
    DMA_SetupDescriptor(&desc[config->count - 1], xfercfg, (void *) src_addr_i,
        (void *) dst_addr_i, NULL);
}

void DMA_ChannelIRQHandle(dma_handle_t *handle)
{
    DMA_Type *base = handle->base;
    uint32_t index = handle->channel;

    uint32_t start_channel = 0;

    if (DMA_GetInstance(base) == 1) {
        start_channel = 8;
    }

    handle = s_DMAHandle[index + start_channel];

    if ((handle->base->STATUSERR_LO >> index) & 0x1) {
        handle->base->CLEARERR_LO = (1 << index);
        if (handle->callback != NULL) {
            (handle->callback)(handle, handle->user_data, DMA_IntError);
        }
    }
    if ((handle->base->STATUSSRCTRAN_LO >> index) & 0x1) {
        handle->base->CLEARSRCTRAN_LO = (1 << index);
        if (handle->callback != NULL) {
            (handle->callback)(handle, handle->user_data, DMA_IntSrcTran);
        }
    }
    if ((handle->base->STATUSDSTTRAN_LO >> index) & 0x1) {
        handle->base->CLEARDSTTRAN_LO = (1 << index);
        if (handle->callback != NULL) {
            (handle->callback)(handle, handle->user_data, DMA_IntDstTran);
        }
    }
    if ((handle->base->STATUSBLOCK_LO >> index) & 0x1) {
        handle->base->CLEARBLOCK_LO = (1 << index);
        if (handle->callback != NULL) {
            (handle->callback)(handle, handle->user_data, DMA_IntBlock);
        }
    }
    if ((handle->base->STATUSTFR_LO >> index) & 0x1) {
        handle->base->CLEARTFR_LO = (1 << index);
        if (handle->callback != NULL) {
            (handle->callback)(handle, handle->user_data, DMA_IntTfr);
        }
    }
}

/*!
 * @}
 */
