/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup dualtimer_driver Драйвер модуля DTIM
 *
 * @brief Драйвер сдвоенного таймера
 *
 * Драйвер модуля сдвоенного таймера управляет сдвоенным таймером DTIM.
 */

/*!
 * @addtogroup dualtimer_driver
 * @{
 */

/*!
 * @file hal_dualtimer.h
 *
 * @brief Интерфейс драйвера сдвоенного таймера
 */

#ifndef HAL_DUALTIMER_H
#define HAL_DUALTIMER_H

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#include "hal_common.h"

#define DUALTIMER_NUMBER_OF_DUALTIMERS 1            /*!< Количество сдвоенных таймеров */
#define DUALTIMER_MAX_INDEX            1            /*!< Максимальный индекс таймера */

/*!
 * @brief Статусы драйвера сдвоенного таймера
 */
enum dualtimer_status {
    DUALTIMER_Status_Ok              = 0, /*!< Нет ошибок */
    DUALTIMER_Status_InvalidArgument = 1, /*!< Недопустимый аргумент */
    DUALTIMER_Status_TimerBusy       = 2, /*!< Таймер уже занят */
    DUALTIMER_Status_BadConfigure    = 3, /*!< Недопустимая конфигурация */
};

/*!
 * @brief Разрешение работы таймера
 */
enum dualtimer_work_enable {
    DUALTIMER_Disable = 0, /*!< Запрет работы таймера */
    DUALTIMER_Enable  = 1, /*!< Разрешение работы таймера */
};

/*!
 * @brief Режим работы таймера
 */
enum dualtimer_mode {
    DUALTIMER_FreeRunning = 0, /*!< Свободный счет (с переполнением) */
    DUALTIMER_Periodic    = 1, /*!< Счет с заданным периодом */
};

/*!
 * @brief Управление прерываниями
 */
enum dualtimer_interrupt_control {
    DUALTIMER_InterruptDisable = 0, /*!< Запрещение прерывания */
    DUALTIMER_InterruptEnable  = 1, /*!< Разрешение прерывания */
};

/*!
 * @brief Предделители частоты
 */
enum dualtimer_prescale {
    DUALTIMER_Prescale1   = 0, /*!< На 1 */
    DUALTIMER_Prescale16  = 1, /*!< На 16 */
    DUALTIMER_Prescale256 = 2, /*!< На 256 */
};

/*!
 * @brief Размер счетчика
 */
enum dualtimer_timer_size {
    DUALTIMER_TimerSize16 = 0, /*!< 16-битный */
    DUALTIMER_TimerSize32 = 1, /*!< 32-битный */
};

/*!
 * @brief Количество запусков
 */
enum dualtimer_number_of_repetitions {
    DUALTIMER_WrappingMode = 0, /*!< Многократный автоматический запуск */
    DUALTIMER_OneShot      = 1, /*!< Одиночный запуск */
};

/*!
 * @brief  Конфигурация аппаратной части сдвоенного таймера
 */
struct dualtimer_hardware_config {
    uint32_t                             load;         /*!< Стартовое значение счетчика */
    uint32_t                             bg_load;      /*!< Тип работы */
    enum dualtimer_work_enable           enable;       /*!< Разрешение работы */
    enum dualtimer_mode                  mode;         /*!< Режим работы */
    enum dualtimer_interrupt_control     int_ctrl;     /*!< Управление прерыванием */
    enum dualtimer_prescale              prescale;     /*!< Делитель частоты */
    enum dualtimer_timer_size            size;         /*!< Разрядность счетчика */
    enum dualtimer_number_of_repetitions cyclicity;    /*!< Количество повторений запусков */
};

/*!
 * @name Интерфейс драйвера
 * @{
 */
 
/*!
 * @brief Создание конфигурации по умолчанию
 * 
 * Создание конфигурации по умолчанию заполняет структуру такими значениями,
 * которые находятся в регистрах после сброса.
 * 
 * @note Если применить эти значения, вызвав функцию DUALTIMER_Init, 
 * то сразу возникнет прерывание, так как значение регистра LOAD = 0 и 
 * прерывания разрешены.
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 */
enum dualtimer_status DUALTIMER_GetDefaultConfig(
    struct dualtimer_hardware_config *config);

/*!
 * @brief Инициализация сдвоенного таймера
 *
 * @param base     Сдвоенный таймер
 * @param index    Индекс таймера в сдвоенном таймере
 * @param config   Конфигурация таймера в сдвоенном таймере
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 * @retval #DUALTIMER_Status_TimerBusy
 */
enum dualtimer_status DUALTIMER_Init(DTIM_Type *base, uint32_t index,
    struct dualtimer_hardware_config config);

/*!
 * @brief Деинициализация сдвоенного таймера
 *
 * Останов сдвоенного таймера, ружим таймера становится как после сброса
 *
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 */
enum dualtimer_status DUALTIMER_Deinit(DTIM_Type *base, uint32_t index);

/*!
 * @brief Запуск сдвоенного таймера
 *
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 */
enum dualtimer_status DUALTIMER_Run(DTIM_Type *base, uint32_t index);

/*!
 * @brief Останов сдвоенного таймера
 *
 * Останов сдвоенного таймера режимы таймера остаются прежними
 *
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 */
enum dualtimer_status DUALTIMER_Stop(DTIM_Type *base, uint32_t index);

/*!
 * @brief Получение немаскированного статуса сдвоенного таймера
 *
 * Получение немаскированного статуса сдвоенного таймера.
 * Корректность выполнения функции можно проверить вызовом
 *
 * @ref DUALTIMER_GetAPIStatus
 *
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 *
 * @return Статус
 */
uint32_t DUALTIMER_GetRawStatus(DTIM_Type *base, uint32_t index);

/*!
 * @brief Получение маскированного статуса сдвоенного таймера
 *
 * Получение маскированного статуса сдвоенного таймера
 * Корректность выполнения функции можно проверить вызовом
 * @ref DUALTIMER_GetAPIStatus
 *
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 *
 * @return Статус
 */
uint32_t DUALTIMER_GetStatus(DTIM_Type *base, uint32_t index);

/*!
 * @brief Получение количества тиков
 *
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 *
 * @retval Количество подсчитанных тактов
 */
uint32_t DUALTIMER_GetTick(DTIM_Type* base, uint32_t index);

/*!
 * @brief Получение результата последнего выполнения функции
 *
 * Получает ошибки выполнения функций, у которых тип возвращаемого результата
 * отличен от enum dualtimer_status
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 * @retval #DUALTIMER_Status_TimerBusy
 * @retval #DUALTIMER_Status_BadConfigure
 */
enum dualtimer_status DUALTIMER_GetAPIStatus();

/*!
 * @brief Немедленная перезапись значения таймера
 *
 * Перезаписывает значение таймера. В режиме @ref DUALTIMER_OneShot 
 * перезапускает таймер, если тот был остановлен.
 *
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 * @param value Загружаемое значение
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 */
enum dualtimer_status DUALTIMER_Reload(DTIM_Type* base,
    uint32_t index, uint32_t value);

/*!
 * @brief Сброс прерывания от таймера
 *
 * Сброс прерывания от таймера, не влияет на NVIC
 * 
 * @param base  Таймер
 * @param index Индекс таймера в сдвоенном таймере
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 */
enum dualtimer_status DUALTIMER_IrqClr(DTIM_Type* base, uint32_t index);

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_DUALTIMER_H */

/*!
 * @}
 */
