/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup flash_driver Драйвер модуля FLASH.
 *
 * @brief Драйвер накристалльной FLASH-памяти.
 *
 * @note Драйвер накристалльной FLASH-памяти позволяет стирать и записывать данные в
 * память.
 *
 */

/*!
 * @addtogroup flash_driver
 * @{
 */

/*!
 * @file hal_flash.h
 *
 * @brief Интерфейс драйвера модуля FLASH.
 */

#ifndef HAL_FLASH_H
#define HAL_FLASH_H

#include "hal_common.h"

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#define FCTR_IRQ_STS_SET_RESULT_FLAGS                      \
    (FCTR_IRQ_STATUS_SET_READ_OVERFLOW_IRQ_STS_SET_Msk     \
        | FCTR_IRQ_STATUS_SET_CMD_REJECT_IRQ_STS_SET_Msk   \
        | FCTR_IRQ_STATUS_SET_CMD_FAIL_IRQ_STS_SET_Msk     \
        | FCTR_IRQ_STATUS_SET_CMD_SUCCESS_IRQ_STS_SET_Msk)   /*!< Объединение всех статусов, кроме ACCEPT. */

#define FCTR_IRQ_STS_CLR_SUCCESS_FLAGS                     \
    (FCTR_IRQ_STATUS_CLR_CMD_SUCCESS_IRQ_STS_CLR_Msk       \
        | FCTR_IRQ_STATUS_CLR_CMD_ACCEPT_IRQ_STS_CLR_Msk)    /*!< Объединение статусов SUCCESS и ACCEPT. */

/*!
 * @name Команды для накристалльной FLASH-памяти.
 * @{
 */
#define FCTR_CMD_READ       (0x1) /*!< Чтение слова. */
#define FCTR_CMD_WRITE      (0x2) /*!< Запись слова. */
#define FCTR_CMD_ROW_WRITE  (0x3) /*!< Быстрая запись слова. */
#define FCTR_CMD_ERASE      (0x4) /*!< Стирание страницы. */
#define FCTR_CMD_MASS_ERASE (0x7) /*!< Стирание всего региона FLASH-памяти. */
/*!
 * @}
 */

/*!
 * @brief Статусы драйвера модуля FLASH.
 */
enum flash_status {
    FLASH_Status_Ok                     = 0, /*!< Нет ошибок. */
    FLASH_Status_InvalidArgument        = 1, /*!< Недопустимый параметр. */
    FLASH_Status_CheckError             = 2, /*!< Получена ошибка от оборудования. */
    FLASH_Status_VerifyError            = 3, /*!< Ошибка верификации данных. */
    FLASH_Status_ConfigureError         = 4, /*!< Недопустимая конфигурация или ошибка в описании оборудования. */
    FLASH_Status_AddressAlignmentError  = 5, /*!< Адрес не выровнен по странице/слову. */
    FLASH_Status_AddressOutOfRange      = 6, /*!< Адрес вне допустимого диапазона. */
};

/*!
 * @brief Регионы накристалльной FLASH-памяти.
 */
enum flash_region {
    FLASH_MainRegion    = FLASH_BASE_ADDR,  /*!< Основной раздел. */
    FLASH_SystemRegion  = FLASH_BASE_ADDR
        + FLASH_SYSTEM_REGION_OFFSET,       /*!< Системный раздел. */
};

/*!
 * @name API HAL для драйвера модуля FLASH.
 * @{
 */

/*!
 * @brief Проверка корректности адреса и количества
 * записываемых/стираемых байтов.
 *
 * @param address Стартовый адрес записи/стирания.
 * @param length  Количество записываемых/стираемых байтов.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_AddressAlignmentError
 * @retval #FLASH_Status_AddressOutOfRange
 */

#define FLASH_TEST_ADDRESSES(address, length) \
    ( \
        /* Проверка на выравнивание по слову. */\
        ((address & FLASH_WORD_ALIGN_Msk) > 0x0U) \
        || ( \
            /* Проверка на выравнивание по странице. */\
            ((length & FLASH_PAGE_ALIGN_Msk) == 0x0U) \
            && ( \
                ( \
                    (((address + length) & \
                            FLASH_PAGE_ALIGN_Msk) > 0x0U) \
                    && ((address & FLASH_PAGE_ALIGN_Msk) == 0x0U) \
                ) \
                || ((address & FLASH_PAGE_ALIGN_Msk) > 0x0U) \
            ) \
        ) \
    ) ? FLASH_Status_AddressAlignmentError : \
    ( \
        ( \
            /* Проверка попадания в основной раздел. */\
            (address >= FLASH_BASE_ADDR) \
            && ((address + length) \
                <= FLASH_BASE_ADDR + FLASH_MAIN_REGION_SIZE_IN_BYTE) \
        ) ? FLASH_Status_Ok : \
        ( \
            ( \
                /* Проверка попадания в системный раздел. */\
                (address >= FLASH_BASE_ADDR \
                    + FLASH_SYSTEM_REGION_OFFSET) \
                && ((address + length) <= FLASH_BASE_ADDR \
                    + FLASH_SYSTEM_REGION_OFFSET \
                    + FLASH_SYSTEM_REGION_SIZE_IN_BYTE) \
            ) ? FLASH_Status_Ok : FLASH_Status_AddressOutOfRange \
        ) \
    )

/*!
 * @brief Инициализация накристалльной FLASH-памяти.
 *
 * @param base Базовый адрес регистров FLASH-памяти.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_ConfigureError
 */
enum flash_status FLASH_Init(FCTR_Type *base);

/*!
 * @brief Запись 32-битного слова во встроенную FLASH-память.
 *
 * @note К аргументам функции применяются следующие ограничения:
 * - адрес должен быть выровнен по границе 32-битного слова;
 * - в значении адреса учитываются только младшие биты [20:0] (младшие 21 бит).
 *
 * @param base  Базовый адрес регистров FLASH-памяти.
 * @param addr  Адрес для записи.
 * @param data  Записываемое слово.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_CheckError
 * @retval #FLASH_Status_AddressAlignmentError
 * @retval #FLASH_Status_AddressOutOfRange
 */
enum flash_status FLASH_WriteWord(FCTR_Type *base, uint32_t addr,
    uint32_t data);

/*!
 * @brief Запись данных во встроенную FLASH-память.
 *
 * @note К аргументам функции применяются следующие ограничения:
 *  - в значении адреса учитываются только младшие биты [20:0] (младшие 21 бит);
 *  - записываемые данные должны быть выровнены по границе 32-битного
 * слова;
 *  - количество записываемых байтов должно быть кратно размеру
 * страницы.
 *
 * @param base      Базовый адрес регистров FLASH-памяти.
 * @param addr      Начальный адрес записи.
 * @param src       Записываемые данные.
 * @param length    Количество записываемых байтов.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_ConfigureError
 * @retval #FLASH_Status_AddressAlignmentError
 * @retval #FLASH_Status_AddressOutOfRange
 */
enum flash_status FLASH_Program(FCTR_Type *base, uint32_t *addr, uint32_t *src,
    uint32_t length);

/*!
 * @brief Проверка корректности данных, записанных во внутреннюю FLASH-память.
 *
 * @note К аргументам функции применяются следующие ограничения:
 *  - начальный адрес чтения должен быть выровнен на размер страницы;
 *  - в значении адреса учитываются только младшие биты [20:0] (младшие 21 бит);
 *  - количество проверяемых байтов должно быть кратно размеру страницы;
 *  - адрес эталонных данных должен быть выровнен по границе 32-битного
 * слова.
 *
 * @param base              Базовый адрес регистров FLASH-памяти.
 * @param addr              Начальный адрес проверяемых данных.
 * @param length            Количество проверяемых байтов.
 * @param expected_data     Эталонные данные.
 * @param failed_address    Адрес первых несовпадающих данных.
 * @param failed_data       Первые несовпадающие данные.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_InvalidArgument
 * @retval #FLASH_Status_ConfigureError
 * @retval #FLASH_Status_AddressAlignmentError
 * @retval #FLASH_Status_AddressOutOfRange
 */
enum flash_status FLASH_VerifyProgram(uint32_t *addr, uint32_t length,
    uint32_t *expected_data, uint32_t *failed_address, uint32_t *failed_data);

/*!
 * @brief Стирание сектора накристалльной FLASH-памяти.
 * @note К аргументам функции применяются следующие ограничения:
 *  - адрес должен быть выровнен по границе 8 KB страницы;
 *  - в значении адреса учитываются только младшие биты [20:0] (младшие 21 бит);
 *  - количество стираемых байтов должно быть кратно размеру страницы.
 *
 * @param base      Базовый адрес регистров FLASH-памяти.
 * @param addr      Начальный адрес стирания.
 * @param length    Количество стираемых байтов.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_InvalidArgument
 * @retval #FLASH_Status_CheckError
 * @retval #FLASH_Status_AddressAlignmentError
 * @retval #FLASH_Status_AddressOutOfRange
 */
enum flash_status FLASH_Erase(FCTR_Type *base, uint32_t *addr,
    uint32_t length);

/*!
 * @brief Стирание раздела накристалльной FLASH-памяти.
 *
 * @param base      Базовый адрес регистров FLASH-памяти.
 * @param region    Раздел памяти.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_CheckError
 */
enum flash_status FLASH_MassErase(FCTR_Type *base, enum flash_region region);

/*!
 * @brief Проверка корректности стирания данных накристалльной FLASH-памяти.
 *
 * @note К аргументам функции применяются следующие ограничения:
 *  - адрес должен быть выровнен на размер страницы;
 *  - в значении адреса учитываются только младшие биты [20:0] (младшие 21 бит);
 *  - количество байтов должно быть кратно размеру страницы.
 *
 * @param base      Базовый адрес регистров FLASH-памяти.
 * @param addr      Начальный адрес проверямой памяти.
 * @param length    Количество проверямых байтов.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_ConfigureError
 * @retval #FLASH_Status_VerifyError
 * @retval #FLASH_Status_AddressAlignmentError
 * @retval #FLASH_Status_AddressOutOfRange
 */
enum flash_status FLASH_VerifyErase(uint32_t *addr, uint32_t length);

/*!
 * @brief Чтение данных из накристалльной FLASH-памяти.
 *
 * @note К аргументам функции применяются следующие ограничения:
 *  - адрес чтения должен быть выровнен на размер страницы;
 *  - в значении адреса учитываются только младшие биты [20:0] (младшие 21 бит);
 *  - aдрес буфера под данные должен быть выровнен по границе 32-битного
 * слова;
 *  - количество cчитываемых байтов должно быть кратно размеру страницы.
 *
 * @param addr      Начальный адрес чтения.
 * @param dest      Считываемые данные.
 * @param length    Количество считываемых байтов.
 *
 * @retval #FLASH_Status_Ok
 * @retval #FLASH_Status_InvalidArgument
 * @retval #FLASH_Status_CheckError
 * @retval #FLASH_Status_AddressAlignmentError
 * @retval #FLASH_Status_AddressOutOfRange
 * @retval #FLASH_Status_AddressAlignmentError
 */
enum flash_status FLASH_Read(uint32_t *addr, uint32_t *dest, uint32_t length);

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_FLASH_H */

/*!
 * @}
 */
