/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup i2c_driver Драйвер модуля I2C
 *
 * @brief Драйвер последовательной асимметричной шины для связи между
 *        интегральными схемами внутри приборов.
 *
 * Драйвер поддерживает обмен по интерфейсу I2C по прерыванию и в режиме опроса.
 *
 * Используя специальные функции из @ref hal_i2c_dma.h, передача данных из
 * буфера / в буфер UART будет происходить с помощью DMA.
 */

/*!
 * @addtogroup i2c_driver
 * @{
 */

/*!
 * @file hal_i2c.h
 *
 * @brief Интерфейс драйвера модуля I2C.
 */

#ifndef HAL_I2C_H
#define HAL_I2C_H

#include "hal_common.h"

#define I2C_CFG_MASK 0x1f /*!< TODO */

/*!
 * @brief Версия драйвера.
 */
#define HAL_I2C_DRIVER_VERSION (MAKE_VERSION(1, 0, 0))

/*!
 * @brief Количество повторов при ожидании флага.
 */
#ifndef I2C_RETRY_TIMES
#define I2C_RETRY_TIMES 0U /*!< 0 - ожидание до получения значения */
#endif

/*!
 * @brief Игнорировать ли сигнал NACK последнего байта во время передачи Master.
 */
#ifndef I2C_MASTER_TRANSMIT_IGNORE_LAST_NACK
#define I2C_MASTER_TRANSMIT_IGNORE_LAST_NACK 0U /*!< 1 - Master игнорирует NACK последнего байта и считает передачу успешной */
#endif

/* TODO Master: Определения битов MSTCODE в регистре состояния I2C STAT */
#define I2C_STAT_MSTCODE_IDLE    (0U) /*!< Master код состояния Idle */
#define I2C_STAT_MSTCODE_RXREADY (1U) /*!< Master код состояния Receive Ready */
#define I2C_STAT_MSTCODE_TXREADY (2U) /*!< Master код состояния Transmit Ready */
#define I2C_STAT_MSTCODE_NACKADR (3U) /*!< Master код состояния NACK от Slave при отправке адреса */
#define I2C_STAT_MSTCODE_NACKDAT (4U) /*!< Master код состояния NACK от Slave при отправке данных */

/* TODO Slave: Определения битов SLVSTATE в регистре состояния I2C STAT */
#define I2C_STAT_SLVST_ADDR (0) /*!< TODO */
#define I2C_STAT_SLVST_RX   (1) /*!< TODO */
#define I2C_STAT_SLVST_TX   (2) /*!< TODO */

/*!
 * @brief Коды возврата функций модуля I2C.
 */
typedef enum {
    I2C_Status_Ok                   =   0U,  /*!< Успешное завершение */
    I2C_Status_Busy                 =   1U,  /*!< Master уже выполняет передачу */
    I2C_Status_Idle                 =   2U,  /*!< Slave-драйвер в состоянии ожидания */
    I2C_Status_Nack                 =   3U,  /*!< Slave-устройство отправило NACK в ответ на байт */
    I2C_Status_InvalidParameter     =   4U,  /*!< Невозможно продолжить из-за недопустимого параметра */
    I2C_Status_BreakTransfer        =   5U,  /*!< Обрыв обмена */
    I2C_Status_ArbitrationLost      =   6U,  /*!< Потеря арбитража */
    I2C_Status_NoTransferInProgress =   7U,  /*!< Нет активной передачи */
    I2C_Status_DmaRequestFail       =   8U,  /*!< DMA запрос не выполнен */

    I2C_Status_Timeout              =  11U,  /*!< Тайм-аут при ожидании установки бита статуса в Master/Slave для продолжения передачи */
    I2C_Status_AddrNack             =  12U,  /*!< NACK получен для адреса */
    I2C_Status_UserError            =  13U,  /*!< Некорректная работа с модулем I2C */
    I2C_Status_SetStartError        =  14U,  /*!< Master получил подтверждение на [Start] условие */
    I2C_Status_HwError              =  15U,  /*!< Аппаратная ошибка */
    I2C_Status_HsCodeError          =  16U,  /*!< Master в HS режиме получил подтверждение на HS code */

    I2C_Status_UnexpectedState      = 100U,  /*!< Неожиданное состояние   */
    I2C_Status_UnexpectedState1     = 101U,  /*!< Неожиданное состояние 1 */
    I2C_Status_UnexpectedState2     = 102U,  /*!< Неожиданное состояние 2 */
    I2C_Status_UnexpectedState3     = 103U,  /*!< Неожиданное состояние 3 */
    I2C_Status_UnexpectedState4     = 104U,  /*!< Неожиданное состояние 4 */
    I2C_Status_UnexpectedState5     = 105U,  /*!< Неожиданное состояние 5 */
    I2C_Status_UnexpectedState6     = 106U,  /*!< Неожиданное состояние 6 */
    I2C_Status_UnexpectedState7     = 107U,  /*!< Неожиданное состояние 7 */
    I2C_Status_UnexpectedState8     = 108U,  /*!< Неожиданное состояние 8 */
    I2C_Status_UnexpectedState9     = 109U,  /*!< Неожиданное состояние 9 */

    I2C_Status_UnDef                = 255U,  /*!< Статус не определен */
} i2c_status_t;

/*!
 * @brief Статусы модуля I2C (read-only регистр IC_STATUS).
 */
enum i2c_status_flags {
    I2C_Stat_Active          = I2C_IC_STATUS_ACTIVITY_Msk,     /*!< Флаг активности шины */
    I2C_Stat_TxFifo_NotFull  = I2C_IC_STATUS_TFNF_Msk,         /*!< TxFifo не полон */
    I2C_Stat_TxFifo_Empty    = I2C_IC_STATUS_TFE_Msk,          /*!< TxFifo пуст */
    I2C_Stat_RxFifo_NotEmpty = I2C_IC_STATUS_RFNE_Msk,         /*!< RxFifo не пуст */
    I2C_Stat_RxFifo_Full     = I2C_IC_STATUS_RFF_Msk,          /*!< RxFifo полон */
    I2C_Stat_Master_Active   = I2C_IC_STATUS_MST_ACTIVITY_Msk, /*!< Статус активности состояния Master */
    I2C_Stat_Slave_Active    = I2C_IC_STATUS_SLV_ACTIVITY_Msk  /*!< Статус активности состояния Slave */
};

/*!
 * @brief Причины обрыва передачи (регистр IC_TX_ABRT_SOURCE).
 */
enum i2c_abort_flags {
    I2C_Abort_7B_Addr_Nack       = I2C_IC_TX_ABRT_SOURCE_ABRT_7B_ADDR_NOACK_Msk,   /*!< Master Tx c 7 битной адресацией, NACK от Slave после отправления адреса */
    I2C_Abort_10B_Addr1_Nack     = I2C_IC_TX_ABRT_SOURCE_ABRT_10ADDR1_NOACK_Msk,   /*!< Master Tx c 10 битной адресацией, NACK от Slave после отправления 1го адреса */
    I2C_Abort_10B_Addr2_Nack     = I2C_IC_TX_ABRT_SOURCE_ABRT_10ADDR2_NOACK_Msk,   /*!< Master Tx c 10 битной адресацией, NACK от Slave после отправления 1го адреса */
    I2C_Abort_TxData_Nack        = I2C_IC_TX_ABRT_SOURCE_ABRT_TXDATA_NOACK_Msk,    /*!< Master Tx не получил ACK от Slave после отправки байта данных */
    I2C_Abort_GenCall_Nack       = I2C_IC_TX_ABRT_SOURCE_ABRT_GCALL_NOACK_Msk,     /*!< Master Tx отправил General Call, и ни один Slave не ответил */
    I2C_Abort_GenCall_Read       = I2C_IC_TX_ABRT_SOURCE_ABRT_GCALL_READ_Msk,      /*!< Master Rx попытка чтения с адреса General Call */
    I2C_Abort_HsCode_Ack         = I2C_IC_TX_ABRT_SOURCE_ABRT_HS_ACKDET_Msk,       /*!< Master в HS режиме получил подтверждение на HS code */
    I2C_Abort_StartByte_Ack      = I2C_IC_TX_ABRT_SOURCE_ABRT_SBYTE_ACKDET_Msk,    /*!< Master получил подтверждение на [Start] условие */
    I2C_Abort_HS_RStart_Dis      = I2C_IC_TX_ABRT_SOURCE_ABRT_HS_NORSTRT_Msk,      /*!< Master в HS режиме пытается отправить [RStart] условие, но возможность отключена */
    I2C_Abort_RStart_Dis         = I2C_IC_TX_ABRT_SOURCE_ABRT_SBYTE_NORSTRT_Msk,   /*!< Master пытается отправить [RStart] условие, но возможность отключена */
    I2C_Abort_Read_RStart_Dis    = I2C_IC_TX_ABRT_SOURCE_ABRT_10B_RD_NORSTRT_Msk,  /*!< Master пытается осуществить чтение режиме 10 битной адресации, но возможность отключена */
    I2C_Abort_Master_Dis         = I2C_IC_TX_ABRT_SOURCE_ABRT_MASTER_DIS_Msk,      /*!< Попытка инициализировать Master-обмен при выключенном Master-режиме */
    I2C_Abort_Arbitr_Lost        = I2C_IC_TX_ABRT_SOURCE_ARB_LOST_Msk,             /*!< Master или Slave (если IC_TX_ABRT_SOURCE[14] == 1) - передатчик проигрывает арбитраж */
    I2C_Abort_RxFifo_NotEmpty    = I2C_IC_TX_ABRT_SOURCE_ABRT_SLVFLUSH_TXFIFO_Msk, /*!< Slave получил запрос на чтение, но в RxFifo уже есть данные */
    I2C_Abort_SlaveArbitr_Lost   = I2C_IC_TX_ABRT_SOURCE_ABRT_SLV_ARBLOST_Msk,     /*!< Slave теряет шину во время передачи данных */
    I2C_Abort_SlaveDataCmd_Error = I2C_IC_TX_ABRT_SOURCE_ABRT_SLVRD_INTX_Msk,      /*!< Slave есть запрос на передачу данных удаленному Master, но пользователь пытается произвести чтение в режиме мастера (пишет 1 в IC_DATA_CMD.CMD) */
    I2C_Abort_MasterDetect       = I2C_IC_TX_ABRT_SOURCE_ABRT_USER_ABRT_Msk,       /*!< Master определил обрыв передачи */
    I2C_Abort_Tx_FlushCnt        = I2C_IC_TX_ABRT_SOURCE_TX_FLUSH_CNT_Msk          /*!< Master/Slave-счетчик оставшихся в TxFifo данны после обрыва передачи */
};

/*!
 * @brief Флаги прерываний I2C.
 *
 * @note Предназначены для объединения по ИЛИ для формирования битовой маски для
 *       регистров:
 *       - IC_RAW_INTR_STAT
 *         - Cтатус немаскированных прерываний
 *       - IC_INTR_STAT
 *         - Регистр статуса прерываний
 *       - IC_INTR_MASK
 *         - Регистр маскирования прерываний
 *
 * Прерывания:
 * - I2C_IRQ_RdReq
 *   - Устанавливается в Slave-режиме при запросе данных удаленным Master. Slave
 *     удерживает состояние ожидания (SCL = 0), пока прерывание обрабатывается.
 *     Процессор должен ответить на это прерывание и начать выдавать данные в
 *     IC_DATA_CMD регистр.
 *     Сброс: чтение IC_CLR_RD_REQ
 *
 * - I2C_IRQ_TxAbrt
 *   - Устанавливается если модуль работает в режиме передатчика и не может
 *     произвести передачу. Когда этот бит устанавливается в 1, регистр
 *     IC_TX_ABRT_SOURCE отображает причину обрыва передачи.
 *     Сброс: чтение IC_CLR_TX_ABRT
 *
 * - I2C_IRQ_Activity
 *   - Устанавливается, если модуль проявил какую-либо активность.
 *     Сброс:
 *     - Выключение модуля I2C
 *     - Чтение IC_CLR_ACTIVITY
 *     - Чтение IC_CLR_INTR
 *     - Системный сброс
 */
enum i2c_interrupt {
    I2C_IRQ_RxUnder  = I2C_IC_INTR_MASK_M_RX_UNDER_Msk,  /*!< [0] Чтение из пустого RxFifo. Сброс: чтение IC_CLR_RX_UNDER */
    I2C_IRQ_RxOver   = I2C_IC_INTR_MASK_M_RX_OVER_Msk,   /*!< [1] Переполнение RxFifo. Сброс: чтение IC_CLR_RX_OVER */
    I2C_IRQ_RxFull   = I2C_IC_INTR_MASK_M_RX_FULL_Msk,   /*!< [2] RxFifo заполнен до уровня IC_RX_TL. Сброс: уменьшение уровня RxFifo ниже IC_RX_TL */
    I2C_IRQ_TxOver   = I2C_IC_INTR_MASK_M_TX_OVER_Msk,   /*!< [3] Попытка записать в заполненный TxFifo. Сброс: чтение IC_CLR_TX_OVER */
    I2C_IRQ_TxEmpty  = I2C_IC_INTR_MASK_M_TX_EMPTY_Msk,  /*!< [4] Опустошение TxFifo ниже уровня IC_TX_TL */
    I2C_IRQ_RdReq    = I2C_IC_INTR_MASK_M_RD_REQ_Msk,    /*!< [5] Устанавливается при запросе данных удаленным Master */
    I2C_IRQ_TxAbrt   = I2C_IC_INTR_MASK_M_TX_ABRT_Msk,   /*!< [6] Передача прервана */
    I2C_IRQ_RxDone   = I2C_IC_INTR_MASK_M_RX_DONE_Msk,   /*!< [7] В режиме Slave-передатчика устанавливается в 1, если Master не подтверждает передачу байта */
    I2C_IRQ_Activity = I2C_IC_INTR_MASK_M_ACTIVITY_Msk,  /*!< [8] Активность на шине I2C */
    I2C_IRQ_StopDet  = I2C_IC_INTR_MASK_M_STOP_DET_Msk,  /*!< [9] В Slave- или Master-режиме устанавливается, если на шине возникает состояние STOP. Сброс: чтение IC_CLR_STOP_DET */
    I2C_IRQ_StartDet = I2C_IC_INTR_MASK_M_START_DET_Msk, /*!< [10] На шине START или RESTART условия. Сброс: чтение IC_CLR_START_DET */
    I2C_IRQ_GenCall  = I2C_IC_INTR_MASK_M_GEN_CALL_Msk,  /*!< [11] Получен адрес General Call и отправлено подтверждение. Сброс: 1) чтение IC_CLR_GEN_CALL 2) выключение модуля */

    I2C_IRQ_Slave    =   I2C_IRQ_RxUnder | I2C_IRQ_RxOver  | I2C_IRQ_RxFull  |
        I2C_IRQ_TxOver |                   I2C_IRQ_RdReq   |
        I2C_IRQ_TxAbrt | I2C_IRQ_RxDone  |
        I2C_IRQ_StopDet | I2C_IRQ_StartDet | I2C_IRQ_GenCall,

    I2C_IRQ_SlaveTst =                                       I2C_IRQ_RxFull  | /*TODO*/
        I2C_IRQ_RdReq  |
        I2C_IRQ_TxAbrt | I2C_IRQ_RxDone  |
        I2C_IRQ_StopDet | I2C_IRQ_StartDet | I2C_IRQ_GenCall,

    I2C_IRQ_All      =   I2C_IRQ_RxUnder | I2C_IRQ_RxOver   | I2C_IRQ_RxFull  |
        I2C_IRQ_TxOver | I2C_IRQ_TxEmpty | I2C_IRQ_RdReq    |
        I2C_IRQ_TxAbrt | I2C_IRQ_RxDone  | I2C_IRQ_Activity |
        I2C_IRQ_StopDet | I2C_IRQ_StartDet | I2C_IRQ_GenCall
};

/*!
 * @brief Направление передачи.
 */
typedef enum {
    I2C_Write = 0U, /*!< Master передает */
    I2C_Read  = 1U  /*!< Master принимает */
} i2c_direction_t;

/*!
 * @brief Разрядность адреса.
 */
typedef enum {
    I2C_Address7Bit  = 0U, /*!< Для обмена используется 7 битная адресация */
    I2C_Address10Bit = 1U  /*!< Для обмена используется 10 битная адресация */
} i2c_addr_size_t;


/*!
 * @brief Состояния для конечного автомата, используемого при обмене.
 * FSM
 * TODO
 */
typedef enum  {
    I2C_MTS_Idle              = 0U, /*!< Состояние: бездействия */
    I2C_MTS_TransmitSubaddr   = 1U, /*!< Состояние: передача адреса */
    I2C_MTS_TransmitData      = 2U, /*!< Состояние: передача данных */
    I2C_MTS_ReceiveDataBegin  = 3U, /*!< Состояние: начало приема данных */
    I2C_MTS_ReceiveData       = 4U, /*!< Состояние: прием данных */
    I2C_MTS_ReceiveLastData   = 5U, /*!< Состояние: заверешение приема данных */
    I2C_MTS_Start             = 6U, /*!< Состояние: Start-условие */
    I2C_MTS_Stop              = 7U, /*!< Состояние: Stop-условие */
    I2C_MTS_WaitForCompletion = 8U, /*!< Состояние: Ожидание завершения */
} i2c_master_transfer_states;


/*!
 * @brief Скоростной режим работы модуля в Master-режиме.
 */
typedef enum {
    I2C_UndefinedSpeedMode = 0U, /*!< Режим не задан */
    I2C_StandardSpeedMode  = 1U, /*!< Скорость от 0 до 100 Кб/с */
    I2C_FastSpeedMode      = 2U, /*!< Скорость до 400 Кб/с */
    I2C_HighSpeedMode      = 3U, /*!< Скорость меньше либо равна 3.4 Мб/с */
} i2c_speed_mode_t;

/*!
 * @brief Структура с настройками для инициализации Master-модуля I2C.
 *
 * @note Cтруктура содержит настройки конфигурации для периферийного
 *       модуля I2C. Чтобы инициализировать структуру с значениями по
 *       умолчанию, необходимо вызвать функцию @ref I2C_MasterGetDefaultConfig
 *       и передать ей указатель на экземпляр структуры конфигурации.
 */
 typedef struct {
    bool     enable_master; /*!< Включить ли модуль при инициализации */
    uint32_t baudrate_bps;  /*!< Скорость передачи в битах в секунду */
    uint8_t  sda_setup;     /*!< Количество тактов удерживания SDA после переднего фронта SCL */
    uint16_t sda_hold;      /*!< Количество тактов удерживания SDA после заднего фронта SCL */
} i2c_master_config_t;

/*!
 * @brief Тип I2C Master дескриптор
 */
typedef struct _i2c_master_handle i2c_master_handle_t;

/*!
 * @brief Указатель на функцию обратного вызова при завершении Master-передачи.
 *
 * Используется только для неблокирующей передачи. Для задания функции
 * обратного вызова, вызывающейся для обработки событий, используется
 * @ref I2C_MasterTransferCreateHandle.
 *
 * @param base              Базовый адрес модуля I2C
 * @param handle            Дескриптор обработчика прерывания
 * @param completion_status Результат завершения операции
 * @param user_data         Данные пользователя
 */
typedef void (*i2c_master_transfer_callback_t)(I2C_Type *base,
    i2c_master_handle_t *handle, i2c_status_t completion_status,
    void *user_data);

/*!
 * @brief Флаги вариантов передачи.
 *
 * @note Предназначены для объединения по ИЛИ, чтобы сформировать битовую маску.
 */
typedef enum {
    I2C_TransferDataFlag    = 0x01U, /*!< Передача пакета данных, без установки Start-условия и Stop-условия */
    I2C_TransferStartFlag   = 0x02U, /*!< Запуск нового цикла передачи, начинающийся со Start-условия */
    I2C_TransferStopFlag    = 0x04U, /*!< Остановка цикла передачи Stop-условием */
    I2C_TransferReStartFlag = 0x08U, /*!< Отправка RStart-условия. Может быть только (!) продолжением передачи. В случае когда предыдущая передача была без I2C_TransferStopFlag и цикл предыдущей передачи не завершился (процесс передачи еще идет) */
} i2c_master_transfer_flags_t;

/*!
 * @brief Структура дескриптора для неблокирующего обмена.
 *
 * Используется для передачи параметров обмена в
 * @ref I2C_MasterTransferNonBlocking.
 *
 * Описание поля flags:
 * - I2C_TransferStartFlag | I2C_TransferStopFlag
 *   - Стандартная транзакция начинающаяся со Start и заканчивающаяся Stop
 *     условием.
 *
 * - I2C_TransferStartFlag
 *   - Начальный пакет транзакции начинающаяся со Start и подразумающий
 *     продолжение.
 *
 * - I2C_TransferDataFlag
 *   - Пакет передачи данных без Start и Stop условия.
 *
 * - I2C_TransferStopFlag
 *   - Завершающий пакет транзакции, заканчивающейся Stop условием.
 *
 * - I2C_TransferReStartFlag
 *   - Продолжение транзакции с выдачей повторного старта и подразумевающей
 *     продолжение. Используется для смены направления обмена.
 *
 * - I2C_TransferReStartFlag | I2C_TransferStopFlag
 *   - Продолжение транзакции с выдачей повторного старта и не допускающей
 *     продолжение. Используется для смены направления обмена.
 */
typedef struct {
    uint32_t        flags;           /*!< Флаги управления передачей для управления специальным поведением, см. описание */
    uint16_t        slave_address;   /*!< Slave-адрес */
    i2c_addr_size_t addr_size;       /*!< Разрядность Slave-адреса */
    i2c_direction_t direction;       /*!< Направление передачи Master -> Slave или Master <- Slave */
    uint32_t        subaddress;      /*!< Дополнительный адрес. Сначала передан MSB */
    size_t          subaddress_size; /*!< Длина дополнительного адрес для отправки в байтах. Максимальный размер 4 байта */
    void            *data;           /*!< Данные для передачи */
    size_t          data_size;       /*!< Количество байтов для передачи */
} i2c_master_transfer_t;

/*!
 * @brief Дескриптор для работы по прерыванию.
 */
struct _i2c_master_handle {
    uint8_t                        state;               /*!< Текущее состояние конечного автомата */
    uint32_t                       transfer_count;      /*!< Указывает на ход передачи */
    uint32_t                       remaining_bytes;     /*!< Количество оставшихся байтов в текущем состоянии */
    uint8_t                        *buf;                /*!< Буфера для текущего состояния */
    uint32_t                       remaining_subaddr;   /*!< Оставшийся дополнительный адрес */
    uint8_t                        subaddr_buf[4];      /*!< TODO */
    bool                           check_addr_nack;     /*!< Проверять сигнал NACK после передачи адреса */
    i2c_master_transfer_t          transfer;            /*!< Копия текущего статуса передачи */
    i2c_master_transfer_callback_t completion_callback; /*!< Функция обратного вызова */
    void                           *user_data;          /*!< Данные пользователя для передачи в функцию обратного вызова */
};

/*!
 * @brief Структура с настройками для инициализации Slave-модуля I2C.
 *
 * Эта структура содержит параметры конфигурации для Slave-устройства I2C. Чтобы
 * инициализировать ее значения по умолчанию, необходимо вызывать
 * @ref I2C_SlaveGetDefaultConfig и передать указатель на экземпляр структуры
 * конфигурации.
 */
typedef struct {
    uint16_t        address;       /*!< Slave-адрес */
    bool            ack_gen_call;  /*!< Отмечать на General Call адрес */
    i2c_addr_size_t i2c_addr_size; /*!< Разрядность адреса: 7 или 10 бит */
    bool            enable_slave;  /*!< Включить Slave-режим */
} i2c_slave_config_t;

/*!
 * @brief Cобытия, вызывающие функцию обратного вызова для неблокирующих передач.
 *
 * Эти перечисления событий используются для двух взаимосвязанных целей:
 * - Битовая маска (по ИЛИ) передается в @ref I2C_SlaveTransferNonBlocking,
 *   чтобы указать, какие события включить.
 * - При вызове функции обратного вызова, в нее передаеются событие через параметр
 *   функции.
 *
 * @note Эти перечисления предназначены для объединения по ИЛИ, чтобы
 *       сформировать битовую маску событий.
 */
typedef enum {
    I2C_SlaveEvent_MatchAddrSlave = 0x01, /*!< Получен Slave-адрес после Start или RStart условия */
    I2C_SlaveEvent_MatchAddrGen   = 0x02, /*!< Получен General-адрес */
    I2C_SlaveEvent_Transmit       = 0x08, /*!< Slave-передает: Запрошен вызов функции обратного вызова для предоставления данных на передачу */
    I2C_SlaveEvent_Receive        = 0x10, /*!< Slave-принимает: Запрошен вызов функции обратного вызова для предоставления буфера, в который будут помещены принятые данные */
    I2C_SlaveEvent_Completion     = 0x20, /*!< Текущий обмен был завершен completion_status содержит статус завершения */

    I2C_SlaveEvents_Ordinary      = I2C_SlaveEvent_Transmit          |   /*!< Битовая маска событий достаточная для большинства задач. */
        I2C_SlaveEvent_Receive           |
        I2C_SlaveEvent_Completion,

    I2C_SlaveEvents_All           = I2C_SlaveEvent_MatchAddrSlave    |   /*!< Битовая маска всех доступных событий. */
        I2C_SlaveEvent_MatchAddrGen      |
        I2C_SlaveEvent_Transmit          |
        I2C_SlaveEvent_Receive           |
        I2C_SlaveEvent_Completion,
} i2c_slave_event_transfer_t;

/*!
 * @brief Тип I2C Slave дескриптор
 */
typedef struct _i2c_slave_handle i2c_slave_handle_t;

/*!
 * @brief I2C Slave структура обмена данными
 */
typedef struct {
    i2c_slave_handle_t         *handle;           /*!< Дескриптор, содержащий эту передачу */
    i2c_slave_event_transfer_t event;             /*!< Причина, по которой вызывается функция обратного вызова */
    uint32_t                   event_mask;        /*!< Маска событий для вызова функции обратного вызова */
    uint8_t                    *rx_data;          /*!< Буфер обмена для приема данных */
    size_t                     rx_size;           /*!< Количество данных на прием */
    const uint8_t              *tx_data;          /*!< Буфер обмена для передачи данных */
    size_t                     tx_size;           /*!< Количество данных на передачу */
    size_t                     transferred_count; /*!< Количество байтов переданных во время этого обмена */
    i2c_status_t               completion_status; /*!< Код успеха или ошибки, описывающий завершение передачи. Применимо только для #I2C_SlaveEvent_Completion */
} i2c_slave_transfer_t;

/*!
 * @brief Slave тип функции обратного вызова.
 *
 * Этот тип функции обратного вызова используется только для неблокирующей передачи. Чтобы
 * установить функцию обратного вызова, используйте функцию @ref I2C_SlaveSetCallback
 * после создания дескриптора.
 *
 * @param base      Базовый адрес экземпляра I2C, на котором произошло событие.
 * @param transfer  Дескриптор передачи, содержащий значения,
 *                  переданные в/из функции обратного вызова.
 * @param user_data Пользовательские данные.
 */
typedef void (*i2c_slave_transfer_callback_t)(I2C_Type *base,
    volatile i2c_slave_transfer_t *transfer, void *user_data);

/*!
 * @brief I2C Slave состояния конечного автомата
 */
typedef enum {
    I2C_SlaveFsm_Idle     = 0U, /*!< Модуль I2C находится в состоянии ожидания, текущего обмена нет */
    I2C_SlaveFsm_Init     = 1U, /*!< Инициализирован процесс обмена, ожидание событий на шине */
    I2C_SlaveFsm_Receive  = 4U, /*!< Модуль I2C в фазе получения данных */
    I2C_SlaveFsm_Transmit = 6U, /*!< Модуль I2C в фазе передачи данных */
    I2C_SlaveFsm_Stop     = 8U, /*!< Модуль I2C получил Stop */
} i2c_slave_fsm_t;

/*!
 * @brief I2C Slave-дескриптор
 */
struct _i2c_slave_handle {
    int32_t irq_num;                          /*!< Номер IRQ */
    volatile i2c_slave_transfer_t transfer;   /*!< Структура обмена данными */
    volatile i2c_slave_fsm_t      slave_fsm;  /*!< Slave конечный автомат передачи */
    i2c_slave_transfer_callback_t callback;   /*!< Функция обратного вызова, вызываемая при обмене */
    void                          *user_data; /*!< Параметр функции обратного вызова */
};

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @name Инициализация и деинициализация.
 * @{
 */

/*!
 * @brief Возврат состояния модуля I2C.
 *
 * @param base Базовый адрес модуля I2C.
 *
 * @retval true  Mодуль I2C в состоянии enable.
 * @retval false Mодуль I2C в состоянии disable.
 */
static inline bool I2C_IsEnable(I2C_Type *base)
{
    return (bool) GET_VAL_MSK(base->IC_ENABLE_STATUS,
            I2C_IC_ENABLE_STATUS_IC_EN_Msk, I2C_IC_ENABLE_STATUS_IC_EN_Pos);
}

/*!
 * @brief Включение и отключение модуля I2C.
 *
 * @note Общий алгоритм отключения модуля I2C:
 *       -# Определить интервал таймера SYSWAIT, равный 10-кратному периоду
 *          высшей скорости I2C. Например: если высшая передача I2C режим 400
 *          кбит/с, то этот интервал SYSWAIT равен 25 мкс.
 *
 *       -# Определить параметр максимального времени ожидания,
 *          I2C_RETRY_TIMES_FOR_DISABLE_UNITS * SYSWAIT, при превышении этого
 *          значения, сообщать об ошибке.
 *
 *       -# Выполнить блокирующий поток/процесс/функцию, которая предотвращает
 *          дальнейшие Master-транзакции I2C.
 *
 *       -# Переменная count инициализируется I2C_RETRY_TIMES_FOR_DISABLE_UNITS.
 *
 *       -# Установить бит 0 регистра IC_ENABLE в 0.
 *
 *       -# Вызвать функцию @ref I2C_IsEnable для проверки текущего состояния
 *          модуля. Уменьшить POLL_COUNT на один. Если POLL_COUNT == 0, выход с
 *          кодом ошибки.
 *
 *       -# Если @ref I2C_IsEnable выдает true, то ожидание времени SYSWAIT и
 *          переход к предыдущему шагу. В противном случае, выйти с кодом
 *          успеха.
 *
 * @param base   Базовый адрес модуля I2C.
 * @param enable Передайте true, чтобы включить, или false, чтобы отключить
 *               указанный модуль I2C.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t I2C_Enable(I2C_Type *base, bool enable);

/*!
 * @brief Возврат номера экземпляра по базовому адресу.
 *
 * Если передан недопустимый базовый адрес, то в режиме отладки будет
 * выполена assert. В режиме релиза будет возвращен номер экземпляра 0.
 *
 * @param base Базовый адрес модуля I2C.
 *
 * @return I2C номер экземпляра, начиная с 0.
 */
uint32_t I2C_GetInstance(I2C_Type *base);

/*!
 * @brief Сброс модуля I2C.
 *
 * Завершение всех операций, очищение TxFifo и RxFifo.
 *
 * @param base Базовый адрес модуля I2C.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t I2C_Reset(I2C_Type *base);

/*!
 * @brief Получение конфигурации по умолчанию для Master-режима модуля I2C.
 *
 * Эта функция обеспечивает следующую конфигурацию по умолчанию для модуля I2C:
 * @code{.c}
 *   master_config->enable_master = true;
 *   master_config->baudrate_bps  = 100000U;
 * @endcode
 *
 * После вызова этой функции вы можете переопределить любые настройки, перед
 * инициализацией помощью @ref I2C_MasterInit.
 *
 * @param master_config Конфигурация модуля для Master-режима.
 */
void I2C_MasterGetDefaultConfig(i2c_master_config_t *master_config);

/*!
 * @}
 */

/*!
 * @name Прерывания.
 * @{
 */

/*!
 * @brief Включение запросов на прерывание.
 *
 * @note Битовая маска прерываний составляется из флагов, #i2c_interrupt,
 *       которые могут быть соединены с использованием оператора побитового ИЛИ.
 *
 * @param base Базовый адрес модуля.
 * @param mask Битовая маска прерываний.
 */
static inline void I2C_EnableInterrupts(I2C_Type *base, uint32_t mask)
{
    base->IC_INTR_MASK |= mask & I2C_IRQ_All;
}

/*!
 * @brief Отключение запросов на прерывание.
 *
 * @note Битовая маска прерываний составляется из флагов, #i2c_interrupt,
 *       которые могут быть соединены с использованием оператора побитового ИЛИ.
 *
 * @param base Базовый адрес модуля.
 * @param mask Битовая маска прерываний.
 */
static inline void I2C_DisableInterrupts(I2C_Type *base, uint32_t mask)
{
    base->IC_INTR_MASK &= ~(mask & I2C_IRQ_All);
}

/*!
 * @brief Возврат набора текущих разрешенных запросов на прерывание.
 *
 * @note Битовая маска прерываний составляется из флагов, #i2c_interrupt,
 *       которые могут быть соединены с использованием оператора побитового ИЛИ.
 *
 * @param base Базовый адрес модуля.
 *
 * @return Битовая маска прерываний.
 */
static inline uint32_t I2C_GetEnabledInterrupts(I2C_Type *base)
{
    return base->IC_INTR_MASK & I2C_IRQ_All;
}

/*!
 * @brief Очищение флагов прерываний.
 *
 * @note Не все флаги прерываний можно очистить этой функцией. За дополнительной
 *       информацией обратитесь к документации на MCU.
 *
 * @param base Базовый адрес модуля.
 * @param irq  Битовая маска прерывания.
 */
static inline void I2C_ClearInterrupt(I2C_Type *base, enum i2c_interrupt irq)
{
    volatile uint32_t temp = 0;
    UNUSED(temp);

    switch (irq) {
        case I2C_IRQ_RxUnder:   /* [0] Чтении из пустого RxFifo. */
            temp = base->IC_CLR_RX_UNDER;  /* Сброс: чтение IC_CLR_RX_UNDER */
            break;

        case I2C_IRQ_RxOver:   /* [1] Переполнение RxFifo. */
            temp = base->IC_CLR_RX_OVER;   /* Сброс: чтение IC_CLR_RX_OVER */
            break;

        case I2C_IRQ_RxFull:   /* [2] RxFifo заполнен до уровня IC_RX_TL. */
            assert(0);
            break;

        case I2C_IRQ_TxOver:   /* [3] Попытка записать в заполненный TxFifo */
            temp = base->IC_CLR_TX_OVER;   /* Сброс: чтение IC_CLR_TX_OVER */
            break;

        case I2C_IRQ_TxEmpty:  /* [4] Опустошение TxFifo ниже уровня IC_TX_TL */
            assert(0);
            break;

        case I2C_IRQ_RdReq:    /* [5] Уст. при запросе данных удаленным Master. */
            temp = base->IC_CLR_RD_REQ;    /* Сброс чтением: IC_CLR_RD_REQ.CLR_RD_REQ */
            break;

        case I2C_IRQ_TxAbrt:   /* [6] Передача прервана */
            temp = base->IC_CLR_TX_ABRT;   /* Сброс чтением: IC_CLR_TX_ABRT.CLR_TX_ABRT */
            break;

        case I2C_IRQ_RxDone:   /* [7] В режиме Slave-передатчика устанавливается в 1, если мастер не подтверждает передачу байта */
            temp = base->IC_CLR_RX_DONE;   /* Сброс чтением: IC_CLR_RX_DONE.CLR_RX_DONE */
            break;

        case I2C_IRQ_Activity: /* [8] Активность на шине I2C */
            temp = base->IC_CLR_ACTIVITY;  /* Сброс чтением: 1) IC_CLR_INTR 2) IC_CLR_ACTIVITY.CLR_ACTIVITY 3) Выключение I2C 4) Системный сброс */
            break;

        case I2C_IRQ_StopDet:  /* [9] Устанавливается в Slave- или Master-режиме, если на шине возникает состояние STOP */
            temp = base->IC_CLR_STOP_DET;  /* Сброс: чтение IC_CLR_STOP_DET */
            break;

        case I2C_IRQ_StartDet: /* [10] На шине START или RESTART условия */
            temp = base->IC_CLR_START_DET; /* Сброс: чтение IC_CLR_START_DET */
            break;

        case I2C_IRQ_GenCall:  /* [11] Получен адрес General Call и отправлено подтверждение */
            temp = base->IC_CLR_GEN_CALL;  /* Сброс: 1) чтением IC_CLR_GEN_CALL 2) выключение модуля */
            break;

        default:
            assert(0);
            break;
    }

}

/*!
 * @brief Очищение всех доступных для очистки флагов прерываний.
 *
 * @note Не все флаги прерываний можно очистить этой функцией. За дополнительной
 *       информацией обратитесь к документации на MCU.
 *
 * @param base Базовый адрес модуля.
 */
static inline void I2C_ClearAllInterrupts(I2C_Type *base)
{
    volatile uint32_t temp;
    temp = base->IC_CLR_INTR;

    UNUSED(temp);
}

/*!
 * @}
 */

/*!
 * @name Master-режим работы.
 * @{
 */

/*!
 * @name Инициализация и деинициализация.
 * @{
 */

/*!
 * @brief Инициализация Master-устройства I2C.
 *
 * Функция инициализирует Master-устройство I2C, в соответствии с
 * пользовательской конфигурацией. Для инициализации конфигурации значениями
 * по умолчанию используйте функцию @ref I2C_MasterGetDefaultConfig.
 *
 * @note Частота синхронизации I2C модуля используется для расчета делителей
 *       скорости передачи данных и периодов ожидания.
 *
 * @param base          Базовый адрес модуля I2C.
 * @param master_config Конфигурация модуля для Master-режима.
 * @param src_clock_hz  Частота синхронизации I2C модуля.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t I2C_MasterInit(I2C_Type *base,
    const i2c_master_config_t *master_config, uint32_t src_clock_hz);

/*!
 * @brief Деинициализация Master-устройства I2C.
 *
 * @param base Базовый адрес модуля I2C.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t I2C_MasterDeinit(I2C_Type *base);

/*!
 * @}
 */

/*!
 * @name Операции на шине
 * @{
 */

/*!
 * @brief Установка частоты шины для Master-режима модуля I2C.
 *
 * Master-режим модуля I2C автоматически отключается и снова включается при
 * необходимости для настройки скорости передачи данных.
 *
 * @note Не вызывайте эту функцию во время передачи, иначе передача будет
 *       прервана.
 *
 * @param base         Базовый адрес модуля.
 * @param baudrate_bps Запрашиваемая частота шины в битах в секунду.
 * @param src_clock_hz Частота синхронизации модуля в Герцах.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 * @retval #I2C_Status_InvalidParameter
 */
i2c_status_t I2C_MasterSetBaudRate(I2C_Type *base, uint32_t baudrate_bps,
    uint32_t src_clock_hz);

/*!
 * @brief Установка адреса Slave-устройства на шине I2C, к которому будет обращение
 *        в цикле обмена.
 *
 * Функция устанавливает адрес Slave-устройства и размер адреса. Следющий обмен
 * данными будет происходить с устройством по указанному адресу.
 *
 * @param base      Базовый адрес модуля.
 * @param address   Адрес Slave-устройства, если адрес равен 0U то General Call
 * @param addr_size Размер адреса (7- или 10-битный адрес).
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy
 * @retval #I2C_Status_HwError
 */
i2c_status_t I2C_MasterAddrSet(I2C_Type *base, uint32_t address,
    i2c_addr_size_t addr_size);

/*!
 * @brief Возврат статуса активность шины.
 *
 * Необходимо, чтобы был включен Master-режим.
 *
 * @param base Базовый адрес модуля.
 *
 * @retval true  Шина занята.
 * @retval false Шина свободна.
 */
static inline bool I2C_MasterGetBusActiveState(I2C_Type *base)
{
    /* Получить значение флага: Активность на шине */
    return (bool) GET_VAL_MSK(base->IC_STATUS, I2C_IC_STATUS_ACTIVITY_Msk,
            I2C_IC_STATUS_ACTIVITY_Pos);
}

/*!
 * @brief Выполнение блокирующей передачи по шине I2C.
 *
 * Отправляет до tx_size байтов на ранее адресованное Slave-устройство. Slave
 * может ответить NACK на любой байт, чтобы досрочно завершить передачу.
 * Если это произойдет, функция возвращает #I2C_Status_Nack.
 *
 * @note Флаги управления передачей перечислены в
 *       @ref i2c_master_transfer_flags_t и могут быть объединены операцией ИЛИ.
 *       Допустимые комбинации:
 *       - I2C_TransferStartFlag | I2C_TransferStopFlag
 *         - Стандартная транзакция. Начинается со Start и заканчивается Stop
 *           условием.
 *       - I2C_TransferStartFlag
 *         - Начальный пакет транзакции. Начинается со Start и подразумевает
 *           продолжение.
 *       - I2C_TransferDataFlag
 *         - Пакет передачи данных без Start и Stop условия.
 *       - I2C_TransferStopFlag
 *         - Завершающий пакет транзакции. Не начинается со Start и
 *           заканчивается Stop условием.
 *       - I2C_TransferReStartFlag
 *         - Продолжение транзакции с выдачей повторного старта. Подразумевает
 *           продолжение. Используется для смены направления обмена.
 *       - I2C_TransferReStartFlag | I2C_TransferStopFlag
 *         - Завершение транзакции с выдачей повторного старта. Не допускает
 *           продолжения. Используется для смены направления обмена.
 *
 * @note В случае, если Slave не подтверждает последний принятый байт в
 *       составной транзакции со сменой направления, статус отказа
 *       (#I2C_Status_Nack) проигнорирован не будет вне зависимости от
 *       #I2C_MASTER_TRANSMIT_IGNORE_LAST_NACK.
 *
 * @param base    Базовый адрес модуля.
 * @param tx_buff Буфер для хранения передаваемых данных.
 * @param tx_size Размер буфера передаваемых данных в байтах.
 * @param flags   Флаги управления передачей для управления специальным
 *                поведением.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy
 * @retval #I2C_Status_Nack
 * @retval #I2C_Status_ArbitrationLost
 * @retval #I2C_Status_Timeout
 * @retval #I2C_Status_UserError
 * @retval #I2C_Status_AddrNack
 * @retval #I2C_Status_BreakTransfer
 * @retval #I2C_Status_UnexpectedState
 * @retval #I2C_Status_SetStartError
 * @retval #I2C_Status_HsCodeError
 */
i2c_status_t I2C_MasterWriteBlocking(I2C_Type *base, const void *tx_buff,
    size_t tx_size, uint32_t flags);

/*!
 * @brief Выполнение блокирующего приема на шине I2C.
 *
 * @note Флаги управления передачей перечислены в
 *       @ref i2c_master_transfer_flags_t и могут быть объединены операцией ИЛИ.
 *       Допустимые комбинации:
 *       - I2C_TransferStartFlag | I2C_TransferStopFlag
 *         - Стандартная транзакция. Начинается со Start и заканчивается Stop
 *           условием.
 *       - I2C_TransferStartFlag
 *         - Начальный пакет транзакции. Начинается со Start и подразумевает
 *           продолжение.
 *       - I2C_TransferDataFlag
 *         - Пакет передачи данных без Start и Stop условия.
 *       - I2C_TransferStopFlag
 *         - Завершающий пакет транзакции. Не начинается со Start и
 *           заканчивается Stop условием.
 *       - I2C_TransferReStartFlag
 *         - Продолжение транзакции с выдачей повторного старта. Подразумевает
 *           продолжение. Используется для смены направления обмена.
 *       - I2C_TransferReStartFlag | I2C_TransferStopFlag
 *         - Завершение транзакции с выдачей повторного старта. Не допускает
 *           продолжения. Используется для смены направления обмена.
 *
 * @note В случае, если Slave не подтверждает последний принятый байт в
 *       составной транзакции со сменой направления, статус отказа
 *       (#I2C_Status_Nack) проигнорирован не будет вне зависимости от
 *       #I2C_MASTER_TRANSMIT_IGNORE_LAST_NACK.
 *
 * @param base    Базовый адрес модуля.
 * @param rx_buff Буфер для хранения принимаемых данных.
 * @param rx_size Размер буфера принимаемых данных в байтах.
 * @param flags   Флаги управления передачей для управления специальным
 *                поведением.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy
 * @retval #I2C_Status_ArbitrationLost
 * @retval #I2C_Status_Timeout
 * @retval #I2C_Status_UserError
 * @retval #I2C_Status_AddrNack
 * @retval #I2C_Status_Nack
 * @retval #I2C_Status_BreakTransfer
 * @retval #I2C_Status_UnexpectedState
 * @retval #I2C_Status_SetStartError
 * @retval #I2C_Status_HsCodeError
 */
i2c_status_t I2C_MasterReadBlocking(I2C_Type *base, void *rx_buff,
    size_t rx_size, uint32_t flags);

/*!
 * @brief Выполнение обмена данными на шине I2C в режиме блокировки.
 *
 * @note Из функции управление не возвращается до тех пор, пока передача не
 *       завершится успешно или неуспешно из-за того, что предыдущий обмен еще
 *       не завершен или из-за в проигранного арбитража или получения NACK во
 *       время передачи адреса или данных.
 *
 * @param base Базовый адрес модуля.
 * @param xfer Структура передачи.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy
 * @retval #I2C_Status_Nack
 * @retval #I2C_Status_ArbitrationLost
 * @retval #I2C_Status_Timeout
 * @retval #I2C_Status_UserError
 * @retval #I2C_Status_AddrNack
 * @retval #I2C_Status_BreakTransfer
 * @retval #I2C_Status_UnexpectedState
 * @retval #I2C_Status_SetStartError
 * @retval #I2C_Status_HsCodeError
 */
i2c_status_t I2C_MasterTransferBlocking(I2C_Type *base,
    i2c_master_transfer_t *xfer);

/*!
 * @}
 */

/*!
 * @name Неблокирующий обмен (по прерыванию).
 * @{
 */

/*!
 * @brief Создание нового дескриптора для неблокирующего Master-режима работы.
 *
 * Создание дескриптора предназначено для использования с неблокирующими API.
 * Однажды созданный дескриптор не требуется специально уничтожать отдельной
 * функцией. Если пользователь хочет завершить передачу, должна быть вызвана
 * @ref I2C_MasterTransferAbort.
 *
 * @param base      Базовый адрес модуля.
 * @param handle    Дескриптор для Master-режима.
 * @param callback  Функция обратного вызова.
 * @param user_data Пользовательские данные для функции обратного вызова.
 */
void I2C_MasterTransferCreateHandle(I2C_Type *base, i2c_master_handle_t *handle,
    i2c_master_transfer_callback_t callback, void *user_data);

/*!
 * @brief Выполнение неблокирующей транзакции на шине I2C.
 *
 * @param base   Базовый адрес модуля.
 * @param handle Дескриптор для Master-режима.
 * @param xfer   Структура передачи.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy Либо другой Master в настоящее время использует
 *                          шину, либо неблокирующая транзакция уже выполняется.
 */
i2c_status_t I2C_MasterTransferNonBlocking(I2C_Type *base,
    i2c_master_handle_t *handle, i2c_master_transfer_t *xfer);

/*!
 * @brief Получение количества байтов, переданных неблокирующей транзакцией
 *        на данный момент.
 *
 * @param base   Базовый адрес модуля.
 * @param handle Дескриптор для Master-режима.
 * @param count  Количество байтов, переданных на данный момент.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy
 */
i2c_status_t I2C_MasterTransferGetCount(I2C_Type *base,
    i2c_master_handle_t *handle, size_t *count);

/*!
 * @brief Досрочное завершаение основной неблокирующей передачи I2C.
 *
 * @note Небезопасно вызывать эту функцию из обработчика IRQ, который имеет
 *       более высокий приоритет, чем приоритет IRQ периферийного устройства
 *       I2C.
 *
 * @param base   Базовый адрес модуля.
 * @param handle Дескриптор для Master-режима.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Timeout Тайм-аут во время опроса флагов.
 */
i2c_status_t I2C_MasterTransferAbort(I2C_Type *base,
    i2c_master_handle_t *handle);

/*!
 * @}
 */

/*!
 * @name Обработчик запросов на прерывание для Master-режима.
 * @{
 */

/*!
 * @brief Обработчик запросов на прерывание для Master-режима.
 *
 * @note Эту функцию не нужно вызывать.
 *
 * @param base   Базовый адрес модуля.
 * @param handle Дескриптор для Master-режима.
 */
void I2C_MasterTransferHandleIRQ(I2C_Type *base, i2c_master_handle_t *handle);

/*!
 * @}
 */

/*!
 * @}
 */

/*!
 * @name Slave-режим работы.
 * @{
 */

/*!
 * @name Инициализация и деинициализация в Slave-режиме.
 * @{
 */

/*!
 * @brief Получение конфигурации по умолчанию для Slave-режима модуля I2C.
 *
 * Эта функция обеспечивает следующую конфигурацию по умолчанию Slave-устройства
 * I2C:
 * @code{.c}
 *   сonfig->address       = I2C_SLAVE_ADDR_DEFAULT;
 *   сonfig->ack_gen_call  = false;
 *   сonfig->i2c_addr_size = I2C_Address7Bit;
 *   сonfig->enable_slave  = true;
 * @endcode
 *
 * После вызова этой функции возможно изменить настройки, если требуется
 * изменить конфигурацию, перед инициализацией с помощью @ref I2C_SlaveInit.
 * Необходимо переопределить Slave-адрес желаемым адресом.
 *
 * @param config Пользовательская конфигурация.
 */
void I2C_SlaveGetDefaultConfig(i2c_slave_config_t *config);

/*!
 * @brief Инициализация Slave-модуля I2C.
 *
 * Функция инциализирует ведомое периферийное устройство I2C, как описано
 * пользователем в переданной конфигурации.
 *
 * @note Для получения значений по умолчанию для пользовательской конфигурации
 *       необходимо воспользоваться функцией @ref I2C_SlaveGetDefaultConfig, а
 *       затем переопределить их при необходимости.
 *
 * @param base   Базовый адрес модуля.
 * @param config Пользовательская конфигурация.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t I2C_SlaveInit(I2C_Type *base, const i2c_slave_config_t *config);

/*!
 * @brief Установка адреса Slave-устройства.
 *
 * Эта функция записывает новое значение в регистр адреса Slave-устройства.
 *
 * @param base         Базовый адрес модуля.
 * @param address      Адрес Slave-устройства.
 * @param ack_gen_call Отвечать или нет General Call.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t  I2C_SlaveSetAddress(I2C_Type *base, uint16_t address,
    bool ack_gen_call);

/*!
 * @brief Деинициализация периферийного Slave-устройства I2C.
 *
 * Эта функция отключает Slave-устройство I2C, а также выполняет программный
 * сброс для восстановления модуля I2C до состояния сброса.
 *
 * @param base Базовый адрес модуля.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
static inline i2c_status_t I2C_SlaveDeinit(I2C_Type *base)
{
    return I2C_Enable(base, false);
}

/*!
 * @brief Включение/отключение модуля I2C.
 *
 * @note  Функция используется для Slave-режима.
 *
 * @param base   Базовый адрес модуля.
 * @param enable true - включить, false - выключить указанный модуль.
 */
static inline i2c_status_t I2C_SlaveEnable(I2C_Type *base, bool enable)
{
    return I2C_Enable(base, enable);
}

/*!
 * @}
 */

/*!
 * @name Операции на шине в Slave-режиме.
 * @{
 */

/*!
 * @brief Выполнение передачи из модуля на шину I2C в режиме блокировки.
 *
 * Функция выполняет фазу передачи адреса и фазу передачи данных.
 *
 * @param base    Базовый адрес модуля.
 * @param tx_buff Буфер для хранения передаваемых данных.
 * @param tx_size Размер буфера передаваемых данных в байтах.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_UnexpectedState Master пытается считать данные из Slave,
 *                                     в то время как Slave ожидает запись.
 */
i2c_status_t I2C_SlaveWriteBlocking(I2C_Type *base, const uint8_t *tx_buff,
    size_t tx_size);

/*!
 * @brief Выполнение приема по шине I2C в модуль в режиме блокировки.
 *
 * Функция выполняет фазу передачи адреса и фазу передачи данных.
 *
 * @param base    Базовый адрес модуля.
 * @param rx_buff Буфер для хранения получаемых данных.
 * @param rx_size Размер буфера получаемых данных в байтах.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_UnexpectedState Master пытается считать данные из Slave,
 *                                     в то время как Slave ожидает запись.
 */
i2c_status_t I2C_SlaveReadBlocking(I2C_Type *base, uint8_t *rx_buff,
    size_t rx_size);

/*!
 * @}
 */

/*!
 * @name Неблокирующий обмен (по прерыванию) в Slave-режиме.
 * @{
 */

/*!
 * @brief Создание нового дескриптора для I2C в Slave-режиме.
 *
 * @note Используется для работы в неблокирующих функциях (по прерыванию).
 *
 * Создание дескриптора для использования в неблокирующих функциях. Однажды
 * созданный дескриптор не требуется специально уничтожать. Если требуется
 * завершить обмен данными используется функция @ref I2C_SlaveTransferAbort.
 *
 * @param base      Базовый адрес модуля.
 * @param handle    Дескриптор для Slave-режима.
 * @param callback  Функция обратного вызова.
 * @param user_data Пользовательские данные для функции обратного вызова.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_UnexpectedState
 */
i2c_status_t I2C_SlaveTransferCreateHandle(I2C_Type *base,
    i2c_slave_handle_t *handle, i2c_slave_transfer_callback_t callback,
    void *user_data);

/*!
 * @brief Инициализация приема для Slave-режима.
 *
 * Вызовите эту функцию после вызова @ref I2C_SlaveInit и
 * @ref I2C_SlaveTransferCreateHandle для начала обмена, управляемого Master-
 * устройством. Модуль в Slave-режиме отслеживает I2C шину и передает
 * событие в функцию обратного вызова, которая была зарегистрирована при
 * вызове @ref I2C_SlaveTransferCreateHandle. Функция обратного вызова вызывается
 * из контекста прерывания.
 *
 * Набор событий, получаемых функцией обратного вызова, настраивается:
 * для этого в параметр event_mask передается комбинация из элементов
 * #i2c_slave_event_transfer_t для событий, которые требуется получать.
 * События #I2C_SlaveEvent_Transmit и #I2C_SlaveEvent_Receive всегда включены и
 * не требуют быть включенным в маску. Константа #I2C_SlaveEvents_All включает
 * все события. Константа #I2C_SlaveEvents_Ordinary включает все события для
 * работы в режиме без General Call.
 *
 * @param base       Базовый адрес модуля.
 * @param handle     Дескриптор для Slave-режима.
 * @param event_mask Битовая маска, определяющая события, приводящие к вызову
 *                   функции обратного вызова.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy На этом дескрипторе уже идет обмен данными.
 */
i2c_status_t I2C_SlaveTransferNonBlocking(I2C_Type *base,
    i2c_slave_handle_t *handle, uint32_t event_mask);

/*!
 * @brief Запуск передачи данных из Slave в Master по прерыванию.
 *
 * Функция может быть вызвана в ответ на событие #I2C_SlaveEvent_Transmit,
 * переданное в функцию обратного вызова, чтобы начать новую передачу
 * из Slave в Master.
 *
 * Набор событий, получаемых функцией обратного вызова, настраивается:
 * для этого в параметр event_mask передается комбинация из элементов
 * #i2c_slave_event_transfer_t для событий, которые требуется получать.
 * События #I2C_SlaveEvent_Transmit и #I2C_SlaveEvent_Receive всегда включены и
 * не требуют быть включенным в маску. Константа #I2C_SlaveEvents_All включает
 * все события. Константа #I2C_SlaveEvents_Ordinary включает все события для
 * работы в режиме без General Call.
 *
 * @param base       Базовый адрес модуля.
 * @param transfer   Структура обмена данными.
 * @param tx_data    Данные для отправки в Master.
 * @param tx_size    Количество данных для отправки в Master в байтах.
 * @param event_mask Битовая маска, определяющая события, приводящие к вызову
 *                   функции обратного вызова.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy Slave передача уже запущена для этого декскриптора.
 */
i2c_status_t I2C_SlaveSetSendBuffer(I2C_Type *base,
    volatile i2c_slave_transfer_t *transfer, const void *tx_data,
    size_t tx_size, uint32_t event_mask);

/*!
 * @brief Иниициирование Slave-устройсвом приема запросов от Master-устройства.
 *
 * Функция может быть вызвана в ответ на вызов функции обратного вызова
 * по событию #I2C_SlaveEvent_Receive для старта нового приема данных.
 *
 * Набор событий, получаемых функцией обратного вызова, настраивается:
 * для этого в параметр event_mask передается комбинация из элементов
 * #i2c_slave_event_transfer_t для событий, которые требуется получать.
 * События #I2C_SlaveEvent_Transmit и #I2C_SlaveEvent_Receive всегда включены и
 * не требуют быть включенным в маску. Константа #I2C_SlaveEvents_All включает
 * все события. Константа #I2C_SlaveEvents_Ordinary включает все события для
 * работы в режиме без General Call.
 *
 * @param base       Базовый адрес модуля.
 * @param transfer   Структура обмена данными.
 * @param rx_data    Буфер для хранения данных от Master.
 * @param rx_size    Размер буфера для хранения данных от Master в байтах.
 * @param event_mask Битовая маска, определяющая события, приводящие к вызову
 *                   функции обратного вызова.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy
 */
i2c_status_t I2C_SlaveSetReceiveBuffer(
    I2C_Type *base, volatile i2c_slave_transfer_t *transfer, void *rx_data,
    size_t rx_size, uint32_t event_mask);

/*!
 * @brief Завершение неблокирующей передачи Slave-устройства.
 *
 * @note Функцию можно вызвать в любое время, чтобы остановить Slave-устройство
 *       для обработки событий шины.
 *
 * @param base   Базовый адрес модуля.
 * @param handle Дескриптор для Slave-режима.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Idle
 */
i2c_status_t I2C_SlaveTransferAbort(I2C_Type *base, i2c_slave_handle_t *handle);

/*!
 * @brief Получение количества оставшихся байтов во время неблокирующей
 *        передачи (через прерывание) в Slave-режиме.
 *
 * @param base   Базовый адрес модуля.
 * @param handle Дескриптор для Slave-режима.
 * @param count  Количество байтов, переданных на данный момент неблокирующей
 *               транзакцией.
 *
 * @retval #I2C_Status_InvalidParameter
 * @retval #I2C_Status_Ok
 */
i2c_status_t I2C_SlaveTransferGetCount(I2C_Type *base,
    i2c_slave_handle_t *handle, size_t *count);

/*!
 * @}
 */

/*!
 * @name Обработчик IRQ для Slave-режима.
 * @{
 */

/*!
 * @brief Процедура для обработки прерываний в Slave-режиме.
 *
 * @note Эту функцию не нужно вызывать самостоятельно.
 *
 * @param base   Базовый адрес модуля.
 * @param handle Дескриптор для Slave-режима.
 */
void I2C_SlaveTransferHandleIRQ(I2C_Type *base, i2c_slave_handle_t *handle);

/*!
 * @}
 */

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif

#endif /* HAL_I2C_H */

/*!
 * @}
 */
