/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup i2c_driver
 * @{
 */

/*!
 * @file hal_i2c_dma.c
 *
 * @brief Имплементация дополнения драйвера I2C с пересылкой данных через DMA
 */

#include "hal_i2c_dma.h"

#define I2C_UNITS_NUM (2U)  /*!< Число I2C модулей */

/*!
 * @brief Дескриптор для I2C-DMA передач
 */
typedef struct _i2c_master_dma_private_handle {
    I2C_Type *base;                  /*!< Базовый адрес I2C */
    i2c_master_dma_handle_t *handle; /*!< Контекст данных прерывания I2C-DMA */
} i2c_master_dma_private_handle_t;

/*! @brief Таблица дескрипторов для I2C-DMA передач */
static i2c_master_dma_private_handle_t s_dma_private_handle[I2C_UNITS_NUM];

/*!
 * @brief Функция приема/передачи данных между I2C и DMA контроллерами
 *
 * @param base   Базовый адрес
 * @param handle Дескриптор I2C-DMA передач
 * @param xfer   Конфигурация I2C-master передачи
 *
 * @retval #I2C_DMA_Status_Success
 * @retval #I2C_DMA_Status_Fail
 * @retval #I2C_DMA_Status_DMA_Busy
 * @retval #I2C_DMA_Status_InvalidArgument
 */
static i2c_status_t I2C_RunDMATransfer(I2C_Type *base,
    i2c_master_dma_handle_t *handle, i2c_master_transfer_t *xfer);

void I2C_MasterTransferCreateHandleDMA(I2C_Type *base,
    i2c_master_dma_handle_t *handle,
    i2c_master_dma_transfer_callback_t callback,
    void *user_data,
    dma_handle_t *tx_dma, dma_handle_t *rx_dma)
{
    assert(handle != NULL);
    assert(tx_dma != NULL);
    assert(rx_dma != NULL);

    (void) memset(handle, 0, sizeof(i2c_master_dma_handle_t));

    uint32_t instance = I2C_GetInstance(base);

    handle->completion_callback = callback;
    handle->user_data = user_data;

    handle->tx_dma = tx_dma;
    handle->rx_dma = rx_dma;

    s_dma_private_handle[instance].base   = base;
    s_dma_private_handle[instance].handle = handle;
}

i2c_status_t I2C_MasterTransferDMA(I2C_Type *base,
    i2c_master_dma_handle_t *handle, i2c_master_transfer_t *xfer)
{
    assert(handle != NULL);
    assert(xfer != NULL);
    assert(xfer->subaddress_size <= sizeof(xfer->subaddress));
    assert(xfer->data_size > 0);

    i2c_status_t status = I2C_Status_Ok;
    uint32_t IC_DATA_CMD = 0UL;

    I2C_MasterAddrSet(base, xfer->slave_address, xfer->addr_size);

    if (xfer->subaddress_size != 0) {
        status = I2C_MasterWriteBlocking(base, (const void *)xfer->subaddress,
            xfer->subaddress_size, 0);
        if (status != I2C_Status_Ok) {
            return status;
        }
    }

    I2C_Enable(base, false);
    /* Направление передачи. */
    SET_VAL_MSK(base->IC_DATA_CMD, I2C_IC_DATA_CMD_CMD_Msk,
        I2C_IC_DATA_CMD_CMD_Pos, xfer->direction);

    if (xfer->flags & I2C_TransferReStartFlag) {
        /* Выдача RESTART. */
        SET_VAL_MSK(IC_DATA_CMD, I2C_IC_DATA_CMD_RSVD_RESTART_Msk,
            I2C_IC_DATA_CMD_RSVD_RESTART_Pos, 1);
    }
    base->IC_DATA_CMD = IC_DATA_CMD;
    I2C_Enable(base, true);
    handle->remaining_bytes_DMA = xfer->data_size;

    switch (xfer->direction) {
        case I2C_Write:
            handle->state = I2C_MTS_TransmitData;
            break;
        case I2C_Read:
            handle->state = I2C_MTS_ReceiveData;
            break;
        default:
            break;
    }

    status = I2C_RunDMATransfer(base, handle, xfer);
    if (status != I2C_Status_Ok) {
        handle->state = I2C_MTS_Idle;
        return status;
    }

    return status;
}

static i2c_status_t I2C_RunDMATransfer(
    I2C_Type *base,
    i2c_master_dma_handle_t *handle,
    i2c_master_transfer_t *xfer)
{
    uint32_t desc_count = 0UL;
    uint32_t instance = I2C_GetInstance(base);
    /* Включение интерфейса взаимодействия с контроллером DMA. */
    SET_VAL_MSK(base->IC_DMA_CR, I2C_IC_DMA_CR_TDMAE_Msk,
        I2C_IC_DMA_CR_TDMAE_Pos, 1);
    /* Включение аппаратного запроса DMA. */
    DMA_HardwareHandshakeEnable(handle->tx_dma, true,
        DMA_HANDSHAKE_I2C0_TX + instance * 2);
    
    uint32_t transfer_size = handle->remaining_bytes_DMA;
    if (transfer_size > DMA_AHB_MAX_BLOCK_SIZE) {
        desc_count = DMA_GetDescriptorCount(transfer_size,
            DMA_Transfer8BitWidth);
    }

    switch (xfer->direction) {
        case I2C_Write:
            if (desc_count != 0UL) {
                I2C_DMADescriptorInitTX(base, handle->tx_desc, desc_count,
                    transfer_size, DMA_Incr, xfer->data);
                DMA_SubmitChannelDescriptor(handle->tx_dma, handle->tx_desc);
            } else {
                uint64_t xfercfg = DMA_CHANNEL_CTL(transfer_size,
                    0, 0,
                    DMA_MemoryToPeripheral_DMA,
                    0, 0,
                    DMA_BurstSize1, DMA_BurstSize1,
                    DMA_Incr, DMA_NoChange,
                    DMA_Transfer8BitWidth, DMA_Transfer8BitWidth,
                    1
                );
                DMA_SubmitChannelTransferParameter(handle->tx_dma, xfercfg,
                    xfer->data, (void *)&base->IC_DATA_CMD);
            }
            DMA_StartTransfer(handle->tx_dma);
            break;

        case I2C_Read:
            /* Включение интерфейса взаимодействия с контроллером DMA. */
            SET_VAL_MSK(base->IC_DMA_CR, I2C_IC_DMA_CR_RDMAE_Msk,
                I2C_IC_DMA_CR_RDMAE_Pos, 1);
            /* Включение аппаратного запроса DMA. */
            DMA_HardwareHandshakeEnable(handle->rx_dma, true,
                DMA_HANDSHAKE_I2C0_RX + instance * 2);
            if (desc_count != 0UL) {
                I2C_DMADescriptorInitRX(base, handle->rx_desc, desc_count,
                    transfer_size, DMA_Incr, xfer->data);
                DMA_SubmitChannelDescriptor(handle->rx_dma, handle->rx_desc);
                I2C_DMADescriptorInitTX(base, handle->tx_desc, desc_count,
                    transfer_size, DMA_NoChange, xfer->data);
                DMA_SubmitChannelDescriptor(handle->tx_dma, handle->tx_desc);
            } else {
                uint64_t xfercfg1 = DMA_CHANNEL_CTL(transfer_size,
                    0, 0,
                    DMA_PeripheralToMemory_DMA,
                    0, 0,
                    DMA_BurstSize1, DMA_BurstSize1,
                    DMA_NoChange, DMA_Incr,
                    DMA_Transfer8BitWidth, DMA_Transfer8BitWidth,
                    1
                );
                uint64_t xfercfg2 = DMA_CHANNEL_CTL(transfer_size,
                    0, 0,
                    DMA_MemoryToPeripheral_DMA,
                    0, 0,
                    DMA_BurstSize1, DMA_BurstSize1,
                    DMA_NoChange, DMA_NoChange,
                    DMA_Transfer32BitWidth, DMA_Transfer32BitWidth,
                    0
                );
                DMA_SubmitChannelTransferParameter(handle->rx_dma, xfercfg1,
                    (void *)&base->IC_DATA_CMD, xfer->data);
                DMA_SubmitChannelTransferParameter(handle->tx_dma, xfercfg2,
                    handle->dummy_data, (void *)&base->IC_DATA_CMD);
            }
            *handle->dummy_data = I2C_IC_DATA_CMD_CMD_Msk;
            DMA_StartTransfer(handle->rx_dma);
            DMA_StartTransfer(handle->tx_dma);
            break;

        default:
            break;
    }

    return I2C_Status_Ok;
}

void I2C_MasterTransferAbortDMA(I2C_Type *base, i2c_master_dma_handle_t *handle)
{
    assert(base != NULL);
    assert(handle != NULL);

    if (I2C_GetInstance(base) > 1) {
        return;
    }

    if (handle->state != (uint8_t) I2C_MTS_Idle) {
        /* Если канал DMA активен, прекращение транзакции. */
        if (DMA_ChannelIsActive(handle->tx_dma->base,
                handle->tx_dma->channel)) {
            DMA_AbortTransfer(handle->tx_dma);
        }
        /* Если контроллер активен, перезапуск контроллера. */
        if (base->IC_STATUS & I2C_IC_STATUS_ACTIVITY_Msk) {
            I2C_Enable(base, false);
            I2C_Enable(base, true);
        }
    }
}

/*!
 * @}
 */
