/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup jtm_driver Драйвер модуля JTM
 *
 * @brief Драйвер модуля встроенных датчиков температуры и напряжения
 *
 * Драйвер содержит функции для измерения температуры кристалла и напряжения в
 * контрольных точках с помощью АЦП и интегрированного датчика температуры
 *
 */

/*!
 * @addtogroup jtm_driver
 * @{
 */

/*!
 * @file jtm_can.h
 *
 * @brief Интерфейс драйвера модуля JTM
 */

#ifndef HAL_JTM_H
#define HAL_JTM_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"

/*!
 * @brief Версия драйвера CAN
 */
#define HAL_JTM_DRIVER_VERSION (MAKE_VERSION(0, 1, 0))

/*!
 * @brief Перечень параметров, значения которых можно прочитать с помощью драйвера JTM
 */
typedef enum
{
    JTM_Temperature = 0U, /*!< Температура */
    JTM_Vcasn       = 4U, /*!< Напряжение Vcasn */
    JTM_Vcore       = 7U, /*!< Напряжение питания ядра с вывода VDDC */
} jtm_parameter_t;

/*!
 * @brief Коды возврата функций драйвера JTM
 */
typedef enum 
{
    JTM_Status_Ok           = 0U, /*!< Успешно */
    JTM_Status_Fail         = 1U, /*!< Провал */
    JTM_Status_BadParameter = 2U, /*!< Неправильный параметр */
    JTM_Status_Busy         = 3U, /*!< Контроллер занят (идет преобразование) */
} jtm_status_t;

/*!
 * @brief Структура конфигурации JTM
 */
typedef struct 
{
    int16_t tcal;      /*!< Калибровочный параметр TCAL */
    int16_t wcal;      /*!< Калибровочный параметр WCAL */
    int16_t wtcalconf; /*!< Калибровочный параметр WTCALCONF */
    int16_t wtconf;    /*!< Калибровочный параметр WTCONF */
} jtm_config_t;

/*!
 * @brief Декларация типа дескриптора драйвера JTM
 */
typedef struct _jtm_handle jtm_handle_t;

/*!
 * @brief Функция обратного вызова JTM
 *
 * @param handle    Дескриптор драйвера JTM
 * @param parameter Параметр, прочитанный с помощью драйвера JTM
 * @param value     Значение параметра. Температура выдается в тысячных долях градуса Цельсия, напряжение - в мВ.
 * @param user_data Указатель на произвольные пользовательские данные
 */
typedef void (*jtm_callback_t)(jtm_handle_t *handle, jtm_parameter_t parameter,
    int32_t value, void *user_data);

/*!
 * @brief Структура обработчика событий JTM
 */
struct _jtm_handle {
    jtm_callback_t  callback;   /*!< Функция обратного вызова */
    jtm_parameter_t parameter;  /*!< Запрашиваемый параметр JTM */
    void            *user_data; /*!< Указатель на пользовательские данные */
};

/*!
 * @brief Инициализация драйвера JTM
 *
 * @param base      Базовый адрес контроллера
 * @param config    Структура с параметрами конфигурации
 */
void JTM_Init(JTM_Type *base, jtm_config_t *config);

/*!
 * @brief Блокирующее чтение параметра JTM
 *
 * Функция возвращает значение указанного параметра JTM, прочитанного с помощью
 * встроенного АЦП, через указатель в параметре value.Температура выдается
 * в тысячных долях градуса Цельсия, напряжение - в мВ.
 *
 * @param base      Базовый адрес контроллера
 * @param parameter Тип параметра для чтения
 * @param value     Значение параметра
 *
 * @retval #JTM_Status_Ok
 * @retval #JTM_Status_Busy
 * @retval #JTM_Status_BadParameter
 */
jtm_status_t JTM_GetParameterValue(JTM_Type *base, jtm_parameter_t parameter,
    int32_t *value);

/*!
 * @brief Инициализация обработчика событий JTM
 *
 * @param base      Базовый адрес контроллера
 * @param handle    Обработчик
 * @param callback  Функция обратного вызова
 * @param user_data Аргумент функции обратного вызова
 *
 * @retval #JTM_Status_Ok
 * @retval #JTM_Status_Fail
 */
jtm_status_t JTM_CreateHandle(JTM_Type *base, jtm_handle_t *handle,
    jtm_callback_t callback, void *user_data);

/*!
 * @brief Неблокирующее чтение параметра JTM
 *
 * Функция инициирует чтение указанного параметра JTM с помощью встроенного АЦП.
 * После завершения процедуры чтения вызывается функция обратного вызова, указанная
 * при создании дескриптора JTM.
 *
 * @param base      Базовый адрес контроллера
 * @param handle    Функция обратного вызова
 * @param parameter Тип параметра для чтения
 *
 * @retval #JTM_Status_Ok
 * @retval #JTM_Status_Busy
 * @retval #JTM_Status_BadParameter
 */
jtm_status_t JTM_GetParameterValueNonBlocking(JTM_Type *base,
    jtm_handle_t *handle, jtm_parameter_t parameter);

#ifdef __cplusplus
}
#endif

#endif /* HAL_JTM_H */

/*!
 * @}
 */
