/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup pwm_driver Драйвер модуля PWM
 *
 * @brief Драйвер широтно-импульсного модулятора
 *
 * Драйвер модуля широтно-импульсного модулятора управляет блоком генерации
 * широтно-импульсного модулированного сигнала
 */

/*!
 * @addtogroup pwm_driver
 * @{
 */

/*!
 * @file hal_pwm.h
 *
 * @brief Интерфейс драйвера модуля широтно-импульсного модулятора
 */

#ifndef HAL_PWM_H
#define HAL_PWM_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"

#define PWM_COUNT  (3) /*!< Количество блоков широтно-импульсного модулятора */

/*!
 * @brief Статусы драйвера широтно-импульсного модулятора
 */
enum pwm_status {
    PWM_Status_Ok              = 0, /*!< Нет ошибок */
    PWM_Status_InvalidArgument = 1, /*!< Недопустимый аргумент */
    PWM_Status_BadConfigure    = 2, /*!< Недопустимая конфигурация */
};

/*!
 * @brief Управление режимом работы предделителя канала
 */
enum pwm_prescaler_mode {
    pwm_PrescModeTimerIsRun = 0, /*!< Предделитель формирует частоту только при включенном таймере */
    pwm_PrescModeAlways     = 1, /*!< Предделитель формирует частоту не зависимо от включенности таймера */

    pwm_PrescMax   = pwm_PrescModeAlways,
};

/*!
 * @brief Управление состоянием предделителя канала
 */
enum pwm_prescaler_cmd {
    pwm_PrescCmdReset = 0, /*!< Счетчик предделителя сбрасывается в «0» */
    pwm_PrescCmdSave  = 1, /*!< Счетчик предделителя сохраняет состояние на момент останова */

    pwm_PrescalerMax   = pwm_PrescCmdSave,
};

/*!
 * @brief Управление пуском/остановкой канала
 */
enum pwm_run_command {
    pwm_RunCmdStop      = 0, /*!< Остановка после следующего переключения счетчика CTRCNT */
    pwm_RunCmdStopEvent = 1, /*!< Остановка при совершении следующих событий:
                                  up-count режим: остановка при CTRCNT==CTRPRD
                                  down-count режим: остановка при CTRCNT==0
                                  up-down-count режим: остановка при CTRCNT==0 */
    pwm_RunCmdRun       = 2, /*!< Запуск */

    pwm_RunCmdMax       = pwm_RunCmdRun,
};

/*!
 * @brief Управление мультиплексором делителя частоты (деление частоты после делителя)
 */
enum pwm_prescaler_divmux {
    pwm_PrescalerDivMux1      = 0, /*!< Частота от предделителя делится на 1 */
    pwm_PrescalerDivMux2      = 1, /*!< Частота от предделителя делится на 2 */
    pwm_PrescalerDivMux4      = 2, /*!< Частота от предделителя делится на 4 */
    pwm_PrescalerDivMux8      = 3, /*!< Частота от предделителя делится на 8 */
    pwm_PrescalerDivMux16     = 4, /*!< Частота от предделителя делится на 16 */
    pwm_PrescalerDivMuxPWMClk = 5, /*!< Используется внешняя частота PWM_CLK */

    pwm_PrescalerDivMuxMax    = pwm_PrescalerDivMuxPWMClk,
};

/*!
 * @brief Разрешения сброса предделителя при возникновения событий SYNCI или SWFSYNC
 */
enum pwm_prescaler_syncrst {
    pwm_PrescalerSyncRstDis   = 0, /*!< Сброс запрещен */
    pwm_PrescalerSyncRstEn    = 1, /*!< Сброс разрешен */

    pwm_PrescalerSyncRstEnMax = pwm_PrescalerSyncRstEn,
};

/*!
 * @brief Направление счета после синхронизации
 * @note Этот бит используется, только когда счетчик работает в up-down режиме.
 */
enum pwm_dirsync {
    pwm_DirSyncDown = 0, /*!< После синхронизации счетчик декрементируется */
    pwm_DirSyncUp   = 1, /*!< После синхронизации счетчик инкрементируется */

    pwm_DirSyncMax  = pwm_DirSyncUp,
};

/*!
 * @brief Выбор источника выходного сигнала SYNCO
 */
enum pwm_syncosel {
    pwm_SyncoSelSynci         = 0, /*!< SYNCI */
    pwm_SyncoSelCtrcntZero    = 1, /*!< Счетчик равен нулю (CTRCNT==0) */
    pwm_SyncoSelCtrcntEquCmpb = 2, /*!< Счетчик равен значению регистра сравнения B (CTRCNT==CMPB) */
    pwm_SyncoSelOff           = 3, /*!< SYNCO отключен */

    pwm_SyncoSelMax           = pwm_SyncoSelOff,
};

/*!
 * @brief Управление моментом переписи данных из теневого регистра периода в активный
 */
enum pwm_loadprd {
    pwm_LoadPrdRegister = 0, /*!< Регистр периода (CTRPRD) загружается из
                                  теневого регистра, когда счетчик (CTRCNT)
                                  равен нулю (запись или чтение CTRPRD
                                  осуществляется через теневой регистр) */
    pwm_LoadPrdDirect   = 1, /*!< CTRPRD загружается напрямую без использования
                                  теневого регистра (запись или чтение CTRPRD
                                  осуществляется напрямую) */

    pwm_LoadPrdMax      = pwm_LoadPrdDirect,
};

/*!
 * @brief Сигнал разрешения загрузки счетчика из регистра фазы
 */
enum pwm_syncphsen {
    pwm_SyncPhsEnDis = 0, /*!< Загрузка CTRCNT из регистра фазы CTRPHS запрещена */
    pwm_SyncPhsEnEn  = 1, /*!< Загрузка CTRCNT из регистра фазы CTRPHS во время синхронизации разрешена */

    pwm_SyncPhsEnMax = pwm_SyncPhsEnEn,
};

/*!
 * @brief Режим работы счетчика CRTCNT
 */
enum pwm_cntmode {
    pwm_CntModeUp     = 0, /*!< Up-count режим */
    pwm_CntModeDown   = 1, /*!< Down-count режим */
    pwm_CntModeUpDown = 2, /*!< Up-down-count режим */
    pwm_CntModeOff    = 3, /*!< Счет не осуществляется */

    pwm_CntModeMax    = pwm_CntModeOff,
};

/*!
 * @brief Режим работы регистра CMPx
 */
enum pwm_scmpxmode {
    pwm_SCmpxModeReg    = 0, /*!< Работа с теневым регистром, все запросы
                                  CPU проходят через теневой регистр */
    pwm_SCmpxModeDirect = 1, /*!< Прямой режим, используется только
                                  активный регистр CMPx */

    pwm_SCmpxModeMax = pwm_SCmpxModeDirect,
};

/*!
 * @brief Выбор режима загрузки данных из теневого регистра в активный CMPx
 */
enum pwm_ldxmode {
    pwm_LdxModeCtrcntZero            = 0, /*!< Загрузка при CTRCNT=0 */
    pwm_LdxModeCtrcntEquCtrprd       = 1, /*!< Загрузка при CTRCNT=CTRPRD */
    pwm_LdxModeCtrcntZeroOrEquCtrprd = 2, /*!< Загрузка при CTRCNT=0 или CTRCNT=CTRPRD */
    pwm_LdxModeNoLoad                = 3, /*!< Загрузка не осуществляется */

    pwm_LdxModeMax                   = pwm_LdxModeNoLoad,
};

/*!
 * @brief Управление выходом OUTx
 */
enum pwm_outx_cmd {
    pwm_OutxCmdNo     = 0, /*!< Не выполнять действий */
    pwm_OutxCmdClear  = 1, /*!< Сбросить OUTx в 0 */
    pwm_OutxCmdSet    = 2, /*!< Установить OUTx в 1 */
    pwm_OutxCmdToggle = 3, /*!< Инвертировать OUTx */

    pwm_OutxCmdMax    = pwm_OutxCmdToggle,
};

/*!
 * @brief Механизм загрузки активного регистра из теневого регистра для регистра программного управления выходами
 */
enum pwm_ldcswrf {
    pwm_LdcswrfCtrcntZero          = 0, /*!< Загрузка при CTRCNT=0 */
    pwm_LdcswrfCtrcntEquCtrprd     = 1, /*!< Загрузка при CTRCNT=CTRPRD */
    pwm_LdcswrfCtrcntZeroEquCtrprd = 2, /*!< Загрузка при CTRCNT=0 или CTRCNT=CTRPRD */
    pwm_LdcswrfDirect              = 3, /*!< Загружать напрямую при обращении CPU без использования теневого регистра */

    pwm_LdcswrfMax                 = pwm_LdcswrfDirect,
};

/*!
 * @brief Источник сигнала для генерации запрещенной зоны
 */
enum pwm_dz_signal {
    pwm_DzSignalOutA = 0, /*!< Сигнал OUTA in */
    pwm_DzSignalOutB = 1, /*!< Сигнал OUTB in */

    pwm_DzSignalMax  = pwm_DzSignalOutB,
};

/*!
 * @brief Полярность OUTx после генерации запрещенной зоны
 */
enum pwm_dz_outx_inv {
    pwm_DzSignalOutxInvOff = 0, /*!< Нет инверсии */
    pwm_DzSignalOutxInvOn  = 1, /*!< Есть инверсия */

    pwm_DzSignalOutxInvMax = pwm_DzSignalOutxInvOn,
};

/*!
 * @brief Выбор режима работы блока запрещенной зоны при формирования OUTx
 */
enum pwm_dz_mode {
    pwm_DzModeOff = 0, /*!< Генератор выключен */
    pwm_DzModeOn  = 1, /*!< Генератор включен */

    pwm_DzModeMax = pwm_DzModeOn,
};

/*!
 * @brief Скважность дробящего сигнала
 */
enum pwm_chopper_duty {
    pwm_ChopperDuty_1_8 = 0, /*!< 1/8 */
    pwm_ChopperDuty_2_8 = 1, /*!< 2/8 */
    pwm_ChopperDuty_3_8 = 2, /*!< 3/8 */
    pwm_ChopperDuty_4_8 = 3, /*!< 4/8 */
    pwm_ChopperDuty_5_8 = 4, /*!< 5/8 */
    pwm_ChopperDuty_6_8 = 5, /*!< 6/8 */
    pwm_ChopperDuty_7_8 = 6, /*!< 7/8 */

    pwm_chopper_dutyMax = pwm_ChopperDuty_7_8,
};

/*!
 * @brief Частота дробящего сигнала
 */
enum pwm_chopper_freq {
    pwm_ChopperFreqClk_8  = 0, /*!< CLK/8  */
    pwm_ChopperFreqClk_16 = 1, /*!< CLK/16 */
    pwm_ChopperFreqClk_24 = 2, /*!< CLK/24 */
    pwm_ChopperFreqClk_32 = 3, /*!< CLK/32 */
    pwm_ChopperFreqClk_40 = 4, /*!< CLK/40 */
    pwm_ChopperFreqClk_48 = 5, /*!< CLK/48 */
    pwm_ChopperFreqClk_56 = 6, /*!< CLK/56 */
    pwm_ChopperFreqClk_64 = 7, /*!< CLK/64 */

    pwm_ChopperFreqMax    = pwm_ChopperFreqClk_64,
};

/*!
 * @brief Ширина первого импульса
 */
enum pwm_chopper_first_width {
    pwm_ChopperFirstWidth_0_8  = 0,  /*!< 0xCLK/8 */
    pwm_ChopperFirstWidth_1_8  = 1,  /*!< 1xCLK/8 */
    pwm_ChopperFirstWidth_2_8  = 2,  /*!< 2xCLK/8 */
    pwm_ChopperFirstWidth_3_8  = 3,  /*!< 3xCLK/8 */
    pwm_ChopperFirstWidth_4_8  = 4,  /*!< 4xCLK/8 */
    pwm_ChopperFirstWidth_5_8  = 5,  /*!< 5xCLK/8 */
    pwm_ChopperFirstWidth_6_8  = 6,  /*!< 6xCLK/8 */
    pwm_ChopperFirstWidth_7_8  = 7,  /*!< 7xCLK/8 */
    pwm_ChopperFirstWidth_8_8  = 8,  /*!< 8xCLK/8 */
    pwm_ChopperFirstWidth_9_8  = 9,  /*!< 9xCLK/8 */
    pwm_ChopperFirstWidth_10_8 = 10, /*!< 10xCLK/8 */
    pwm_ChopperFirstWidth_11_8 = 11, /*!< 11xCLK/8 */
    pwm_ChopperFirstWidth_12_8 = 12, /*!< 12xCLK/8 */
    pwm_ChopperFirstWidth_13_8 = 13, /*!< 13xCLK/8 */
    pwm_ChopperFirstWidth_14_8 = 14, /*!< 14xCLK/8 */
    pwm_ChopperFirstWidth_15_8 = 15, /*!< 15xCLK/8 */

    pwm_ChopperFirstWidthMax   = pwm_ChopperFirstWidth_15_8,
};

/*!
 * @brief Работа блока Chopper
 */
enum pwm_chopper_work {
    pwm_ChopperWorkOff = 0, /*!< Выключена функция дробления */
    pwm_ChopperWorkOn  = 1, /*!< Включена функция дробления */

    pwm_ChopperWorkMax = pwm_ChopperWorkOn,
};

/*!
 * @brief Работа блока trip unit
 */
enum pwm_trip_unit_signal {
    pwm_TripUnitSignalNotUsed = 0, /*!< Вход не используется */
    pwm_TripUnitSignalUsed    = 1, /*!< Вход используется */

    pwm_TripUnitMax           = pwm_TripUnitSignalUsed,
};

/*!
 * @brief Реакции на событие блока trip unit
 */
enum pwm_trip_unit_action {
    pwm_TripUnitActionHigh = 0, /*!< OUTx переводится в высокоомное состояние */
    pwm_TripUnitActionOne  = 1, /*!< OUTx подтягивается к 1 */
    pwm_TripUnitActionZero = 2, /*!< OUTx подтягивается к 0 */
    pwm_TripUnitActionNo   = 3, /*!< Действий не производится */

    pwm_TripUnitActionMax  = pwm_TripUnitActionNo,
};

/*!
 * @brief Разрешение прерывания блока
 */
enum pwm_int_en {
    pwm_IntEnNo  = 0, /*!< Прерывание запрещено */
    pwm_IntEnYes = 1, /*!< Прерывание разрешено */

    pwm_IntEnMax = pwm_IntEnYes,
};

/*!
 * @brief Выбор периода прерываний PWM_INT
 */
enum pwm_eventprd {
    pwm_EventPrdNo    = 0, /*!< Нет генерации прерывания */
    pwm_EventPrdOne   = 1, /*!< Генерация прерывания каждый раз при возникновении события */
    pwm_EventPrdTwo   = 2, /*!< Генерация прерывания каждый второй раз при возникновении события */
    pwm_EventPrdThree = 3, /*!< Генерация прерывания каждый третий раз при возникновении события */

    pwm_EventPrd      = pwm_EventPrdThree,
};

/*!
 * @brief Источник прерывания
 */
enum pwm_int_source {
    pwm_IntSourceNo               = 0, /*!< Резерв */
    pwm_IntSourceCtrcntEquZero    = 2, /*!< Признак равенства счетчика нулю CTRCNT=0 */
    pwm_IntSourceCtrcntEquCtrprd  = 3, /*!< Признак равенства счетчика периоду счета CTRCNT=CTRPRD */
    pwm_IntSourceCtrcntEquCmpAInc = 4, /*!< Признак равенства счетчика регистру сравнения CMPA во время инкремента */
    pwm_IntSourceCtrcntEquCmpADec = 5, /*!< Признак равенства счетчика регистру сравнения CMPA во время декремента */
    pwm_IntSourceCtrcntEquCmpBInc = 6, /*!< Признак равенства счетчика регистру сравнения CMPB во время инкремента */
    pwm_IntSourceCtrcntEquCmpBDec = 7, /*!< Признак равенства счетчика регистру сравнения CMPB во время декремента */

    pwm_IntSourceMax              = pwm_IntSourceCtrcntEquCmpBDec,
};

/*!
 * @brief Конфигурация канала широтно-импульсного модулятора
 */
struct pwm_channel_config {
    uint32_t channel;                                       /*!< Конфигурируемый канал */
    /* Группа 1 */
    /* Предделитель */
    enum pwm_prescaler_mode      prescaler_mode;            /*!< Режим управления  */
    enum pwm_prescaler_cmd       prescaler_cmd;             /*!< Режим сброса счетчика предделителя */
    enum pwm_prescaler_syncrst   prescaler_syncrst;         /*!< Режим сброса при событиях */
    uint8_t                      prescaler;                 /*!< Предделитель (PRESPRD) */
    enum pwm_prescaler_divmux    prescaler_divmux;          /*!< Мультиплексор чатоты */
    /* Основной счетчик */
    enum pwm_cntmode             cntmode;                   /*!< Режим работы */
    uint32_t                     counter;                   /*!< Значение счетчика */
    uint32_t                     period;                    /*!< Период */
    enum pwm_loadprd             loadprd;                   /*!< Моментом переписи данных */
    uint32_t                     ctrphs;                    /*!< Фазы синхронизации блока */
    enum pwm_syncphsen           syncphsen;                 /*!< Загрузки счетчика из регистра фазы */
    /* Блок сравнения */
    uint32_t                     cmpa;                      /*!< Регистр сравнения CMPA */
    uint32_t                     cmpb;                      /*!< Регистр сравнения CMPB */
    enum pwm_scmpxmode           scmpamode;                 /*!< Режим работы регистра CMPA */
    enum pwm_scmpxmode           scmpbmode;                 /*!< Режим работы регистра CMPB */
    enum pwm_ldxmode             ldamode;                   /*!< режима загрузки данных из теневого регистра в активный CMPA */
    enum pwm_ldxmode             ldbmode;                   /*!< режима загрузки данных из теневого регистра в активный CMPB */
    /* Группа 2 */
    /* Блок реакции на событие */
    enum pwm_outx_cmd            cnt_eq_prd_outa;           /*!< Событие канала OUTA по совпадению счетчика и периода */
    enum pwm_outx_cmd            cnt_eq_prd_outb;           /*!< Событие канала OUTB по совпадению счетчика и периода */
    enum pwm_outx_cmd            cnt_eq_cmpa_dec_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPA при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpa_inc_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPA при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpa_dec_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPA при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpa_inc_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPA при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_dec_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPB при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_inc_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPB при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_dec_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPB при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_inc_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPB при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_zero_outa;          /*!< Событие канала OUTA по достижению счетчиком 0 */
    enum pwm_outx_cmd            cnt_eq_zero_outb;          /*!< Событие канала OUTB по достижению счетчиком 0 */
    enum pwm_outx_cmd            sw_forced_outa;            /*!< Событие канала OUTA при кратковременном программном событии */
    enum pwm_outx_cmd            sw_forced_outb;            /*!< Событие канала OUTB при кратковременном программном событии */
    enum pwm_outx_cmd            sw_forced_long_outa;       /*!< Событие канала OUTA при долговременном программном событии */
    enum pwm_outx_cmd            sw_forced_long_outb;       /*!< Событие канала OUTB при долговременном программном событии */
    enum pwm_ldcswrf             ldcswrf;                   /*!< Загрузка активного регистра из теневого регистра для регистра программного управления выходами */
    /* Блок прерываний */
    enum pwm_int_en              pwm_int_enable;            /*!< Разрешение прерывания */
    enum pwm_int_source          pwm_int_source;            /*!< Источник прерывания */
    enum pwm_eventprd            eventprd;                  /*!< Период прерывания */
    /* Группа 3 */
    /* Генератор запретной зоны */
    uint16_t                     dz_rising_edge_delay_clk;  /*!< Запратная зона в тактах около возрастающего фронта */
    uint16_t                     dz_falling_edge_delay_clk; /*!< Запратная зона в тактах около ниспадающего фронта */
    enum pwm_dz_signal           dz_rising_edge_source;     /*!< Источник сигнала для генерации запрещенной зоны около возрастающего фронта */
    enum pwm_dz_signal           dz_falling_edge_source;    /*!< Источник сигнала для генерации запрещенной зоны около ниспадающего фронта */
    enum pwm_dz_outx_inv         dz_rising_edge_outa_inv;   /*!< Полярность OUTA после генерации запрещенной зоны */
    enum pwm_dz_outx_inv         dz_falling_edge_outb_inv;  /*!< Полярность OUTB после генерации запрещенной зоны */
    enum pwm_dz_mode             dz_outa_enable;            /*!< Выбор режима работы блока запрещенной зоны при формирования OUTA */
    enum pwm_dz_mode             dz_outb_enable;            /*!< Выбор режима работы блока запрещенной зоны при формирования OUTB */
    /* Блок дробления выходного сигнала */
    enum pwm_chopper_duty        chopper_duty;              /*!< Скважность дробящего сигнала */
    enum pwm_chopper_freq        chopper_freq;              /*!< Частота дробящего сигнала */
    enum pwm_chopper_first_width chopper_first_width;       /*!< Ширина первого импульса */
    enum pwm_chopper_work        chopper_work;              /*!< Работа блока дробления */
    /* Блок реакции на внешнее воздействие */
    uint8_t                      inputs_mask_one;           /*!< Выбор используемых TU[7:0] сигналов для канала PWM работающих в режиме однократного срабатывания */
    uint8_t                      inputs_mask_mult;          /*!< Выбор используемых TU[7:0] сигналов для канала PWM работающих в режиме многократного срабатывания */
    enum pwm_trip_unit_action    trip_unit_action_outa;     /*!< Реакции на событие блока trip unit для OUTA */
    enum pwm_trip_unit_action    trip_unit_action_outb;     /*!< Реакции на событие блока trip unit для OUTB */
    enum pwm_int_en              pwmtu_int_one;             /*!< Разрешение прерывания блока */
    enum pwm_int_en              pwmtu_int_mult;            /*!< Разрешение прерывания блока */

    /* Запуск/останов канала*/
    enum pwm_run_command cmd;                               /*!< Разрешение работы */
};

/*!
 * @name Интерфейс драйвера
 * @{
 */

/*!
 * @brief Инициализация структуры "по умолчанию" для канала блока ШИМ
 *
 * @param cfg   Конфигурация
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 */
enum pwm_status PWM_GetChannelDefaultConfig(struct pwm_channel_config * cfg);

/*!
 * @brief Инициализация канала блока широтно-импульсного модулятора
 *
 * @param base Блок широтно-импульсного модулятора
 * @param cfg  Конфигурация
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 */
enum pwm_status PWM_InitChannel(PWM_Type *base,
    struct pwm_channel_config cfg);

/*!
 * @brief Деинициализация блока широтно-импульсного модулятора
 *
 * @param base Блок широтно-импульсного модулятора
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 */
enum pwm_status PWM_Deinit(PWM_Type *base);

/*!
 * @brief Запуск/останов канала блока широтно-импульсного модулятора
 *
 * @param base    Блок широтно-импульсного модулятора
 * @param channel Канал
 * @param cmd     Команда запуска для канала широтно-импульсного модулятора
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 */
enum pwm_status PWM_Enable(PWM_Type *base, uint32_t channel, enum pwm_run_command cmd);

/*!
 * @brief Запуск/останов всех каналов блока широтно-импульсного модулятора
 *
 * @param base         Блок широтно-импульсного модулятора
 * @param channel_mask Каналы - 1 бит означает управление каналом
 * @param cmd0         Команда запуска 0 канала широтно-импульсного модулятора
 * @param cmd1         Команда запуска 1 канала широтно-импульсного модулятора
 * @param cmd2         Команда запуска 2 канала широтно-импульсного модулятора
 * @param cmd3         Команда запуска 3 канала широтно-импульсного модулятора
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 */
enum pwm_status PWM_CmdForAllChannels(PWM_Type *base,
    uint32_t channel_mask,
    enum pwm_run_command cmd0,
    enum pwm_run_command cmd1,
    enum pwm_run_command cmd2,
    enum pwm_run_command cmd3);

/*!
 * @brief Программная не длительная установка значения выходов канала
 *
 * @note  В отличие от длительной приводит к переключению значения выходов
 *        которое будет изменено аппаратным событием. Значения устанавливаются
 *        ранее предопределённые в конфигурации.
 *
 * @param base      Блок широтно-импульсного модулятора
 * @param channel   Канал
 * @param mask_outs Маски выводов: бит 0 - OutA; 1 - OutB
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 * @retval #PWM_Status_BadConfigure
 */
enum pwm_status PWM_ApplySoftOuts(PWM_Type *base, uint32_t channel,
    int8_t mask_outs);

/*!
 * @brief Программная длительная установка значения выходов канала
 *
 * @param base      Блок широтно-импульсного модулятора
 * @param channel   Канал
 * @param mask_outs Маски выводов: бит 0 - OutA; 1 - OutB
 * @param outa      Значение вывода OutA
 * @param outb      Значение вывода OutB
 *
 * @note Значения pwm_OutxCmdToggle для outa и outb имеют эффект
 *       как pwm_OutxCmdNo.
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 * @retval #PWM_Status_BadConfigure
 */
enum pwm_status PWM_ApplyLongSoftOuts(PWM_Type *base, uint32_t channel,
    int8_t mask_outs, enum pwm_outx_cmd outa, enum pwm_outx_cmd outb);

/*!
 * @brief Определяет работает ли счетчик в канале
 *
 * @param base      Блок широтно-импульсного модулятора
 * @param channel   Канал
 *
 * @retval 1 - Счетчик работает
 * @retval 0 - Счетчик не работает
 */
uint32_t PWM_GetCntStat(PWM_Type *base, uint32_t channel);

/*!
 * @brief Устанавливает значение периода
 *
 * @param base      Блок широтно-импульсного модулятора
 * @param channel   Канал
 * @param period    Период в клоках частоты тактирования
 *
 * @retval #PWM_Status_Ok
 * @retval #PWM_Status_InvalidArgument
 * @retval #PWM_Status_BadConfigure
 */
enum pwm_status PWM_SetPeriod(PWM_Type *base, uint32_t channel,
    uint32_t period);
/*!
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* HAL_PWM_H */

/*!
 * @}
 */
