/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup qspi_driver Драйвер модуля QSPI
 *
 * @brief Драйвер контроллера QSPI
 *
 * Драйвер поддерживает обмен данными с устройствами, подключенными к выходам
 * QSPI в режимах Normal, Dual, Quad SPI и шириной поля данных от 4 до 32 битов.
 */

/*!
 * @addtogroup qspi_driver
 * @{
 */

/*!
 * @file hal_qspi.h
 *
 * @brief Интерфейс драйвера модуля QSPI
 */

#ifndef HAL_QSPI_H
#define HAL_QSPI_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"

/*!
 * @brief Режим работы контроллера QSPI
 */
typedef enum _qspi_qmode {
    QSPI_NormalSPI = 0x0, /*!< Стандартный режим SPI */
    QSPI_DualSPI   = 0x2, /*!< DUAL SPI */
    QSPI_QuadSPI   = 0x3  /*!< QUAD SPI*/
} qspi_qmode_t;

/*!
 * @brief Количество бит во фрейме
 */
#if defined(QSPI_DRIVER_VERSION_1_6_0)
typedef enum _qspi_bit_size_t {
    QSPI_FRAME_BITS_4  = 0x03, /*!< 4 бита */
    QSPI_FRAME_BITS_5  = 0x04, /*!< 5 бит */
    QSPI_FRAME_BITS_6  = 0x05, /*!< 6 бит */
    QSPI_FRAME_BITS_7  = 0x06, /*!< 7 бит */
    QSPI_FRAME_BITS_8  = 0x07, /*!< 8 бит */
    QSPI_FRAME_BITS_9  = 0x08, /*!< 9 бит */
    QSPI_FRAME_BITS_10 = 0x09, /*!< 10 бит */
    QSPI_FRAME_BITS_11 = 0x0A, /*!< 11 бит */
    QSPI_FRAME_BITS_12 = 0x0B, /*!< 12 бит */
    QSPI_FRAME_BITS_13 = 0x0C, /*!< 13 бит */
    QSPI_FRAME_BITS_14 = 0x0D, /*!< 14 бит */
    QSPI_FRAME_BITS_15 = 0x0E, /*!< 15 бит */
    QSPI_FRAME_BITS_16 = 0x0F, /*!< 16 бит */
    QSPI_FRAME_BITS_17 = 0x10, /*!< 17 бит */
    QSPI_FRAME_BITS_18 = 0x11, /*!< 18 бит */
    QSPI_FRAME_BITS_19 = 0x12, /*!< 19 бит */
    QSPI_FRAME_BITS_20 = 0x13, /*!< 20 бит */
    QSPI_FRAME_BITS_21 = 0x14, /*!< 21 бит */
    QSPI_FRAME_BITS_22 = 0x15, /*!< 22 бита */
    QSPI_FRAME_BITS_23 = 0x16, /*!< 23 бита */
    QSPI_FRAME_BITS_24 = 0x17, /*!< 24 бита */
    QSPI_FRAME_BITS_25 = 0x18, /*!< 25 бит */
    QSPI_FRAME_BITS_26 = 0x19, /*!< 26 бит */
    QSPI_FRAME_BITS_27 = 0x1A, /*!< 27 бит */
    QSPI_FRAME_BITS_28 = 0x1B, /*!< 28 бит */
    QSPI_FRAME_BITS_29 = 0x1C, /*!< 29 бит */
    QSPI_FRAME_BITS_30 = 0x1D, /*!< 30 бит */
    QSPI_FRAME_BITS_31 = 0x1E, /*!< 31 бит */
    QSPI_FRAME_BITS_32 = 0x1F  /*!< 32 бита */
} qspi_bit_size_t;
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
typedef enum _qspi_bit_size_t {
    QSPI_FRAME_BITS_4  = 0x00, /*!< 4 бита */
    QSPI_FRAME_BITS_8  = 0x01, /*!< 8 бит */
    QSPI_FRAME_BITS_12 = 0x02, /*!< 12 бит */
    QSPI_FRAME_BITS_16 = 0x03, /*!< 16 бит */
    QSPI_FRAME_BITS_20 = 0x04, /*!< 20 бит */
    QSPI_FRAME_BITS_24 = 0x05, /*!< 24 бита */
    QSPI_FRAME_BITS_28 = 0x06, /*!< 28 бит */
    QSPI_FRAME_BITS_32 = 0x07  /*!< 32 бита */
} qspi_bit_size_t;
#else
    #error "Please define QSPI controller version"
#endif

#if defined(QSPI_DRIVER_VERSION_1_6_0)
/*!
 * @brief Выбор формата SPI 
 */
typedef enum _qspi_format_spi_t {
    QSPI_FORMAT_MOTOROLA  = 0x00, /*!< Motorola SPI */
    QSPI_FORMAT_TI        = 0x02, /*!< Texas Instruments Synchronous Serial Frame(только одна линия) */
    QSPI_FORMAT_MICROWIRE = 0x03  /*!< National Microwire Frame(только одна линия) */
} qspi_format_spi_t;
#endif

/*!
 * @brief Структура, определяющая параметры конфигурации контроллера QSPI
 */
typedef struct _qspi_config {
    uint32_t delay_en;          /*!< Включение задержки между передачами для работы в режиме Master */
    uint32_t cpol;              /*!< Полярность тактового сигнала */
    uint32_t cpha;              /*!< Фаза тактового сигнала */
    uint32_t msb;               /*!< Порядок передачи битов */
    uint32_t cont_trans_en;     /*!< Бит непрерывной передачи */
    uint16_t cont_transfer_ext; /*!< Бит продление непрерывной передачи */
    qspi_qmode_t spi_mode;      /*!< Режим работы SPI */
    uint32_t slave_select;      /*!< Выбор slave-устройства */
    uint32_t slave_pol;         /*!< Полярность сигнала SS */
    qspi_bit_size_t bit_size;   /*!< Количество битов в передаче */
    uint32_t mode;              /*!< Режим работы контроллера (Master/Slave) */
    uint32_t dma_en;            /*!< Включение режима DMA */
    uint32_t inhibit_din;       /*!< Запрет записи в RX FIFO */
    uint32_t inhibit_dout;      /*!< Запрет чтение из TX FIFO */
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    uint32_t xfer_format;       /*!< Формат SPI */
#endif
} qspi_config_t;

/*!
 * @brief Структура параметров конфигурации XIP контроллера QSPI
 */
typedef struct _qspi_xip_config {
    uint32_t cmd;               /*!< Тип команды чтения режима XIP */
    uint32_t hpen;              /*!< Включение режима высокой производительности */
    uint32_t cpha;              /*!< Фаза тактового сигнала */
    uint32_t cpol;              /*!< Полярность тактового сигнала */
    uint32_t addr4;             /*!< 4-байтовый режим адресов */
    uint32_t le32;              /*!< 32-битная организация данных */
    uint32_t hp_mode;           /*!< Командный байт режима XIP */
    uint32_t dummy_cycles;      /*!< Количество dummy тактов */
    uint32_t hp_end_dummy;      /*!< Количество dummy тактов для выхода из HP режима */
} qspi_xip_config_t;

/*!
 * @brief Получение номера блока QSPI
 *
 * @param base Адрес QSPI
 * @return Номер блока QSPI
 */
uint32_t QSPI_GetInstance(QSPI_Type *base);

/*!
 * @brief Получение конфигурации QSPI по умолчанию
 *
 * @param config Конфигурационная структура QSPI
 */
void QSPI_GetDefaultConfig(qspi_config_t *config);

/*!
 * @brief Инициализация контроллера QSPI
 *
 * @param base   Базовый адрес контроллера QSPI
 * @param config Структура с настройками контроллера по умолчанию
 */
void QSPI_Init(QSPI_Type *base, const qspi_config_t *config);

/*!
 * @brief Установка количества передаваемых бит
 *
 * @param base     Базовый адрес контроллера QSPI
 * @param bit_size Количество передаваемых бит
 */
void QSPI_SetBitSize(QSPI_Type *base, qspi_bit_size_t bit_size);

#if defined(QSPI_DRIVER_VERSION_1_6_0)
/*!
 * @brief Установка формата SPI
 *
 * @param base        Базовый адрес контроллера QSPI
 * @param xfer_format Формат SPI
 */
void QSPI_SetFormatSPI(QSPI_Type *base, qspi_format_spi_t xfer_format);
#endif

/*!
 * @brief Установка режима SPI
 *
 * @param base     Базовый адрес контроллера QSPI
 * @param spi_mode Режим SPI
 */
void QSPI_SetQMode(QSPI_Type *base, qspi_qmode_t spi_mode);

/*!
 * @brief Установка запрета записи в Tx FIFO
 *
 * @param base        Базовый адрес контроллера QSPI
 * @param inhibit_din Запрет (1) или нет запрета (0) на запись
 */
void QSPI_SetInhibitDin(QSPI_Type *base, bool inhibit_din);

/*!
 * @brief Установка запрета чтения из Rx FIFO
 *
 * @param base         Базовый адрес контроллера QSPI
 * @param inhibit_dout Запрет (1) или нет запрета (0) на чтение
 */
void QSPI_SetInhibitDout(QSPI_Type *base, bool inhibit_dout);

/*!
 * @brief Включение контроллера QSPI
 *
 * @param base Базовый адрес контроллера QSPI
 */
static inline void QSPI_Enable(QSPI_Type *base)
{
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    base->ENABLE = QSPI_ENABLE_ENABLEREQ_Msk;

    while (base->ENABLE != QSPI_ENABLE_ENABLEREQ_Msk)
        ;
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
    base->ENABLE = QSPI_ENABLE_ENABLE_Msk;

    while (base->ENABLE != QSPI_ENABLE_ENABLE_Msk)
        ;
#else
    #error "Please define QSPI controller version"
#endif
}

/*!
 * @brief Деинициализация контроллера QSPI
 *
 * @param base Базовый адрес контроллера QSPI
 */
static inline void QSPI_DeInit(QSPI_Type *base)
{
    base->ENABLE = 0x0;

    while (base->ENABLE != 0)
        ;
}

/*!
 * @brief Включение DMA
 *
 * @param base Базовый адрес контроллера QSPI
 */
static inline void QSPI_EnableDMA(QSPI_Type *base)
{
    base->CTRL |= QSPI_CTRL_DMA_Msk;
}

/*!
 * @brief Выключение DMA
 *
 * @param base Базовый адрес контроллера QSPI
 */
static inline void QSPI_DisableDMA(QSPI_Type *base)
{
    base->CTRL &= ~QSPI_CTRL_DMA_Msk;
}

/*!
 * @brief Получение значения статусного регистра
 *
 * @param base Базовый адрес контроллера QSPI
 * @return Значение регистра STAT
 */
static inline uint32_t QSPI_GetStatusFlag(QSPI_Type *base)
{
    return base->STAT;
}

/*!
 * @brief Переключение ведомого устройства
 *
 * @param base         Базовый адрес контроллера QSPI
 * @param slave_select Выбор ведомого устройства по битовой маске
 */
static inline void QSPI_SetSlaveSelect(QSPI_Type *base, uint32_t slave_select)
{
    base->SS = slave_select;
}

/*!
 * @brief Включение прерываний
 *
 * @param base Базовый адрес контроллера QSPI
 * @param mask Маска прерываний
 */
static inline void QSPI_EnableInterrupt(QSPI_Type *base, uint32_t mask)
{
    base->INTR_EN |= mask;
}

/*!
 * @brief Отключение прерываний
 *
 * @param base Базовый адрес контроллера QSPI
 * @param mask Маска прерываний
 */
static inline void QSPI_DisableInterrupt(QSPI_Type *base, uint32_t mask)
{
    base->INTR_EN &= ~mask;
}

/*!
 * @brief Сброс прерываний
 *
 * @param base Базовый адрес контроллера QSPI
 * @param mask Маска прерываний
 */
static inline void QSPI_ClearInterrupt(QSPI_Type *base, uint32_t mask)
{
    base->INTR_CLR = mask;
}

/*!
 * @brief Передача 32-битного слова в Tx FIFO
 *
 * @param base Базовый адрес контроллера QSPI
 * @param data 32-битное слово для передачи
 */
static inline void QSPI_WriteData(QSPI_Type *base, uint32_t data)
{
    base->TX_DATA = data;
}

/*!
 * @brief Передача байта данных в Tx FIFO
 *
 * @param base Базовый адрес контроллера QSPI
 * @param data Байт данных для передачи
 */
static inline void QSPI_WriteDataByte(QSPI_Type *base, uint8_t data)
{
    *((volatile uint8_t *)(&(base->TX_DATA))) = data;
}

/*!
 * @brief Чтение 32-битного слова из Rx FIFO
 *
 * @param base Базовый адрес контроллера QSPI
 * @return Считанное 32-битное слово
 */
static inline uint32_t QSPI_ReadData(QSPI_Type *base)
{
    return base->RX_DATA;
}

/*!
 * @brief Чтение байта данных из Rx FIFO
 *
 * @param base Базовый адрес контроллера QSPI
 * @return Считанный байт данных
 */
static inline uint8_t QSPI_ReadDataByte(QSPI_Type *base)
{
    return *((volatile uint8_t *)(&(base->RX_DATA)));
}

/*!
 * @brief Чтение уровня заполнения буфера передачи Tx FIFO
 *
 * @param base Базовый адрес контроллера QSPI
 * @return Количество фреймов данных в буфере передачи
 */
static inline uint32_t QSPI_GetTXLVL(QSPI_Type *base)
{
    return base->TX_FIFO_LVL;
}

/*!
 * @brief Чтение уровня заполнения буфера приема Rx FIFO
 *
 * @param base Базовый адрес контроллера QSPI
 * @return Количество фреймов данных в буфере приема
 */
static inline uint32_t QSPI_GetRXLVL(QSPI_Type *base)
{
    return base->RX_FIFO_LVL;
}

#ifdef __cplusplus
}
#endif

#endif /* HAL_QSPI_H */

/*!
 * @}
 */
