/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup rwc_driver
 * @{
 */

/*!
 * @file hal_rwc.c
 *
 * @brief Имплементация драйвера модуля RWC
 */

#include "hal_rwc.h"

/*!
 * @name Макросы значений исходного состояния таймера
 * @{
 */
#define RWC_CONFIG_DEFAULT      (0x80006021U) /*!< Значение регистра CONFIG по умолчанию */
#define RWC_TRIM_DEFAULT        (0x84019800U) /*!< Значение регистра TRIM по умолчанию */
#define RWC_WAKE_CONFIG_DEFAULT (0x0U)        /*!< Значение регистра WAKE_CONFIG по умолчанию */
#define RWC_TRIM_LOAD_CMD       (0x0U)        /*!< Значение регистра TRIM по умолчанию */
/*!
 * @}
 */

static enum rwc_status last_status = RWC_Status_Ok;

/*!
 * @brief Преобразование месяца в количество дней в нём
 *
 * @note Год используется для проверки на високосность и коррекции
 *       дней в феврале високосного года.
 *
 * @param month Месяц
 * @param year  Год
 *
 * @return Количество дней в переданном месяце указанного года
 */
static uint8_t RWC_ConvertMonthToDays(const uint8_t month, const uint16_t year);

/*!
 * @brief Проверка года на високосность
 *
 * @param year Проверяемый год
 *
 * @retval 1 Если год является високосным
 * @retval 0 Если год не является високосным
 */
static uint8_t RWC_CheckLeapYear(const uint16_t year);

/*!
 * @brief Проверка введенных дат и времени на корректность
 *
 * @param datetime Указатель на структуру, в которой хранятся сведения
 *                 о дате и времени
 *
 * @return Возвращает значение false, если сведения о дате и времени находятся
 *         вне диапазона; значение true, если находятся в диапазоне
 */
static bool RWC_CheckDatetimeFormat(const rtc_datetime_t *datetime);

/*!
 * @brief Преобразование данных из даты/времени в секунды
 *
 * @param datetime Указатель на структуру даты и времени, в которой хранятся
 *                 сведения о дате и времени
 *
 * @return Результат преобразования в секундах
 */
uint32_t RWC_ConvertDatetimeToSeconds(const rtc_datetime_t *datetime);

/*!
 * @brief Преобразование данных о времени из секунд в структуру даты и времени
 *
 * @param seconds  Значение секунд, которое необходимо преобразовать в формат
 *                 даты и времени
 * @param datetime Указатель на структуру даты и времени, в которой хранится
 *                 результат преобразования
 */
static void RWC_ConvertSecondsToDatetime(uint32_t seconds,
    rtc_datetime_t *datetime);

static bool RWC_CheckDatetimeFormat(const rtc_datetime_t *datetime)
{
    uint8_t leap_year = 0U;
    assert(datetime);

    /* Проверка года, месяца, часа, минуты, секунды. */
    if ((datetime->year < YEAR_RANGE_START)
        || (datetime->year > YEAR_RANGE_END)
        || (datetime->month > 12U)
        || (datetime->month < 1U)
        || (datetime->hour >= 24U)
        || (datetime->minute >= 60U)
        || (datetime->second >= 60U)) {
        return false;
    }

    /* Регулировка дней в феврале для високосного года. */
    if ((((datetime->year & 3U) == 0U) && (datetime->year % 100U != 0U))
        || (datetime->year % 400U == 0U)) {
            leap_year = 1U;
    }

    /* Проверка корректности дня. */
    if ((datetime->day > RWC_ConvertMonthToDays(datetime->month, datetime->year))
        || (datetime->day < 1U) || ((leap_year == 1U)
        && (datetime->month == 2U) && (datetime->day > 30U))) {
            return false;
    }

    return true;
}

static uint8_t RWC_CheckLeapYear(const uint16_t year)
{
    if (((year % 4U) == 0U && (year % 100U) != 0U)
        || (year % 400U == 0U)) {
            return  1U;
    } else {
        return 0U;
    }
}

static uint8_t RWC_ConvertMonthToDays(const uint8_t month, const uint16_t year)
{
    if (month == 0 || month > 12) {
        return 0U;
    }

    switch (month) {
        case 1:
        case 3:
        case 5:
        case 7:
        case 8:
        case 10:
        case 12:
            return 31U;
            break;
        case 2:
            if (RWC_CheckLeapYear(year) != 0U) {
                return 29U;
            } else {
                return 28U;
            }
        default:
            return 30U;
    }
}

uint32_t RWC_ConvertDatetimeToSeconds(const rtc_datetime_t *datetime)
{
    assert(datetime);
    uint32_t seconds, days;

    /* Вычисление количества дней с 1970 по данный год. */
    days = ((uint32_t) datetime->year - YEAR_RANGE_START) * DAYS_IN_A_YEAR;

    /* Добавление дней високосного года. */
    days += (((uint32_t) datetime->year / 4U) - (YEAR_RANGE_START / 4U));

    /* Добавление количества дней до заданного месяца. */
    for (uint8_t i = 1; i < datetime->month; i++) {
        days += RWC_ConvertMonthToDays(i, datetime->year);
    }

    /*
     * Добавление дней в данном месяце. Вычитается текущий день,
     * какой он есть для отображения в поле часов, минут и секунд.
     */
    days += ((uint32_t)datetime->day - 1U);

    /*
     * Для високосного года, если месяц меньше или равен февралю,
     * уменьшается счетчик дней.
     */
    if ((RWC_CheckLeapYear(datetime->year)) && (datetime->month <= 2U)) {
        days--;
    }

    seconds = (days * SECONDS_IN_A_DAY) + (datetime->hour * SECONDS_IN_A_HOUR) +
        (datetime->minute * SECONDS_IN_A_MINUTE) + datetime->second;

    return seconds;
}

static void RWC_ConvertSecondsToDatetime(uint32_t seconds, rtc_datetime_t *datetime)
{
    assert(datetime);

    uint8_t i;
    uint8_t leap_year = 0U;
    uint16_t days_in_year;
    uint32_t seconds_remaining;
    uint32_t days;

    /*
     * Стартовое значение секунд, которое передается для преобразования
     * в формат даты и времени.
     */
    seconds_remaining = seconds;

    /*
     * Расчет количества дней и добавление 1 для текущего дня,
     * который представлен в поле часы и секунды.
     */
    days = seconds_remaining / SECONDS_IN_A_DAY + 1U;

    /* Обновление секунд. */
    seconds_remaining = seconds_remaining % SECONDS_IN_A_DAY;

    /* Расчет количества часов, минут и секунд. */
    datetime->hour = (uint8_t)(seconds_remaining / SECONDS_IN_A_HOUR);
    seconds_remaining = seconds_remaining % SECONDS_IN_A_HOUR;
    datetime->minute = (uint8_t)(seconds_remaining / 60U);
    datetime->second = (uint8_t)(seconds_remaining % SECONDS_IN_A_MINUTE);

    /* Вычисление года. */
    days_in_year = DAYS_IN_A_YEAR;
    datetime->year = YEAR_RANGE_START;
    while (days > days_in_year) {
        /* Уменьшение количества дней на год и увеличение года на 1. */
        days -= days_in_year;
        datetime->year++;

        /* Регулировка количества дней для високосного года. */
        if (RWC_CheckLeapYear(datetime->year) != 0U) {
            days_in_year = DAYS_IN_A_YEAR + 1U;
        } else {
            days_in_year = DAYS_IN_A_YEAR;
        }
    }

    leap_year = RWC_CheckLeapYear(datetime->year);

    for (i = 1U; i <= 12U; i++) {
        uint8_t days_per_month = RWC_ConvertMonthToDays(i, datetime->year);
        if (days <= days_per_month) {
            datetime->month = i;
            break;
        } else {
            /* Регулировка дней в феврале для високосного года */
            if (i == 2 && leap_year) {
                days -= days_per_month + 1;
            } else {
                days -= days_per_month;
            }
        }
    }

    datetime->day = (uint8_t) days;
}


static enum rwc_status RWC_WaitSyncDone(RWC_Type *base)
{
#if RWC_RETRY_TIMES
    uint32_t wait_times;
#endif
#if RWC_RETRY_TIMES
    wait_times = RWC_RETRY_TIMES;
    while ((GET_VAL_MSK(base->CTRL, RWC_CTRL_SYNC_DONE_Msk,
                RWC_CTRL_SYNC_DONE_Pos) == 0) && (--wait_times != 0U))
#else
    while (GET_VAL_MSK(base->CTRL, RWC_CTRL_SYNC_DONE_Msk,
            RWC_CTRL_SYNC_DONE_Pos) == 0)
#endif
    {
        ;
    }
#if RWC_RETRY_TIMES
    if (wait_times == 0U) {
        return RWC_Status_Timeout;
    }
#endif
    return RWC_Status_Ok;
}

static enum rwc_status RWC_WaitCmdWaitAndSyncDone(RWC_Type *base)
{
#if RWC_SYNC_RETRY_TIMES
    uint32_t wait_times;
#endif
#if RWC_SYNC_RETRY_TIMES
    wait_times = RWC_SYNC_RETRY_TIMES;
    while (((GET_VAL_MSK(base->CTRL, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos)
                != RWC_CmdWait)
            || (GET_VAL_MSK(base->CTRL, RWC_CTRL_SYNC_DONE_Msk,
                    RWC_CTRL_SYNC_DONE_Pos) == 0U)) && (--wait_times != 0U))
#else
    while ((GET_VAL_MSK(base->CTRL, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos)
            != RWC_CmdWait)
        || (GET_VAL_MSK(base->CTRL, RWC_CTRL_SYNC_DONE_Msk,
                RWC_CTRL_SYNC_DONE_Pos) == 0U))
#endif
    {
        ;
    }
#if RWC_SYNC_RETRY_TIMES
    if (wait_times == 0U) {
        return RWC_Status_Timeout;
    }
#endif
    return RWC_Status_Ok;
}

static enum rwc_status RWC_WaitCmdWait(RWC_Type *base)
{
#if RWC_SYNC_RETRY_TIMES
    uint32_t wait_times;
#endif
#if RWC_SYNC_RETRY_TIMES
    wait_times = RWC_SYNC_RETRY_TIMES;
    while ((GET_VAL_MSK(base->CTRL, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos)
            != RWC_CmdWait) && (--wait_times != 0U))
#else
    uint32_t CTRL_CMD_current_value = 0x0;

    while (CTRL_CMD_current_value != RWC_CmdWait) {
        CTRL_CMD_current_value = GET_VAL_MSK(base->CTRL, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos);
    }
#endif
#if RWC_SYNC_RETRY_TIMES
        if (wait_times == 0U) {
            return RWC_Status_Timeout;
        }
#endif
    return RWC_Status_Ok;
}

enum rwc_status RWC_GetInternalRegister(RWC_Type *base,
    enum rwc_internal_register reg, union rwc_union_reg *value)
{
    uint32_t reg_ctrl;

    if ((value == NULL)
        || (reg < RWC_TrimLoad)
        || (reg > RWC_WakeConfig)) {
        return RWC_Status_InvalidArgument;
    }

    while ((GET_VAL_MSK(base->CTRL, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos)
            != RWC_CmdWait)
        || (GET_VAL_MSK(base->CTRL, RWC_CTRL_SYNC_DONE_Msk,
                RWC_CTRL_SYNC_DONE_Pos) == 0)) {
        ;
    }

    reg_ctrl = base->CTRL;
    SET_VAL_MSK(reg_ctrl, ~RWC_CTRL_PRESC_Msk, 0, 0);
    SET_VAL_MSK(reg_ctrl, RWC_CTRL_RSEL_Msk, RWC_CTRL_RSEL_Pos, reg);
    SET_VAL_MSK(reg_ctrl, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos, RWC_CmdRead);
    base->CTRL = reg_ctrl;

    while (GET_VAL_MSK(base->CTRL, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos)
        != RWC_CmdWait) {
        ;
    }

    value->reg_value = base->RWDATA;

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetInternalRegister(RWC_Type *base,
    enum rwc_internal_register reg, union rwc_union_reg value)
{
    enum rwc_status res;
    uint32_t reg_ctrl;

    if ((reg < RWC_TrimLoad)
        || (reg > RWC_WakeConfig)) {
        return RWC_Status_InvalidArgument;
    }

    res = RWC_WaitCmdWaitAndSyncDone(base);
    if (RWC_Status_Ok != res) {
        return res;
    }

    reg_ctrl = base->CTRL;
    base->RWDATA = value.reg_value;
    SET_VAL_MSK(reg_ctrl, ~RWC_CTRL_PRESC_Msk, 0, 0);
    SET_VAL_MSK(reg_ctrl, RWC_CTRL_RSEL_Msk, RWC_CTRL_RSEL_Pos, reg);
    SET_VAL_MSK(reg_ctrl, RWC_CTRL_CMD_Msk, RWC_CTRL_CMD_Pos, RWC_CmdWrite);
    base->CTRL = reg_ctrl;

    return RWC_WaitCmdWait(base);
}

enum rwc_status RWC_GetRTCClkParam(uint32_t *div, enum rwc_rtcclk_type *src)
{
    union rwc_union_reg reg; /* Для хранения RWC_Config */

    if ((div == NULL) || (src == NULL) || (div == (uint32_t *) src)) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(RWC_Secure, RWC_Config, &reg)) {
        return RWC_Status_CheckError;
    }

    *div = 1 << reg.config.clk_div; /* Хранится показатель степени 2*/
    *src = reg.config.osc_sel;

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetRTCClkParam(uint32_t div, enum rwc_rtcclk_type src)
{
    union rwc_union_reg reg; /* Для хранения RWC_Config */
    uint32_t offset = 0;     /* Смещение единичного бита */

    /*
     * В условии проверяется, что div - это степень 2 из диапазона
     * от 2^0 до 2^RWC_DivMax, а также допустимость значения источника частоты.
     */
    if ((div > (1 << RWC_DivMax)) || (!div) || ((div & (div - 1)) != 0)
        || ((src != RWC_RTCClkTypeLFI) && (src != RWC_RTCClkTypeLFE))) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(RWC_Secure, RWC_Config, &reg)) {
        return RWC_Status_CheckError;
    }

    /* При div = 2^x возвратит x */
    while (!(div & 1)) {
        offset++;
        div = div >> 1;
    }

    reg.config.clk_div = offset;
    reg.config.osc_sel = src;

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(RWC_Secure, RWC_Config, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

uint32_t RWC_GetTime(RWC_Type *base)
{
    enum rwc_status res;
    last_status = RWC_Status_Ok;

    if (NULL == base) {
        last_status = RWC_Status_InvalidArgument;
        return 0;
    }

    res = RWC_WaitSyncDone(base);
    if (RWC_Status_Ok != res) {
        last_status = res;
        return 0;
    }
    return base->STR;
}

enum rwc_status RWC_Init(RWC_Type *base, struct rwc_config cfg)
{
    union rwc_union_reg temp;

    union rwc_union_reg time;
    union rwc_union_reg alarm;
    union rwc_union_reg trim;
    union rwc_union_reg config;
    union rwc_union_reg general;
    union rwc_union_reg wake_config;
    enum rwc_status res;

    if (NULL == base)
        return RWC_Status_InvalidArgument;

    /* Приостановка таймера. */
    temp.reg_value = RWC_CONFIG_DEFAULT;
    if ((last_status
        = RWC_SetInternalRegister(base, RWC_Config, temp)) != RWC_Status_Ok) {
            return last_status;
    }

    temp.reg_value = RWC_WAKE_CONFIG_DEFAULT;
    if ((last_status
        = RWC_SetInternalRegister(base, RWC_WakeConfig, temp)) != RWC_Status_Ok) {
        return last_status;
    }

    /* Формирование регистров */
    /* Регистр time */
    time.reg_value = cfg.time;

    /* Регистр alarm_time */
    alarm.reg_value = cfg.alarm_time;

    /* Регистр trim */
    trim.reg_value = 0;
    trim.trim.trim_lfe = cfg.trim_lfe;
    trim.trim.trim_lfi = cfg.trim_lfi;
    trim.trim.lfe_bypass = cfg.lfe_bypass;

    /* Регистр config */
    config.reg_value = 0;
    config.config.time_clk_sel = cfg.time_clk_sel;
    config.config.osc_sel = cfg.osc_sel;
    config.config.clk_div = cfg.clkdiv;
    config.config.reset_en = cfg.reset_en;
    config.config.alarm_en = cfg.alarm_en;

    /* Начало особой обработки полей pz и pl */
    if ((last_status
        = RWC_GetInternalRegister(base, RWC_Config, &temp)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }
    if (cfg.pz)
        config.config.pz = cfg.pz;
    else
        config.config.pz = temp.config.pz;
    if (cfg.pl)
        config.config.pl = cfg.pl;
    else
        config.config.pl = temp.config.pl;
    /* Конец особой обработки полей pz и pl */
    config.config.wake_in_en = cfg.wake_in_en;
    config.config.shutdown_force = RWC_ShutdownNoSet; /* Этот бит должен записываться отдельно! */
    config.config.wake_stat1 = cfg.wake_stat1;
    /* Регистр general */
    general.reg_value = cfg.general;
    /* Регистр wake_config */
    wake_config.reg_value = 0;
    wake_config.wake_config.wake_pol = cfg.wake_pol;
    wake_config.wake_config.wake_en = cfg.wake_en;

    /* Настройка */
    if ((last_status
        = RWC_SetInternalRegister(base, RWC_Config, config)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }

    if ((last_status
        = RWC_SetInternalRegister(base, RWC_GeneralReg, general)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }

    if ((last_status
        = RWC_SetInternalRegister(base, RWC_Time, time)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }
    SET_VAL_MSK(base->CTRL, RWC_CTRL_RESYNC_Msk, RWC_CTRL_RESYNC_Pos, 1);
    res = RWC_WaitSyncDone(base);
    if (RWC_Status_Ok != res) {
        return res;
    }

    if ((last_status
        = RWC_SetInternalRegister(base, RWC_Alarm, alarm)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }

    if ((last_status
        = RWC_SetInternalRegister(base, RWC_Trim, trim)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }

    if (cfg.trimload) {
        temp.reg_value = RWC_TRIM_LOAD_CMD;
        if (RWC_Status_Ok
            != RWC_SetInternalRegister(base, RWC_TrimLoad, temp)) {
            return RWC_Status_CheckError;
        }
    }

    if ((last_status
        = RWC_SetInternalRegister(base, RWC_WakeConfig, wake_config)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }

    if (cfg.shutdown_force == RWC_ShutdownSet) {
        config.config.shutdown_force = cfg.shutdown_force;
        if ((last_status
            = RWC_SetInternalRegister(base, RWC_Config, config)) != RWC_Status_Ok) {
            return RWC_Status_CheckError;
        }
    }

    SET_VAL_MSK(base->CTRL, RWC_CTRL_PRESC_Msk, RWC_CTRL_PRESC_Pos, cfg.presc);

    return RWC_Status_Ok;
}

enum rwc_status RWC_Deinit(RWC_Type *base)
{
    union rwc_union_reg temp;

    if (NULL == base)
        return RWC_Status_InvalidArgument;

    /* Приостановка таймера */
    temp.reg_value = RWC_CONFIG_DEFAULT;
    if ((last_status
        = RWC_SetInternalRegister(base, RWC_Config, temp)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }
    temp.reg_value = RWC_WAKE_CONFIG_DEFAULT;
    if ((last_status
        = RWC_SetInternalRegister(base, RWC_WakeConfig, temp)) != RWC_Status_Ok) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetDatetime(RWC_Type *base, const rtc_datetime_t *datetime)
{
    union rwc_union_reg reg;

    if ((NULL == base) || (NULL == datetime))
        return RWC_Status_InvalidArgument;

    /* Ошибка формата */
    if (!(RWC_CheckDatetimeFormat(datetime))) {
        return RWC_Status_InvalidArgument;
    }

    /* Установка времени */
    reg.reg_value = RWC_ConvertDatetimeToSeconds(datetime);
    return RWC_SetSecondsTimerCount(base, reg.reg_value);
}

enum rwc_status RWC_GetDatetime(RWC_Type *base, rtc_datetime_t *datetime)
{
    uint32_t seconds = 0;

    if ((NULL == base) || (NULL == datetime))
        return RWC_Status_InvalidArgument;

    RWC_GetSecondsTimerCount(base, &seconds);

    RWC_ConvertSecondsToDatetime(seconds, datetime);

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetAlarm(RWC_Type *base, const rtc_datetime_t *alarm_time)
{
    uint32_t alarm_seconds = 0;
    uint32_t curr_seconds  = 0;

    if ((NULL == base) || (NULL == alarm_time))
        return RWC_Status_InvalidArgument;

    if (!(RWC_CheckDatetimeFormat(alarm_time))) {
        return RWC_Status_InvalidArgument;
    }

    alarm_seconds = RWC_ConvertDatetimeToSeconds(alarm_time);

    /* Получение текущего времени. */
    RWC_GetSecondsTimerCount(base, &curr_seconds);

    /* Возврат ошибки, если будильник просрочен. */
    if (alarm_seconds < curr_seconds) {
        return RWC_Status_ConfigureError;
    }

    /* Установка будильника. */
    RWC_SetSecondsTimerMatch(base, alarm_seconds);

    return RWC_Status_Ok;
}

enum rwc_status RWC_GetAlarm(RWC_Type *base, rtc_datetime_t *datetime)
{
    if ((NULL == base) || (NULL == datetime))
        return RWC_Status_InvalidArgument;

    union rwc_union_reg alarm_seconds;

    /* Получение времени будильника в секундах */
    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Alarm, &alarm_seconds)) {
        return RWC_Status_CheckError;
    }

    RWC_ConvertSecondsToDatetime(alarm_seconds.reg_value, datetime);
    return RWC_Status_Ok;
}

enum rwc_status RWC_SetSecondsTimerMatch(RWC_Type *base, uint32_t match_value)
{
    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_Alarm,
            (union rwc_union_reg) match_value)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

uint32_t RWC_GetSecondsTimerMatch(RWC_Type *base)
{
    union rwc_union_reg result;
    last_status = RWC_Status_Ok;

    if (NULL == base) {
        last_status = RWC_Status_InvalidArgument;
        return 0;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Alarm, &result)) {
        last_status = RWC_Status_CheckError;
        return 0;
    }

    return result.reg_value;
}

enum rwc_status RWC_SetSecondsTimerCount(RWC_Type *base, uint32_t count_value)
{
    uint32_t sec;
    uint16_t retry_times = 0;
    enum rwc_status res;
    if (NULL == base)
        return RWC_Status_InvalidArgument;

    while (1) {
        res = RWC_SetInternalRegister(base, RWC_Time,
                (union rwc_union_reg) count_value);
        if (RWC_Status_Ok != res) {
            return res;
        }

        SET_VAL_MSK(base->CTRL, RWC_CTRL_RESYNC_Msk, RWC_CTRL_RESYNC_Pos, 1);
        res = RWC_WaitSyncDone(base);

        if (RWC_Status_Ok != res) {
            return res;
        }

        sec = RWC_GetTime(base);
        res = RWC_GetLastAPIStatus();

        if (RWC_Status_Ok != res) {
            return res;
        }

        if (sec == count_value) {
            break;
        }

        if (retry_times == RWC_SET_RETRY_TIMES) {
            return RWC_Status_Timeout;
        } else {
            retry_times++;
        }
    }

    return RWC_Status_Ok;
}

enum rwc_status RWC_GetSecondsTimerCount(RWC_Type *base, uint32_t *sec)
{
    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    *sec = RWC_GetTime(base);
    return RWC_Status_Ok;
}

enum rwc_status RWC_EnableWakeUpTimerInterruptFromDPD(RWC_Type *base,
    bool enable)
{
    union rwc_union_reg reg;

    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_WakeConfig, &reg)) {
        return RWC_Status_CheckError;
    }

    if (enable) {
        reg.wake_config.wake_en = RWC_IRQWkUpEnable;
    } else {
        reg.wake_config.wake_en = RWC_IRQWkUpDisable;
    }

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_WakeConfig, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

enum rwc_status RWC_EnableAlarmTimerInterruptFromDPD(RWC_Type *base,
    bool enable)
{
    union rwc_union_reg reg;

    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Config, &reg)) {
        return RWC_Status_CheckError;
    }

    if (enable) {
        reg.config.alarm_en = 1;
    } else {
        reg.config.alarm_en = 0;
    }

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_Config, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

uint32_t RWC_GetStatusFlags(RWC_Type *base)
{
    uint32_t result = 0;
    union rwc_union_reg reg;

    last_status = RWC_Status_Ok;

    if (NULL == base) {
        last_status = RWC_Status_InvalidArgument;
        return 0;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Trim, &reg)) {
        last_status = RWC_Status_CheckError;
        return 0;
    }

    if (reg.trim.wake_stat3) {
        result |= RWC_WakeStat3;
    }

    if (reg.trim.wake_stat2) {
        result |= RWC_WakeStat2;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Config, &reg)) {
        last_status = RWC_Status_CheckError;
        return 0;
    }

    if (reg.config.wake_stat1) {
        result |= RWC_WakeStat1;
    }

    return result;
}

enum rwc_status RWC_InterruptClear(RWC_Type *base)
{
    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    SET_VAL_MSK(base->INTCLR, RWC_INTCLR_ALARM_CLR_Msk,
        RWC_INTCLR_ALARM_CLR_Pos, 1);

    return RWC_Status_Ok;
}

enum rwc_status RWC_GetDefaultConfig(struct rwc_config *config)
{
    if (NULL == config) {
        return RWC_Status_InvalidArgument;
    }

    config->trimload = true;
    config->time = 0;
    config->alarm_time = 0;
    config->lfe_bypass = RWC_QuartzResonator;
    config->trim_lfi = 0x033;
    config->trim_lfe = 0x000;
    config->wake_stat1 = 1;
    config->shutdown_force = RWC_ShutdownNoSet;
    config->wake_in_en = RWC_WkUpDisable;
    config->pl = 1;
    config->pz = 1;
    config->alarm_en = RWC_AlarmDisable;
    config->reset_en = RWC_ResetOnlyWakeConfigAndConfig;
    config->clkdiv = RWC_Div1;
    config->osc_sel = RWC_RTCClkTypeLFI;
    config->time_clk_sel = RWC_TimeClock1Hz;
    config->general = 0;
    config->wake_pol = RWC_WkUpPolarityHigh;
    config->wake_en = RWC_IRQWkUpDisable;
    config->presc = RWC_FS1MHz;

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetWakeUpEnable(RWC_Type *base, bool enable)
{
    union rwc_union_reg reg;

    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Config, &reg)) {
        return RWC_Status_CheckError;
    }

    if (enable) {
        reg.config.wake_in_en = RWC_WkUpEnable;
    } else {
        reg.config.wake_in_en = RWC_WkUpDisable;
    }

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_Config, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetWakeUpActiveLewel(RWC_Type *base,
    enum rwc_wake_up_polarity value)
{
    union rwc_union_reg reg;

    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_WakeConfig, &reg)) {
        return RWC_Status_CheckError;
    }

    reg.wake_config.wake_pol = value;

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_WakeConfig, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetLFEBypass(RWC_Type *base,
    enum rwc_lfe_bypass value)
{
    union rwc_union_reg reg;

    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Trim, &reg)) {
        return RWC_Status_CheckError;
    }

    reg.trim.lfe_bypass = value;

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_Trim, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

enum rwc_status RWC_SetLFx(RWC_Type *base,
    enum rwc_rtcclk_type value)
{
    union rwc_union_reg reg;

    if (NULL == base) {
        return RWC_Status_InvalidArgument;
    }

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Config, &reg)) {
        return RWC_Status_CheckError;
    }

    reg.config.osc_sel = value;

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_Config, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

enum rwc_status RWC_GetLastAPIStatus()
{
    return last_status;
}

enum rwc_status RWC_SetResetCtrl(RWC_Type *base,
    enum rwc_reset_type value)
{
    union rwc_union_reg reg;

    if (NULL == base)
        return RWC_Status_InvalidArgument;

    if (RWC_Status_Ok
        != RWC_GetInternalRegister(base, RWC_Config, &reg)) {
        return RWC_Status_CheckError;
    }

    reg.config.reset_en = value;

    if (RWC_Status_Ok
        != RWC_SetInternalRegister(base, RWC_Config, reg)) {
        return RWC_Status_CheckError;
    }

    return RWC_Status_Ok;
}

/*!
 * @}
 */
