/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup sdmmc_driver
 * @{
 */

/*!
 * @file hal_sdmmc.c
 *
 * @brief Имплементация драйвера модуля SDMMC
 */

#include "hal_gpio.h"
#include "hal_sdmmc.h"
#include "hal_clkctr.h"
#include "private/hal_sdmmc_spec.h"

/*!
 * @name Включение отладочной печати драйвера SDMMC
 * @{
 */
#define SDMMC_DEBUG_PRINT_EN     0 /*!< Включение печати для отладки */

#if SDMMC_DEBUG_PRINT_EN
#define SDMMC_DEBUG_PRINTF       printf                         /*!< Форматированная отладочная печать */
#define SDMMC_DEBUG_MSG(a)       printf("%s\r\n", a)            /*!< Печать текстового сообщения */
#define SDMMC_DEBUG_MSG_VAL(s,a) printf("%s : 0x%lx\r\n", s, a) /*!< Печать текстового сообщения и шестнадцатеричного значения */
#else
#define SDMMC_DEBUG_PRINTF(...)                                 /*!< Форматированная отладочная печать */
#define SDMMC_DEBUG_MSG(a)                                      /*!< Печать текстового сообщения */
#define SDMMC_DEBUG_MSG_VAL(s,a)                                /*!< Печать текстового сообщения и шестнадцатеричного значения */
#endif
/*!
 * @}
 */

#define SDMMC_FREQ_1MHZ 1000000 /*!< Частота 1МГц */

/* Маски используемых прерываний контроллера SDMMC. */
#define SDMMC_NORMAL_INT_MASK 0xFEDF /*!< NORMAL INT */
#define SDMMC_ERROR_INT_MASK  0xFFFF /*!< ERROR INT */

/*!
 * @brief Флаги причины блокировки контроллера SDMMC
 */
typedef enum {
    SDMMC_NoLock             = 0, /* Блокировки нет. */
    SDMMC_StartOperationLock = 1, /* Блокировка по началу операции. */
    SDMMC_ReadAsyncLock      = 2, /* Блокировка асинхронной операции чтения. */
    SDMMC_WriteAsyncLock     = 3, /* Блокировка асинхронной операции записи. */
    SDMMC_CardDisabledLock   = 4  /* Блокировка по отсутствию или отключению карты. */
} sdmmc_lock_t;

/*!
 * @brief Включение и определение входной тактовой частоты
 *
 * @param sd Контекст драйвера SDMMC
 */
static void SDMMC_EnableInputClock(sdmmc_card_t *sd);

/*!
 * @brief Инициализация внешних выводов GPIO
 *
 * @param sd Контекст драйвера SDMMC
 */
static void SDMMC_InitGPIO(sdmmc_card_t *sd);

/*!
 * @brief Инициализация дополнительных внешних выводов GPIO для режима 8бит MMC
 *
 * @param sd Контекст драйвера SDMMC
 */
static void SDMMC_InitGPIO_8Bit(sdmmc_card_t *sd);

/*!
 * @brief Отключение внешних выводов GPIO
 *
 * @param sd Контекст драйвера SDMMC
 */
static void SDMMC_DisableGPIO(sdmmc_card_t *sd);

/*!
 * @brief Перевод внешнего вывода CMD в режим OpenDrain (только для MMC)
 *
 * @param sd    Контекст драйвера SDMMC
 * @param od_en OpenDrain режим: 0 - отключен, 1 - включен
 */
static void SDMMC_MMC_SetOdCmdPin(sdmmc_card_t *sd, int32_t od_en);

/*!
 * @brief Определение входной тактовой частоты
 *
 * @return Значение тактовой частоты в Гц
 */
static int32_t SDMMC_GetInputClockHz(void);

/*!
 * @brief Перевод карты в исходное состояние (сброс)
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_GoIdleState(sdmmc_card_t *sd);

/*!
 * @brief Функция зыадержка времени в микросекундах
 *
 * @param sd Контекст драйвера SDMMC
 * @param us Время в микросекундах
 */
static void SDMMC_Udelay(sdmmc_card_t *sd, uint32_t us);

/*!
 * @name Отправка команд
 * @{
 */
static sdmmc_status_t SDMMC_MMC_CmdSendOpCond(sdmmc_card_t *sd);
static sdmmc_status_t SDMMC_CmdSelectCard(sdmmc_card_t *sd);
static sdmmc_status_t SDMMC_CmdSetBlockLength(sdmmc_card_t *sd,
    uint32_t block_size);
static sdmmc_status_t SDMMC_CmdReadMultiple(sdmmc_card_t *sd,
    uint32_t start_block, uint32_t baddr, uint32_t nblocks);
static sdmmc_status_t SDMMC_CmdWriteMultiple(sdmmc_card_t *sd,
    uint32_t start_block, uint32_t baddr, uint32_t nblocks);
static sdmmc_status_t SDMMC_CmdSetBusWidth(sdmmc_card_t *sd);
static sdmmc_status_t SDMMC_CmdSendOpCond(sdmmc_card_t *sd);
static sdmmc_status_t SDMMC_CmdClearCardDetect(sdmmc_card_t *sd);
static sdmmc_status_t SDMMC_CmdExecute(sdmmc_card_t *sd, uint32_t sdma_addr,
    uint32_t blk_sz_cnt, uint32_t arg1, uint32_t trans_mod_com);
static sdmmc_status_t SDMMC_CmdSend(sdmmc_card_t *sd, uint32_t cmd,
    uint32_t arg1, uint32_t resp_sz, uint32_t *resp);
static sdmmc_status_t SDMMC_CmdR2Send(sdmmc_card_t *sd, uint32_t cmd,
    uint32_t arg1, uint32_t *reg128);
/*!
 * @}
 */

/*!
 * @brief Получение относительного адреса карты
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_GetRCA(sdmmc_card_t *sd);

/*!
 * @brief Отправка команды CMD55 для перевода карты в режим приема AppCMD
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_PrepareAcmd(sdmmc_card_t *sd);

/*!
 * @brief Перевод карты в режим TRAN
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_WaitTRANState(sdmmc_card_t *sd);

/*!
 * @brief Установка делителя выходной частоты
 *
 * @param sd      Контекст драйвера SDMMC
 * @param clk_div Делитель выходной частоты
 */
static void SDMMC_SetClockDivider(sdmmc_card_t *sd, int32_t clk_div);

/*!
 * @brief Cброс контроллера и карты
 *
 * @param sd Контекст драйвера SDMMC
 */
static void SDMMC_Reset(sdmmc_card_t *sd);

/*!
 * @brief Перезапуск контроллера с новым делителем частоты
 *
 * @param sd      Контекст драйвера SDMMC
 * @param clk_div Делитель выходной частоты
 */
static void SDMMC_RestartController(sdmmc_card_t *sd, int32_t clk_div);

/*!
 * @brief Чтение регистров Response
 *
 * @param sd  Контекст драйвера SDMMC
 * @param num Номер ответа
 *
 * @return 32-битное значение ответа
 */
static uint32_t SDMMC_GetResponse(sdmmc_card_t *sd, int32_t num);

/*!
 * @brief Чтение регистров EXT_CSD карты MMC
 *
 * @param sd     Контекст драйвера SDMMC
 * @param reg512 Буфер сохранения регистра размером 512 бит
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_MMC_GetExtCSD(sdmmc_card_t *sd, void *reg512);

/*!
 * @brief Включение режима High Speed
 *
 * @param sd     Контекст драйвера SDMMC
 * @param reg512 Буфер размером 512 байт
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_EnableHS(sdmmc_card_t *sd, void *reg512);

/*!
 * @brief Запрос поддерживаемых рабочих напряжений питания у карты
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_CheckSupportedVoltage(sdmmc_card_t *sd);

/*!
 * @brief Блокировка контроллера SDMMC
 *
 * Выставляется флаг SDMMC_StartOperationLock.
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_Lock(sdmmc_card_t *sd);

/*!
 * @brief Попытка снятия блокировки контроллера SDMMC
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_Unlock(sdmmc_card_t *sd);

/*!
 * @brief Процесс переключения напряжения питания карты SD с 3,3 В на 1,8 В
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
static sdmmc_status_t SDMMC_SD_1v8Enabling(sdmmc_card_t *sd);

/*!
 * @brief Установка выходного напряжения питания карты
 *
 * @param sd  Контекст драйвера SDMMC
 * @param vol Выбор напряжения питания карты
 */
static void SDMMC_SetBusVoltage(sdmmc_card_t *sd, sdmmc_voltage_t vol);

/*!
 * @brief Включение питания карты
 *
 * @param sd Контекст драйвера SDMMC
 */
static void SDMMC_BusPowerOn(sdmmc_card_t *sd);

/*!
 * @brief Отключение питания карты
 *
 * @param sd Контекст драйвера SDMMC
 */
static void SDMMC_BusPowerOff(sdmmc_card_t *sd);

static void SDMMC_SetBusVoltage(sdmmc_card_t *sd, sdmmc_voltage_t vol)
{
    SDMMC_POWERCONTROL_PWRCTRL_SDBUSVOLTAGE_PTYPE_SET(sd->regs,
        (vol == SDMMC_3v3) ? SDMMC_HostPWR_3V3 : SDMMC_HostPWR_1V8);
}

static void SDMMC_BusPowerOn(sdmmc_card_t *sd)
{
    /* Включение тестового режима вывода CD. */
    SDMMC_HOSTCONTROL1_HOSTCTRL1_CDTESTLEVEL_PTYPE_SET(sd->regs, 1);
    SDMMC_HOSTCONTROL1_HOSTCTRL1_CDSIGSELECT_PTYPE_SET(sd->regs, 1);
    SDMMC_POWERCONTROL_PWRCTRL_SDBUSPOWER_PTYPE_SET(sd->regs, 1);
}

static void SDMMC_BusPowerOff(sdmmc_card_t *sd)
{
    /* Выключение тестового режима вывода CD. */
    SDMMC_HOSTCONTROL1_HOSTCTRL1_CDTESTLEVEL_PTYPE_SET(sd->regs, 0);
    SDMMC_HOSTCONTROL1_HOSTCTRL1_CDSIGSELECT_PTYPE_SET(sd->regs, 0);
    SDMMC_POWERCONTROL_PWRCTRL_SDBUSPOWER_PTYPE_SET(sd->regs, 0);
}

static void SDMMC_Reset(sdmmc_card_t *sd)
{
    const uint8_t reset_mask = SDMMC_SOFTWARERESET_SWRESET_FOR_ALL_Msk
        | SDMMC_SOFTWARERESET_SWRESET_FOR_CMD_Msk
        | SDMMC_SOFTWARERESET_SWRESET_FOR_DAT_Msk;

    SDMMC_BusPowerOff(sd);
    sd->regs->SOFTWARERESET |= reset_mask;

    while ((sd->regs->SOFTWARERESET & reset_mask) != 0)
        ;

    SDMMC_TIMEOUTCONTROL_TIMEOUT_CTRVALUE_PTYPE_SET(sd->regs,
        SDMMC_TIMEOUTCONTROL_MAX_VALUE);
    SDMMC_SetBusVoltage(sd,
        (sd->gpio_vol ==  SDMMC_3v3To1v8) ? SDMMC_3v3 : sd->gpio_vol);

    SDMMC_Udelay(sd, SDMMC_Timeout_Rst);
    SDMMC_BusPowerOn(sd);
    SDMMC_Udelay(sd, SDMMC_Timeout_Rst);

    SDMMC_CORECFG0_BASEFREQ_PTYPE_SET(sd->regs,
        sd->freq_input / SDMMC_FREQ_1MHZ);
    SDMMC_CORECFG0_SLOTTYPE_PTYPE_SET(sd->regs, sd->cfg->slot_type);
    SDMMC_NORMALINTRSTSENA_SDHCREGSET_CARDINSSTSENA_PTYPE_SET(sd->regs, 1);
}

void SDMMC_DisableCard(sdmmc_card_t *sd)
{
    if ((sd == NULL) || (sd->lock == SDMMC_CardDisabledLock)) {
        return;
    }

    SDMMC_DisableGPIO(sd);
    const uint8_t reset_mask = SDMMC_SOFTWARERESET_SWRESET_FOR_ALL_Msk
        | SDMMC_SOFTWARERESET_SWRESET_FOR_CMD_Msk
        | SDMMC_SOFTWARERESET_SWRESET_FOR_DAT_Msk;
    SDMMC_BusPowerOff(sd);
    sd->regs->SOFTWARERESET |= reset_mask;

    while ((sd->regs->SOFTWARERESET & reset_mask) != 0)
        ;

    sd->lock = SDMMC_CardDisabledLock;
}

static sdmmc_status_t SDMMC_SD_1v8Enabling(sdmmc_card_t *sd)
{
    uint32_t cmd_resp;

    if (SDMMC_CmdSend(sd, SDMMC_SD_VOLTAGE_SWITCH, 0, 1, &cmd_resp)
        != SDMMC_Status_Ok) {
        SDMMC_DEBUG_MSG("CMD11 : ERROR");

        return SDMMC_Status_Err;
    }

    /* Отключение частоты SDCLK. */
    SDMMC_CLOCKCONTROL_CLKCTRL_SDCLKENA_PTYPE_SET(sd->regs, 0);
    /* Проверка выводов DAT[3:0]. */
    const uint32_t data_pin_mask = SDMMC_PRESENTSTATE_SDIF_DAT0IN_DSYNC_Msk
        | SDMMC_PRESENTSTATE_SDIF_DAT0IN_DSYNC_Msk
        | SDMMC_PRESENTSTATE_SDIF_DAT0IN_DSYNC_Msk
        | SDMMC_PRESENTSTATE_SDIF_DAT0IN_DSYNC_Msk;

    if ((sd->regs->PRESENTSTATE & data_pin_mask) != 0) {
        SDMMC_DEBUG_MSG("1V8 EN : ERROR : DAT[3:0] LVL != 0");

        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG("1V8 EN : Switch voltage");
    SDMMC_SetBusVoltage(sd, SDMMC_1v8);
    SDMMC_HOSTCONTROL2_HOSTCTRL2_1P8VSIGNALLINGENA_PTYPE_SET(sd->regs, 1);
    SDMMC_HOSTCONTROL2_HOSTCTRL2_UHSMODESELECT_PTYPE_SET(sd->regs,
        SDMMC_SD_UHS_MODE_DEFAULT_SDR12);

    /* Ожидание переключения напряжения. */
    SDMMC_Udelay(sd, SDMMC_Timeout_Vol1);
    /* Включение частоты SDCLK. */
    SDMMC_CLOCKCONTROL_CLKCTRL_SDCLKENA_PTYPE_SET(sd->regs, 1);
    SDMMC_Udelay(sd, SDMMC_Timeout_Vol2);

    /* Проверка выводов DAT[3:0]. */
    if ((sd->regs->PRESENTSTATE & data_pin_mask) != data_pin_mask) {
        SDMMC_DEBUG_MSG("1V8 EN : ERROR : DAT[3:0] LVL != 0xF");

        return SDMMC_Status_Err;
    }

    /* Проверка вывода CMD. */
    if ((sd->regs->PRESENTSTATE & SDMMC_PRESENTSTATE_SDIF_CMDIN_DSYNC_Msk)
        == 0) {
        SDMMC_DEBUG_MSG("1V8 EN : ERROR : CMD LVL != 0x1");

        return SDMMC_Status_Err;
    }

    return SDMMC_Status_Ok;
}

static void SDMMC_Udelay(sdmmc_card_t *sd, uint32_t us)
{
    if (sd != NULL && sd->cfg->delay_us != NULL) {
        sd->cfg->delay_us(us);
    }
}

sdmmc_status_t SDMMC_InitCard(sdmmc_card_t *sd, uint32_t num,
    sdmmc_voltage_t vol, void *init_buf)
{
    int32_t timeout;

    if (sd == NULL) {
        SDMMC_DEBUG_MSG("SDMMC : ctx is NULL");

        return SDMMC_Status_Err;
    }

    if (sd->cfg == NULL) {
#ifdef SDMMC_CFG
        sd->cfg = SDMMC_CFG;
    }

    if (sd->cfg == NULL || sd->cfg->reg_base[0] == 0) {
#endif
        SDMMC_DEBUG_MSG("SDMMC_InitCard() : Port config is not found");

        return SDMMC_Status_Err;
    }

    if (num >= sd->cfg->slot_count) {
        SDMMC_DEBUG_MSG("SDMMC_InitCard() : incorrect SDMMC port number");

        return SDMMC_Status_Err;
    }

    if (sd->cfg->hs_en && !SDMMC_SDMA_IS_BLOCK_ALIGN_ADDR(init_buf)) {
        SDMMC_DEBUG_MSG("SDMMC : init_buffer is not aligned");

        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG("SDMMC_InitCard()");

    sd->dev_num = num;
    sd->rca = 0;
    sd->type = SDMMC_TypeSD;
    sd->sdhc_mode = 0;
    sd->hs_mode = 0;
    sd->ddr_mode = 0;
    sd->version = 0;
    sd->gpio_vol = vol;
    sd->need_1V8en = 0;
    sd->regs = (SDMMC_Type *)sd->cfg->reg_base[0];
    sd->lock = SDMMC_NoLock;

    SDMMC_EnableInputClock(sd);
    SDMMC_InitGPIO(sd);
    SDMMC_Reset(sd);

    SDMMC_Udelay(sd, SDMMC_Timeout_Min);

    do {
        /* Проверка, что карта вставлена. */
        if (SDMMC_PRESENTSTATE_SDIF_CD_N_DSYNC_PTYPE_GET(sd->regs) == 0) {
            SDMMC_DEBUG_MSG("ERROR : No card detect");
            break;
        }

        SDMMC_DEBUG_MSG("SDMMC : Waiting interrupt...");
        /* Ожидание события Card Inserted. */
        timeout = sd->cfg->timeout_cd;
        while (SDMMC_PRESENTSTATE_SDIF_CD_N_DSYNC_PTYPE_GET(sd->regs) != 0
            && SDMMC_PRESENTSTATE_SDHCCARDDET_INSERTED_DSYNC_PTYPE_GET(sd->regs)
            == 0) {
            SDMMC_Udelay(sd, SDMMC_Timeout_1ms);
            if (timeout-- == 0) {
                SDMMC_DEBUG_MSG("ERROR : Timeout card detect");
                break;
            }
        }

        if (sd->gpio_vol == SDMMC_1v8) {
            sd->type = SDMMC_TypeMMC;
        }

        sd->freq_divider = SDMMC_CALC_DIVIDER(sd->freq_input,
                sd->cfg->freq_out_init);
        SDMMC_RestartController(sd, sd->freq_divider);

        if (SDMMC_GoIdleState(sd) == SDMMC_Status_Err) {
            break;
        }

        /* Проверка поддержки необходимого напряжения питания. */
        if (SDMMC_CheckSupportedVoltage(sd) == SDMMC_Status_Err) {
            break;
        }

        /* Запрос OCR регистра. */
        if (SDMMC_CmdSendOpCond(sd) == SDMMC_Status_Err) {
            break;
        }

        /* Переключение на 1,8 В, если это необходимо. */
        if (sd->need_1V8en) {
            if (SDMMC_SD_1v8Enabling(sd) == SDMMC_Status_Err) {
                break;
            }
        }

        if (SDMMC_GetRCA(sd) == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("ERROR : SDMMC_GetRCA()");
            break;
        }

        if (SDMMC_CmdR2Send(sd, SDMMC_SEND_CID, sd->rca, sd->cid)
            == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("SDMMC : Error get CID!");
        }

        if (SDMMC_CmdR2Send(sd, SDMMC_SEND_CSD, sd->rca, sd->csd)
            == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("SDMMC : Error get CSD!");
        }

        if (SDMMC_IS_MMC(sd)) {
            SDMMC_DEBUG_MSG("SDMMC : is MMC");
            /* Отключение режима OD у вывода CMD. */
            SDMMC_MMC_SetOdCmdPin(sd, 0);
        }

        if (sd->sdhc_mode) {
            SDMMC_DEBUG_MSG("SDMMC : SDHC mode enabled");
        }

        if (SDMMC_CmdSelectCard(sd) == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("ERROR : SDMMC_CmdSelectCard()");
            break;
        }

        /* Отключение pull-up резистора на выводе CD/DAT3. */
        if (SDMMC_CmdClearCardDetect(sd) == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("ERROR : SDMMC_CmdClearCardDetect()");
            break;
        }

        if (SDMMC_IS_MMC(sd)) {
            if (sd->cfg->hs_en) {
                if (SDMMC_EnableHS(sd, (uint8_t *) init_buf) == SDMMC_Status_Err) {
                    SDMMC_DEBUG_MSG("ERROR : SDMMC_EnableHS()");
                    break;
                }
            }

            /* Установка 4-битного или 8-битного режима. */
            if (SDMMC_CmdSetBusWidth(sd) == SDMMC_Status_Err) {
                SDMMC_DEBUG_MSG("ERROR : SDMMC_CmdSetBusWidth()");
                break;
            }
        } else {
            /* Установка 4-битного режима. */
            if (SDMMC_CmdSetBusWidth(sd) == SDMMC_Status_Err) {
                SDMMC_DEBUG_MSG("ERROR : SDMMC_CmdSetBusWidth()");
                break;
            }

            if (sd->cfg->hs_en) {
                if (SDMMC_EnableHS(sd, (uint8_t *) init_buf) == SDMMC_Status_Err) {
                    SDMMC_DEBUG_MSG("ERROR : SDMMC_EnableHS()");
                    break;
                }
            }
        }

        if (SDMMC_CmdSetBlockLength(sd, SDMMC_SDHC_SECTOR_SIZE)
            == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("ERROR : SDMMC_CmdSetBlockLength()");
            break;
        }

        if (!sd->sdhc_mode && !SDMMC_IS_MMC(sd)) {
            SDMMC_DEBUG_MSG("SDMMC_InitCard() : Error, unsupportable type card. Support only SDHC/SDXC or MMC");
            break;
        }

        sd->freq_divider = SDMMC_CALC_DIVIDER(sd->freq_input,
                sd->cfg->freq_out);
        SDMMC_SetClockDivider(sd, sd->freq_divider);

        return SDMMC_Status_Ok;
    } while (0);

    sd->lock = SDMMC_CardDisabledLock;
    SDMMC_BusPowerOff(sd);

    return SDMMC_Status_Err;
}

static sdmmc_status_t SDMMC_PrepareAcmd(sdmmc_card_t *sd)
{
    uint32_t cmd_resp;

    if (SDMMC_IS_MMC(sd)) {
        SDMMC_DEBUG_MSG("ERROR : APP_CMD not allowed for MMC");

        return SDMMC_Status_Err;
    }

    if (SDMMC_CmdSend(sd, SDMMC_APP_CMD, sd->rca, 1, &cmd_resp)
        == SDMMC_Status_Err) {
        SDMMC_DEBUG_MSG("ERROR : APP_CMD failed");

        return SDMMC_Status_Err;
    }

    if ((cmd_resp & SDMMC_APP_CMD) == 0) {
        SDMMC_DEBUG_MSG("ERROR : APP_CMD not accepted");

        return SDMMC_Status_Err;
    }

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_WaitTRANState(sdmmc_card_t *sd)
{
    uint32_t cmd_resp;

    do {
        if (SDMMC_CmdSend(sd, SDMMC_SEND_STATUS, sd->rca, 1,
                (uint32_t *) &cmd_resp) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }
    } while ((cmd_resp & SDMMC_R1_READY_FOR_DATA) != SDMMC_R1_READY_FOR_DATA
        || SDMMC_R1_CURRENT_STATE(cmd_resp) != SDMMC_R1_STATE_TRAN);

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_EnableHS(sdmmc_card_t *sd, void *reg512)
{
    uint8_t *ext_csd_regs = (uint8_t *) reg512;

    if (SDMMC_IS_MMC(sd)) {
        uint32_t cmd_resp;
        sdmmc_mmc_ext_csd_arg_t arg1;

        arg1.arg = 0;
        arg1.index = SDMMC_EXT_CSD_HS_TIMING_OFFSET;
        arg1.value = SDMMC_EXT_CSD_HS_TIMING_EN;
        arg1.access = SDMMC_EXT_CSD_ACCESS_WRITE_BYTE;

        if (SDMMC_CmdSend(sd, SDMMC_SWITCH, arg1.arg, 1, (uint32_t *) &cmd_resp)
            == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }

        if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }

        if (SDMMC_MMC_GetExtCSD(sd, ext_csd_regs) != SDMMC_Status_Ok) {
            return SDMMC_Status_Err;
        }

        if (ext_csd_regs[SDMMC_EXT_CSD_HS_TIMING_OFFSET]
            == SDMMC_EXT_CSD_HS_TIMING_EN) {
            sd->hs_mode = 1;
            if (((ext_csd_regs[SDMMC_EXT_CSD_CARD_TYPE_OFFSET]
                        & SDMMC_EXT_CSD_CARD_TYPE_HS_MASK)
                    & SDMMC_EXT_CSD_CARD_TYPE_HS_DDR_MMC_52) != 0) {
                sd->ddr_mode = 1;
            }
        }
    } else {
        if (sd->version == 1) {
            return SDMMC_Status_Ok;
        }

        sdmmc_sd_switch_cmd_t sw_func;

        sw_func.value = 0;
        sw_func.mode = SDMMC_SD_ACCESS_MODE_SET_FUNCTION;
        if (sd->need_1V8en) {
            sw_func.access_mode = sd->cfg->sd_uhs_mode;
        } else {
            sw_func.access_mode = SDMMC_SD_UHS_MODE_HIGHSPEED_SDR25;
        }

        if (SDMMC_CmdSend(sd, SDMMC_SWITCH, sw_func.value, 1, 0)
            == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }

        if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }

        sd->hs_mode = 1;
        SDMMC_HOSTCONTROL1_HOSTCTRL1_HIGHSPEEDENA_PTYPE_SET(sd->regs, 1);
    }

    return SDMMC_Status_Ok;
}

sdmmc_status_t SDMMC_CalcMemorySpace(sdmmc_card_t *sd, void *sector_buf,
    bool unsafe)
{
    int32_t i, step;
    sdmmc_status_t result = SDMMC_Status_Err;
    uint32_t csize, mbi;
    uint64_t block_addr;
    uint32_t *sdmmc_buf = (uint32_t *) sector_buf;
    sdmmc_emmc_csd_reg_t *emmc_csd = (sdmmc_emmc_csd_reg_t *) sd->csd;
    sdmmc_sd_csd_reg_v2_t *sd_csd_v2 = (sdmmc_sd_csd_reg_v2_t *) sd->csd;
    sdmmc_sd_csd_reg_v1_t *sd_csd_v1 = (sdmmc_sd_csd_reg_v1_t *) sd->csd;

    SDMMC_DEBUG_PRINTF("SDMMC Card %ld : Calculate size ...\r\n", sd->dev_num);

    do {
        if (sd == NULL) {
            break;
        }
        if (SDMMC_IS_MMC(sd)) {
            if (emmc_csd->csd_structure != 0) {
                /* Регистр CSD формата версии 1.1 или выше. */
                csize = emmc_csd->c_size_low | (emmc_csd->c_size_high << 10);

                if (csize != 0xFFF) {
                    /* Низкая емкость <= 4Gb. */
                    sd->total_size = ((1 + csize)
                            << (emmc_csd->c_size_mult + 2))
                        << emmc_csd->read_bl_len;
                    result = SDMMC_Status_Ok;
                    break;
                } else {
                    /* Чтение регистра EXT_CSD. */
                    if (SDMMC_MMC_GetExtCSD(sd, sdmmc_buf) == SDMMC_Status_Ok) {
                        sd->total_size =
                            (uint64_t)
                            sdmmc_buf[SDMMC_EXT_CSD_SEC_COUNT_OFFSET / 4]
                            * SDMMC_SDHC_SECTOR_SIZE;
                        result = SDMMC_Status_Ok;
                        break;
                    }
                }
            }
        } else {
            if (sd_csd_v2->csd_structure != 0) {
                sd->total_size = (uint64_t)(sd_csd_v2->c_size + 1)
                    * 512 * 1024;
                result = SDMMC_Status_Ok;
                break;
            } else {
                /* Низкая емкость <= 4Gb. */
                csize = sd_csd_v1->c_size_low | (sd_csd_v1->c_size_high << 10);
                sd->total_size = ((1 + csize) << (sd_csd_v1->c_size_mult + 2))
                    << sd_csd_v1->read_bl_len;
                result = SDMMC_Status_Ok;
                break;
            }
        }
        if (!unsafe || (sdmmc_buf == NULL)) {
            break;
        }
        for (i = 0; i < SDMMC_SDHC_SECTOR_SIZE / 4; i++) {
            sdmmc_buf[i] = 0;
        }

        /*
         * Итеративный метод определения размера памяти карты, путем записи
         * значений в блоки с шагом 256МБ, затем с шагом 1МБ в последнем блоке.
         */
        for (mbi = 0, step = 256; step > 0; step -= 255, mbi -= 256) {
            for (; mbi < (128 * 1024); mbi += step) {
                sdmmc_buf[0] = mbi;
                block_addr = ((uint64_t) mbi * 1024 * 1024)
                    / SDMMC_SDHC_SECTOR_SIZE;

                if (SDMMC_Write(sd, block_addr, sdmmc_buf, 1)
                    != SDMMC_Status_Ok) {
                    SDMMC_DEBUG_PRINTF(
                        "SDMMC Card %ld : Warning! Sector %ldMb : Write Error!\r\n",
                        sd->dev_num, mbi);
                    break;
                }

                if (SDMMC_Read(sd, block_addr, sdmmc_buf, 1)
                    != SDMMC_Status_Ok) {
                    SDMMC_DEBUG_PRINTF(
                        "SDMMC Card %ld : Warning! Sector %ldMb : Read Error\r!\n",
                        sd->dev_num, mbi);
                    break;
                }

                if (sdmmc_buf[0] != mbi) {
                    SDMMC_DEBUG_PRINTF("SDMMC Card %ld : EOF cause\r\n",
                        sd->dev_num);
                    break;
                }
            }

            SDMMC_InitCard(sd, sd->dev_num, sd->gpio_vol, sdmmc_buf);

            if ((mbi == 0) || (step == 1)) {
                break;
            }
        }

        if (mbi != 0) {
            sd->total_size = (uint64_t) mbi * 1024 * 1024;
            result = SDMMC_Status_Ok;
        }
    } while (0);

    if (result == SDMMC_Status_Ok) {
        SDMMC_DEBUG_PRINTF("SDMMC Card %ld : size : %lluMb\r\n",
            sd->dev_num, sd->total_size / (1024 * 1024));
    } else {
        SDMMC_DEBUG_PRINTF("SDMMC Card : unknown size\r\n");
    }

    return result;
}

sdmmc_status_t SDMMC_Read(sdmmc_card_t *sd, uint32_t start_block, void *data,
    uint32_t nblocks)
{
    if (!SDMMC_SDMA_IS_BLOCK_ALIGN_ADDR(data)) {
        return SDMMC_Status_Err;
    }

    if (!sd->sdhc_mode) {
        start_block *= SDMMC_SDHC_SECTOR_SIZE;
    }

    if (sd->lock != SDMMC_NoLock) {
        if (SDMMC_Unlock(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }
    }

    if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    if (SDMMC_CmdReadMultiple(sd, start_block, (uint32_t) data, nblocks)
        == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG("SDMMC : Data received");

    return SDMMC_Status_Ok;
}

sdmmc_status_t SDMMC_ReadAsync(sdmmc_card_t *sd, uint32_t start_block,
    void *data, uint32_t nblocks)
{
    if (!SDMMC_SDMA_IS_BLOCK_ALIGN_ADDR(data)) {
        return SDMMC_Status_Err;
    }

    if (!sd->sdhc_mode) {
        start_block *= SDMMC_SDHC_SECTOR_SIZE;
    }

    if (sd->lock != SDMMC_NoLock) {
        if (SDMMC_Unlock(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }
    }

    if (SDMMC_Lock(sd) == SDMMC_Status_Err || SDMMC_WaitTRANState(sd)
        == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    if (SDMMC_CmdReadMultiple(sd, start_block, (uint32_t) data, nblocks)
        == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG("SDMMC : Async read DATA");

    return SDMMC_Status_Ok;
}

sdmmc_status_t SDMMC_ReadWait(sdmmc_card_t *sd)
{
    return SDMMC_Unlock(sd);
}

sdmmc_status_t SDMMC_Write(sdmmc_card_t *sd, uint32_t start_block,
    const void *data, uint32_t nblocks)
{
    if ((sd == NULL) || (nblocks == 0)) {
        return SDMMC_Status_Err;
    }

    if (!sd->sdhc_mode) {
        start_block *= SDMMC_SDHC_SECTOR_SIZE;
    }

    if (sd->lock != SDMMC_NoLock) {
        if (SDMMC_Unlock(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }
    }

    if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    if (SDMMC_CmdWriteMultiple(sd, start_block, (uint32_t) data, nblocks)
        == SDMMC_Status_Err) {
        SDMMC_DEBUG_PRINTF("ERROR : CMD25 return false. %s, %d\r\n",
            __FILE__, __LINE__);

        return SDMMC_Status_Err;
    }

    if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG("SDMMC : Data written");

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_Lock(sdmmc_card_t *sd)
{
    if (sd == NULL) {
        return SDMMC_Status_Err;
    }

    if (sd->lock == SDMMC_NoLock) {
        sd->lock = SDMMC_StartOperationLock;

        return SDMMC_Status_Ok;
    }

    return SDMMC_Status_Err;
}

static sdmmc_status_t SDMMC_Unlock(sdmmc_card_t *sd)
{
    if ((sd == NULL) || (sd->lock == SDMMC_CardDisabledLock)) {
        return SDMMC_Status_Err;
    }

    if (sd->lock == SDMMC_NoLock) {
        return SDMMC_Status_Ok;
    }

    uint32_t adma_addr = SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_GET(sd->regs)
        | (SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_GET(sd->regs) << 16);

    while (SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_PTYPE_GET(sd->regs)
        == 0) {
        if (sd->regs->ERRORINTRSTS != 0) {
            SDMMC_DEBUG_PRINTF(
                "ERROR : Unlock has some errors 0x%08x. %s, %d\r\n",
                sd->regs->ERRORINTRSTS, __FILE__, __LINE__);

            return SDMMC_Status_Err;
        }
    }

    while (1) {
        if (SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_PTYPE_GET(sd->regs)
            != 0) {

            /* Передача завершена. */
            sd->regs->NORMALINTRSTS |=
                SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_Msk;
            break;
        }

        if (SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_PTYPE_GET(sd->regs)
            != 0) {
            SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_PTYPE_SET(sd->regs,
                1);
            adma_addr += SDMMC_SDMA_BLOCK_SIZE;
            SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, adma_addr);
            SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_SET(sd->regs,
                adma_addr >> 16);
        }

        if (sd->regs->ERRORINTRSTS != 0) {
            SDMMC_DEBUG_PRINTF(
                "ERROR : Unlock has some errors 0x%08x. %s, %d\r\n",
                sd->regs->ERRORINTRSTS, __FILE__, __LINE__);

            return SDMMC_Status_Err;
        }
    }

    sd->lock = SDMMC_NoLock;

    return SDMMC_Status_Ok;
}

sdmmc_status_t SDMMC_WriteAsync(sdmmc_card_t *sd, uint32_t start_block,
    const void *data, uint32_t nblocks)
{
    if ((sd == NULL) || (nblocks == 0)) {
        return SDMMC_Status_Err;
    }

    if (!sd->sdhc_mode) {
        start_block *= SDMMC_SDHC_SECTOR_SIZE;
    }

    if (sd->lock != SDMMC_NoLock) {
        if (SDMMC_Unlock(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }
    }

    if ((SDMMC_Lock(sd) == SDMMC_Status_Err)
        || (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err)) {
        return SDMMC_Status_Err;
    }

    if (SDMMC_CmdWriteMultiple(sd, start_block, (uint32_t) data, nblocks)
        == SDMMC_Status_Err) {
        SDMMC_DEBUG_PRINTF("ERROR : CMD25 return false. %s, %d\r\n",
            __FILE__, __LINE__);

        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG("SDMMC : Async write DATA");

    return SDMMC_Status_Ok;
}

sdmmc_status_t SDMMC_WriteWait(sdmmc_card_t *sd)
{
    return SDMMC_Unlock(sd);
}

static int32_t SDMMC_GetInputClockHz(void)
{
    return CLKCTR_GetSysClk(CLOCK_BASE);
}

static sdmmc_status_t SDMMC_CmdReadMultiple(sdmmc_card_t *sd,
    uint32_t start_block, uint32_t baddr, uint32_t nblocks)
{
    uint32_t trans_mod, cmd_reg, blk_size, blk_count;
    uint32_t adma_addr = baddr;

    blk_size = SDMMC_BLOCKSIZE_XFER_BLOCKSIZE_VAL(SDMMC_SDHC_SECTOR_SIZE)
        | SDMMC_BLOCKSIZE_SDMA_BUFBOUNDARY_VAL(SDMMC_SDMA_ALIGN);
    blk_count = SDMMC_BLOCKCOUNT_XFER_BLOCKCOUNT_VAL(nblocks);
    trans_mod = SDMMC_TRANSFERMODE_XFERMODE_DMAENABLE_VAL(1)                   /* DMA - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_BLKCNTENA_VAL(1)                         /* BLK - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_AUTOCMDENA_VAL(1)                        /* AUTO CMD12 - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_DATAXFERDIR_VAL(SDMMC_SDMA_TransferRead) /* READ */
        | SDMMC_TRANSFERMODE_XFERMODE_MULTIBLKSEL_VAL(1);                      /* MUL SEL - EN */
    cmd_reg   = SDMMC_COMMAND_COMMAND_RESPONSETYPE_VAL(SDMMC_ResponceLength48) /* 48 bit */
        | SDMMC_COMMAND_COMMAND_DATAPRESENT_VAL(1)                             /* DATA RDY - EN */
        | SDMMC_COMMAND_COMMAND_CMDINDEX_VAL(SDMMC_READ_MULTIPLE_BLOCK);       /* CMD */

    sd->regs->NORMALINTRSTS |= SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_Msk
        | SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_Msk
        | SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_Msk;

    SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, baddr);
    SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, baddr >> 16);
    sd->regs->BLOCKSIZE  = blk_size;
    sd->regs->BLOCKCOUNT = blk_count;
    SDMMC_ARGUMENT1LO_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, start_block);
    SDMMC_ARGUMENT1HI_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, start_block >> 16);
    sd->regs->TRANSFERMODE = trans_mod;
    sd->regs->COMMAND = cmd_reg;

    if (sd->lock == SDMMC_StartOperationLock) {
        sd->lock = SDMMC_ReadAsyncLock;

        return SDMMC_Status_Ok;
    } else if (sd->lock != SDMMC_NoLock) {
        SDMMC_DEBUG_PRINTF("ERROR : CMD18 : Locked operation %s, %d\r\n",
            __FILE__, __LINE__);

        return SDMMC_Status_Ok;
    }

    while (SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_PTYPE_GET(sd->regs)
        == 0) {
        if (sd->regs->ERRORINTRSTS != 0) {
            SDMMC_DEBUG_PRINTF(
                "ERROR : CMD18 has some errors 0x%08x. %s, %d\r\n",
                sd->regs->ERRORINTRSTS, __FILE__, __LINE__);

            return SDMMC_Status_Err;
        }
    }

    while (1) {
        if (SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_PTYPE_GET(sd->regs)
            != 0) {
            /* Передача завершена. */
            sd->regs->NORMALINTRSTS |=
                SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_Msk;
            break;
        }

        if (SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_PTYPE_GET(sd->regs)
            != 0) {
            SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_PTYPE_SET(sd->regs,
                1);
            adma_addr += SDMMC_SDMA_BLOCK_SIZE;
            SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, adma_addr);
            SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_SET(sd->regs,
                adma_addr >> 16);
        }

        if (sd->regs->ERRORINTRSTS != 0) {
            SDMMC_DEBUG_PRINTF(
                "ERROR : CMD18 has some errors 0x%08x. %s, %d\r\n",
                sd->regs->ERRORINTRSTS, __FILE__, __LINE__);

            return SDMMC_Status_Err;
        }
    }

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_CmdExecute(sdmmc_card_t *sd, uint32_t sdma_addr,
    uint32_t blk_sz_cnt, uint32_t arg1, uint32_t trans_mod_com)
{
    SDMMC_DEBUG_PRINTF(
        "EXEC : ADDR 0x%lx, blk_sz_cnt 0x%lx, arg1 0x%lx, trans_mod_com 0x%lx\r\n",
        sdma_addr, blk_sz_cnt, arg1, trans_mod_com);

    SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, sdma_addr);
    SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, sdma_addr >> 16);
    sd->regs->BLOCKSIZE  = blk_sz_cnt;
    sd->regs->BLOCKCOUNT = blk_sz_cnt >> 16;
    SDMMC_ARGUMENT1LO_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, arg1);
    SDMMC_ARGUMENT1HI_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, arg1 >> 16);
    sd->regs->TRANSFERMODE = trans_mod_com;
    sd->regs->COMMAND = trans_mod_com >> 16;

    while (SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_PTYPE_GET(sd->regs)
        == 0) {
        if (sd->regs->ERRORINTRSTS != 0) {
            SDMMC_DEBUG_MSG_VAL("SDMMC : CMD Error",
                SDMMC_COMMAND_COMMAND_CMDINDEX_PTYPE_GET(sd->regs));

            SDMMC_DEBUG_PRINTF("SDMMC : EINTSTS 0x%08x\r\n",
                sd->regs->ERRORINTRSTS);

            return SDMMC_Status_Err;
        }
    }

    /* Сброс бита прерывания завершения команды. */
    sd->regs->NORMALINTRSTS |= SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_Msk
        | SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_Msk
        | SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_Msk;

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_CmdSend(sdmmc_card_t *sd, uint32_t cmd,
    uint32_t arg1, uint32_t resp_sz, uint32_t *resp)
{
    SDMMC_DEBUG_MSG_VAL("CMD", cmd);

    uint32_t i, block = 0;

    cmd = SDMMC_COMMAND_COMMAND_CMDINDEX_VAL(cmd);

    if (resp_sz == 1) {
        cmd |= SDMMC_COMMAND_COMMAND_RESPONSETYPE_VAL(SDMMC_ResponceLength48);  /* 48 bit */
    } else if (resp_sz > 1) {
        cmd |= SDMMC_COMMAND_COMMAND_RESPONSETYPE_VAL(SDMMC_ResponceLength136); /* 136 bit */
    }

    if (SDMMC_CmdExecute(sd, 0x00000000, block, arg1, cmd << 16)
        == SDMMC_Status_Ok) {
        if (resp != 0) {
            for (i = 0; i < resp_sz; i++) {
                resp[i] = SDMMC_GetResponse(sd, i + 1);
            }
        }

        return SDMMC_Status_Ok;
    } else {
        return SDMMC_Status_Err;
    }
}

static sdmmc_status_t SDMMC_CmdR2Send(sdmmc_card_t *sd, uint32_t cmd,
    uint32_t arg1, uint32_t *reg128)
{
    uint32_t i, blk_size, blk_count, cmd_reg, trans_mod;

    if (reg128 == NULL) {
        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG_VAL("R2 CMD", cmd);

    blk_size  = SDMMC_BLOCKSIZE_XFER_BLOCKSIZE_VAL(16)
        | SDMMC_BLOCKSIZE_SDMA_BUFBOUNDARY_VAL(SDMMC_SDMA_ALIGN);
    blk_count = SDMMC_BLOCKCOUNT_XFER_BLOCKCOUNT_VAL(1);
    cmd_reg   = SDMMC_COMMAND_COMMAND_RESPONSETYPE_VAL(SDMMC_ResponceLength136) /* 48 bit */
        | SDMMC_COMMAND_COMMAND_CRCCHKENA_VAL(1)                                /* CRC CHK EN */
        | SDMMC_COMMAND_COMMAND_CMDINDEX_VAL(cmd);                              /* CMD */
    trans_mod =
        SDMMC_TRANSFERMODE_XFERMODE_DATAXFERDIR_VAL(SDMMC_SDMA_TransferRead);

    if (SDMMC_CmdExecute(sd, 0x00000000, blk_size | (blk_count << 16),
            arg1, trans_mod | (cmd_reg << 16)) == SDMMC_Status_Ok) {
        for (i = 0; i < 4; i++) {
            reg128[i] = SDMMC_GetResponse(sd, i + 1);
        }

        return SDMMC_Status_Ok;
    } else {
        return SDMMC_Status_Err;
    }
}

static sdmmc_status_t SDMMC_MMC_GetExtCSD(sdmmc_card_t *sd, void *reg512)
{
    uint32_t blk_size, blk_count, cmd_reg, trans_mod;
    uint32_t baddr = (uint32_t) reg512;

    if (reg512 == NULL) {
        return SDMMC_Status_Err;
    }

    SDMMC_DEBUG_MSG("SEND_CSD_EXT");

    if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    blk_size = SDMMC_BLOCKSIZE_XFER_BLOCKSIZE_VAL(512)
        | SDMMC_BLOCKSIZE_SDMA_BUFBOUNDARY_VAL(SDMMC_SDMA_ALIGN);
    blk_count = SDMMC_BLOCKCOUNT_XFER_BLOCKCOUNT_VAL(1);
    trans_mod = SDMMC_TRANSFERMODE_XFERMODE_DMAENABLE_VAL(1)                    /* DMA - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_BLKCNTENA_VAL(1)                          /* BLK - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_DATAXFERDIR_VAL(SDMMC_SDMA_TransferRead); /* READ */
    cmd_reg   = SDMMC_COMMAND_COMMAND_RESPONSETYPE_VAL(SDMMC_ResponceLength48)  /* 48 bit */
        | SDMMC_COMMAND_COMMAND_DATAPRESENT_VAL(1)                              /* DATA RDY - EN */
        | SDMMC_COMMAND_COMMAND_CMDINDEX_VAL(SDMMC_MMC_SEND_EXT_CSD);           /* CMD */

    sd->regs->NORMALINTRSTS |= SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_Msk
        | SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_Msk
        | SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_Msk;
    SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, baddr);
    SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, baddr >> 16);
    sd->regs->BLOCKSIZE  = blk_size;
    sd->regs->BLOCKCOUNT = blk_count;
    SDMMC_ARGUMENT1LO_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, 0);
    SDMMC_ARGUMENT1HI_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, 0);
    sd->regs->TRANSFERMODE = trans_mod;
    sd->regs->COMMAND = cmd_reg;

    while (SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_PTYPE_GET(sd->regs)
        == 0) {
        if (sd->regs->ERRORINTRSTS != 0) {
            return SDMMC_Status_Err;
        }
    }

    while (1) {
        if (SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_PTYPE_GET(sd->regs)
            != 0) {
            /* Передача завершена. */
            sd->regs->NORMALINTRSTS |=
                SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_Msk;
            break;
        }

        if (sd->regs->ERRORINTRSTS != 0) {
            return SDMMC_Status_Err;
        }
    }

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_GoIdleState(sdmmc_card_t *sd)
{
    SDMMC_DEBUG_MSG("CMD0");

    return SDMMC_CmdSend(sd, SDMMC_GO_IDLE_STATE, 0, 0, 0);
}

static sdmmc_status_t SDMMC_MMC_CmdSendOpCond(sdmmc_card_t *sd)
{
    sdmmc_mmc_ocr_reg_t ocr;
    sdmmc_mmc_ocr_reg_t arg1;
    uint32_t timeout = sd->cfg->timeout_init;
    int32_t hc = 0;

    SDMMC_DEBUG_MSG("CMD1");

    arg1.value = 0;
    arg1.access_mode = SDMMC_MMC_ACCESS_MODE_SECTOR;

    if (sd->gpio_vol == SDMMC_1v8) {
        arg1.vol_1v7_1v95 = 0x1;
    } else {
        arg1.vol_2v7_3v6 = 0x1FF;
    }

    do {
        if (timeout == 0) {
            return SDMMC_Status_Err;
        }

        if (SDMMC_CmdSend(sd, SDMMC_MMC_SEND_OP_COND, arg1.value, 1, &ocr.value)
            == SDMMC_Status_Err) {
            if (!hc) {
                hc = 1;
                SDMMC_RestartController(sd, sd->freq_divider);

                if (!SDMMC_GoIdleState(sd)) {
                    return SDMMC_Status_Err;
                }
                arg1.access_mode = SDMMC_MMC_ACCESS_MODE_BYTE;
                ocr.value = 0;
            } else {
                return SDMMC_Status_Err;
            }
        }
        --timeout;
        SDMMC_Udelay(sd, SDMMC_Timeout_1ms);
    } while (ocr.busy == 0);

    sd->sdhc_mode = (ocr.access_mode == SDMMC_MMC_ACCESS_MODE_SECTOR);

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_CmdSelectCard(sdmmc_card_t *sd)
{
    return SDMMC_CmdSend(sd, SDMMC_SELECT_CARD, sd->rca, 1, 0);
}

static sdmmc_status_t SDMMC_CheckSupportedVoltage(sdmmc_card_t *sd)
{
    uint32_t resp = 0;

    if (SDMMC_IS_MMC(sd) || (sd->version == 1)) {
        return SDMMC_Status_Ok;
    }

    SDMMC_DEBUG_MSG("CMD8");

    if (SDMMC_CmdSend(sd, SDMMC_SD_SEND_IF_COND, SDMMC_SD_SEND_IF_COND_PATTERN,
            1, &resp) == SDMMC_Status_Err) {
        SDMMC_DEBUG_MSG("ERROR : CMD8 not response");
        sd->version = 1;
        SDMMC_RestartController(sd, sd->freq_divider);

        return SDMMC_GoIdleState(sd);
    } else {
        sd->version = 2;

        return SDMMC_Status_Ok;
    }
}

static sdmmc_status_t SDMMC_CmdSetBlockLength(sdmmc_card_t *sd,
    uint32_t block_size)
{
    if (SDMMC_IS_MMC(sd) && sd->ddr_mode) {
        return SDMMC_Status_Ok;
    }

    return SDMMC_CmdSend(sd, SDMMC_SET_BLOCKLEN, block_size, 1, 0);
}

static sdmmc_status_t SDMMC_CmdWriteMultiple(sdmmc_card_t *sd,
    uint32_t start_block, uint32_t baddr, uint32_t nblocks)
{
    uint32_t trans_mod, cmd_reg, blk_size, blk_count;
    uint32_t adma_addr = baddr;

    blk_size = SDMMC_BLOCKSIZE_XFER_BLOCKSIZE_VAL(SDMMC_SDHC_SECTOR_SIZE)
        | SDMMC_BLOCKSIZE_SDMA_BUFBOUNDARY_VAL(SDMMC_SDMA_ALIGN);
    blk_count = SDMMC_BLOCKCOUNT_XFER_BLOCKCOUNT_VAL(nblocks);
    trans_mod = SDMMC_TRANSFERMODE_XFERMODE_DMAENABLE_VAL(1)                    /* DMA - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_BLKCNTENA_VAL(1)                          /* BLK - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_AUTOCMDENA_VAL(1)                         /* AUTO CMD12 - EN */
        | SDMMC_TRANSFERMODE_XFERMODE_DATAXFERDIR_VAL(SDMMC_SDMA_TransferWrite) /* WRITE */
        | SDMMC_TRANSFERMODE_XFERMODE_MULTIBLKSEL_VAL(1);                       /* MUL SEL - EN */
    cmd_reg   = SDMMC_COMMAND_COMMAND_RESPONSETYPE_VAL(SDMMC_ResponceLength48)  /* 48 bit */
        | SDMMC_COMMAND_COMMAND_DATAPRESENT_VAL(1)                              /* DATA RDY - EN */
        | SDMMC_COMMAND_COMMAND_CMDINDEX_VAL(SDMMC_WRITE_MULTIPLE_BLOCK);       /* CMD */

    SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, baddr);
    SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, baddr >> 16);
    sd->regs->BLOCKSIZE  = blk_size;
    sd->regs->BLOCKCOUNT = blk_count;
    SDMMC_ARGUMENT1LO_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, start_block);
    SDMMC_ARGUMENT1HI_COMMAND_ARGUMENT1_PTYPE_SET(sd->regs, start_block >> 16);
    sd->regs->TRANSFERMODE = trans_mod;
    sd->regs->COMMAND = cmd_reg;

    if (sd->lock == SDMMC_StartOperationLock) {
        sd->lock = SDMMC_WriteAsyncLock;

        return SDMMC_Status_Ok;
    } else if (sd->lock != SDMMC_NoLock) {
        SDMMC_DEBUG_PRINTF("ERROR : CMD25 : Locked operation %s, %d\r\n",
            __FILE__, __LINE__);

        return SDMMC_Status_Ok;
    }
    sd->regs->ERRORINTRSTS = 0xFFFF;
    while (SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_PTYPE_GET(sd->regs)
        == 0) {
        if (sd->regs->ERRORINTRSTS != 0) {
            SDMMC_DEBUG_PRINTF(
                "ERROR : CMD25 has some errors 0x%08x. %s, %d\r\n",
                sd->regs->ERRORINTRSTS, __FILE__, __LINE__);

            return SDMMC_Status_Err;
        }
    }

    while (1) {
        if (SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_PTYPE_GET(sd->regs)
            != 0) {
            /* Передача завершена. */
            sd->regs->NORMALINTRSTS |=
                SDMMC_NORMALINTRSTS_NORMALINTRSTS_CMDCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_XFERCOMPLETE_Msk
                | SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_Msk;
            break;
        }

        if (SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_PTYPE_GET(sd->regs)
            != 0) {
            SDMMC_NORMALINTRSTS_NORMALINTRSTS_DMAINTERRUPT_PTYPE_SET(sd->regs,
                1);
            adma_addr += SDMMC_SDMA_BLOCK_SIZE;
            SDMMC_SDMASYSADDRLO_SDMA_SYSADDRESS_PTYPE_SET(sd->regs, adma_addr);
            SDMMC_SDMASYSADDRHI_SDMA_SYSADDRESS_PTYPE_SET(sd->regs,
                adma_addr >> 16);
        }

        if (sd->regs->ERRORINTRSTS != 0) {
            SDMMC_DEBUG_PRINTF(
                "ERROR : CMD25 has some errors 0x%08x. %s, %d\r\n",
                sd->regs->ERRORINTRSTS, __FILE__, __LINE__);

            return SDMMC_Status_Err;
        }
    }

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_CmdSetBusWidth(sdmmc_card_t *sd)
{
    if (SDMMC_IS_MMC(sd)) {
        sdmmc_mmc_ext_csd_arg_t arg1;

        arg1.arg = 0;
        arg1.index = SDMMC_EXT_CSD_DATA_WIDTH_OFFSET;
        arg1.access = SDMMC_EXT_CSD_ACCESS_WRITE_BYTE;
        if (sd->cfg->emmc_8bit_en) {
            SDMMC_InitGPIO_8Bit(sd);
            arg1.value = SDMMC_EXT_CSD_DATA_WIDTH_8BIT;
        } else {
            arg1.value = SDMMC_EXT_CSD_DATA_WIDTH_4BIT;
        }

#if SDMMC_IN_PROGRESS_CODE
        if (sd->ddr_mode) {
            arg1.value |= SDMMC_EXT_CSD_DATA_WIDTH_DDR_EN;
        }
#endif

        if (SDMMC_CmdSend(sd, SDMMC_SWITCH, arg1.arg, 1, 0)
            == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }

        if (SDMMC_WaitTRANState(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }

        if (sd->cfg->emmc_8bit_en) {
            SDMMC_HOSTCONTROL1_HOSTCTRL1_EXTDATAWIDTH_PTYPE_SET(sd->regs,
                SDMMC_ExtDataBusTransferWidth_8bit);
        }

#if SDMMC_IN_PROGRESS_CODE
        sd->regs->HOST_CTRL_2 =
            SET_SDMMC_HOST_CTRL_2_V_1_8_SIG_EN(sd->regs->HOST_CTRL_2, 1);
        sd->regs->CORECFG5 = sd->freq_input / SDMMC_FREQ_1MHZ;
        sd->regs->HOST_CTRL_2 =
            SET_SDMMC_HOST_CTRL_2_UHS_MODE_SEL(sd->regs->HOST_CTRL_2,
                SDMMC_SD_UHS_MODE_DDR50);
        sd->regs->HOST_POW_CNTL |= (1 << 2);
#endif
    } else {
        if (SDMMC_PrepareAcmd(sd) == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }

        if (SDMMC_CmdSend(sd, SDMMC_ACMD_SET_BUS_WIDTH,
                SDMMC_SD_ACMD_SET_BUS_WIDTH_4BIT, 1, 0)
            == SDMMC_Status_Err) {
            return SDMMC_Status_Err;
        }
    }

    SDMMC_HOSTCONTROL1_HOSTCTRL1_DATAWIDTH_PTYPE_SET(sd->regs,
        SDMMC_DataBusTransferWidth_4bit);

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_CmdSendOpCond(sdmmc_card_t *sd)
{
    sdmmc_sd_acmd41_arg_t arg1;
    sdmmc_sd_acmd41_resp_t resp1;
    uint32_t timeout = sd->cfg->timeout_init;

    SDMMC_DEBUG_MSG("ACMD41");

    if (SDMMC_IS_MMC(sd)) {
        return SDMMC_MMC_CmdSendOpCond(sd);
    }

    arg1.arg = 0;

    if (sd->version > 1) {
        arg1.ocr = SDMMC_SD_OCR_INIT_VALUE;
        arg1.hcs = 1;

        if (sd->gpio_vol == SDMMC_1v8) {
            arg1.s18r = 1;
        }
    }
    do {
        if (timeout == 0) {
            break;
        }

        if (SDMMC_PrepareAcmd(sd) == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("SDMMC : can't ACMD55");

            if (sd->version == 1) {
                sd->type = SDMMC_TypeMMC;
                SDMMC_MMC_SetOdCmdPin(sd, 1);
                SDMMC_RestartController(sd, sd->freq_divider);

                if (SDMMC_GoIdleState(sd) == SDMMC_Status_Err) {
                    return SDMMC_Status_Err;
                }

                return SDMMC_MMC_CmdSendOpCond(sd);
            } else {
                return SDMMC_Status_Err;
            }
        }

        --timeout;
        SDMMC_Udelay(sd, SDMMC_Timeout_1ms);

        if (SDMMC_CmdSend(sd, SDMMC_ACMD_SD_SEND_OP_COND, arg1.arg, 1,
                &resp1.value) == SDMMC_Status_Err) {
            SDMMC_DEBUG_MSG("SDMMC : can't ACMD41");

            return SDMMC_Status_Err;
        }
    } while (resp1.busy == 0);

    if (resp1.busy == 0) {
        SDMMC_DEBUG_MSG("ERROR : Resp[31] = 0");

        return SDMMC_Status_Err;
    }

    sd->sdhc_mode = (resp1.ccs != 0);

    if ((resp1.s18r == 1) && (sd->gpio_vol ==  SDMMC_3v3To1v8)) {
        sd->need_1V8en = 1;
    }

    return SDMMC_Status_Ok;
}

static sdmmc_status_t SDMMC_CmdClearCardDetect(sdmmc_card_t *sd)
{
    if (SDMMC_IS_MMC(sd)) {
        return SDMMC_Status_Ok;
    }

    if (SDMMC_PrepareAcmd(sd) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    return SDMMC_CmdSend(sd, SDMMC_ACMD_SET_CLR_CARD_DETECT, 0, 1, 0);
}

static sdmmc_status_t SDMMC_GetRCA(sdmmc_card_t *sd)
{
    if (SDMMC_CmdSend(sd, SDMMC_ALL_SEND_CID, 0, 4, 0) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    if (SDMMC_CmdSend(sd, SDMMC_SET_RELATIVE_ADDR, SDMMC_MMC_RCA_ADDR, 1,
            (uint32_t *) &sd->rca) == SDMMC_Status_Err) {
        return SDMMC_Status_Err;
    }

    if (SDMMC_IS_MMC(sd)) {
        sd->rca = SDMMC_MMC_RCA_ADDR;
    } else {
        sd->rca &= 0xFFFF0000;
    }

    SDMMC_DEBUG_MSG_VAL("RCA", sd->rca);

    return SDMMC_Status_Ok;
}

static void SDMMC_EnableInputClock(sdmmc_card_t *sd)
{
    sd->freq_input = SDMMC_GetInputClockHz();
}

static void SDMMC_InitGPIO(sdmmc_card_t *sd)
{
    /* Используются выводы D0-D3, CMD, CK, не используются выводы CD, WP. */
    GPIO_PinMode_Function(sd->cfg->pin_map.CK,  GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.CMD, GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.D0,  GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.D1,  GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.D2,  GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.D3,  GPIO_ALT_FUNC_SDMMC_SMC);

    /* Максимальный ток выводов 12 mA. */
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.CK,  sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.CMD, sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D0,  sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D1,  sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D2,  sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D3,  sd->cfg->pin_map.max_current);

    /* Включение резистивных подтяжек на выводах. */
    GPIO_PinSet_PUPD(sd->cfg->pin_map.CK,  GPIO_PULL_NONE);
    GPIO_PinSet_PUPD(sd->cfg->pin_map.CMD, GPIO_PULL_NONE);

    if (sd->cfg->pin_map.pull_en) {
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D0, GPIO_PULL_UP);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D1, GPIO_PULL_UP);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D2, GPIO_PULL_UP);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D3, GPIO_PULL_UP);
    } else {
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D0, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D1, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D2, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D3, GPIO_PULL_NONE);
    }
}

static void SDMMC_InitGPIO_8Bit(sdmmc_card_t *sd)
{
    /* Используются выводы D4-D7. */
    GPIO_PinMode_Function(sd->cfg->pin_map.D4, GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.D5, GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.D6, GPIO_ALT_FUNC_SDMMC_SMC);
    GPIO_PinMode_Function(sd->cfg->pin_map.D7, GPIO_ALT_FUNC_SDMMC_SMC);

    /* Максимальный ток выводов 12mA. */
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D4, sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D5, sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D6, sd->cfg->pin_map.max_current);
    GPIO_PinSet_MaxCurrent(sd->cfg->pin_map.D7, sd->cfg->pin_map.max_current);

    /* Включение резистивных подтяжек на выводах. */
    if (sd->cfg->pin_map.pull_en) {
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D4, GPIO_PULL_UP);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D5, GPIO_PULL_UP);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D6, GPIO_PULL_UP);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D7, GPIO_PULL_UP);
    } else {
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D4, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D5, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D6, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D7, GPIO_PULL_NONE);
    }
}

static void SDMMC_DisableGPIO(sdmmc_card_t *sd)
{
    GPIO_PinSet_PUPD(sd->cfg->pin_map.D0, GPIO_PULL_NONE);
    GPIO_PinSet_PUPD(sd->cfg->pin_map.D1, GPIO_PULL_NONE);
    GPIO_PinSet_PUPD(sd->cfg->pin_map.D2, GPIO_PULL_NONE);
    GPIO_PinSet_PUPD(sd->cfg->pin_map.D3, GPIO_PULL_NONE);
    GPIO_PinMode_HiZ(sd->cfg->pin_map.CK);
    GPIO_PinMode_HiZ(sd->cfg->pin_map.CMD);
    GPIO_PinMode_HiZ(sd->cfg->pin_map.D0);
    GPIO_PinMode_HiZ(sd->cfg->pin_map.D1);
    GPIO_PinMode_HiZ(sd->cfg->pin_map.D2);
    GPIO_PinMode_HiZ(sd->cfg->pin_map.D3);

    if (sd->cfg->emmc_8bit_en) {
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D4, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D5, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D6, GPIO_PULL_NONE);
        GPIO_PinSet_PUPD(sd->cfg->pin_map.D7, GPIO_PULL_NONE);
        GPIO_PinMode_HiZ(sd->cfg->pin_map.D4);
        GPIO_PinMode_HiZ(sd->cfg->pin_map.D5);
        GPIO_PinMode_HiZ(sd->cfg->pin_map.D6);
        GPIO_PinMode_HiZ(sd->cfg->pin_map.D7);
    }
}

static void SDMMC_MMC_SetOdCmdPin(sdmmc_card_t *sd, int32_t od_en)
{
    GPIO_PinSet_PUPD(sd->cfg->pin_map.CD,
        od_en != 0 ? GPIO_PULL_UP_OD : GPIO_PULL_NONE);
}

static void SDMMC_SetClockDivider(sdmmc_card_t *sd, int32_t clk_div)
{
    SDMMC_CLOCKCONTROL_CLKCTRL_SDCLKFREQSEL_PTYPE_SET(sd->regs, clk_div);
    SDMMC_CLOCKCONTROL_CLKCTRL_SDCLKFREQSEL_UPPERBITS_PTYPE_SET(sd->regs, 0);
    SDMMC_CLOCKCONTROL_CLKCTRL_CLKGENSEL_PTYPE_SET(sd->regs, 0);
    SDMMC_CLOCKCONTROL_CLKCTRL_SDCLKENA_PTYPE_SET(sd->regs, 1);
}

static void SDMMC_RestartController(sdmmc_card_t *sd, int32_t clk_div)
{
    SDMMC_DEBUG_MSG("SDMMC : Start reset");

    SDMMC_Reset(sd);

    SDMMC_CLOCKCONTROL_CLKCTRL_INTCLKENA_PTYPE_SET(sd->regs, 1);
    while (SDMMC_CLOCKCONTROL_SDHCCLKGEN_INTCLKSTABLE_DSYNC_PTYPE_GET(sd->regs)
        == 0)
        ;

    SDMMC_DEBUG_MSG("SDMMC : Clock enable");

    SDMMC_SetClockDivider(sd, clk_div);
    sd->regs->NORMALINTRSTSENA = SDMMC_NORMAL_INT_MASK;
    sd->regs->ERRORINTRSTSENA = SDMMC_ERROR_INT_MASK;
    SDMMC_Udelay(sd, SDMMC_Timeout_Min);
}

static uint32_t SDMMC_GetResponse(sdmmc_card_t *sd, int32_t num)
{
    switch (num) {
        case 1:
            return SDMMC_RESPONSE0_COMMAND_RESPONSE_PTYPE_GET(sd->regs)
                | (SDMMC_RESPONSE1_COMMAND_RESPONSE_PTYPE_GET(sd->regs) << 16);
        case 2:
            return SDMMC_RESPONSE2_COMMAND_RESPONSE_PTYPE_GET(sd->regs)
                | (SDMMC_RESPONSE3_COMMAND_RESPONSE_PTYPE_GET(sd->regs) << 16);
        case 3:
            return SDMMC_RESPONSE4_COMMAND_RESPONSE_PTYPE_GET(sd->regs)
                | (SDMMC_RESPONSE5_COMMAND_RESPONSE_PTYPE_GET(sd->regs) << 16);
        case 4:
            return SDMMC_RESPONSE6_COMMAND_RESPONSE_PTYPE_GET(sd->regs)
                | (SDMMC_RESPONSE7_COMMAND_RESPONSE_PTYPE_GET(sd->regs) << 16);
        default:
            SDMMC_DEBUG_MSG(
                "ERROR : Wrong argument num for SDMMC_GetResponse()");
            break;
    }

    return 0;
}

/*!
 * @}
 */
