/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup spi_driver
 * @{
 */

/*!
 * @file hal_spi.c
 * @brief Имплементация драйвера модуля SPI
 */

#include "hal_spi.h"
#include "hal_ioim.h"

#define SPI_FIFO_DEPTH 8 /*!< Глубина FIFO. */

/*!
 * @brief Маска для доверенного/недоверенного SPI
 */
#define SPI_ADDRESS_SEC_MSK (0x10000000UL)

/*!
 * @brief Тип модуля SPI
 */
typedef enum {
    Master = 0, /*!< Модуль SPI работает в режиме Master */
    Slave  = 1, /*!< Модуль SPI работает в режиме Slave */
} spi_unit_type_t;

/*!
 * @brief Преобразование размера кадра данных (#spi_frame_width_t) в 8-битное
 *        беззнаковое число
 */
#define SPI_DATA_WIDTH(x) (((uint8_t) (x)) + 1)

/*!
 * @brief Массив внутренних конфигураций SPI
 */
static spi_config_internal_t spi_config_internal[SPI_BASE_ADDRS_COUNT] = {0};

/*!
 * @brief Массив базовых адресов SPI модулей
 */
static const uint32_t spi_unit_base_addrs[SPI_BASE_ADDRS_COUNT] = SPI_BASE_ADDRS;

/*!
 * @brief Максимальная разрядность данных для SPI модулей
 */
static const uint32_t spi_units_frame_width[SPI_BASE_ADDRS_COUNT] =
    SPI_UNITS_DATA_WIDTH;

/*!
 * @brief Режим работы модуля (задан аппаратно)
 */
static const spi_unit_type_t spi_unit_type[SPI_BASE_ADDRS_COUNT] = SPI_UNITS_TYPE;

/*!
 * @brief Фиктивные данные для каждого модуля SPI
 *
 * Используются когда нет пользовательских данных для передачи.
 */
volatile uint8_t spi_dummy_data[SPI_BASE_ADDRS_COUNT] = {0};

/*!
 * @brief Сброс содержимого TxFIFO и RxFIFO
 *
 * @param base Базовый адрес SPI
 */
static inline void SPI_FifoClr(SPI_Type *base)
{
    uint8_t enable;

    assert(NULL != base);

    /* Очистка TxFIFO и RxFIFO (регистры сбрасываются при SSI_EN = 0). */

    /* Получить текущее состояние  SPI. */
    enable = GET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk,
            SPI_SSIENR_SSI_EN_Pos);

    SET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk, SPI_SSIENR_SSI_EN_Pos,
        0U);

    SET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk, SPI_SSIENR_SSI_EN_Pos,
        enable);
}

/*!
 * @brief Извлечение заданного размера кадра данных
 *
 * @param frame_width_bits Размер кадра данных в битах
 *
 * @return Размер кадра данных в байтах
 */
static uint32_t _DataWidth_Bytes(uint32_t frame_width_bits)
{
    uint32_t frame_width_bytes;

    assert(frame_width_bits <= 32);

    /* Вычисление ширины данных в байтах. */
    if (frame_width_bits > 16) {
        frame_width_bytes = 4;
    } else if (frame_width_bits > 8) {
        frame_width_bytes = 2;
    } else {
        frame_width_bytes = 1;
    }

    return frame_width_bytes;
}

uint8_t SPI_GetInstance(SPI_Type *base)
{
    uint8_t i;

    for (i = 0U; i < (uint32_t) SPI_BASE_ADDRS_COUNT; i++) {
        if ((((uint32_t) base) & ~SPI_ADDRESS_SEC_MSK)
            == spi_unit_base_addrs[i]) {
            break;
        }
    }

    assert(i < ((uint32_t) SPI_BASE_ADDRS_COUNT));

    return i;
}

void SPI_SetDummyData(SPI_Type *base, uint8_t dummy_data)
{
    uint8_t instance = SPI_GetInstance(base);

    spi_dummy_data[instance] = dummy_data;
}

spi_config_internal_t *SPI_GetConfig(SPI_Type *base)
{
    uint8_t instance;

    instance = SPI_GetInstance(base);

    return &spi_config_internal[instance];
}

static void SPI_GetDefaultConfig(spi_config_t *config)
{
    assert(NULL != config);

    (void) memset(config, 0, sizeof(*config));

    /* Включение закольцовывания (в целях тестирования). */
    config->master.loopback_enable = false;
    /* Скорость обмена SPI в Hz. */
    config->master.baud_rate_bps = 500000U;
    /* Размер кадра данных. */
    config->data_width_bits = SPI_Data8Bits;
    /* Формат передачи данных, MSB или LSB. */
    config->direction = SPI_ShiftDirMsbFirst;
    /* Определение формата кадра протокола передачи данных. */
    config->frame_format = SPI_FfMotorola;

    /* Конфигурация для протокола National Semiconductor Microwire. */

    /* Выбор длины управляющего слова для протокола Microwire. */
    config->microwire_cfg.ctrl_word_len = SPI_MicrowireCtrlWordLen8Bit;

    /* Установка проверки busy/ready флага. */
    config->microwire_cfg.busy_ready_check = SPI_MicrowireBusyReadyCheckDisable;

    /* Направление передачи слова данных. */
    config->microwire_cfg.tx_rx = SPI_MicrowireTx;

    /* Выбор типа передачи. */
    config->microwire_cfg.single_serial = SPI_MicrowireSingle;

    /* Конфигурация для протокола Motorola. */

    /*
     * Полярность тактового сигнала, линия синхронизации до начала цикла
     * передачи и после его окончания имеют низкий уровень.
     */
    config->motorola_cfg.clk_pol = SPI_MotorolaClkPolLow;

    /* Захват данных происходит по переднему фронту тактового сигнала. */
    config->motorola_cfg.cap_data = SPI_MotorolaCapDataRising;
}

void SPI_MasterGetDefaultConfig(spi_config_t *config)
{
    SPI_GetDefaultConfig(config);
}

static enum spi_status SPI_Init(SPI_Type *base, const spi_config_t *config,
    uint32_t src_clock_hz)
{
    enum spi_status result = SPI_Status_Ok;

    uint8_t instance = SPI_GetInstance(base);

    /* Проверка внутренних настроек драйвера. */
    assert((spi_units_frame_width[instance] == 16)
        || (spi_units_frame_width[instance] == 32));

    /* Проверка параметров. */
    if ((NULL == base) || (NULL == config)) {
        return SPI_Status_InvalidArgument;
    }

    if ((spi_unit_type[instance] == Master)
        && (src_clock_hz == 0)) {
        return SPI_Status_InvalidArgument;
    }

    /* Отключение прерываний. */
    SPI_DisableInterrupts(base, SPI_IRQ_All);

    /* Выключение SPI. */
    SPI_Enable(base, false);

    /* Для Master режима модуля SPI. */
    if (spi_unit_type[instance] == Master) {
        /* Установить делитель частоты. */
        result = SPI_MasterSetBaud(base, config->master.baud_rate_bps,
                src_clock_hz);
        if (SPI_Status_Ok != result) {
            return result;
        }
    }

    /* Настройка регистра CTRLR0. */
    {
        uint32_t reg = 0;

        /* Установка общей конфигурации SPI. */
        if (spi_units_frame_width[instance] == 16) {
            /*
             * Размер кадра данных при 16-битном обмене
             */
            SET_VAL_MSK(reg, SPI_CTRLR0_DFS_Msk, SPI_CTRLR0_DFS_Pos,
                config->data_width_bits);
        } else {
            /*
             * Размер кадра данных при 32-битном обмене
             */
            SET_VAL_MSK(reg, SPI_CTRLR0_DFS_32_Msk, SPI_CTRLR0_DFS_32_Pos,
                config->data_width_bits);
        }

        /* Выбор длины управляющего слова для протокола Microwire. */
        SET_VAL_MSK(reg, SPI_CTRLR0_CFS_Msk, SPI_CTRLR0_CFS_Pos,
            config->microwire_cfg.ctrl_word_len);
        /* Включить закольцовывание (в целях тестирования). */
        SET_VAL_MSK(reg, SPI_CTRLR0_SRL_Msk, SPI_CTRLR0_SRL_Pos,
            config->master.loopback_enable);
        /*
         * Установка режима работы модуля: дуплекс, симплекс (только передача), симплекс
         * (только прием), полудуплекс.
         */
        SET_VAL_MSK(reg, SPI_CTRLR0_TMOD_Msk, SPI_CTRLR0_TMOD_Pos,
            SPI_ModeDuplex);
        /*
         * Полярность тактового сигнала при отсутствии передаваемых данных в режиме
         * Master; только для режима (FRF) Motorola.
         */
        SET_VAL_MSK(reg, SPI_CTRLR0_SCPOL_Msk, SPI_CTRLR0_SCPOL_Pos,
            config->motorola_cfg.clk_pol);
        /*
         * Захват данных происходит по переднему или по заднему фронту; только для
         * режима (FRF) Motorola.
         */
        SET_VAL_MSK(reg, SPI_CTRLR0_SCPH_Msk, SPI_CTRLR0_SCPH_Pos,
            config->motorola_cfg.cap_data);

        /* Определение формата кадра (протокола) передачи данных. */
        SET_VAL_MSK(reg, SPI_CTRLR0_FRF_Msk, SPI_CTRLR0_FRF_Pos,
            config->frame_format);

        base->CTRLR0 = reg;
    }

    /* Настройка регистра MWCR. */
    /* Установка конфигурации для протокола Microwire National Semiconductor. */
    {
        uint32_t reg = 0;
        /*
         * Включение/отключение busy/ready проверки протокола Microwire.
         * В активном состоянии SPI проверяет готовность Slave.
         */
        SET_VAL_MSK(reg, SPI_MWCR_MHS_Msk, SPI_MWCR_MHS_Pos,
            config->microwire_cfg.busy_ready_check);
        /* Направление передачи слова данных. */
        SET_VAL_MSK(reg, SPI_MWCR_MDD_Msk, SPI_MWCR_MDD_Pos,
            config->microwire_cfg.tx_rx);
        /* Определяет одиночная или последовательная передача. */
        SET_VAL_MSK(reg, SPI_MWCR_MWMOD_Msk, SPI_MWCR_MWMOD_Pos,
            config->microwire_cfg.single_serial);

        base->MWCR = reg;
    }

    /* Сохранение конфигурации во внутренней структуре. */
    {
        uint32_t frame_width_bits = SPI_DATA_WIDTH(config->data_width_bits);
        /* Размер кадра данных в битах. */
        spi_config_internal[instance].frame_width_bits = frame_width_bits;

        /* Размер кадра данных в байтах. */
        spi_config_internal[instance].frame_width_bytes =
            _DataWidth_Bytes(frame_width_bits);

        /* Направление сдвига. */
        spi_config_internal[instance].shift_dir = config->direction;
    }

    /* Установка байта пустых данных по умолчанию. */
    SPI_SetDummyData(base, (uint8_t) SPI_DUMMYDATA);

    return SPI_Status_Ok;
}

enum spi_status SPI_MasterInit(SPI_Type *base, const spi_config_t *config,
    uint32_t src_clock_hz)
{
    enum spi_status status;

    /* Проверка корректности вызова функции с заданным модулем SPI. */
    assert(spi_unit_type[SPI_GetInstance(base)] == Master);

    status = SPI_Init(base, config, src_clock_hz);

    /*
     * Выставление сигнала Slave-Select произойдет только при передаче данных.
     */
    SET_VAL_MSK(base->SER, SPI_SER_SER_Msk, SPI_SER_SER_Pos, 1U);

    return status;
}

void SPI_SlaveGetDefaultConfig(spi_config_t *config)
{
    SPI_GetDefaultConfig((spi_config_t *) config);
}

enum spi_status SPI_SlaveInit(SPI_Type *base, const spi_config_t *config)
{
    /* Проверка корректности вызова функции с заданным модулем SPI. */
    assert(spi_unit_type[SPI_GetInstance(base)] == Slave);

    /*
     * src_clock_hz = 0, потому что фактически для режима Slave инициализация
     * генератора clk не производится.
     */
    return SPI_Init(base, config, 0);
}

void SPI_Deinit(SPI_Type *base)
{
    /* Проверка аргументов. */
    assert(NULL != base);

    /* Отключение прерываний. */
    SPI_DisableInterrupts(base, SPI_IRQ_All);

    /* Отключение DMA запросов. */
    base->DMACR = 0UL;

    /* Выключение SPI. */
    SPI_Enable(base, false);
}

void SPI_EnableTxDMA(SPI_Type *base, bool enable)
{
    SET_VAL_MSK(base->DMACR, SPI_DMACR_TDMAE_Msk, SPI_DMACR_TDMAE_Pos,
        enable);
}

void SPI_EnableRxDMA(SPI_Type *base, bool enable)
{
    SET_VAL_MSK(base->DMACR, SPI_DMACR_RDMAE_Msk, SPI_DMACR_RDMAE_Pos,
        enable);
}

enum spi_status SPI_MasterSetBaud(SPI_Type *base, uint32_t baud_rate_bps,
    uint32_t src_clock_hz)
{
    uint32_t tmp_div;
    bool unit_is_enable;

    /* Проверка параметров. */
    if ((NULL == base) || (0U == baud_rate_bps) || (0U == src_clock_hz)) {
        return SPI_Status_InvalidArgument;
    }

    unit_is_enable = SPI_IsEnable(base);

    /* Выключение SPI. */
    SPI_Enable(base, false);

    /*
     * Вычисление делителя частоты; допустимо любое четное число от 2 до 65534.
     */
    tmp_div = src_clock_hz / baud_rate_bps;

    if ((tmp_div < 2) || (tmp_div > 65534U) || (tmp_div % 2)) {
        return SPI_Status_BaudrateNotSupport;
    }

    /*
     * Делитель частоты тактового сигнала SPI (SCKDV); определяется формулой:
     * fsclk_out = fssi_clk / SCKDV.
     */
    SET_VAL_MSK(base->BAUDR, SPI_BAUDR_SCKDV_Msk, SPI_BAUDR_SCKDV_Pos,
        tmp_div);

    /* Возвращение состояния SPI бывшего до входа в функцию. */
    SPI_Enable(base, unit_is_enable);

    return SPI_Status_Ok;
}

uint32_t SPI_ReadData(SPI_Type *base)
{
    uint32_t data;
    uint8_t instance;

    /* Проверка параметров. */
    assert(NULL != base);
    instance = SPI_GetInstance(base);

    /* Чтение данных из RxFIFO. */
    data = base->DR0;

    /* Преобразование формата представления: big-endian <-> little-endian. */
    if (SPI_ShiftDirLsbFirst == spi_config_internal[instance].shift_dir) {
        LBBLEndian(&data, spi_config_internal[instance].frame_width_bits);
    }

    return data;
}

void SPI_WriteData(SPI_Type *base, uint32_t data)
{
    uint8_t instance;

    /* Проверка параметров. */
    assert(NULL != base);

    instance = SPI_GetInstance(base);

    /* Преобразование формата представления: big-endian <-> little-endian. */
    if (SPI_ShiftDirLsbFirst == spi_config_internal[instance].shift_dir) {
        LBBLEndian(&data, spi_config_internal[instance].frame_width_bits);
    }

    /* Запись данных в TxFIFO. */
    base->DR0 = data;
}

/*!
 * @brief Чтение данных из RxFifo
 *
 * @note Выход по окончании буфера на прием (*size) или после опустошения RxFIFO.
 * Функция перемещает указатель на буфер **rx_data.
 *
 * @param base Базовый адрес SPI
 * @param instance Порядковый номер экземпляра устройства
 * @param rx_data Указатель на указатель на приемный буфер
 * @param size Размер приемного буфера
 */
void _RxFifo_Read(SPI_Type *base, uint32_t instance, volatile uint8_t **rx_data,
    volatile size_t *size)
{
    uint32_t frame_width_bytes;
    uint32_t tmp32;

    /* Размер кадра данных в байтах. */
    frame_width_bytes = spi_config_internal[instance].frame_width_bytes;

    /* Пока есть место в буфере на прием. */
    while (*size != 0) {
        /* Если RxFIFO не пуст. */
        if (GET_VAL_MSK(base->SR, SPI_SR_RFNE_Msk, SPI_SR_RFNE_Pos) != 0U) {

            tmp32 = base->DR0;

            if (SPI_ShiftDirLsbFirst
                == spi_config_internal[instance].shift_dir) {
                /*
                 * Преобразование формата представления
                 * Big-endian <--> Little-endian.
                 */
                LBBLEndian(&tmp32,
                    spi_config_internal[instance].frame_width_bits);
            }

            /*
             * В зависимости от размер кадра данных запись в приемный буфер
             * 1,2 или 4 байта.
             */
            for (uint32_t i = 0; i < frame_width_bytes; i++) {
                **rx_data = (uint8_t) (tmp32 >> (i * 8));
                (*rx_data)++;
                (*size)--;
            }
        } else {
            return;
        }
    }

    return;
}

/*!
 * @brief Запись данных в TxFifo
 *
 * @note Выход по окончании данных на передачу(*size) или после заполения TxFIFO.
 * Функция перемещает указатель на буфер **tx_data
 *
 * @param base Базовый адрес SPI
 * @param instance Порядковый номер экземпляра устройства
 * @param tx_data Указатель на указатель на буфер для передачи
 * @param size Размер буфера
 */
void _TxFifo_Write(SPI_Type *base, uint32_t instance,
    volatile uint8_t **tx_data, volatile size_t *size)
{
    uint32_t frame_width_bytes;
    uint32_t tmp32;
    uint8_t  fifo_depth = SPI_FIFO_DEPTH;

    /* Размер кадра данных в байтах. */
    frame_width_bytes = spi_config_internal[instance].frame_width_bytes;

    /* Пока есть данные на передачу и пока FIFO-буфер не заполнен. */
    while ((*size != 0) && (fifo_depth--)) {
        /* Если в TxFIFO есть место. */
        if (GET_VAL_MSK(base->SR, SPI_SR_TFNF_Msk, SPI_SR_TFNF_Pos) != 0U) {

            tmp32 = 0;
            /*
             * В зависимости от размера кадра данных запись в tmp32 1, 2 или
             * 4 байта.
             */
            for (uint32_t i = 0; i < frame_width_bytes; i++) {
                tmp32 |= ((uint32_t) (**tx_data)) << (i * 8);
                (*tx_data)++;
                (*size)--;
            }

            if (spi_config_internal[instance].shift_dir ==
                SPI_ShiftDirLsbFirst) {
                /*
                 * Преобразование формата представления
                 * Big-endian <--> Little-endian.
                 */
                LBBLEndian(&tmp32,
                    spi_config_internal[instance].frame_width_bits);
            }

            /* Отправка данных. */
            base->DR0 = tmp32;
        } else {
            return;
        }
    }

    return ;
}

enum spi_status SPI_MasterTransferBlocking(SPI_Type *base,
    spi_transfer_t *xfer)
{
    uint8_t  instance;
    uint32_t frame_width_bytes;     /* Размер кадра данных в байтах. */
    uint32_t tmp32;
    size_t   rx_remaining_bytes, tx_remaining_bytes;
    uint32_t to_receive_frame = 0;  /* Количество кадров на прием. */
    uint8_t *tx_data, *rx_data;

#if SPI_RETRY_TIMES
    uint32_t wait_times;
#endif

    /* Проверка параметров. */
    if ((NULL == base) || (NULL == xfer)
        || ((NULL == xfer->tx_data) && (NULL == xfer->rx_data))) {
        return SPI_Status_InvalidArgument;
    }

    instance = SPI_GetInstance(base);
    /* Размер кадра данных в байтах. */
    frame_width_bytes = spi_config_internal[instance].frame_width_bytes;

    /* Проверка кратности. */
    assert((xfer->data_size % frame_width_bytes) == 0);

    /* Ожидание завершения передачи всех данных. */
#if SPI_RETRY_TIMES
    wait_times = SPI_RETRY_TIMES;
#endif
    while (GET_VAL_MSK(base->SR, SPI_SR_BUSY_Msk, SPI_SR_BUSY_Pos) != 0) {
#if SPI_RETRY_TIMES
        if (--wait_times == 0U) {
            return SPI_Status_Timeout;
        }
#endif
    }

    /* Отключение модуля SPI. */
    /* Очистка FIFO. */
    /* Сброс прерываний. */
    SPI_Enable(base, 0U);

    /* Включение режима дуплексного обмена. */
    SET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_TMOD_Msk, SPI_CTRLR0_TMOD_Pos, SPI_ModeDuplex);

    /* Включение модуля SPI. */
    SPI_Enable(base, 1U);

    /* Отключение прерываний. */
    SPI_DisableInterrupts(base, SPI_IRQ_All);

    /* Буфер на передачу. */
    tx_data = xfer->tx_data;
    /* Буфер на прием. */
    rx_data = xfer->rx_data;

    /* Определяется режим работы: передача, прием, передача и прием. */
    /* Количество данных для приема. */
    tx_remaining_bytes = (tx_data != NULL) ? xfer->data_size : 0U;
    /* Количество данных для передачи. */
    rx_remaining_bytes = (rx_data != NULL) ? xfer->data_size : 0U;


#if SPI_RETRY_TIMES
    wait_times = SPI_RETRY_TIMES;
#endif
    /* Цикл пока есть данные на прием или отправку. */
    while ((tx_remaining_bytes != 0U) || (rx_remaining_bytes != 0U)
        || (to_receive_frame != 0U)) {
#if SPI_RETRY_TIMES
        if (--wait_times == 0U) {
            return SPI_Status_Timeout;
        }
#endif

        /* Прием 1 кадра данных. */

        /* Если RxFIFO не пуст. */
        if (GET_VAL_MSK(base->SR, SPI_SR_RFNE_Msk, SPI_SR_RFNE_Pos) != 0U) {
            tmp32 = base->DR0;

            if (spi_config_internal[instance].shift_dir ==
                SPI_ShiftDirLsbFirst) {
                /*
                 * Преобразование формата представления
                 * Big-endian <--> Little-endian.
                 */
                LBBLEndian(&tmp32,
                    spi_config_internal[instance].frame_width_bits);
            }

            /* RxBuffer не пуст (значит требуется принимать данные). */
            if (rx_remaining_bytes != 0U) {
                /*
                 * В зависимости от размер кадра данных запись в приемный буфер
                 * 1,2 или 4 байта.
                 */
                for (uint32_t i = 0; i < frame_width_bytes; i++) {
                    *(rx_data++) = (uint8_t) (tmp32 >> (i * 8));
                    rx_remaining_bytes--;
                }
            }

            /* Уменьшить количество ожидаемых для приема данных. */
            to_receive_frame -= 1U;
        }

        /* Передача 1 кадра данных. */

        /* Если TxFIFO есть место. */
        if ((GET_VAL_MSK(base->SR, SPI_SR_TFNF_Msk, SPI_SR_TFNF_Pos) != 0U)
            && (to_receive_frame < SPI_FIFO_DEPTH)
            /* и есть данные для передачи. */
            && ((tx_remaining_bytes != 0U) ||
                /* или требуется принять данные. */
                (rx_remaining_bytes
                    >= (to_receive_frame * frame_width_bytes + 1U)))) {

            /* Если есть данные на отправку. */
            if (tx_remaining_bytes != 0U) {
                tmp32 = 0;
                /*
                 * В зависимости от размера кадра данных запись в tmp32 1, 2 или
                 * 4 байта.
                 */
                for (uint32_t i = 0; i < frame_width_bytes; i++) {
                    tmp32 |= ((uint32_t) * (tx_data++)) << (i * 8);
                    tx_remaining_bytes--;
                }
            }
            /* Если нет данных на отправку, отправляем фиктивные данные. */
            else {
                tmp32 = 0;
                for (uint32_t i = 0; i < frame_width_bytes; i++) {
                    tmp32 |= ((uint32_t) spi_dummy_data[instance]) << (i * 8U);
                }
            }

            if (spi_config_internal[instance].shift_dir
                == SPI_ShiftDirLsbFirst) {
                /*
                 * Преобразование формата представления
                 * Big-endian <--> Little-endian.
                 */
                LBBLEndian(&tmp32,
                    spi_config_internal[instance].frame_width_bits);
            }

            /* Запись данных в TxFIFO. */
            base->DR0 = tmp32;
            to_receive_frame += 1U;
        }
    }

    /* Ожидание завершения передачи всех данных. */
#if SPI_RETRY_TIMES
    wait_times = SPI_RETRY_TIMES;
#endif
    while (GET_VAL_MSK(base->SR, SPI_SR_BUSY_Msk, SPI_SR_BUSY_Pos) != 0) {
#if SPI_RETRY_TIMES
        if (--wait_times == 0U) {
            return SPI_Status_Timeout;
        }
#endif
    }

    return SPI_Status_Ok;
}

enum spi_status SPI_MasterHalfDuplexTransferBlocking(SPI_Type *base,
    spi_half_duplex_transfer_t *xfer)
{
    uint8_t instance;
    uint32_t frame_width_bytes; /* Размер кадра данных в байтах. */
    volatile size_t rx_remaining_bytes, tx_remaining_bytes;
    volatile uint8_t *tx_data, *rx_data;
    uint32_t rx_remaining_frame;
    uint32_t temp;
#if SPI_RETRY_TIMES
    uint32_t wait_times;
#endif

    assert(base != NULL);
    assert(xfer != NULL);

    instance = SPI_GetInstance(base);
    /* Размер кадра данных в байтах. */
    frame_width_bytes = spi_config_internal[instance].frame_width_bytes;

    /* Проверка кратности. */
    assert(0 == xfer->tx_data_size % frame_width_bytes);
    assert(0 == xfer->rx_data_size % frame_width_bytes);

    /* Проверка аппаратных настроек модуля.
     * Формат кадра передачи данных.
     */
    temp = GET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_FRF_Msk, SPI_CTRLR0_FRF_Pos);
    if (SPI_FfTexas == temp) {
        /*
         * Формат кадра передачи данных для полудуплексного режима
         * не может быть SPI_FfTexas.
         * Используйте SPI_MasterTransferNonBlocking().
         */
        return SPI_Status_IncorrectCall;
    }

    /* Защита от вызова функции без запроса на передачу и прием. */
    if ((xfer->tx_data == NULL) || (xfer->tx_data_size == 0) ||
        (xfer->rx_data == NULL) || (xfer->rx_data_size == 0)) {

        return SPI_Status_InvalidArgument;
    }

    /* Ожидание завершения передачи всех данных. */
#if SPI_RETRY_TIMES
    wait_times = SPI_RETRY_TIMES;
#endif
    while (GET_VAL_MSK(base->SR, SPI_SR_BUSY_Msk, SPI_SR_BUSY_Pos) != 0) {
#if SPI_RETRY_TIMES
        if (--wait_times == 0U) {
            return SPI_Status_Timeout;
        }
#endif
    }

    /* Отключение модуля SPI. */
    /* Очистка TxFIFO и RxFIFO (регистры сбрасываются при SSI_EN = 0). */
    /* Сброс прерываний. */
    SPI_Enable(base, 0U);

    /* Отключение прерываний. */
    SPI_DisableInterrupts(base, SPI_IRQ_All);

    /* Включение режима дуплексного обмена. */
    SET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_TMOD_Msk, SPI_CTRLR0_TMOD_Pos, SPI_ModeHalfDuplex);

    /* Буфер на передачу. */
    tx_data = xfer->tx_data;
    /* Буфер на прием. */
    rx_data = xfer->rx_data;
    /* Количество данных для передачи. */
    tx_remaining_bytes = (tx_data != NULL) ? xfer->tx_data_size : 0U;
    /* Количество данных для приема. */
    rx_remaining_bytes = (rx_data != NULL) ? xfer->rx_data_size : 0U;

    /* Установка количества фреймов данных для приема. */
    rx_remaining_frame = rx_remaining_bytes / frame_width_bytes;
    SET_VAL_MSK(base->CTRLR1, SPI_CTRLR1_NDF_Msk, SPI_CTRLR1_NDF_Pos,
        rx_remaining_frame - 1);

    /* Включение SPI. */
    SET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk, SPI_SSIENR_SSI_EN_Pos,
        1U);

    /* Отправка данных. */

#if SPI_RETRY_TIMES
    wait_times = SPI_RETRY_TIMES;
#endif

    /* Пока есть данные на отправку. */
    while (tx_remaining_bytes != 0U) {

        /* Выход по таймауту, если включен. */
#if SPI_RETRY_TIMES
        if (--wait_times == 0U) {
            return SPI_Status_Timeout;
        }
#endif
        /* Заполнить TxFifo. */
        _TxFifo_Write(base, instance, &tx_data, &tx_remaining_bytes);
    }

    /* Прием запрошенных данных. */

#if SPI_RETRY_TIMES
    wait_times = SPI_RETRY_TIMES;
#endif
    /* Цикл пока требуется принимать. */
    while (rx_remaining_bytes != 0U) {

        /* Выход по таймауту, если включен. */
#if SPI_RETRY_TIMES
        if (--wait_times == 0U) {
            return SPI_Status_Timeout;
        }
#endif
        /* Вычитывание RxFifo. */
        _RxFifo_Read(base, instance, &rx_data,  &rx_remaining_bytes);
    }

    /* Ожидание завершения передачи всех данных. */

#if SPI_RETRY_TIMES
    wait_times = SPI_RETRY_TIMES;
#endif

    while (GET_VAL_MSK(base->SR, SPI_SR_BUSY_Msk, SPI_SR_BUSY_Pos)) {

        /* Выход по таймауту, если включен. */
#if SPI_RETRY_TIMES
        if (--wait_times == 0U) {
            return SPI_Status_Timeout;
        }
#endif
    }

    return SPI_Status_Ok;
}

enum spi_status SPI_MasterTransferCreateHandle(SPI_Type *base,
    spi_master_handle_t *handle, spi_master_callback_t callback,
    void *user_data)
{
    assert(NULL != base);
    assert(NULL != handle);

    (void) memset(handle, 0, sizeof(*handle));

    /* Получение номер модуля SPI по базовому адресу. */
    handle->instance = SPI_GetInstance(base);

    /* Маскировка всех прерываний модуля. */
    SPI_DisableInterrupts(base, (uint32_t) SPI_IRQ_All);

    /* Установка обработчика прерывания. */
    if (IOIM_SetIRQHandler(base, SPI_MasterTransferHandleIRQ, handle)
        != IOIM_Status_Ok) {
        return SPI_Status_UnexpectedState;
    }

    /* Размер кадра данных в битах. */
    handle->frame_width_bits =
        (uint8_t) (spi_config_internal[handle->instance].frame_width_bits);
    /* Размер кадра данных в байтах. */
    handle->frame_width_bytes =
        (uint8_t) (spi_config_internal[handle->instance].frame_width_bytes);

    handle->callback = callback;
    handle->user_data = user_data;
    handle->state = (uint32_t) SPI_TransStatus_Idle;

    return SPI_Status_Ok;
}

enum spi_status SPI_MasterTransferNonBlocking(SPI_Type *base,
    spi_master_handle_t *handle, spi_transfer_t *xfer)
{

    /* Проверка параметров. */
    assert(!((NULL == base) || (NULL == handle) || (NULL == xfer)
            || ((NULL == xfer->tx_data) && (NULL == xfer->rx_data))));

    if ((base == NULL)
        || (handle == NULL)
        || (xfer == NULL)
        || ((xfer->tx_data == NULL) && (xfer->rx_data == NULL))
        || (xfer->data_size == 0))
    { /* Защита от вызова без запроса на обмен данными. */

        return SPI_Status_InvalidArgument;
    }

    /* Проверка кратности. */
    assert(xfer->data_size % handle->frame_width_bytes == 0);

    /* Проверка занятости SPI. */
    if (handle->state == (uint32_t) SPI_TransStatus_Busy) {
        return SPI_Status_Busy;
    }

    /* Отключение модуля SPI. */
    /* Очистка FIFO. */
    /* Сброс прерываний. */
    SPI_Enable(base, 0U);

    /* Настройка дескриптора: указатели на буферы. */
    handle->tx_data = xfer->tx_data;
    handle->rx_data = xfer->rx_data;

    /* Настройка дескриптора: счетчики байт. */
    handle->tx_remaining_bytes = (xfer->tx_data != NULL) ? xfer->data_size : 0U;
    handle->rx_remaining_bytes = (xfer->rx_data != NULL) ? xfer->data_size : 0U;
    /* Настройка дескриптора: Общее количество байтов для обмена. */
    handle->total_byte_to_transfer = xfer->data_size;

    /* Настройка дескриптора: Установка режима обмена. */
    if (handle->rx_remaining_bytes == 0) {
        /* Требуется только передача. */
        handle->mode = SPI_ModeSimplexTx;
    } else {
        /* Требуется только прием. НЕ ПОДДЕРЖИВАЕТСЯ. */
        if (handle->tx_remaining_bytes == 0) {
            return SPI_Status_InvalidArgument;
        } else {
            /* Требуется передача и прием. */
            handle->mode = SPI_ModeDuplex;
        }
    }

    /* Настройка режима обмена. */
    SET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_TMOD_Msk, SPI_CTRLR0_TMOD_Pos, handle->mode);

    /* Включение модуля SPI. */
    SPI_Enable(base, 1U);

    /* Сброс прерываний. */
    SPI_TakeDownInterrupts(base, SPI_IRQ_MultiMaster
        | SPI_IRQ_RxFifoOverflow
        | SPI_IRQ_RxFifoUnderflow
        | SPI_IRQ_TxFifoOverflow);

    /* TxFIFO триггер прерывания по уровню заполнения: опустошение. */
    SET_VAL_MSK(base->TXFTLR, SPI_TXFTLR_TFT_Msk, SPI_TXFTLR_TFT_Pos, 0);
    /* RxFIFO триггер прерывания по уровню заполнения: один кадр. */
    SET_VAL_MSK(base->RXFTLR, SPI_RXFTLR_RFT_Msk, SPI_RXFTLR_RFT_Pos, 0);

    /* Настройка дескриптора: Установка состояния модуля как занятого. */
    handle->state = (uint32_t) SPI_TransStatus_Busy;

    /* Включение прерываний. */
    switch (handle->mode) {
        /* Только передача. */
        case SPI_ModeSimplexTx:
            SPI_EnableInterrupts(base, SPI_IRQ_TxOnly);
            break;
        /* Режим дуплекса. */
        case SPI_ModeDuplex:
            SPI_EnableInterrupts(base, SPI_IRQ_All);
            break;

        /* Режим полудуплекса. */
        /* Режим только прием.*/
        default:
            return SPI_Status_UnexpectedState;
            break;

    }

    /* Здесь должно сработать прерывание по пустому буферу TxFIFO. */

    return SPI_Status_Ok;
}

status_t SPI_MasterHalfDuplexTransferNonBlocking(SPI_Type *base,
    spi_master_handle_t *handle,
    spi_half_duplex_transfer_t *xfer)
{
    uint32_t temp;
    uint32_t rx_remaining_frame;

    assert(xfer != NULL);
    assert(handle != NULL);

    /* Проверка кратности. */
    assert(xfer->tx_data_size % handle->frame_width_bytes == 0);
    assert(xfer->rx_data_size % handle->frame_width_bytes == 0);

    /*
     * Проверка аппаратных настроек модуля.
     * Формат кадра передачи данных.
     */
    temp = GET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_FRF_Msk, SPI_CTRLR0_FRF_Pos);
    if (SPI_FfTexas == temp) {
        /*
         * Формат кадра передачи данных для полудуплексного режима
         * не может быть SPI_FfTexas.
         * Используйте SPI_MasterTransferNonBlocking().
         */
        return SPI_Status_IncorrectCall;
    }

    /* Проверка текущего состояния модуля. */
    if (handle->state == (uint32_t) SPI_TransStatus_Busy)
        /* Модуль занят.*/
        return SPI_Status_Busy;


    /* Проверка входных параметров. */
    if ((xfer->tx_data != NULL) && (xfer->tx_data_size != 0)) {
        handle->tx_data            = xfer->tx_data;
        handle->tx_remaining_bytes = xfer->tx_data_size;
    } else {
        handle->tx_remaining_bytes = 0;
    }

    if ((xfer->rx_data != NULL) && (xfer->rx_data_size != 0)) {
        handle->rx_data            = xfer->rx_data;
        handle->rx_remaining_bytes = xfer->rx_data_size;
    } else {
        handle->rx_remaining_bytes = 0;
    }

    /* Защита от вызова функции без запроса на передачу и прием. */
    if ((handle->rx_remaining_bytes == 0)
        || (handle->tx_remaining_bytes == 0))
    {
        return SPI_Status_InvalidArgument;
    }

    /* Подготовка к началу обмена. */

    /* Настройка дескриптора: Установка состояния модуля как занятый. */
    handle->state = (uint32_t) SPI_TransStatus_Busy;

    /* Настройка дескриптора: Общее количество байт для передачи. */
    handle->total_byte_to_transfer = handle->tx_remaining_bytes + handle->rx_remaining_bytes;

    /* Настройка дескриптора: Установка режима обмена. */
    /* Требуется передача и прием. */
    handle->mode = SPI_ModeHalfDuplex;

    /* Отключение модуля SPI. */
    /* Очистка FIFO. */
    /* Сброс прерываний. */
    SPI_Enable(base, 0U);

    /* Включение режима полудуплексного обмена. */
    SET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_TMOD_Msk, SPI_CTRLR0_TMOD_Pos, SPI_ModeHalfDuplex);

    /* Задать количество фреймов данных для приема. */
    rx_remaining_frame = handle->rx_remaining_bytes / handle->frame_width_bytes;
    SET_VAL_MSK(base->CTRLR1, SPI_CTRLR1_NDF_Msk, SPI_CTRLR1_NDF_Pos,
        rx_remaining_frame - 1);

    /* Включение модуля SPI. */
    SPI_Enable(base, 1U);

    /* TxFIFO триггер прерывания по уровню заполнения. */
    SET_VAL_MSK(base->TXFTLR, SPI_TXFTLR_TFT_Msk, SPI_TXFTLR_TFT_Pos, 0);
    /* RxFIFO триггер прерывания по уровню заполнения. */
    SET_VAL_MSK(base->RXFTLR, SPI_RXFTLR_RFT_Msk, SPI_RXFTLR_RFT_Pos, 0);
    /* Включение всех прерываний по уровню FIFO. */
    SPI_EnableInterrupts(base, SPI_IRQ_All);

    /* Здесь должно сработать прерывание по пустому буферу TxFIFO. */

    return SPI_Status_Ok;
}

status_t SPI_MasterTransferGetRemainingByte(spi_master_handle_t *handle,
    size_t *count)
{
    assert(handle != NULL);

    if (count == NULL) {
        return SPI_Status_InvalidArgument;
    }

    /* Возврат, если нет текущего обмена. */
    if (handle->state != (uint32_t) SPI_TransStatus_Busy) {
        *count = 0;
        return SPI_Status_NoTransferInProgress;
    }

    switch (handle->mode) {

        /* Только передача. */
        case SPI_ModeSimplexTx:
        /* Режим только прием. */
        case SPI_ModeSimplexRx:
        /* Режим полудуплекса. */
        case SPI_ModeHalfDuplex:
            *count = handle->tx_remaining_bytes + handle->rx_remaining_bytes;
            break;

        /* Режим дуплекса. */
        case SPI_ModeDuplex:
            *count = handle->tx_remaining_bytes;
            break;

        default:
            return SPI_Status_UnexpectedState;
            break;

    }

    return SPI_Status_Ok;
}

status_t SPI_MasterTransferGetByte(spi_master_handle_t *handle, size_t *count)
{
    assert(handle != NULL);

    if (count == NULL) {
        return SPI_Status_InvalidArgument;
    }

    /* Возврат, если нет текущего обмена. */
    if (handle->state != (uint32_t) SPI_TransStatus_Busy) {
        return SPI_Status_NoTransferInProgress;
    }

    switch (handle->mode) {
        /* Только передача. */
        case SPI_ModeSimplexTx:
        /* Режим только прием. */
        case SPI_ModeSimplexRx:
        /* Режим полудуплекса. */
        case SPI_ModeHalfDuplex:
            *count = handle->total_byte_to_transfer -
                (handle->tx_remaining_bytes + handle->rx_remaining_bytes);
            break;

        /* Режим дуплекса. */
        case SPI_ModeDuplex:
            *count = handle->total_byte_to_transfer - handle->tx_remaining_bytes;
            break;

        default:
            return SPI_Status_UnexpectedState;
            break;
    }

    return SPI_Status_Ok;
}

status_t SPI_MasterTransferGetTotalByte(spi_master_handle_t *handle,
    size_t *count)
{
    assert(handle != NULL);

    if (count == NULL) {

        return SPI_Status_InvalidArgument;
    }

    /* Возврат, если нет текущего обмена. */
    if (handle->state != (uint32_t) SPI_TransStatus_Busy) {

        *count = 0;

        return SPI_Status_NoTransferInProgress;
    }

    *count = handle->total_byte_to_transfer;

    return SPI_Status_Ok;
}

void SPI_MasterTransferAbort(SPI_Type *base, spi_master_handle_t *handle)
{
    assert(handle != NULL);

    /* Отключить прерывания. */
    SPI_DisableInterrupts(base, SPI_IRQ_All);

    /* Очистка FIFO. */
    SPI_FifoClr(base);

    handle->state              = (uint32_t) SPI_TransStatus_Idle;
    handle->tx_remaining_bytes = 0U;
    handle->rx_remaining_bytes = 0U;
}

void SPI_MasterTransferHandleIRQ(SPI_Type *base, spi_master_handle_t *handle)
{
    assert((base != NULL) && (handle != NULL));
    uint32_t isr;

    isr = SPI_CurrentStatusInterrupts(base);

    /* Если ошибка при обмене. */
    if ((isr & (SPI_IRQ_MultiMaster
                | SPI_IRQ_TxFifoOverflow
                | SPI_IRQ_RxFifoOverflow
                | SPI_IRQ_RxFifoUnderflow)) != 0) {

        handle->state = (uint32_t) SPI_TransStatus_Error;

        /* Multi Master Конфликт. */
        if ((isr & SPI_IRQ_MultiMaster) != 0) {
            handle->state |= (uint32_t) SPI_TransStatus_Error_1;
        }

        /* TxFIFO Overflow. */
        if ((isr & SPI_IRQ_TxFifoOverflow) != 0) {
            handle->state |= (uint32_t) SPI_TransStatus_Error_2;
        }

        /* RxFIFO Overflow. */
        if ((isr & SPI_IRQ_RxFifoOverflow) != 0) {
            handle->state |= (uint32_t) SPI_TransStatus_Error_3;
        }

        /* RxFIFO Underflow. */
        if ((isr & SPI_IRQ_RxFifoUnderflow) != 0) {
            handle->state |= (uint32_t) SPI_TransStatus_Error_4;
        }

        /* Если есть пользовательский обработчик. */
        if (handle->callback != NULL) {
            (handle->callback)(base, handle, handle->state, handle->user_data);
        }

        SPI_TakeDownInterrupts(base, SPI_IRQ_MultiMaster
            | SPI_IRQ_RxFifoOverflow
            | SPI_IRQ_RxFifoUnderflow
            | SPI_IRQ_TxFifoOverflow);

        /* Останавливает обмен. */
        SPI_MasterTransferAbort(base, handle);

        return;
    }

    /* Опустошение TxFIFO. */
    if ((isr & SPI_IRQ_TxFifoTrigger) != 0) {

        /* Есть данные для передачи. */
        if (handle->tx_remaining_bytes != 0) {
            /* Запись в TxFifo. */
            _TxFifo_Write(base, handle->instance, &(handle->tx_data),
                &(handle->tx_remaining_bytes));
        }
    }

    /* Есть данные в RxFIFO. */
    if ((isr & SPI_IRQ_RxFifoTrigger) != 0) {

        /* Если есть требование на прием данных. */
        if (handle->rx_remaining_bytes != 0) {
            /* Считывание всех данных из RxFifo. */
            _RxFifo_Read(base, handle->instance, &(handle->rx_data),
                &(handle->rx_remaining_bytes));
        }
    }

    /* Все передано? */
    if (handle->tx_remaining_bytes == 0) {
        /* Отключение прерывания по опустошению TxFifo. */
        SPI_DisableInterrupts(base, SPI_IRQ_TxFifoTrigger);
    }

    /* Все принято? */
    if (handle->rx_remaining_bytes == 0) {
        /* Отключение прерывания по наполениею RxFifo. */
        SPI_DisableInterrupts(base, SPI_IRQ_RxFifoTrigger);
    } else {

        /* RxFIFO триггер прерывания по уровню заполнения. */

        /*
         * На высоких частотах возможно будет происходит переполнение аппартного FIFO,
         * поэтому при инициализации обмена значение RXFTLR устанавливается на ноль,
         * это означает, что прерывание будет возникать при попадании в FIFO одного байта.
         * Под свою задачу можно настроить порог срабатывания, если буфер будет переполнен
         * обмен закончится с кодом SPI_TransStatus_Error_3 (по "или" так как
         * могут быть и другие ошибки).
         */
    }

    /* Весь обмен выполнен. */
    if ((handle->tx_remaining_bytes == 0) && (handle->rx_remaining_bytes == 0)) {
        /* Отключение прерываний. */
        SPI_DisableInterrupts(base, SPI_IRQ_All);

        /* Вызов пользовательской функция обратного вызова. */
        handle->state = (uint32_t) SPI_TransStatus_Idle;
        if (handle->callback != NULL) {
            (handle->callback)(base, handle, handle->state, handle->user_data);
        }
    }

    return;
}

/*!
 * @}
 */
