/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup spi_driver Драйвер модуля SPI
 *
 * @brief Последовательный синхронный стандарт передачи данных в режиме полного дуплекса, полудуплекса или симплекса
 *
 * @note Драйвер поддерживает обмены по интерфейсу SPI по прерыванию и в режиме опроса,
 * ширину поля данных от 4 до 32 битов, форматы кадров: Motorola SPI,
 * Texas Instruments, Synchronous Serial Protocol (SSP) и NS Microwire.
 *
 */

/*!
 * @addtogroup spi_driver
 * @{
 */

/*!
 * @file hal_spi.h
 *
 * @brief Интерфейс драйвера модуля SPI
 */

#ifndef HAL_SPI_H
#define HAL_SPI_H

#include "hal_common.h"

/*!
 * @brief Версия драйвера SPI
 */
#define HAL_SPI_DRIVER_VERSION (MAKE_VERSION(1, 1, 0))

/*!
 * @brief SPI фиктивные данные для передачи по умолчанию. 
 * @note Фиктивные данные передаются в @SPI_MasterTransferBlocking,
 *       когда буфер Tx пуст, но есть запрос на прием.
 */
#ifndef SPI_DUMMYDATA
#define SPI_DUMMYDATA (0xA5U)
#endif

#ifndef SPI_RETRY_TIMES
#define SPI_RETRY_TIMES 0U /*!< Время повтора для флага ожидания. Ноль означает продолжать ждать, пока флаг не будет установлен/снят */
#endif

/*!
 * @brief Статусы возврата из функций для драйвера SPI
 */ 
enum spi_status { 
    SPI_Status_Ok                   =  0,   /*!< Успешно */
    SPI_Status_Fail                 =  1,   /*!< Провал */
    SPI_Status_ReadOnly             =  2,   /*!< Только чтение */
    SPI_Status_InvalidArgument      =  3,   /*!< Неверный аргумент */
    SPI_Status_Timeout              =  4,   /*!< Отказ по таймауту */
    SPI_Status_BaudrateNotSupport   =  5,   /*!< Частота не поддерживается*/
    SPI_Status_Busy                 =  6,   /*!< SPI модуль занят */
    SPI_Status_Idle                 =  9,   /*!< SPI модуль простаивает */
    SPI_Status_TxError              = 10,   /*!< Ошибка в TxFIFO */
    SPI_Status_RxError              = 11,   /*!< Ошибка в RxFIFO */
    SPI_Status_RxRingBufferOverrun  = 12,   /*!< Ошибка в кольцевом буфере Rx */
    SPI_Status_RxFifoBufferOverrun  = 13,   /*!< Ошибка переполнения hw RxFIFO буферa */
    SPI_Status_NoTransferInProgress = 14,   /*!< Нет текущего обмена */
    SPI_Status_IncorrectCall        = 15,   /*!< Некорректный вызов с текущими настройками модуля */
    SPI_Status_UnexpectedState      = 128U, /*!< Неожиданное состояние   */
};

/*!
 * @brief Формат передачи данных (MSB или LSB)
 */
typedef enum {
    SPI_ShiftDirMsbFirst = 0, /*!< Передача данных начинается со старшего бита */
    SPI_ShiftDirLsbFirst = 1, /*!< Передача данных начинается с младшего бита */
} spi_shift_direction_t;

/*!
 * @brief Триггер уровня заполнения TxFIFO
 */
typedef enum {
    SPI_TxFifoWatermark0 = 0, /*!< TxFIFO пуст */
    SPI_TxFifoWatermark1 = 1, /*!< 1 элемент в TxFIFO */
    SPI_TxFifoWatermark2 = 2, /*!< 2 элемента в TxFIFO */
    SPI_TxFifoWatermark3 = 3, /*!< 3 элемента в TxFIFO */
    SPI_TxFifoWatermark4 = 4, /*!< 4 элемента в TxFIFO */
    SPI_TxFifoWatermark5 = 5, /*!< 5 элементов в TxFIFO */
    SPI_TxFifoWatermark6 = 6, /*!< 6 элементов в TxFIFO */
    SPI_TxFifoWatermark7 = 7, /*!< 7 элементов в TxFIFO */
} spi_txfifo_watermark_t;

/*!
 * @brief Триггер уровня заполнения RxFIFO
 */
typedef enum {
    SPI_RxFifoWatermark1 = 0, /*!< 1 элемент в RxFIFO */
    SPI_RxFifoWatermark2 = 1, /*!< 2 элемента в RxFIFO */
    SPI_RxFifoWatermark3 = 2, /*!< 3 элемента в RxFIFO */
    SPI_RxFifoWatermark4 = 3, /*!< 4 элемента в RxFIFO */
    SPI_RxFifoWatermark5 = 4, /*!< 5 элементов в RxFIFO */
    SPI_RxFifoWatermark6 = 5, /*!< 6 элементов в RxFIFO */
    SPI_RxFifoWatermark7 = 6, /*!< 7 элементов в RxFIFO */
    SPI_RxFifoWatermark8 = 7, /*!< 8 элементов в RxFIFO */
} spi_rxfifo_watermark_t;

/*!
 * @brief Размер кадра данных в 32-х битном режиме передачи данных
 *        (CTRLR0.DFS_32)
 */
typedef enum {
    SPI_Data4Bits  =  3, /*!<  4 бита */
    SPI_Data5Bits  =  4, /*!<  5 бит */
    SPI_Data6Bits  =  5, /*!<  6 бит */
    SPI_Data7Bits  =  6, /*!<  7 бит */
    SPI_Data8Bits  =  7, /*!<  8 бит */
    SPI_Data9Bits  =  8, /*!<  9 бит */
    SPI_Data10Bits =  9, /*!< 10 бит */
    SPI_Data11Bits = 10, /*!< 11 бит */
    SPI_Data12Bits = 11, /*!< 12 бит */
    SPI_Data13Bits = 12, /*!< 13 бит */
    SPI_Data14Bits = 13, /*!< 14 бит */
    SPI_Data15Bits = 14, /*!< 15 бит */
    SPI_Data16Bits = 15, /*!< 16 бит */
    SPI_Data17Bits = 16, /*!< 17 бит */
    SPI_Data18Bits = 17, /*!< 18 бит */
    SPI_Data19Bits = 18, /*!< 19 бит */
    SPI_Data20Bits = 19, /*!< 20 бит */
    SPI_Data21Bits = 20, /*!< 21 бит */
    SPI_Data22Bits = 21, /*!< 22 бита */
    SPI_Data23Bits = 22, /*!< 23 бита */
    SPI_Data24Bits = 23, /*!< 24 бита */
    SPI_Data25Bits = 24, /*!< 25 бит */
    SPI_Data26Bits = 25, /*!< 26 бит */
    SPI_Data27Bits = 26, /*!< 27 бит */
    SPI_Data28Bits = 27, /*!< 28 бит */
    SPI_Data29Bits = 28, /*!< 29 бит */
    SPI_Data30Bits = 29, /*!< 30 бит */
    SPI_Data31Bits = 30, /*!< 31 бит */
    SPI_Data32Bits = 31, /*!< 32 бита */
} spi_frame_width_t;

/*!
 * @brief Формат кадра передачи данных
 *
 * @note Для Motorola SPI - режим Slave-Select выставляется на всю продолжительность
 * обмена данными.
 *
 * Для Texas Instruments Synchronous Serial Protocol (SSP):
 *  - Slave-Select выставляется на 1 такт до начала передачи;
 *  - Установка данных происходит по переднему фронту Clk, а выборка - по
 *    заднему;
 *  - Значение DFS должно быть кратно 2.
 *
 * Для National Semiconductor Microwire:
 *  - Сигнал Slave-Select остается активно-низким на протяжении всей передачи и
 *    переходит в высокое состояние через полтакта после окончании передачи
 *    данных;
 *  - Данные устанавливаются по заднему фронту линии синхронизации, а выборка
 *    по переднему;
 *  - Значение DFS должно быть кратно 4.
 */
typedef enum {
    SPI_FfMotorola  = 0, /*!< Motorola SPI */
    SPI_FfTexas     = 1, /*!< Texas Instruments SSP */
    SPI_FfMicrowire = 2, /*!< National Semiconductor Microwire */
} spi_frame_format_t;

/*!
 * @name National Semiconductor Microwire
 * @{
 */

/*!
 * @brief Выбор длины управляющего слова для формата кадра передачи данных
 *        National Semiconductor Micorwire
 */
typedef enum {
    SPI_MicrowireCtrlWordLen1Bit  =  0, /*!< Длина - 1 бит */
    SPI_MicrowireCtrlWordLen2Bit  =  1, /*!< Длина - 2 бита */
    SPI_MicrowireCtrlWordLen3Bit  =  2, /*!< Длина - 3 бита */
    SPI_MicrowireCtrlWordLen4Bit  =  3, /*!< Длина - 4 бита */
    SPI_MicrowireCtrlWordLen5Bit  =  4, /*!< Длина - 5 бит */
    SPI_MicrowireCtrlWordLen6Bit  =  5, /*!< Длина - 6 бит */
    SPI_MicrowireCtrlWordLen7Bit  =  6, /*!< Длина - 7 бит */
    SPI_MicrowireCtrlWordLen8Bit  =  7, /*!< Длина - 8 бит */
    SPI_MicrowireCtrlWordLen9Bit  =  8, /*!< Длина - 9 бит */
    SPI_MicrowireCtrlWordLen10Bit =  9, /*!< Длина - 10 бит */
    SPI_MicrowireCtrlWordLen11Bit = 10, /*!< Длина - 11 бит */
    SPI_MicrowireCtrlWordLen12Bit = 11, /*!< Длина - 12 бит */
    SPI_MicrowireCtrlWordLen13Bit = 12, /*!< Длина - 13 бит */
    SPI_MicrowireCtrlWordLen14Bit = 13, /*!< Длина - 14 бит */
    SPI_MicrowireCtrlWordLen15Bit = 14, /*!< Длина - 15 бит */
    SPI_MicrowireCtrlWordLen16Bit = 15, /*!< Длина - 16 бит */
} microwire_ctrlword_len_t;

/*!
 * @brief Включить/отключить проверку busy/ready флага (регистр SR) для формата
 *        кадра передачи данных National Semiconductor Microwire
 *
 * @note В активном состоянии модуль SPI проверяет готовность Slave после передачи
 * последнего бита данных, для снятия busy статуса в регистре SR.
 */
typedef enum {
    SPI_MicrowireBusyReadyCheckDisable = 0, /*!< Отключить проверку */
    SPI_MicrowireBusyReadyCheckEnable  = 1, /*!< Включить проверку */
} microwire_busy_ready_check_t;

/*!
 * @brief Направление передачи слова данных для формата кадра передачи данных
 *        National Semiconductor Microwire
 */
typedef enum {
    SPI_MicrowireTx = 0, /*!< SPI передает слово данных */
    SPI_MicrowireRx = 1, /*!< SPI принимает слово данных */
} microwire_tx_rx_t;

/*!
 * @brief Выбор типа передачи (одиночная или последовательная) для формата кадра
 *        передачи данных National Semiconductor Microwire
 */
typedef enum {
    SPI_MicrowireSingle = 0, /*!< Одиночная передача */
    SPI_MicrowireSerial = 1, /*!< Последовательная передача */
} microwire_single_serial_t;

/*!
 * @brief Конфигурация для протокола Microwire National Semiconductor
 */
typedef struct {
    microwire_ctrlword_len_t     ctrl_word_len;    /*!< Выбор длины управляющего слова для протокола Microwire */
    microwire_busy_ready_check_t busy_ready_check; /*!< Включить/отключить проверку busy/ready флаг (регистр SR) */
    microwire_tx_rx_t            tx_rx;            /*!< Направление передачи слова данных */
    microwire_single_serial_t    single_serial;    /*!< Одиночная или последовательная передача */
} spi_microwire_cfg_t;

/*!
 * @}
 */

/*!
 * @name Motorola SPI
 * @{
 */

/*!
 * @brief Выбор полярности тактового сигнала при отсутствия передаваемых данных
 *        в режиме Master для формата кадра передачи данных Motorola SPI
 */
typedef enum {
    SPI_MotorolaClkPolLow = 0, /*!< Линия синхронизации до начала цикла передачи и после его окончания имеет низкий уровень */
    SPI_MotorolaClkPolHi  = 1, /*!< Линия синхронизации до начала цикла передачи и после его окончания имеет высокий уровень */
} spi_motorola_clk_pol_t;

/*!
 * @brief Выбор фронта для захвата данных для формата кадра передачи данных
 *        Motorola SPI
 */
typedef enum {
    SPI_MotorolaCapDataRising  = 0, /*!< Захват данных происходит по переднему фронту тактового сигнала */
    SPI_MotorolaCapDataFalling = 1, /*!< Происходит пропуск одного периода тактового сигнала после установки Slave-Select, захват данных происходит по заднему фронту тактового сигнала */
} spi_motorola_cap_data_t;

/*!
 * @brief Конфигурация для протокола Motorola SPI
 */
typedef struct {
    spi_motorola_clk_pol_t  clk_pol;  /*!< Полярность тактового сигнала, при отсутствия передаваемых данных в режиме Master */
    spi_motorola_cap_data_t cap_data; /*!< Захват данных происходит по переднему фронту или по заднему фронту */
} spi_motorola_cfg_t;

/*!
 * @}
 */

/*!
 * @brief SPI флаги статусов
 */
enum spi_status_flags {
    SPI_TxNotFullFlag  = SPI_SR_TFNF_Msk, /*!< TxFIFO не полон */
    SPI_TxEmptyFlag    = SPI_SR_TFE_Msk,  /*!< TxFIFO пуст */
    SPI_RxNotEmptyFlag = SPI_SR_RFNE_Msk, /*!< RxFIFO не пуст */
    SPI_RxFullFlag     = SPI_SR_RFF_Msk,  /*!< RxFIFO полон */
};

/*!
 * @brief Режим передачи (CTRLR0.TMOD)
 *
 * @note
 * - Дуплекс (Duplex) - передача продолжится до последнего слова в FIFO
 *   передатчика;
 * - Симплекс, только передача (SimplexTx) - принимаемые данные не поступают в
 *   RxFIFO; при использовании этого режима необходимо маскировать прерывания
 *   от приемника;
 * - Симплекс, только прием (SimplexRx) - передаваемые данные не валидны; при
 *   использовании этого режима необходимо маскировать прерывания от
 *   передатчика;
 * - Полудуплекс (Halfduplex) - режим чтения EEPROM. Сначала передаются все
 *   данные из TxFIFO, принимаемые в этот момент данные игнорируются; когда все
 *   данные были отправлены, модуль принимает заданное в CTRLR1.NDF+1
 *   количество кадров. Режим недоступен для протокола SSP.
 */
typedef enum {
    SPI_ModeDuplex     = 0, /*!< Дуплекс (передача и прием идут одновременно) */
    SPI_ModeSimplexTx  = 1, /*!< Симплекс (только передача) */
    SPI_ModeSimplexRx  = 2, /*!< Симплекс (только прием) */
    SPI_ModeHalfDuplex = 3, /*!< Полудуплекс (сначала передача потом прием). Режим - чтение EEPROM */
} spi_mode_t;

/*!
 * @brief Структура конфигурации для Master SPI
 */
typedef struct {
    struct {
        bool              loopback_enable; /*!< Включить закольцовывание (в целях тестирования) */
        uint32_t          baud_rate_bps;   /*!< Скорость обмена SPI в Hz */
    } master;                              /*!< Конфигурация, актуальная только для master режима */
    spi_frame_width_t     data_width_bits; /*!< Размер кадра данных */
    spi_shift_direction_t direction;       /*!< Формат передачи данных (MSB или LSB) */
    spi_frame_format_t    frame_format;    /*!< Формат кадра (протокла) передачи данных */
    spi_microwire_cfg_t   microwire_cfg;   /*!< Конфигурация для протокола National Semiconductor Microwire */
    spi_motorola_cfg_t    motorola_cfg;    /*!< Конфигурация для протокола Motorola */
} spi_config_t;

/*!
 * @brief Состояния приемо-передачи SPI
 */
enum spi_trans_status {
    SPI_TransStatus_Busy    = 0,   /*!< Модуль занят */
    SPI_TransStatus_Idle    = 1,   /*!< Модуль простаивает */
    SPI_TransStatus_Error   = 128, /*!< Ошибка */
    SPI_TransStatus_Error_1 = 129, /*!< Multi Master Конфликт */
    SPI_TransStatus_Error_2 = 130, /*!< Внутренняя ошибка, TxFIFO Overflow */
    SPI_TransStatus_Error_3 = 132, /*!< Внутренняя ошибка, RxFIFO Overflow */
    SPI_TransStatus_Error_4 = 144, /*!< Внутренняя ошибка, RxFIFO Underflow */
};

/*!
 * @brief Источники прерываний SPI
 *
 * @note Битовые маски подходят для работы с регистрами:
 *        IMR - регистр маскирования прерываний;
 *        ISR - регистр статуса прерываний после маскирования;
 *        RISR - регистр статуса прерываний до маскирования.
 */
enum spi_interrupt_enable {
    SPI_IRQ_MultiMaster     = SPI_IMR_MSTIM_Msk,  /*!< Бит 5: Мультимастер */
    SPI_IRQ_RxFifoTrigger   = SPI_IMR_RXFIM_Msk,  /*!< Бит 4: Прерывание по триггеру уровня RxFIFO, если уровень RxFIFO больше или равен регистру RXFLTR */
    SPI_IRQ_RxFifoOverflow  = SPI_IMR_RXOIM_Msk,  /*!< Бит 3: Переполнение RxFIFO */
    SPI_IRQ_RxFifoUnderflow = SPI_IMR_RXUIM_Msk,  /*!< Бит 2: Чтение из пустого RxFIFO */
    SPI_IRQ_TxFifoOverflow  = SPI_IMR_TXOIM_Msk,  /*!< Бит 1: Переполнение TxFIFO */
    SPI_IRQ_TxFifoTrigger   = SPI_IMR_TXEIM_Msk,  /*!< Бит 0: Прерывание по триггеру уровня TxFIFO, если уровень TxFIFO меньше или равен установленному значению регистра TXFTLR */

    SPI_IRQ_TxOnly                                /*!< Только передача */
        = SPI_IRQ_MultiMaster                       /* Мультимастер */
            | SPI_IRQ_TxFifoOverflow                /* Переполнение TxFIFO */
            | SPI_IRQ_TxFifoTrigger,                /* Прерывание по триггеру уровня TxFIFO */

    SPI_IRQ_RxOnly                                /*!< Только прием */
        = SPI_IRQ_MultiMaster                       /* Мультимастер */
            | SPI_IRQ_RxFifoTrigger                 /* Прерывание по триггеру уровня RxFIFO */
            | SPI_IRQ_RxFifoOverflow                /* Переполнение RxFIFO */
            | SPI_IRQ_RxFifoUnderflow,              /* Чтение из пустого RxFIFO */

    SPI_IRQ_All                                   /*!< Все прерывания включены */
        = SPI_IRQ_MultiMaster
            | SPI_IRQ_RxFifoTrigger
            | SPI_IRQ_RxFifoOverflow
            | SPI_IRQ_RxFifoUnderflow
            | SPI_IRQ_TxFifoOverflow
            | SPI_IRQ_TxFifoTrigger,
};

/*!
 * @brief Структура SPI для приемо-передачи
 *
 * @note Если tx_data равна NULL, то осуществляется только прием, если
 *       rx_data равна NULL - только передача. Если tx_data и rx_data
 *       одновременно не равны NULL, то осуществляются и прием, и передача.
 */
typedef struct {
    uint8_t  *tx_data;  /*!< Буфер на отправку */
    uint8_t  *rx_data;  /*!< Приемный буфер */
    size_t   data_size; /*!< Количесто байт */
} spi_transfer_t;

/*!
 * @brief Структура SPI для полудуплексной приемо-передачи в режиме Master
 */
typedef struct {
    uint8_t  *tx_data;     /*!< Буфер на отправку */
    uint8_t  *rx_data;     /*!< Приемный буфер */
    size_t   tx_data_size; /*!< Количество байт для передачи */
    size_t   rx_data_size; /*!< Количество принятых байт */
} spi_half_duplex_transfer_t;

/*!
 * @brief Внутренняя структура конфигурации модуля SPI
 */
typedef struct {
    spi_shift_direction_t shift_dir;         /*!< Направление сдвига данных при выдаче в шину */
    uint8_t               frame_width_bits;  /*!< Размер кадра данных в битах (допустимые значения: 4 - 32) */
    uint8_t               frame_width_bytes; /*!< Размер кадра данных в байтах (допустимые значения: 1, 2 и 4) */
} spi_config_internal_t;

/*!
 * @brief Прототип структуры
 */
struct spi_handle;

/*!
 * @brief Дескриптор Master SPI для работы по прерыванию
 */
typedef struct spi_handle spi_master_handle_t;

/*!
 * @brief Дескриптор Slave SPI для работы по прерыванию
 */
typedef struct spi_handle spi_slave_handle_t;

/*!
 * @brief Прототип пользовательской функции обратного вызова Master SPI для вызова по окончанию обмена
 */
typedef void (*spi_master_callback_t)(SPI_Type *base,
    spi_master_handle_t *handle, uint32_t status, void *user_data);

/*!
 * @brief Прототип пользовательской функции обратного вызова Slave SPI для вызова по окончанию обмена
 */
typedef void (*spi_slave_callback_t)(SPI_Type *base, spi_slave_handle_t *handle,
    uint32_t status, void *user_data);

/*!
 * @brief SPI структура дескриптора для работы по прерыванию
 *
 * @note Поскольку количество полученных и отправленных кадров должно
 *       совпадать для завершения передачи, значит, если отправленное количество
 *       равно x, а полученное количество равно y, то #to_receive_frame равно
 *       x-y.
 */
struct spi_handle {
    volatile uint8_t      *tx_data;               /*!< Tx буфер */
    volatile uint8_t      *rx_data;               /*!< Rx буфер */
    volatile size_t       tx_remaining_bytes;     /*!< Количество байт, которые осталось передать */
    volatile size_t       rx_remaining_bytes;     /*!< Количество байт, которые осталось принять */
    int8_t                instance;               /*!< Индекс модуля SPI */
    size_t                total_byte_to_transfer; /*!< Общее количество байтов для обмена, если полудуплекс, то передача + прием */
    int32_t               state;                  /*!< Текущее состояние модуля SPI, может быть комбинацией состояний #spi_trans_status по ИЛИ */
    spi_master_callback_t callback;               /*!< Указатель на пользовательскую функцию обратного вызова */
    void                  *user_data;             /*!< Параметр пользовательской функции обратного вызова */
    uint8_t               frame_width_bits;       /*!< Размер кадра данных в битах (допустимые значения: 4 - 32) */
    uint8_t               frame_width_bytes;      /*!< Размер кадра данных в байтах (допустимые значения: 1, 2 и 4) */
    spi_mode_t            mode;                   /*!< Режим обмена */
};

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Получение индекса модуля SPI
 *
 * @param base Базовый адрес SPI
 *
 * @return Индекс модуля
 */
uint8_t SPI_GetInstance(SPI_Type *base);

/*!
 * @name Инициализация и деинициализация
 * @{
 */

/*!
 * @brief Инициализация конфигурации SPI значениями по умолчанию
 *
 * @note Инициализация структуры конфигурации SPI для использования в #SPI_MasterInit.
 * Инициализированную структуру можно использовать как без изменений, так и
 * изменив какие-либо из полей структуры перед вызовом #SPI_MasterInit.
 *
 * Пример:
 * @code
 *  spi_config_t config;
 *  SPI_MasterGetDefaultConfig(&config);
 * @endcode
 *
 * @param config Конфигурация SPI
 */
void SPI_MasterGetDefaultConfig(spi_config_t *config);

/*!
 * @brief Инициализация SPI модуля как Master с заданной конфигурацией
 *
 * Структура конфигурации может быть заполнена пользователем с нуля или
 * установлена по умолчанию значения с помощью #SPI_MasterGetDefaultConfig.
 * После вызова этого API ведомое устройство готово к передаче.
 *
 * Пример:
 * @code
 *  spi_config_t config = {
 *      .frame_width_bits        = SPI_Data8Bits;
 *      .master.loopback_enable = false;
 *      .enable                 = false;
 *      .master.baud_rate_bps   = 50000;
 *      ...
 *  };
 *  SPI_MasterInit(SPI0, &config);
 * @endcode
 *
 * @param base         Базовый адрес SPI
 * @param config       Основная конфигурация SPI
 * @param src_clock_hz Исходная тактовая частота
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 */
enum spi_status SPI_MasterInit(SPI_Type *base, const spi_config_t *config,
    uint32_t src_clock_hz);

/*!
 * @brief Заполнение структуры конфигурации SPI Slave-устройства значениями
 *        по умолчанию
 *
 * @note Функция инициализирует структуру конфигурации для использования в
 * #SPI_SlaveInit. Пользователь может изменить некоторые поля структуры перед
 * вызовом #SPI_SlaveInit.
 *
 * Пример:
 * @code
 *  spi_config_t config;
 *  SPI_SlaveGetDefaultConfig(&config);
 * @endcode
 *
 * @param config Структура конфигурации Slave
 */
void SPI_SlaveGetDefaultConfig(spi_config_t *config);

/*!
 * @brief Инициализация SPI заданной конфигурацией
 *
 * @note Структура конфигурации может быть заполнена пользователем или установлена
 * функцией SPI_SlaveGetDefaultConfig() в значения по умолчанию. После вызова
 * этой функции устройство готово к работе в режиме Slave.
 *
 * Пример:
 * @code
 *  spi_config_t config = {
 *      .direction = SPIMsbFirst;
 *      ...
 *  };
 *  SPI_SlaveInit(SPI0, &config);
 * @endcode
 *
 * @param base   Базовый адрес SPI
 * @param config Структура конфигурации Slave
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 */
enum spi_status SPI_SlaveInit(SPI_Type *base, const spi_config_t *config);

/*!
 * @brief Деинициализация SPI
 *
 * @note Вызов этого API сбрасывает модуль SPI. Модуль SPI не может работать без
 * вызова функции #SPI_MasterInit / #SPI_SlaveInit для инициализации.
 *
 * @param base Базовый адрес SPI
 */
void SPI_Deinit(SPI_Type *base);

/*!
 * @brief Включение или отключение модуль SPI Master или Slave
 *
 * @param base Базовый адрес SPI
 * @param enable Включить (1) или выключить (0)
 */
static inline void SPI_Enable(SPI_Type *base, bool enable)
{
    /* Включить/выключить SPI (SSIENR.SSI_EN). */
    SET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk, SPI_SSIENR_SSI_EN_Pos,
        enable);
}

/*!
 * @brief Получение статуса модуля SPI включено/выключено
 *
 * @param base Базовый адрес SPI
 *
 * @retval 1 Включено
 * @retval 0 Выключено
 */
static inline bool SPI_IsEnable(SPI_Type *base)
{
    /*  SPI включено/выключено (SSIENR.SSI_EN). */
    return (bool)GET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk,
        SPI_SSIENR_SSI_EN_Pos);
}

/*!
 * @brief Останов всех операций SPI
 *
 * @note Любые операции передачи данных прерываются немедленно.
 * FIFO буферы приемника и передатчика очищаются.
 *
 * @param base Базовый адрес SPI
 */
static inline void SPI_Reset(SPI_Type *base)
{
    /* Выключить SPI. */
    SET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk, SPI_SSIENR_SSI_EN_Pos, 0U);

    /* Включить SPI. */
    SET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk, SPI_SSIENR_SSI_EN_Pos, 1U);
}

/*!
 * @}
 */

/*!
 * @name Статусы
 * @{
 */

/*!
 * @brief Получение флага состояния SPI
 *
 * @note Для получения статуса, соответствующего состоянию SPI, необходимо
 *       использовать полученный флаг состояния и #spi_status_flags.
 *
 * @param base Базовый адрес SPI
 *
 * @return Состояние SPI
 */
static inline uint32_t SPI_GetStatusFlags(SPI_Type *base)
{
    assert(NULL != base);
    return base->SR;
}

/*!
 * @}
 */

/*!
 * @name Прерывания
 * @{
 */

/*!
 * @brief Включение прерываний SPI
 *
 * @note В качестве источника прерывания может быть комбинация следующих
 *       значений:
 *        #SPI_IRQ_MultiMaster
 *        #SPI_IRQ_RxFifoTrigger
 *        #SPI_IRQ_RxFifoOverflow
 *        #SPI_IRQ_RxFifoUnderflow
 *        #SPI_IRQ_TxFifoOverflow
 *        #SPI_IRQ_TxFifoTrigger
 *        #SPI_IRQ_All
 *
 * @param base Базовый адрес SPI
 * @param irqs Источники прерываний
 */
static inline void SPI_EnableInterrupts(SPI_Type *base, uint32_t irqs)
{
    assert(NULL != base);

    /*
     * IMR - регистр маскирования прерываний:
     * 0 - прерывание замаскировано, 1 - активно.
     */
    base->IMR |= irqs;
}

/*!
 * @brief Отключение прерываний SPI
 *
 * @note В качестве источника прерывания может быть комбинация следующих
 *       значений:
 *        #SPI_IRQ_MultiMaster
 *        #SPI_IRQ_RxFifoTrigger
 *        #SPI_IRQ_RxFifoOverflow
 *        #SPI_IRQ_RxFifoUnderflow
 *        #SPI_IRQ_TxFifoOverflow
 *        #SPI_IRQ_TxFifoTrigger
 *        #SPI_IRQ_All
 *
 * @param base Базовый адрес SPI
 * @param irqs Источники прерываний
 */
static inline void SPI_DisableInterrupts(SPI_Type *base, uint32_t irqs)
{
    assert(NULL != base);

    /*
     * IMR - регистр маскирования прерываний:
     * 0 - прерывание замаскировано, 1 - активно.
     */
    base->IMR &= ~irqs;
}

/*!
 * @brief Получение статуса прерываний
 *
 * @param base Базовый адрес SPI
 *
 * @return Статус прерываний
 */
static inline uint32_t SPI_CurrentStatusInterrupts(SPI_Type *base)
{

    assert(NULL != base);

    /* ISR - регистр статуса прерываний после маскирования. */
    return (base->ISR & SPI_IRQ_All);
}

/*!
 * @brief Сброс флагов прерывания
 *
 * @note Флаги @SPI_IRQ_RxFifoTrigger и @SPI_IRQ_TxFifoTrigger
 *       таким образом сбросить нельзя, по этой причине использование
 *       @SPI_IRQ_All для сброса всех прерываний недопустимо.
 *
 * @note Прерывания @SPI_IRQ_RxFifoTrigger и @SPI_IRQ_TxFifoTrigger можно
 *       сбросить, вычитав RxFifo и TxFifo соответственно.
 *
 * @param base Базовый адрес SPI
 * @param irqs Источники прерываний
 */
static inline void SPI_TakeDownInterrupts(SPI_Type *base, uint32_t irqs)
{
    assert(NULL != base);
    assert(0 == (irqs & SPI_IRQ_RxFifoTrigger));
    assert(0 == (irqs & SPI_IRQ_TxFifoTrigger));

    volatile uint32_t temp = 0;
    UNUSED(temp);

    /* xxxICR - Регистры снятия прерываний. */

    /* Требуется сброс всех прерываний (допустимых для сброса таким образом). */
    if ((SPI_IRQ_MultiMaster
            | SPI_IRQ_RxFifoOverflow
            | SPI_IRQ_RxFifoUnderflow
            | SPI_IRQ_TxFifoOverflow) == irqs) {
        temp = GET_VAL_MSK(base->ICR, SPI_ICR_ICR_Msk, SPI_ICR_ICR_Pos);
    }
    /* Требуется сброс некоторых прерываний. */
    else {
        if (0 != (irqs & SPI_IRQ_MultiMaster)) {
            /* MSTICR - Регистр  сброса прерывания: “Возможный конфликт master устройств”. */
            temp = GET_VAL_MSK(base->MSTICR, SPI_MSTICR_MSTICR_Msk, SPI_MSTICR_MSTICR_Pos);
        }

        if (0 != (irqs & SPI_IRQ_RxFifoOverflow)) {
            /* RXOICR - Регистр сброса прерывания: “Переполнен FIFO приемника”. */
            temp = GET_VAL_MSK(base->RXOICR, SPI_RXOICR_RXOICR_Msk, SPI_RXOICR_RXOICR_Pos);
        }

        if (0 != (irqs & SPI_IRQ_RxFifoUnderflow)) {
            /* RXUICR - Регистр сброса прерывания: “FIFO приемника пуст”. */
            temp = GET_VAL_MSK(base->RXUICR, SPI_RXUICR_RXUICR_Msk, SPI_RXUICR_RXUICR_Pos);
        }

        if (0 != (irqs & SPI_IRQ_TxFifoOverflow)) {
            /* TXOICR - Регистр сброса прерывания: “Переполнен FIFO передатчика”.*/
            temp = GET_VAL_MSK(base->TXOICR, SPI_TXOICR_TXOICR_Msk, SPI_TXOICR_TXOICR_Pos);
        }
    }

    return;
}

/*!
 * @}
 */

/*!
 * @name DMA управление
 * @{
 */

/*!
 * @brief Включение или отключение запроса DMA от SPI TxFIFO
 *
 * @param base   Базовый адрес SPI
 * @param enable Включить (1) или выключить (0)
 */
void SPI_EnableTxDMA(SPI_Type *base, bool enable);

/*!
 * @brief Включение или отключение запроса DMA от SPI RxFIFO
 *
 * @param base   Базовый адрес SPI
 * @param enable Включить (1) или выключить (0)
 */
void SPI_EnableRxDMA(SPI_Type *base, bool enable);

/*!
 * @}
 */

/*!
 * @name Операции на шине SPI
 * @{
 */

/*!
 * @brief Получение внутренней конфигурации
 *
 * @param base Базовый адрес SPI
 *
 * @return Внутренняя структура конфигурации модуля SPI
 */
spi_config_internal_t *SPI_GetConfig(SPI_Type *base);

/*!
 * @brief Установка скорости передачи для SPI Master
 *
 * @param base         Базовый адрес SPI
 * @param baudrate_bps Скорость передачи в Hz
 * @param src_clock_hz SPI частота синхронизации в Hz
 *
 * @retval SPI_Status_Ok
 * @retval SPI_Status_BaudrateNotSupport
 * @retval SPI_Status_InvalidArgument
 */
enum spi_status SPI_MasterSetBaud(SPI_Type *base, uint32_t baudrate_bps,
    uint32_t src_clock_hz);

/*!
 * @brief Запись данных в регистр данных SPI
 *
 * @param base Базовый адрес SPI
 * @param data Данные для записи
 */
void SPI_WriteData(SPI_Type *base, uint32_t data);

/*!
 * @brief Получение данных из регистра данных SPI
 *
 * @param base Базовый адрес SPI
 *
 * @return Данные регистра
 */
uint32_t SPI_ReadData(SPI_Type *base);

/*!
 * @brief Запись фиктивных данных для передачи
 *
 * @note Фиктивные данные передаются в @SPI_MasterTransferBlocking,
 * когда буфер Tx пуст, но есть запрос на прием.
 *
 * @param base       Базовый адрес SPI
 * @param dummy_data Фиктивные данные
 */
void SPI_SetDummyData(SPI_Type *base, uint8_t dummy_data);

/*!
 * @}
 */

/*!
 * @name Приемо-передача
 * @{
 */

/*!
 * @brief Инициализация дескриптора SPI Master
 *
 * @param base      Базовый адрес SPI
 * @param handle    SPI дескриптор
 * @param callback  Callback-функция
 * @param user_data Данные пользователя для callback-функции
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 */
enum spi_status SPI_MasterTransferCreateHandle(SPI_Type *base,
    spi_master_handle_t *handle, spi_master_callback_t callback,
    void *user_data);

/*!
 * @brief Блокирующая дуплексная передача данных
 *        (с ожиданием завершения операции)
 *
 * @note Функция не поддерживает симплексный режим обмена.
 * Это значит, что в xfer должны быть переданы указатели на
 * данные для передачи и буфер на прием с указанием длины, неравной нулю.
 * Если требуется симплексный обмен, используйте 
 * @SPI_MasterHalfDuplexTransferBlocking
 *
 * @param base Базовый адрес SPI
 * @param xfer Структура для приемо-передачи
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 * @retval #SPI_Status_Timeout
 */
enum spi_status SPI_MasterTransferBlocking(SPI_Type *base,
    spi_transfer_t *xfer);

/*!
 * @brief Неблокирующий SPI обмен по прерыванию
 *
 * @note Функция используется для разнесенной по времени передачи и приема,
 * первым этапом происходит передача данных заданной ширины,
 * вторым - прием данных. Весь обмен происходит в асинхронном режиме,
 * т.е в прерываниях. По завершении второй части обмена произойдет вызов
 * callback функции.
 *
 * Функция не поддерживает симплексный режим обмена.
 * Это значит, что в xfer должны быть переданы указатели на
 * данные для передачи и буфер на прием с указанием длины, неравной нулю.
 * Если требуется симплексный обмен, используйте 
 * @SPI_MasterHalfDuplexTransferNonBlocking
 *
 * @param base Базовый адрес SPI.
 * @param handle SPI структура дескриптора для работы по прерыванию.
 * @param xfer Структура для полудуплексной приемо-передачи.
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 * @retval #SPI_Status_IncorrectCall
 */
enum spi_status SPI_MasterTransferNonBlocking(SPI_Type *base,
    spi_master_handle_t *handle, spi_transfer_t *xfer);

/*!
 * @brief Блокирующая полудуплексная передача данных (с ожиданием завершения
 *        операции)
 *
 * @note Функция не возвращает управление, пока все данные не будут переданы. Данные
 * передаются в полудуплексном режиме. Пользователь может выбрать, что будет
 * выполняться в первую очередь - передача или прием.
 *
 * @param base Базовый адрес SPI
 * @param xfer Структура для полудуплексной приемо-передачи
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 * @retval #SPI_Status_Timeout
 */
enum spi_status SPI_MasterHalfDuplexTransferBlocking(SPI_Type *base,
    spi_half_duplex_transfer_t *xfer);

/*!
* @brief Выполнение неблокирующего SPI обмена по прерыванию
*
* @note Эта функция использует режим опроса для первый части обмена и использует
* прерывания для выполения второй части обмена. Режим обмена полудуплексный.
* Когда завершится первая часть обмена, сразу произойдет выход из функции,
* по завершении второй части обмена произойдет вызов callback-функции.
*
* @param base Базовый адрес SPI.
* @param handle SPI структура дескриптора для работы по прерыванию.
* @param xfer Структура для полудуплексной приемо-передачи.
*
* @retval #SPI_Status_Ok
* @retval #SPI_Status_InvalidArgument
*/
status_t SPI_MasterHalfDuplexTransferNonBlocking(SPI_Type *base,
    spi_master_handle_t *handle,
    spi_half_duplex_transfer_t *xfer);

/*!
 * @brief Получение количества переданных и принятых байт
 *
 * @param handle SPI структура дескриптора для работы по прерыванию.
 * @param count  Количество байтов, переданных с помощью неблокирующей транзакции.
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 */
status_t SPI_MasterTransferGetByte(spi_master_handle_t *handle, size_t *count);

/*!
 * @brief Получение количества оставшихся байт для передачи и приема
 *
 * @param handle SPI структура дескриптора для работы по прерыванию.
 * @param count  Количество байтов, оставшихся для передачи и приема c помощью неблокирующей транзакции.
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 */
status_t SPI_MasterTransferGetRemainingByte(spi_master_handle_t *handle,
    size_t *count);

/*!
 * @brief Получение общего количества байт на передачу и прием
 *
 * @param handle SPI структура дескриптора для работы по прерыванию.
 * @param count  Общее количество байт на передачу и прием
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 */
status_t SPI_MasterTransferGetTotalByte(spi_master_handle_t *handle,
    size_t *count);

/*!
 * @brief Останов передачи для мастер режима 
 *
 * @note Функция останавливает передачу, происходящую по прерываниям.
 *
 * @param base   Базовый адрес SPI.
 * @param handle SPI структура дескриптора для работы по прерыванию.
 */
void SPI_MasterTransferAbort(SPI_Type *base, spi_master_handle_t *handle);

/*!
 * brief Обработчик прерываний для SPI.
 *
 * @param base   Базовый адрес SPI.
 * @param handle SPI структура дескриптора для работы по прерыванию.
 */
void SPI_MasterTransferHandleIRQ(SPI_Type *base, spi_master_handle_t *handle);

/*!
 * @brief Инициализация slave SPI дескриптор
 *
 * @note Функция инициализирует дескриптор slave SPI. Для указанного модуля SPI
 * достаточно вызвать один раз эту функцию, чтобы получить инициализированный дескриптор.
 *
 * @param base   Базовый адрес SPI.
 * @param handle SPI структура дескриптора для работы по прерыванию.
 * @param callback Callback-функция.
 * @param userData Данные пользователя.
 *
 * @retval #SPI_Status_UnexpectedState
 * @retval #SPI_Status_Ok
 */
static inline status_t SPI_SlaveTransferCreateHandle(SPI_Type *base,
    spi_slave_handle_t *handle,
    spi_slave_callback_t callback,
    void *userData)
{
    return SPI_MasterTransferCreateHandle(base, handle, callback, userData);
}

/*!
 * @brief Неблокирующая дуплексная передача данных (без ожидания завершения
 *        операции)
 *
 * @param base   Базовый адрес SPI.
 * @param handle Структура, сохраняющая состояние приемо-передачи.
 * @param xfer   Структура для приемо-передачи.
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 * @retval #SPI_Status_Busy
 */
static inline status_t SPI_SlaveTransferNonBlocking(SPI_Type *base,
    spi_slave_handle_t *handle, spi_transfer_t *xfer)
{
    return SPI_MasterTransferNonBlocking(base, handle, xfer);
}

static inline status_t SPI_SlaveHalfDuplexTransferNonBlocking(SPI_Type *base,
    spi_slave_handle_t *handle, spi_half_duplex_transfer_t *xfer)
{
    return SPI_MasterHalfDuplexTransferNonBlocking(base, handle, xfer);
}
/*!
 * @brief Получение количества байт для обмена
 *
 * @param handle SPI структура дескриптора для работы по прерыванию.
 * @param count  Количество байтов, переданных с помощью неблокирующей транзакции.
 *
 * @retval #SPI_Status_Ok
 * @retval #SPI_Status_InvalidArgument
 */
static inline status_t SPI_SlaveTransferGetByte(spi_slave_handle_t *handle,
    size_t *count)
{
    return SPI_MasterTransferGetByte((spi_master_handle_t *) handle, count);
}

/*!
 * @brief Останов передачи для режима Slave по прерыванию.
 *
 * @param base   Базовый адрес SPI.
 * @param handle Структура дескриптора для работы по прерыванию.
 */
static inline void SPI_SlaveTransferAbort(SPI_Type *base, spi_slave_handle_t *handle)
{
    SPI_MasterTransferAbort(base, (spi_master_handle_t *) handle);
}

/*!
 * @brief Обработчик прерываний для SPI
 *
 * @param base   Базовый адрес SPI.
 * @param handle SPI структура дескриптора для работы по прерыванию.
 */
static inline void SPI_SlaveTransferHandleIRQ(SPI_Type *base,
    spi_slave_handle_t *handle)
{
    SPI_MasterTransferHandleIRQ(base, handle);
}

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif

#endif /* HAL_SPI_H */

/*!
 * @}
 */
