#!/usr/bin/env python3

# Copyright 2018-2019 RnD Center "ELVEES", JSC

import argparse
import json
import sys
import wave
from collections import OrderedDict

import numpy
from numpy.fft import fft


def read_wav(fname):
    f = wave.open(sys.stdin.buffer if fname == "-" else fname, "rb")
    info = OrderedDict(
        [
            ("nchannels", f.getnchannels()),
            ("sampwidth", f.getsampwidth()),
            ("framerate", f.getframerate()),
            ("nframes", f.getnframes()),
        ]
    )

    data = f.readframes(info["nframes"])
    f.close()
    dtype = "<i{}".format(info["sampwidth"])
    samples = numpy.frombuffer(data, dtype=dtype).reshape(-1, info["nchannels"]).T

    return samples.astype("float"), info


def get_channel_info(samples, sampwidth, framerate, signals_count):
    samples = samples / (256**sampwidth)
    f = numpy.abs(fft(samples, framerate))
    f = f[: len(f) // 2]  # cut mirror
    f = (2.0 / len(f)) * f  # scale
    signals = []
    signals_freq = numpy.argsort(f)[-signals_count:] if signals_count > 0 else []
    for freq in reversed(signals_freq):
        signal_db = 20 * numpy.log10(f[freq])
        signals.append(OrderedDict([("frequency", int(freq)), ("db", signal_db)]))

    return signals


def main():
    description = """Script to analyze WAV file. Showing frequency and amplitude ratio of
several strongest frequencies for each channel.

Example: %(prog)s file.wav -c 3
Example: arecord -f dat -d 1 | %(prog)s - -j"""
    parser = argparse.ArgumentParser(
        description=description, formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument("file_name", help='WAV file to analyze ("-" for use stdin)')
    parser.add_argument("-j", "--json", action="store_true", help="output result in JSON format")
    parser.add_argument(
        "-c",
        "--signals_count",
        type=int,
        default=2,
        help="number of frequncies with maximum amplitude for which to dump "
        "detailed statistics (default: %(default)s)",
    )
    args = parser.parse_args()
    samples, info = read_wav(args.file_name)
    info["channels"] = []
    for channel_samples in samples:
        cinfo = get_channel_info(
            channel_samples, info["sampwidth"], info["framerate"], args.signals_count
        )
        info["channels"].append(cinfo)

    if args.json:
        print(json.dumps(info, indent=4))
    else:
        print(
            "Frame rate: {0[framerate]}\n"
            "Sample width: {0[sampwidth]}\n"
            "Number of frames in file: {0[nframes]}\n"
            "Number of channels: {0[nchannels]}".format(info)
        )
        for ch, cinfo in enumerate(info["channels"]):
            print("Channel {0}:".format(ch))
            for sig_num, sig in enumerate(cinfo):
                amplitude = 10 ** (sig["db"] / 20) * 100
                print(
                    "  Signal {0}\n"
                    "    Frequency: {1[frequency]} Hz\n"
                    "    Amplitude: {1[db]:.3f} dB ({2:.1f}%)".format(sig_num, sig, amplitude)
                )


if __name__ == "__main__":
    main()
