#!/bin/sh
#
# Copyright 2016-2020 RnD Center "ELVEES", JSC
# SPDX-License-Identifier: BSD-2-Clause
#
# Author: Dmitriy Zagrebin <dzagrebin@elvees.com>
#
# This is a simple test for MCom-02 clock management subsystem. It consists of:
#     * Initial value checking for all PLLs and DIVs.
#     * Clock gate testing for most blocks.

# Limitations:
# We can not perform clock gate testing for SDMMC0, SDMMC1, UART0
# due to:
#     * SDMMC0 is used by rootfs.
#     * SDMMC1 is used by Wifi driver which can not be unbound.
#     * UART0 is used by console

# The test can be run on the following boards:
#     salute-el24d1-rev1.3;
#     salute-el24d1-rev1.4;
#     salute-el24d1-rev1.5;
#     salute-el24d2-rev1.1;
#     salute-el24pm2-rev1.0;
#     salute-el24pm2-rev1.1;

REINIT_SLEEP_TIME=2
COMMON_PATH=/sys/bus/platform/drivers
GATE_SYS_CTR=0x3809404c
GATE_CORE_CTR=0x38094048
GATE_DSP_CTR=0x38094068

SALUTE_D1_GATES="0x3809404c:0x0038113D 0x38094048:0x0000004B 0x38094068:0x00000000"
SALUTE_D1_PLLS="0x38094100:0x21 0x38094104:0x0F 0x38094108:0x1B \
0x3809410c:0x0B 0x38094110:0x0F 0x38094114:0x00"
SALUTE_D1_DIVS="0x38094040:0x00 0x38094044:0x01 0x38094004:0x01 \
0x38094008:0x03 0x3809400c:0x01 0x3809402c:0x01 0x38094038:0x02"

SALUTE_D2_GATES="0x3809404c:0x0038103D 0x38094048:0x0000004B 0x38094068:0x00000000"
SALUTE_D2_PLLS="0x38094100:0x21 0x38094104:0x14 0x38094108:0x1B \
0x3809410c:0x0B 0x38094110:0x0F 0x38094114:0x00"
SALUTE_D2_DIVS="0x38094040:0x00 0x38094044:0x01 0x38094004:0x01 \
0x38094008:0x03 0x3809400c:0x01 0x3809402c:0x01 0x38094038:0x02"

SALUTE_PM2_GATES="0x3809404c:0x00281DBF 0x38094048:0x0000005F 0x38094068:0x00000000"
SALUTE_PM2_PLLS="0x38094100:0x21 0x38094104:0x14 0x38094108:0x1B \
0x3809410c:0x0B 0x38094110:0x0F 0x38094114:0x00"
SALUTE_PM2_DIVS="0x38094040:0x00 0x38094044:0x01 0x38094004:0x01 \
0x38094008:0x03 0x3809400c:0x01 0x3809402c:0x01 0x38094038:0x02"

DURATION=1
SIZE=1280x720
TMP_VIDEO=/tmp/input.y4m

echo performance > /sys/devices/system/cpu/cpu0/cpufreq/scaling_governor

BOARD_MODEL=$(tr -d '\0' < /proc/device-tree/model)

match_board () {
    echo "$BOARD_MODEL" | grep -q "$1"
}

get_bit () {
    val=$(devmem $1)
    return $(( $(( val >> $2 )) & 1 ))
}

print_msg () {
    echo "******************************"
    echo $1
    echo "******************************"
}

run_vpu_encoding () {
    ffmpeg -loglevel error -f lavfi \
            -i testsrc=duration=$DURATION:size=$SIZE \
            -pix_fmt yuv420p -y $TMP_VIDEO 1>&-
    m2m-test -d /dev/v4l/by-path/platform-37100000.codec-video-index0 \
                -v -o /dev/null $TMP_VIDEO -t -l 5 1>&- &
    echo $!
}

run_serial_flushing() {
    DEV=$(basename $(readlink -f /sys/bus/platform/devices/$1/tty/tty*))
    timeout 5 cat /dev/random > /dev/$DEV &
    echo $!
}

load_module() {
    if [ ! -z $3 ]; then
        modprobe $3
    else
        echo $1 > ${COMMON_PATH}/$2/bind
    fi
}

unload_module() {
    if [ ! -z $3 ]; then
        modprobe -r $3
    else
        echo $1 > ${COMMON_PATH}/$2/unbind
    fi
}

assert_bit() {
    get_bit $1 $2
    if [ $? -eq "$3" ]; then
        echo $4
        return 1
    fi
    return 0
}

check_runtime_pm_gate() {
    assert_bit $1 $2 "1" "Error: clock must be gated"
    gate_check_rc=$?

    process=$($3 $4)
    sleep $REINIT_SLEEP_TIME

    assert_bit $1 $2 "0" "Error: clock must be enabled"
    clk_check_rc=$?

    while [ -e /proc/$process ]; do
        sleep 1
    done

    if [ $gate_check_rc -eq 1 -o $clk_check_rc -eq 1 ]; then
        print_msg "GATE CHECK IS FAILED"
        return 1
    else
        print_msg "GATE CHECK IS PASSED"
        return 0
    fi
}

check_gate () {
    unload_module $1 $2 $5
    assert_bit $3 $4 "1" "Error: clock must be gated"
    gate_check_rc=$?

    sleep $REINIT_SLEEP_TIME

    load_module $1 $2 $5
    assert_bit $3 $4 "0" "Error: clock must be enabled"
    clk_check_rc=$?

    if [ $gate_check_rc -eq 1 -o $clk_check_rc -eq 1 ]; then
        print_msg "GATE CHECK IS FAILED"
        return 1
    else
        print_msg "GATE CHECK IS PASSED"
        return 0
    fi
}

check_initial_values () {
    fail=0
    for i in $1; do
        addr=$(echo $i | cut -d ":" -f1)
        expected_val=$(echo $i | cut -d ":" -f2)
        real_val=$(devmem $addr $2)
        if [ "$real_val" != "$expected_val" ]; then
            echo "Bad value at $addr : $real_val. Must be $expected_val"
            fail=1
        fi
    done

    if [ $fail -eq "1" ]; then
        print_msg "INITIAL $3 VALUE CHECK IS FAILED"
        return 1
    else
        print_msg "INITIAL $3 VALUE CHECK IS PASSED"
        return 0
    fi
}

case "$1" in
eth)
    check_gate "3800f000.ethernet" "arasan-gemac" $GATE_SYS_CTR 4
;;
gpu)
    check_gate "gpu" "gpu" $GATE_CORE_CTR 6 mali
;;
vinc)
    check_gate "37200000.vinc" "vinc" $GATE_CORE_CTR 3
;;
vpout)
    if match_board "Salute-EL24PM2"; then
        check_gate "38004000.vpout" "vpout-drm" $GATE_CORE_CTR 4 vpout-drm
    else
        # For Salute-EL24D1 or Salute-EL24D2 boards VPOUTFB driver is not loaded by default.
        # According to bugs #MCOM02SW-470, #MCOM02SW-1334, we cannot correctly unload
        # the driver or load after unloading and bring the system to the default state.
        # We check only the presence of frequency if the VPOUTFB driver is loaded
        # or the absence of frequency if the driver is unloaded.
        if $(lsmod | grep -qFe vpoutfb) ; then
            assert_bit $GATE_CORE_CTR 4 "0" "Error: clock must be enabled"
        else
            assert_bit $GATE_CORE_CTR 4 "1" "Error: clock must be gated"
        fi
    fi
;;
vpu)
    check_runtime_pm_gate $GATE_CORE_CTR 5 run_vpu_encoding
;;
usb)
    check_gate "38002000.usb" "dwc2" $GATE_SYS_CTR 5
;;
nfc)
    check_gate "38007000.nand-controller" "arasan_nfc" $GATE_SYS_CTR 21
;;
spi)
    if match_board "Salute-EL24PM2"; then
        check_gate "38032000.spi" "dw_spi_mmio" $GATE_SYS_CTR 19
    else
        check_gate "38032000.spi" "dw_spi_mmio" $GATE_SYS_CTR 19 &&
        check_gate "38033000.spi" "dw_spi_mmio" $GATE_SYS_CTR 20
    fi
;;
uart)
    if match_board "Salute-EL24D2"; then
        check_runtime_pm_gate $GATE_SYS_CTR 14 run_serial_flushing 3802a000.serial
    else
        check_runtime_pm_gate $GATE_SYS_CTR 14 run_serial_flushing 3802a000.serial &&
        check_runtime_pm_gate $GATE_SYS_CTR 15 run_serial_flushing 3802b000.serial0
    fi
;;
dsp)
    modprobe -r avico
    echo 37220000.dma > /sys/bus/amba/drivers/dma-pl330/unbind
    check_gate "37001000.dsp" "delcore30m" $GATE_DSP_CTR 0 delcore30m &&
    check_gate "37001000.dsp" "delcore30m" $GATE_DSP_CTR 1 delcore30m &&
    check_gate "37001000.dsp" "delcore30m" $GATE_DSP_CTR 2 delcore30m &&
    check_gate "37001000.dsp" "delcore30m" $GATE_DSP_CTR 3 delcore30m
    rc=$?
    modprobe -r delcore30m
    echo 37220000.dma > /sys/bus/amba/drivers/dma-pl330/bind
    modprobe avico
    exit $rc
;;
init_values)
    if match_board "Salute-EL24D1"; then
        check_initial_values "$SALUTE_D1_GATES" 32 "SALUTE_D1_GATES" &&
        check_initial_values "$SALUTE_D1_PLLS" 8 "SALUTE_D1_PLLS" &&
        check_initial_values "$SALUTE_D1_DIVS" 8 "SALUTE_D1_DIVS"
    elif match_board "Salute-EL24PM2"; then
        check_initial_values "$SALUTE_PM2_GATES" 32 "SALUTE_PM2_GATES" &&
        check_initial_values "$SALUTE_PM2_PLLS" 8 "SALUTE_PM2_PLLS" &&
        check_initial_values "$SALUTE_PM2_DIVS" 8 "SALUTE_PM2_DIVS"
    elif match_board "Salute-EL24D2"; then
        check_initial_values "$SALUTE_D2_PLLS" 8 "SALUTE_D2_PLLS" &&
        check_initial_values "$SALUTE_D2_DIVS" 8 "SALUTE_D2_DIVS" &&
        check_initial_values "$SALUTE_D2_GATES" 32 "SALUTE_D2_GATES"
    fi
;;
*)
    echo "Unknown option: $1"
    exit 1
;;
esac
