#!/usr/bin/env bash
# Copyright 2019 RnD Center "ELVEES", JSC

NAMES=(RAM XYRAM JPEGRAM DSPRAM0 DSPRAM1 VRAM)
ADDRESES=(0x20000000 0x3a400000 0x3a480000 0x3a600000 0x3a620000 0x3b000000)
SIZES=(64K 256K 64K 32K 32K 1M)
TIMES=("3 sec" "7 sec" "2 sec" "1 sec" "1 sec" "37 sec")
RESULTS=()
MODULES=()
VERBOSE=0

unload_module() {
    lsmod | grep $1 > /dev/null
    if [ "$?" == "0" ]; then
        rmmod $1
        MODULES+=($1)
    fi
}

help() {
    echo "memtester-mcom02.sh - tool to test MCom02 internal memories"
    echo "Usage: memtester-mcom02.sh [options]"
    echo ""
    echo "Options:"
    echo "  -h   Show help"
    echo "  -v   Increase verbosity (show memtester output)"
}

while getopts "vh" opt; do
    case $opt in
        v) VERBOSE=1;;
        h) help; exit 0;;
        ?) exit 1;;
    esac
done

dmesg -D
unload_module "avico"
unload_module "delcore30m"
dmesg -E

# Save GATE_CORE_CTR and enable VPU clock
GATE_CORE_CTR=$(devmem 0x38094048)
devmem 0x38094048 32 $((GATE_CORE_CTR | 0x20))

# Save and set UPLL
UPLL=$(devmem 0x38094114)
UPLL=$((UPLL & 0xff))
devmem 0x38094114 32 0xe

# Save power domain status and enable power for VPU
CORE_PWR_STATUS=$(devmem 0x3809508c)
devmem 0x38095080 32 $CORE_PWR_STATUS

for i in ${!NAMES[*]}; do
    echo "Testing ${NAMES[$i]} (duration: ${TIMES[$i]})..."
    if [ $VERBOSE -eq 0 ]; then
        MEMTESTER_TEST_MASK=0x07fff memtester -p ${ADDRESES[$i]} ${SIZES[$i]} 1 > /dev/null
    else
        MEMTESTER_TEST_MASK=0x07fff memtester -p ${ADDRESES[$i]} ${SIZES[$i]} 1
    fi
    RESULTS[$i]=$?
done

echo ""
echo "Results:"
IS_FAIL=0
for i in ${!NAMES[*]}; do
    NAME=${NAMES[$i]}
    SPACES=$((8 - ${#NAMES[$i]}))
    for _ in $(seq $SPACES); do
        NAME="$NAME "
    done
    if [ ${RESULTS[$i]} -eq 0 ]; then
        RES="ok"
    else
        RES="fail"
        IS_FAIL=1
    fi
    echo "$NAME $RES"
done

echo ""
if [ $IS_FAIL -eq 0 ]; then
    echo "ALL TESTS PASSED"
else
    echo "SOME TESTS FAILED"
fi

# Restore power domain, clocks and loaded modules
if [ $((CORE_PWR_STATUS + 0)) -ne 0 ]; then
    devmem 0x38095084 32 $CORE_PWR_STATUS
fi
devmem 0x38094114 32 $UPLL
devmem 0x38094048 32 $GATE_CORE_CTR
dmesg -D
for m in ${MODULES[*]}; do
    modprobe $m
done
dmesg -E

exit $IS_FAIL
