#!/usr/bin/env bats
# Copyright 2020 RnD Center "ELVEES", JSC
#
# This is Bash BATS test runner for running various ElcoreCL kernels and associated executables.
#
# Test running examples:
# bats /usr/bin/elcorecl-tests.bats - run all tests
# bats /usr/bin/elcorecl-tests.bats -f test - run all tests with 'test' in name
#
# `set -e` is set by bats framework

binaries_dir="/usr/share/elcore50-extra"
platform_name="MCOM03"

@test "elcore50_km_modprobe" {
    for i in $(seq 1 100); do
        modprobe -r elcore50 && modprobe elcore50
        if [ "$platform_name" = "MCOM03" ]; then
            paralleltest /usr/share/elcore50/elf-buf-sum.elf -d 2 2> /dev/null
        else
            paralleltest /usr/share/elcore50/elf-buf-sum.elf -d 16 2> /dev/null
        fi
    done
}

@test "coremark_single_core" {
    elcorecl-run -e /usr/share/elcore50/coremark
}

@test "coremark_multi_core" {
    elcorecl-run -e /usr/share/elcore50/coremark --core=all
}

@test "tinymembench_single_core" {
    elcorecl-run -e $binaries_dir/tinymembench
}

@test "tinymembench_multi_core" {
    elcorecl-run -e $binaries_dir/tinymembench --core=all
}

@test "dsp_dma_rw" {
    if [ "$platform_name" = "MCOM03" ]; then
        elcorecl-run -e $binaries_dir/dsp-dma-12 -s 16384 --core=all
    else
        elcorecl-run -e $binaries_dir/dsp-dma-44 -s 16384 --core=all
    fi
}

@test "test1_syscall" {
    output=$(elcorecl-run -e $binaries_dir/test1-syscall  --core=all)
    [[ ! $(grep -i error <<< $output) ]]
}

@test "test1_syscall_xyram" {
    output=$(elcorecl-run -e $binaries_dir/test1-syscall-xyram --core=all)
    [[ ! $(grep -i error <<< $output) ]]
}

@test "test1_syscall_xyram_interleave" {
    output=$(elcorecl-run -e $binaries_dir/test1-syscall-xyram-interleave --core=all)
    [[ ! $(grep -i error <<< $output) ]]
}

@test "test2_syscall" {
    mkdir -p /tmp/test
    dir=$(pwd)
    # times(2) returns values not in CPU clock cycles, but in
    # tens of microseconds (since CLK_TCL = 100).
    # We write to stdin with delay = 2 seconds to provide execution
    # time exceeding 10 us.
    output=$((sleep 2; echo -n a) | elcorecl-run -e $binaries_dir/test2-syscall --core=0)
    [[ ! $(grep -i error <<< $output) ]]
    cd $dir
}

@test "test_xyram" {
    elcorecl-run -e $binaries_dir/test-xyram --core=all
}

@test "test_xyram_interleave" {
    elcorecl-run -e $binaries_dir/test-xyram-interleave  --core=all
}

@test "test_elcorecl_run" {
    output=$(elcorecl-run -e $binaries_dir/test-kernel-arguments --core=0 \
             -- -5 65536 -14.305 5.1001 foo)
    [[ $output == *"args: -5 65536 -14.305 5.1001 foo"* ]]
}

@test "test_elcorecl_parallel" {
    if [[ "$(uname -r)" =~ 4.*.* && "$platform_name" == "MCOM03" ]]; then
        skip "Workaround for MCOM03SW-2569"
    fi

    FAIL=0
    for i in $(seq 0 15); do
        # Can't run the filtered-out test: it requests 3.6GiB memory for one process
        # and this will spawn 16 processes, triggering OOM-killer.
        output=$(test-elcorecl --gtest_filter=*/$i:-*CheckNonCrossingElfSectionsAndNonCachedPools* &)
    done

    for job in $(jobs -p); do
        wait $job || let "FAIL+=1"
    done
   [[ "$FAIL" == "0" ]]
}
