#!/bin/bash
#
# Copyright 2021 RnD Center "ELVEES", JSC
#
# Script to setup LT86104UX on MONOBOARD r1.2.2.

DEBUG=true
I2C_DEBUG=false

I2C_BUS=1
I2C_ADDR=0x3f

readonly LOW=0
readonly HIGH=1

readonly DRIVE_OFF=0
readonly DRIVE_ON=1

readonly CLK_CHANGE_MARGIN=$(( 8 * 1024 ))        #/*CLK_CHANGE_MARGIN*1KHz */
readonly CLK_ACTIVE_MIN_LIMIT=$(( 16 * 1024 ))    #/*CLK_ACTIVE_MIN_LIMIT should be lower than 25M(480P). */


# /******************** TX Parameters ***********************/

readonly TX_SWING_200MHZ=0x50
readonly TX_SWING_300MHZ=0x50
readonly TX_SWING_600MHZ=0x60            #/*0x20 */
readonly TX_TAP_SWING_200MHZ=0x00
readonly TX_TAP_SWING_300MHZ=0x43
readonly TX_TAP_SWING_600MHZ=0xcf        #/*0x0F */
readonly TX_PREEMPHASIS=0xb0
readonly TX_NOPREEMPHASIS=0x00
# /*****************************************************/
readonly FREQ_DIV_200MHZ=0x030d40
readonly FREQ_DIV_300MHZ=0x053020


# /******************** EDID Parameters ***********************/
if true
then
    # HDMI1.4
    readonly l_aedid_default=( \
        0x00 0xFF 0xFF 0xFF 0xFF 0xFF 0xFF 0x00 0x32 0x8D 0x01 0x04 0x90 0x01 0x03 0x6A \
        0x2F 0x16 0x01 0x03 0x80 0x59 0x32 0x78 0x0A 0x0D 0xC9 0xA0 0x57 0x47 0x98 0x27 \
        0x12 0x48 0x4C 0x21 0x08 0x00 0x81 0x80 0x01 0x01 0x01 0x01 0x01 0x01 0x01 0x01 \
        0x01 0x01 0x01 0x01 0x01 0x01 0x01 0x1D 0x00 0x72 0x51 0xD0 0x1E 0x20 0x6E 0x28 \
        0x55 0x00 0x76 0xF2 0x31 0x00 0x00 0x1E 0x02 0x3A 0x80 0xD0 0x72 0x38 0x2D 0x40 \
        0x10 0x2C 0x45 0x80 0x10 0x09 0x00 0x00 0x00 0x1E 0x00 0x00 0x00 0xFC 0x00 0x4C \
        0x4F 0x4E 0x54 0x49 0x55 0x4D 0x0A 0x20 0x20 0x20 0x20 0x20 0x00 0x00 0x00 0xFD \
        0x00 0x30 0x3E 0x0E 0x46 0x0F 0x00 0x0A 0x20 0x20 0x20 0x20 0x20 0x20 0x01 0xEA \

        0x02 0x03 0x35 0xC1 0x53 0x1F 0x90 0x14 0x05 0x13 0x04 0x20 0x22 0x00 0x00 0x12 \
        0x16 0x03 0x07 0x11 0x15 0x02 0x06 0x01 0x26 0x09 0x07 0x01 0x83 0x01 0x00 0x00 \
        0x74 0x03 0x0C 0x00 0x00 0x00 0x80 0x1E 0x2F 0xC0 0x0A 0x01 0x40 0x00 0x7F 0x20 \
        0x30 0x70 0x80 0x90 0x76 0x02 0x3A 0x80 0xD0 0x72 0x38 0x2D 0x40 0x10 0x2C 0x45 \
        0x80 0x10 0x09 0x00 0x00 0x00 0x1E 0x01 0x1D 0x00 0xBC 0x52 0xD0 0x1E 0x20 0xB8 \
        0x28 0x55 0x40 0x76 0xF2 0x31 0x00 0x00 0x1E 0x01 0x1D 0x80 0xD0 0x72 0x1C 0x16 \
        0x20 0x10 0x2C 0x25 0x80 0x76 0xF2 0x31 0x00 0x00 0x9E 0x00 0x00 0x00 0x00 0x00 \
        0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0xCA \
    )
else
    # HDMI2.0
    readonly l_aedid_default=( \
        0x00 0xFF 0xFF 0xFF 0xFF 0xFF 0xFF 0x00 0x4C 0x2D 0xB4 0x0B 0x01 0x00 0x00 0x00 \
        0x02 0x18 0x01 0x03 0x80 0x6F 0x3E 0x78 0x0A 0xEE 0x91 0xA3 0x54 0x4C 0x99 0x26 \
        0x0F 0x50 0x54 0xBD 0xEF 0x80 0x71 0x4F 0x81 0xC0 0x81 0x00 0x81 0x80 0x95 0x00 \
        0xA9 0xC0 0xB3 0x00 0x01 0x01 0x08 0xE8 0x00 0x30 0xF2 0x70 0x5A 0x80 0xB0 0x58 \
        0x8A 0x00 0x50 0x1D 0x74 0x00 0x00 0x1E 0x02 0x3A 0x80 0x18 0x71 0x38 0x2D 0x40 \
        0x58 0x2C 0x45 0x00 0x50 0x1D 0x74 0x00 0x00 0x1E 0x00 0x00 0x00 0xFD 0x00 0x18 \
        0x4B 0x0F 0x87 0x3C 0x00 0x0A 0x20 0x20 0x20 0x20 0x20 0x20 0x00 0x00 0x00 0xFC \
        0x00 0x53 0x41 0x4D 0x53 0x55 0x4E 0x47 0x0A 0x20 0x20 0x20 0x20 0x20 0x01 0x0A \

        0x02 0x03 0x49 0xF1 0x57 0x61 0x10 0x1F 0x04 0x13 0x05 0x14 0x20 0x21 0x22 0x5D \
        0x5E 0x5F 0x60 0x65 0x66 0x62 0x63 0x64 0x07 0x16 0x03 0x12 0x23 0x09 0x07 0x07 \
        0x83 0x01 0x00 0x00 0xE2 0x00 0x0F 0x6E 0x03 0x0C 0x00 0x40 0x00 0xB8 0x3C 0x21 \
        0x00 0x80 0x01 0x02 0x03 0x04 0x6E 0xd8 0x5d 0xc4 0x01 0x78 0x80 0x00 0x00 0x00 \
        0x00 0x00 0x00 0x00 0x00 0xE3 0x0F 0x01 0xE0 0x01 0x1D 0x80 0xD0 0x72 0x1C 0x16 \
        0x20 0x10 0x2C 0x25 0x80 0x50 0x1D 0x74 0x00 0x00 0x9E 0x66 0x21 0x56 0xAA 0x51 \
        0x00 0x1E 0x30 0x46 0x8F 0x33 0x00 0x50 0x1D 0x74 0x00 0x00 0x1E 0x00 0x00 0x00 \
        0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0xAA \
    )
fi


# /******************** typedef enum ***********************/

# LT86102UX_TX
readonly TX_0=0x01
readonly TX_1=0x02
readonly TX_2=0x04
readonly TX_3=0x08
readonly TX_All=0x0F

# Tx_TypeDef
readonly Tx_0=0x00,
readonly Tx_1=0x01,
readonly Tx_2=0x02,
readonly Tx_3=0x03

# EqModeType
readonly FixEq=0
readonly AutoEq=1



# /******************** lt86104ux.c ***********************/

readonly l_aEQ_level=( \
    0x06 0x0a 0x02 0x0C 0x14 0x04 0x18 0x28 \
    0x08 0x30 0x50 0x10 0x60 0x20 0x40 \
    )



#################### GLOBAL VARIABLES ####################


g_ucRxHdmi5vStatus=0
g_ucTxHpdStatus=(0 0 0 0)
g_ucOldTxPlugStatus=0
g_ucTxPlugNumber=0
g_ulPreTmdsClk=0

g_bFlagChipConfig=false
g_bFlagSystemStart=false
g_bFlagHdmi20True=false
g_bFlagRxPllLock=false
g_bFlagTxChanged=false


#################### FUNCTIONS ####################

gTotalIndents=0
_IndStr=""
IndentsIncrease() {
    gTotalIndents=$((gTotalIndents+1))
    _IndStr=""

    if (( $gTotalIndents > 0 ))
    then
        for counter_i in $(seq $gTotalIndents); do
            _IndStr+="    "
        done
    fi
}

IndentsDecrease() {
    gTotalIndents=$(( gTotalIndents-1))
    _IndStr=""

    if (( $gTotalIndents > 0 ))
    then
        for counter_i in $(seq $gTotalIndents); do
            _IndStr+="    "
        done
    fi
}

# bool bHDMI_WriteI2C_Byte(u8 RegAddr, u8 d)
bHDMI_WriteI2C_Byte() {
    local REG_ADDR=0x$(printf '%02x' ${1})
    local REG_VAL=0x$(printf '%02x' ${2})
    i2cset -y $I2C_BUS $I2C_ADDR $REG_ADDR $REG_VAL

    if [[ "$I2C_DEBUG" = true ]]; then
        echo "i2cset -y ${I2C_BUS} ${I2C_ADDR} ${REG_ADDR} ${REG_VAL}"
    fi
}

# u8 ucHDMI_ReadI2C_Byte(u8 RegAddr)
ucHDMI_ReadI2C_Byte() {
    local __RET=$2
    local REG_ADDR=0x$(printf '%02x' ${1})
    REG_VAL=$(i2cget -y $I2C_BUS $I2C_ADDR $REG_ADDR)
    eval $__RET="'$REG_VAL'"

    if [[ "$I2C_DEBUG" = true ]]; then
        echo "i2cget -y ${I2C_BUS} ${I2C_ADDR} ${REG_ADDR}"
    fi
}

# bool bHDMI_ReadI2C_ByteN(u8 RegAddr,u8 *ucp_data,u8 N)
bHDMI_ReadI2C_ByteN() {
    local -n __RET=$2
    local N=$3
    local REG_ADDR=0x$(printf '%02x' ${1})
    REG_VAL=$(i2ctransfer -y $I2C_BUS w1@$I2C_ADDR $REG_ADDR r$N)
    __RET=("${REG_VAL[@]}")

    if [[ "$I2C_DEBUG" = true ]]; then
        echo "i2ctransfer -y $I2C_BUS w1@$I2C_ADDR $REG_ADDR r$N"
    fi
}

# void vdSet_RX_HPD(HPD_Status State)
vdSet_RX_HPD() {
    local STATE=$1
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdSet_RX_HPD(${STATE})"
    fi

    if (( STATE != 0 ))
    then
        bHDMI_WriteI2C_Byte 0xe7 0x08  # use chip io control RX HPD
    else
        bHDMI_WriteI2C_Byte 0xe7 0x48
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdInit_Lt86104UX_State( void )
vdInit_Lt86104UX_State() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdInit_Lt86104UX_State"
    fi

    bHDMI_WriteI2C_Byte 0xcb 0x00  # power on RX
    bHDMI_WriteI2C_Byte 0xcd 0x00  # power on RX PI
    bHDMI_WriteI2C_Byte 0xa1 0x40  # tx rst
    bHDMI_WriteI2C_Byte 0xaa 0x40  # tx rst
    bHDMI_WriteI2C_Byte 0xb3 0x40  # tx rst
    bHDMI_WriteI2C_Byte 0xbc 0x40  # tx rst
    bHDMI_WriteI2C_Byte 0xc1 0x00  # tx tap0_en
    bHDMI_WriteI2C_Byte 0xb8 0x00  # tx tap0_en
    bHDMI_WriteI2C_Byte 0xaf 0x00  # tx tap0_en
    bHDMI_WriteI2C_Byte 0xa6 0x00  # tx tap0_en
    bHDMI_WriteI2C_Byte 0x81 0xcc  # ldo=1.3v
    bHDMI_WriteI2C_Byte 0x82 0xcc  # ldo=1.3v
    bHDMI_WriteI2C_Byte 0xe7 0x48  # Rx hpd Low,BIT2��BIT1����Ϊ0�������Ӱ��TX_ST���Ƶ�
    bHDMI_WriteI2C_Byte 0xd1 0x70  # TX0_ST driving ability adjust
    bHDMI_WriteI2C_Byte 0xd2 0x77  # TX1_ST/TX2_ST driving ability adjust
    bHDMI_WriteI2C_Byte 0xd3 0x07  # TX3_ST driving ability adjust
    bHDMI_WriteI2C_Byte 0xc7 0xdf  # termination resister adjustment.

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdTX_Drive_Config(TX_Status State )
vdTX_Drive_Config() {
    local STATE=$1
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdTX_Drive_Config(${STATE})"
    fi

    local UCREAD_DATA=0

    if (( STATE != 0 ))
    then
        if (( g_ucTxHpdStatus[0] != 0 ))
        then
            ucHDMI_ReadI2C_Byte 0xa6 UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xa6 $((UCREAD_DATA & 0x10 | 0xa0))
            echo "TX0 ON!"
        else
            ucHDMI_ReadI2C_Byte 0xa6 UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xa6 $((UCREAD_DATA & 0x10))
            echo "TX0 OFF!"
        fi

        if (( g_ucTxHpdStatus[1] != 0 ))
        then
            ucHDMI_ReadI2C_Byte 0xaf UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xaf $((UCREAD_DATA & 0x10 | 0xa0))
            echo "TX1 ON!"
        else
            ucHDMI_ReadI2C_Byte 0xaf UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xaf $((UCREAD_DATA & 0x10))
            echo "TX1 OFF!"
        fi

        if (( g_ucTxHpdStatus[2] != 0 ))
        then
            ucHDMI_ReadI2C_Byte 0xb8 UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xb8 $((UCREAD_DATA & 0x10 | 0xa0))
            echo "TX2 ON!"
        else
            ucHDMI_ReadI2C_Byte 0xb8 UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xb8 $((UCREAD_DATA & 0x10))
            echo "TX2 OFF!"
        fi

        if (( g_ucTxHpdStatus[3] != 0 ))
        then
            ucHDMI_ReadI2C_Byte 0xc1 UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xc1 $((UCREAD_DATA & 0x10 | 0xa0))
            echo "TX3 ON!"
        else
            ucHDMI_ReadI2C_Byte 0xc1 UCREAD_DATA
            bHDMI_WriteI2C_Byte 0xc1 $((UCREAD_DATA & 0x10))
            echo "TX3 OFF!"
        fi
    else
        ucHDMI_ReadI2C_Byte 0xa6 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0xa6 $((UCREAD_DATA & 0x10))
        ucHDMI_ReadI2C_Byte 0xaf UCREAD_DATA
        bHDMI_WriteI2C_Byte 0xaf $((UCREAD_DATA & 0x10))
        ucHDMI_ReadI2C_Byte 0xb8 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0xb8 $((UCREAD_DATA & 0x10))
        ucHDMI_ReadI2C_Byte 0xc1 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0xc1 $((UCREAD_DATA & 0x10))
        echo "ALL TX OFF!"
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdInit_System_State( void )
vdInit_System_State() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdInit_System_State()"
    fi

    vdSet_RX_HPD LOW
    vdInit_Lt86104UX_State

    vdTX_Drive_Config DRIVE_OFF

    #vdWriteEdidToShadow();
    #disableInterrupts();

    # clr_EA;
    echo "LT86102UX initialization OK!"

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# bool bTxHpdStatusDet( void )
bTxHpdStatusDet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}bTxHpdStatusDet()"
    fi

    local ucPreTxHpdStatus=(0 0 0 0)
    local ucNewTxPlugStatus=0
    local uctx=(0 0 0 0)
    local UCREAD_DATA=0

    for i in {0..3}; do
        g_ucTxHpdStatus[$1]=${ucPreTxHpdStatus[$1]}
    done

    for i in {0..3}; do
        ucHDMI_ReadI2C_Byte 0xf5 UCREAD_DATA
        uctx[$i]=$(( (UCREAD_DATA & (0x01 << $i)) >> $i ))

        if (( uctx[$i] != 0 ))
        then
            # check port 10 times
            for j in {1..10}; do
                sleep 0.01  # sleep 0.01 seconds
                ucHDMI_ReadI2C_Byte 0xf5 UCREAD_DATA
                uctx[$i]=$(( (UCREAD_DATA & (0x01 << $i)) >> $i ))
                if (( uctx[$i] != 0 ))
                then
                    echo "Done... UCREAD_DATA=$UCREAD_DATA"
                    break
                fi
            done
        fi

        g_ucTxHpdStatus[$i]=${uctx[$i]}
    done

    ucNewTxPlugStatus=$(( uctx[3] * 8 + uctx[2] * 4 + uctx[1] * 2 + uctx[0] ))

    if (( g_ucOldTxPlugStatus == ucNewTxPlugStatus ))
    then
        if [ "$DEBUG" = true ]; then
            IndentsDecrease
        fi
        return
    fi

    for i in {0..3}; do
        bnew_hpd=$(( ucNewTxPlugStatus & (1 << $i) ))
        bold_hpd=$(( g_ucOldTxPlugStatus & (1 << $i) ))

        if (( bnew_hpd != bold_hpd ))
        then
            # Output has been connected
            if ((bold_hpd == 0)) && ((bnew_hpd != 0))
            then
                echo "Output[$i] has been connected"

                g_ucTxPlugNumber=$((g_ucTxPlugNumber+1))
                g_bFlagTxChanged=true
                if (( g_ucTxPlugNumber == 1 ))
                then
                    echo "read edid!"
                    vdTxHpdChangeHandleEDID
                fi
            # Output has been disconnected (if (old==1 && new==0))
            else
                echo "Output[$i] has been disconnected"

                g_ucTxPlugNumber=$((g_ucTxPlugNumber-1))
                if [[ "$g_bFlagRxPllLock" = true ]]; then
                    vdTX_Drive_Config DRIVE_ON
                fi
            fi
        fi
    done

    g_ucOldTxPlugStatus=$ucNewTxPlugStatus


    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdTxHpdChangeHandleEDID( void )
vdTxHpdChangeHandleEDID() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdTxHpdChangeHandleEDID()"
    fi

    vdSet_RX_HPD LOW
    vdTX_Drive_Config DRIVE_OFF
    g_bFlagSystemStart=true

    echo "g_ucTxHpdStatus[0]=${g_ucTxHpdStatus[0]}"

    # /*ֻ��1��TX���н������ʱ��ֱ�Ӷ�EDID*/
    if (( g_ucTxPlugNumber == 1 ))
    then
        if (( g_ucTxHpdStatus[0] != 0 ))
        then
            bReadEdidFromSink 0x00
        elif (( g_ucTxHpdStatus[1] != 0 ))
        then
            bReadEdidFromSink 0x01
        elif (( g_ucTxHpdStatus[2] != 0 ))
        then
            bReadEdidFromSink 0x02
        elif (( g_ucTxHpdStatus[3] != 0 ))
        then
            bReadEdidFromSink 0x03
        fi
    fi

    vdWriteEdidToShadow

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# bool bReadEdidFromSink( Tx_TypeDef tx )
bReadEdidFromSink() {
    local tx=$1
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}bReadEdidFromSink(${tx})"
    fi

    bHDMI_WriteI2C_Byte 0x02 0xbf
    bHDMI_WriteI2C_Byte 0x02 0xff       #/*Reset DDC */

    bHDMI_WriteI2C_Byte 0x44 $tx        #/*bit[1:0] = tx_ddc_sel  ,  0=tx0 ; 1=tx1 ; 2=tx2 ; 3=tx3 */
    bHDMI_WriteI2C_Byte 0x41 0x39       #/*gurantee one cycle duration */
    bHDMI_WriteI2C_Byte 0x45 0xA0       #/*device address :  EDID_ADDR = 0xA0 ; SCDC_ADDR = 0xA8 */
    bHDMI_WriteI2C_Byte 0x48 0x20       #/*Max read 32Byte */

    local ucOffsetAddress=0
    g_aedid_buff=()
    for ucLoopx in {0..7}; do
        bHDMI_WriteI2C_Byte 0x47 $ucOffsetAddress  #/*initial address */

        bHDMI_WriteI2C_Byte 0x44 $((tx | 0x10))    #/*bit[6:4] = access command ; 1=seq_rd , 2=seq_wr , 7=nop_com */
        bHDMI_WriteI2C_Byte 0x44 $((tx | 0x70))    #/*bit[6:4] = access command ; 1=seq_rd , 2=seq_wr , 7=nop_com */
        sleep 0.1  # sleep 0.1 seconds
        bHDMI_ReadI2C_ByteN 0x62 g_aedid_buff_part 0x20

        g_aedid_buff+=(${g_aedid_buff_part[@]})
        ucOffsetAddress=$((ucOffsetAddress + 0x20))                #/*next 32 bytes */
    done

    echo "g_aedid_buff=${g_aedid_buff[@]}"  # DEBUG

    if ((g_aedid_buff[0] == 0x00)) && ((g_aedid_buff[1] == 0xFF)) && \
        ((g_aedid_buff[2] == 0xFF)) && ((g_aedid_buff[3] == 0xFF)) && \
        ((g_aedid_buff[4] == 0xFF)) && ((g_aedid_buff[5] == 0xFF)) && \
        ((g_aedid_buff[6] == 0xFF)) && ((g_aedid_buff[7] == 0x00))
    then
        if [ "$DEBUG" = true ]; then
            IndentsDecrease
        fi
        return  # SUCCESS
    else
        g_aedid_buff=("${l_aedid_default[@]}")
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi

    return  # ERROR
}

# void vdWriteEdidToShadow( void )
vdWriteEdidToShadow() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdWriteEdidToShadow()"
    fi

    bHDMI_WriteI2C_Byte 0x21 0x00       #/*Set initial Address */
    bHDMI_WriteI2C_Byte 0x20 0x00       #/*disable Source read edid */
    for uicount in {0..255}; do
        #//write EDID buffer for 256 bytes */
        bHDMI_WriteI2C_Byte 0x60 ${g_aedid_buff[$uicount]}
    done
    bHDMI_WriteI2C_Byte 0x20 0x01       #/*enbale Source read edid */


    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# bool bRxHdmi5vStatusDet( void )
bRxHdmi5vStatusDet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}bRxHdmi5vStatusDet()"
    fi

    g_ucRxHdmi5vStatus=true

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# u32 ulFreqIndicate( void )
ulFreqIndicate() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}ulFreqIndicate()"
    fi

    local __RET=$1
    local UCREAD_DATA=0

    ucHDMI_ReadI2C_Byte 0xf2 UCREAD_DATA
    ulValue=$(( UCREAD_DATA & 0x07 ))
    ulValue=$(( ulValue << 8 ))

    ucHDMI_ReadI2C_Byte 0xf3 UCREAD_DATA
    ulValue=$(( ulValue | UCREAD_DATA ))
    ulValue=$(( ulValue << 8 ))

    ucHDMI_ReadI2C_Byte 0xf4 UCREAD_DATA
    ulValue=$(( ulValue | UCREAD_DATA ))


    if [ "$DEBUG" = true ]; then
        echo "ulFreq=${ulValue}"
    fi

    eval $__RET="'$ulValue'"

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# bool bTmdsClkStableDet( void )
bTmdsClkStableDet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}bTmdsClkStableDet()"
    fi

    local __RET=$1
    local ulPreTmdsClk=0
    local ulTmdsClk=0
    local UCREAD_DATA=0
    local retValue=true


    bHDMI_WriteI2C_Byte 0xc4 0x80                   #/*Clock Dc in ��ֹ���� */
    ucHDMI_ReadI2C_Byte 0xF0 UCREAD_DATA
    local ucClkStatus=$(( UCREAD_DATA & 0x20 ))     #/*Clk stable */

    ulFreqIndicate ulPreTmdsClk
    sleep 0.001  # sleep 0.001 seconds
    ulFreqIndicate ulTmdsClk

    bHDMI_WriteI2C_Byte 0xc4 0xa0                   #/*Clock Ac in */


    if (( ulPreTmdsClk >= ulTmdsClk ))
    then
        local clkDiff=$(( ulPreTmdsClk - ulTmdsClk ))
    else
        local clkDiff=$(( ulTmdsClk - ulPreTmdsClk ))
    fi

    if (( clkDiff >= CLK_CHANGE_MARGIN ))
    then
        retValue=false
    fi

    if (( ulTmdsClk < CLK_ACTIVE_MIN_LIMIT ))
    then
        retValue=false
    fi

    # if (( ucClkStatus == 0 ))
    #     retValue=false
    # fi

    eval $__RET="'$retValue'"


    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# bool bHdmiVerChangeDet( void )
bHdmiVerChangeDet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}bHdmiVerChangeDet()"
    fi
    local __RET=$1

    local retValue=false
    local UCREAD_DATA=0


    ucHDMI_ReadI2C_Byte 0x6f UCREAD_DATA
    local UCREAD_DATA=$(( UCREAD_DATA & 0x02 ))

    if $g_bFlagHdmi20True  #/*Hdmi version change */
    then
        if ((UCREAD_DATA == 0x02 ))
        then
            local retValue=false
        else
            local retValue=true
        fi
    else
        if ((UCREAD_DATA == 0x02 ))
        then
            local retValue=true
        else
            local retValue=false
        fi
    fi


    eval $__RET="'$retValue'"

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# bool bTmdsClkChangeDet( void )
bTmdsClkChangeDet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}bTmdsClkChangeDet()"
    fi
    local __RET=$1

    local retValue=false
    local ulTmdsClk=0
    # g_ulPreTmdsClk


    ulFreqIndicate ulTmdsClk
    if (( g_ulPreTmdsClk >= ulTmdsClk ))
    then
        local clkDiff=$(( g_ulPreTmdsClk - ulTmdsClk ))
    else
        local clkDiff=$(( ulTmdsClk - g_ulPreTmdsClk ))
    fi

    if (( clkDiff >= CLK_CHANGE_MARGIN ))
    then
        local retValue=true
    fi


    g_ulPreTmdsClk=$ulTmdsClk
    eval $__RET="'$retValue'"

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdConfigRxScdcStatus( u8 RegAddr, u8 value )
vdConfigRxScdcStatus() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdConfigRxScdcStatus(${1}, ${2})"
    fi

    local _RegAddr=$1
    local _value=$2

    bHDMI_WriteI2C_Byte 0x21 $_RegAddr    # /*WR INIT ADDR */
    bHDMI_WriteI2C_Byte 0x61 $_value      # /*WR SCDC ADDR */

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdHdmiVerStatusDet( void )
vdHdmiVerStatusDet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdHdmiVerStatusDet()"
    fi

    local ucTMDS_Configuration=0

    ucHDMI_ReadI2C_Byte 0x6f ucTMDS_Configuration
    if (( (ucTMDS_Configuration & 0x01) == 0x01 ))
    then
        vdConfigRxScdcStatus 0x21 0x01
    else
        vdConfigRxScdcStatus 0x21 0x00
    fi

    sleep 0.3  # sleep 0.3 seconds
    ucHDMI_ReadI2C_Byte 0x6f ucTMDS_Configuration
    if (( (ucTMDS_Configuration & 0x02) == 0x02 ))  # /*M=1 HDMI2.0 */
    then
        g_bFlagHdmi20True=true
        echo "Source is HDMI2.0!"
    else
        g_bFlagHdmi20True=false
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdWriteScdcToSink( u8 RegAddr, u8 value )
vdWriteScdcToSink() {
    local _RegAddr=$1
    local _value=$2
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdWriteScdcToSink(${_RegAddr}, ${_value})"
    fi

    bHDMI_WriteI2C_Byte 0x43 $_value
    bHDMI_WriteI2C_Byte 0x45 0xA8
    bHDMI_WriteI2C_Byte 0x47 $_RegAddr
    bHDMI_WriteI2C_Byte 0x48 0x01
    bHDMI_WriteI2C_Byte 0x49 0x00
    bHDMI_WriteI2C_Byte 0x44 0x20
    bHDMI_WriteI2C_Byte 0x44 0x70

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdTxScdcWriteHandle( LT86102UX_TX tx )
vdTxScdcWriteHandle() {
    local tx=$1
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdTxScdcWriteHandle(${tx})"
    fi

    local UCREAD_DATA=0

    if (( tx == TX_0 ))
    then
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA & 0x1f ))   # /*disable tx1/tx2/tx3 */
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA | 0x10 ))   # /*enable tx0 */

    elif (( tx == TX_1 ))
    then
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA & 0x2f ))   # /*disable tx0/tx2/tx3 */
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA | 0x20 ))   # /*enable tx1 */

    elif (( tx == TX_2 ))
    then
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA & 0x4f ))   # /*disable tx0/tx1/tx3 */
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA | 0x40 ))   # /*enable tx2 */

    elif (( tx == TX_3 ))
    then
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA & 0x8f ))   # /*disable tx0/tx1/tx2 */
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA | 0x80 ))   # /*enable tx3 */

    elif (( tx == TX_All ))
    then
        ucHDMI_ReadI2C_Byte 0x05 UCREAD_DATA
        bHDMI_WriteI2C_Byte 0x05 $(( UCREAD_DATA | 0xF0 ))   # /*enable tx0&tx1&tx2&tx3 */
    fi

    vdWriteScdcToSink 0x02 0x01
    sleep 0.001
    vdWriteScdcToSink 0x02 0x01
    sleep 0.001
    if [ "$g_bFlagHdmi20True" = true ]; then
        vdWriteScdcToSink 0x20 0x03
        sleep 0.001
        vdWriteScdcToSink 0x20 0x03
    else
        vdWriteScdcToSink 0x20 0x00
        sleep 0.001
        vdWriteScdcToSink 0x20 0x00
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdRxTermCalibration( void )
vdRxTermCalibration() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdRxTermCalibration()"
    fi

    bHDMI_WriteI2C_Byte 0xC9 0x04       # /*power on */
    bHDMI_WriteI2C_Byte 0xC9 0x14       # /*rst */

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdTxTermCalibration( void )
vdTxTermCalibration() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdTxTermCalibration()"
    fi

    bHDMI_WriteI2C_Byte 0xa0 0x50
    bHDMI_WriteI2C_Byte 0xa9 0x50
    bHDMI_WriteI2C_Byte 0xb2 0x50
    bHDMI_WriteI2C_Byte 0xbb 0x50

    # /*enable auto PD */
    bHDMI_WriteI2C_Byte 0x9f 0x4a
    bHDMI_WriteI2C_Byte 0xa8 0x4a
    bHDMI_WriteI2C_Byte 0xb1 0x4a
    bHDMI_WriteI2C_Byte 0xba 0x4a

    # /*rst */
    bHDMI_WriteI2C_Byte 0x9f 0x4b
    bHDMI_WriteI2C_Byte 0xa8 0x4b
    bHDMI_WriteI2C_Byte 0xb1 0x4b
    bHDMI_WriteI2C_Byte 0xba 0x4b

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdRxPllStepSet( void )
vdRxPllStepSet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdRxPllStepSet()"
    fi

    local ucread_data=0
    local ucread_rterm=0
    local ulclk_freq=0
    local uiread_result=0

    if [ "$g_bFlagHdmi20True" = true ]; then
        bHDMI_WriteI2C_Byte 0x86 0x14       # /*xxPLL_CPCUR_SEL */
        bHDMI_WriteI2C_Byte 0x87 0x50       # /*xxPLL_LPF_RSET  xxPLL_LPF_CPSET */
        bHDMI_WriteI2C_Byte 0x8f 0x14       # /*xxPLL_CPCUR_SEL */
        bHDMI_WriteI2C_Byte 0x90 0x50       # /*xxPLL_LPF_RSET  xxPLL_LPF_CPSET */
        ucHDMI_ReadI2C_Byte 0xc4 ucread_data
        bHDMI_WriteI2C_Byte 0xc4 $(( ucread_data & 0xf8 ))
        ucHDMI_ReadI2C_Byte 0xcc ucread_data
        bHDMI_WriteI2C_Byte 0xcc $(( (ucread_data & 0xf8) | 0x04 ))
        ucHDMI_ReadI2C_Byte 0xc5 ucread_data
        bHDMI_WriteI2C_Byte 0xc5 $(( (ucread_data & 0x80) | 0x07 ))
        ucHDMI_ReadI2C_Byte 0xc6 ucread_data
        bHDMI_WriteI2C_Byte 0xc6 $(( ucread_data & 0xfc ))

        bHDMI_WriteI2C_Byte 0xdd 0x02
        ucHDMI_ReadI2C_Byte 0x85 ucread_data
        bHDMI_WriteI2C_Byte 0x85 $(( ucread_data | 0x20 ))  # /*RXPLL_FREQ_SEL */
        ucHDMI_ReadI2C_Byte 0x88 ucread_data
        bHDMI_WriteI2C_Byte 0x88 $(( ucread_data & 0xf3 ))  # /*rxpll_freq_set<1:0> */
        ucHDMI_ReadI2C_Byte 0x8e ucread_data
        bHDMI_WriteI2C_Byte 0x8e $(( ucread_data | 0x20 ))  # /*TX1PLL_FREQ_SEL */
        ucHDMI_ReadI2C_Byte 0x91 ucread_data
        bHDMI_WriteI2C_Byte 0x91 $(( ucread_data & 0xf3 ))  # /*tx1pll_freq_set<1:0> */
        ucHDMI_ReadI2C_Byte 0x97 ucread_data
        bHDMI_WriteI2C_Byte 0x97 $(( ucread_data | 0x20 ))  # /*TX2PLL_FREQ_SEL */
        ucHDMI_ReadI2C_Byte 0x9a ucread_data
        bHDMI_WriteI2C_Byte 0x9a $(( ucread_data & 0xf3 ))  # /*tx2pll_freq_set<1:0> */


        # /*
        #  *  ucHDMI_ReadI2C_Byte(0xce);
        #  *  bHDMI_WriteI2C_Byte(0xce,ucread_data&0xfd);//RG_RXPI_HCUR_PD
        #  */
        bHDMI_WriteI2C_Byte 0xce 0x01               # /*bit0:RG_RXPI_SEL bit1:RXPI high current power down */

        ucHDMI_ReadI2C_Byte 0xf1 ucread_rterm        # /*bfjiang */
        bHDMI_WriteI2C_Byte 0xc9 0x1c               # /*bfjiang */
        uiread_result=$(( ucread_rterm + 0x30 ))
        if (( uiread_result > 0xff ))
        then
            ucread_rterm=0xff
        else
            ucread_rterm=$(( ucread_rterm + 0x30 ))
        fi
        # //bHDMI_WriteI2C_Byte 0xc7, ucread_rterm                         # /*bfjiang */
        # /*bHDMI_WriteI2C_Byte(0xc7,0xdf);//bfjiang */

    else
        bHDMI_WriteI2C_Byte 0x86 0x18                               # /*xxPLL_CPCUR_SEL */
        bHDMI_WriteI2C_Byte 0x87 0x34                               # /*xxPLL_LPF_RSET  xxPLL_LPF_CPSET */
        bHDMI_WriteI2C_Byte 0x8f 0x11                               # /*xxPLL_CPCUR_SEL */
        bHDMI_WriteI2C_Byte 0x90 0x50                               # /*xxPLL_LPF_RSET  xxPLL_LPF_CPSET */
        bHDMI_WriteI2C_Byte 0xc5 0x7f
        bHDMI_WriteI2C_Byte 0xdd 0x0a

        ucHDMI_ReadI2C_Byte 0x85 ucread_data
        bHDMI_WriteI2C_Byte 0x85 $(( ucread_data | 0x20 ))          # /*RXPLL_FREQ_SEL */
        ucHDMI_ReadI2C_Byte 0x88 ucread_data
        ulFreqIndicate ulclk_freq
        if (( ulclk_freq > 0x2ee00 ))                                      # /* 0x2ee00 = 192000 */
        then
            bHDMI_WriteI2C_Byte 0x88 $(( ucread_data & 0xf3 ))             # /*rxpll_freq_set<1:0> */

        elif (( ulclk_freq > 0x17700 )) && (( ulclk_freq <= 0x2ee00 ))     # /* 0x17700 = 96000 */
        then
            bHDMI_WriteI2C_Byte 0x88 $(( (ucread_data & 0xf3) | 0x08 ))     # /*rxpll_freq_set<1:0> */

        elif (( ulclk_freq > 0x40d8 )) && (( ulclk_freq <= 0x17700 ))      # /* 0x40d8 = 16600 */
        then
            bHDMI_WriteI2C_Byte 0x88 $(( ucread_data | 0x0c ))             # /*rxpll_freq_set<1:0> */
        fi

        ucHDMI_ReadI2C_Byte 0x8e ucread_data
        bHDMI_WriteI2C_Byte 0x8e $(( ucread_data | 0x20 ))                # /*TX1PLL_FREQ_SEL */
        ucHDMI_ReadI2C_Byte 0x97 ucread_data
        bHDMI_WriteI2C_Byte 0x97 $(( ucread_data | 0x20 ))                # /*TX2PLL_FREQ_SEL */
        ucHDMI_ReadI2C_Byte 0x91 ucread_data
        ucHDMI_ReadI2C_Byte 0x9a ucread_data
        if (( ulclk_freq > 0x2ee00 ))                                   # /* 0x2ee00 = 192000 */
        then
            ucHDMI_ReadI2C_Byte 0x91 ucread_data
            bHDMI_WriteI2C_Byte 0x91 $(( ucread_data & 0xf3 ))            # /*tx1pll_freq_set<1:0> */
            ucHDMI_ReadI2C_Byte 0x9a ucread_data
            bHDMI_WriteI2C_Byte 0x9a $(( ucread_data & 0xf3 ))            # /*tx2pll_freq_set<1:0> */

        elif (( ulclk_freq > 0x17700 )) && (( ulclk_freq <= 0x2ee00 ))  # /* 0x17700 = 96000 */
        then
            ucHDMI_ReadI2C_Byte 0x91 ucread_data
            bHDMI_WriteI2C_Byte 0x91 $(( (ucread_data & 0xf3) | 0x08 ))    # /*tx1pll_freq_set<1:0> */
            ucHDMI_ReadI2C_Byte 0x9a ucread_data
            bHDMI_WriteI2C_Byte 0x9a $(( (ucread_data & 0xf3) | 0x08 ))     # /*tx2pll_freq_set<1:0> */

        elif (( ulclk_freq > 0x40d8 )) && (( ulclk_freq <= 0x17700 ))   # /* 0x40d8 = 16600 */
        then
            ucHDMI_ReadI2C_Byte 0x91 ucread_data
            bHDMI_WriteI2C_Byte 0x91 $(( ucread_data | 0x0c ))            # /*tx1pll_freq_set<1:0> */
            ucHDMI_ReadI2C_Byte 0x9a ucread_data
            bHDMI_WriteI2C_Byte 0x9a $(( ucread_data | 0x0c ))            # /*tx2pll_freq_set<1:0> */
        fi

        if (( ulclk_freq > 0x33450 ))                             # /* 0x33450 = 210000 */
        then
            ucHDMI_ReadI2C_Byte 0xc4 ucread_data
            bHDMI_WriteI2C_Byte 0xc4 $(( (ucread_data & 0xf8) | 0x01 ))
            ucHDMI_ReadI2C_Byte 0xcc ucread_data
            bHDMI_WriteI2C_Byte 0xcc $(( (ucread_data & 0xf8) | 0x02 ))
            ucHDMI_ReadI2C_Byte 0xc5 ucread_data
            bHDMI_WriteI2C_Byte 0xc5 $(( (ucread_data & 0x80) | 0x0b ))
            ucHDMI_ReadI2C_Byte 0xc6 ucread_data
            bHDMI_WriteI2C_Byte 0xc6 $(( (ucread_data & 0xfc) | 0x01 ))

            ucHDMI_ReadI2C_Byte 0xf1 ucread_rterm                # /*bfjiang */
            bHDMI_WriteI2C_Byte 0xc9 0x1c                       # /*bfjiang */
            # //bHDMI_WriteI2C_Byte 0xc7, ucread_rterm - 0x20          # /*bfjiang */

        elif (( ulclk_freq > 0x19E10 )) && (( ulclk_freq <= 0x33450 ))      # /* 0x19E10  = 106000 */
        then
            ucHDMI_ReadI2C_Byte 0xc4 ucread_data
            bHDMI_WriteI2C_Byte 0xc4 $(( (ucread_data & 0xf8) | 0x02 ))
            ucHDMI_ReadI2C_Byte 0xcc ucread_data
            bHDMI_WriteI2C_Byte 0xcc $(( (ucread_data & 0xf8) | 0x00 ))
            ucHDMI_ReadI2C_Byte 0xc5 ucread_data
            bHDMI_WriteI2C_Byte 0xc5 $(( (ucread_data & 0x80) | 0x09 ))
            ucHDMI_ReadI2C_Byte 0xc6 ucread_data
            bHDMI_WriteI2C_Byte 0xc6 $(( (ucread_data & 0xfc) | 0x02 ))

            ucHDMI_ReadI2C_Byte 0xf1 ucread_rterm        # /*bfjiang */
            bHDMI_WriteI2C_Byte 0xc9 0x1c               # /*bfjiang */
            # //bHDMI_WriteI2C_Byte 0xc7, ucread_rterm - 0x40  # /*bfjiang */

        else
        #   /*if( (clk_freq > 0x40d8) && (clk_freq <= 0x19E10) ){// 0x40d8 = 16600 */
            ucHDMI_ReadI2C_Byte 0xc4 ucread_data
            bHDMI_WriteI2C_Byte 0xc4 $(( (ucread_data & 0xf8) | 0x03 ))
            ucHDMI_ReadI2C_Byte 0xcc ucread_data
            bHDMI_WriteI2C_Byte 0xcc $(( (ucread_data & 0xf8) | 0x00 ))
            ucHDMI_ReadI2C_Byte 0xc5 ucread_data
            bHDMI_WriteI2C_Byte 0xc5 $(( (ucread_data & 0x80) | 0x08 ))
            ucHDMI_ReadI2C_Byte 0xc6 ucread_data
            bHDMI_WriteI2C_Byte 0xc6 $(( (ucread_data & 0xfc) | 0x03 ))


            ucHDMI_ReadI2C_Byte 0xf1 ucread_rterm        # /*bfjiang */
            bHDMI_WriteI2C_Byte 0xc9 0x1c               # /*bfjiang */
            # //bHDMI_WriteI2C_Byte 0xc7, ucread_rterm - 0x40  # /*bfjiang */
        fi


        # /*
        #  * ucHDMI_ReadI2C_Byte(0xce);
        #  * bHDMI_WriteI2C_Byte(0xce,ucread_data|0x02);//RG_RXPI_HCUR_PD
        #  */
        bHDMI_WriteI2C_Byte 0xce 0x03                       # /*bit0:RG_RXPI_SEL bit1:RXPI high current power down */
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdRxPllTestSet( void )
vdRxPllTestSet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdRxPllTestSet()"
    fi

    local ucreddata=0

    bHDMI_WriteI2C_Byte 0xc1 0x00
    bHDMI_WriteI2C_Byte 0xb8 0x00
    bHDMI_WriteI2C_Byte 0xaf 0x00
    bHDMI_WriteI2C_Byte 0xa6 0x00
    bHDMI_WriteI2C_Byte 0xcb 0x00
    bHDMI_WriteI2C_Byte 0x84 0xc1
    bHDMI_WriteI2C_Byte 0x8b 0x48
    bHDMI_WriteI2C_Byte 0x83 0x00
    bHDMI_WriteI2C_Byte 0x03 0xef
    bHDMI_WriteI2C_Byte 0x03 0xff
    ucHDMI_ReadI2C_Byte 0x88 ucreddata
    bHDMI_WriteI2C_Byte 0x88 $(( ucreddata | 0x01 ))
    bHDMI_WriteI2C_Byte 0x88 $(( ucreddata & 0xFE ))

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdRxPllCalSet( void )
vdRxPllCalSet() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdRxPllCalSet()"
    fi

    bHDMI_WriteI2C_Byte 0x37 0x1b
    bHDMI_WriteI2C_Byte 0x03 0xef
    bHDMI_WriteI2C_Byte 0x03 0xff
    bHDMI_WriteI2C_Byte 0x85 0x26
    bHDMI_WriteI2C_Byte 0x8a 0x4a
    # /* bHDMI_WriteI2C_Byte 0xcc,0x00  ); */
    bHDMI_WriteI2C_Byte 0xca 0x01
    bHDMI_WriteI2C_Byte 0xcb 0x06

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdRxPllCalibration( void )
vdRxPllCalibration() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdRxPllCalibration()"
    fi

    local ulFreqValue=0
    local ulPreFreqValue=0
    local ucPllCalibLevel_0=0x00
    local ucLevelMax=0x00
    local ucLevelMin=0x00
    local ucread_data=0


    bHDMI_WriteI2C_Byte 0x89 0x10     # /*reset first level */

    vdRxPllTestSet
    ulFreqIndicate ulPreFreqValue
    vdRxPllCalSet
    sleep 0.005
    ulFreqIndicate ulFreqValue
    # ulPreFreqValue does not need to be multiplied by 32
    ulFreqValue=$(( ulFreqValue << 5 ))       # /* *32 */

    if (( ulFreqValue == ulPreFreqValue ))
    then
        ucHDMI_ReadI2C_Byte 0x89 ucread_data
        ucPllCalibLevel_0=$(( ucread_data & 0xf0 ))

    else
        if (( ulFreqValue < ulPreFreqValue ))
        then
            while (( ulFreqValue < ulPreFreqValue ))
            do
                ucHDMI_ReadI2C_Byte 0x89 ucread_data
                ucPllCalibLevel_0=$(( ucread_data & 0xf0 ))
                ucPllCalibLevel_0=$(( ucPllCalibLevel_0 + 0x10 ))
                if (( ucPllCalibLevel_0 == 0x00 ))
                then
                    ucPllCalibLevel_0=0xff
                    break
                fi
                bHDMI_WriteI2C_Byte 0x89 $ucPllCalibLevel_0
                sleep 0.005
                ulFreqIndicate ulFreqValue
                ulFreqValue=$(( ulFreqValue << 5 ))    # /* *32 */
            done
            ucLevelMax=ucPllCalibLevel_0;
            ucLevelMin=$(( ucPllCalibLevel_0 - 0x10 ))

        elif (( ulFreqValue > ulPreFreqValue ))
        then
            while (( ulFreqValue > ulPreFreqValue ))
            do
                ucHDMI_ReadI2C_Byte 0x89 ucread_data
                ucPllCalibLevel_0=$(( ucread_data & 0xf0 ))
                ucPllCalibLevel_0=$(( ucPllCalibLevel_0 - 0x10 ))
                bHDMI_WriteI2C_Byte 0x89 $ucPllCalibLevel_0
                if (( ucPllCalibLevel_0 == 0xf0 ))
                then
                    ucPllCalibLevel_0=0x00
                    break
                fi
                sleep 0.005
                ulFreqIndicate ulFreqValue
                ulFreqValue=$(( ulFreqValue << 5 ))    # /* *32 */
            done
            ucLevelMin=ucPllCalibLevel_0;
            ucLevelMax=$(( ucPllCalibLevel_0 + 0x10 ))
        fi

        while true
        do
            ucPllCalibLevel_0=$(( (ucLevelMax + ucLevelMin) / 2 ))
            bHDMI_WriteI2C_Byte 0x89 $ucPllCalibLevel_0
            sleep 0.01
            ulFreqIndicate ulFreqValue
            ulFreqValue=$(( ulFreqValue << 5 ))    # /* *32 */

            if (( ulFreqValue < ulPreFreqValue ))
            then
                ucLevelMin=$ucPllCalibLevel_0
            else
                ucLevelMax=$ucPllCalibLevel_0
            fi

            if (( (ucLevelMax - ucLevelMin) <= 0x01 )) || (( ulFreqValue == ulPreFreqValue ))
            then
                break
            fi
        done
    fi

    bHDMI_WriteI2C_Byte 0x89 $ucPllCalibLevel_0
    bHDMI_WriteI2C_Byte 0x92 $ucPllCalibLevel_0
    bHDMI_WriteI2C_Byte 0x9b $ucPllCalibLevel_0

    bHDMI_WriteI2C_Byte 0x37 0x0b
    # /* bHDMI_WriteI2C_Byte(0xcc,0x07); */
    ucHDMI_ReadI2C_Byte 0xcb ucread_data
    bHDMI_WriteI2C_Byte 0xcb $(( ucread_data & 0xfb ))   # /*this must be put before read 0x5c,or 0x5c will be error */
    sleep 0.01
    # /* g_bFlagRxPllLock = ucHDMI_ReadI2C_Byte(0x5c)&0x01; */

    ucHDMI_ReadI2C_Byte 0x5c ucread_data
    if (( ucread_data & 0x01 ))
    then
        g_bFlagRxPllLock=true
        bHDMI_WriteI2C_Byte 0x8a 0x08
    else
        g_bFlagRxPllLock=false
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdTxPllCalibration( void )
vdTxPllCalibration() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdTxPllCalibration()"
    fi

    local ucread_data=0

    # /*Tx Pll 1 */
    ucHDMI_ReadI2C_Byte 0x8e ucread_data
    bHDMI_WriteI2C_Byte 0x8e $(( ucread_data | 0x24 ))
    ucHDMI_ReadI2C_Byte 0x30 ucread_data
    bHDMI_WriteI2C_Byte 0x33 $ucread_data
    bHDMI_WriteI2C_Byte 0x8C 0xc0        # /*disable calibratin */
    bHDMI_WriteI2C_Byte 0x8D 0x41        # /*RXPLL_VREF_EN */
    bHDMI_WriteI2C_Byte 0x94 0x08        # /*RXPLL_VREG25_EN */
    bHDMI_WriteI2C_Byte 0x8D 0xc1        # /*RXPLL_VREG_EN */
    bHDMI_WriteI2C_Byte 0x8C 0x40        # /*RXPLL_BIAS normal */
    bHDMI_WriteI2C_Byte 0x8C 0x00        # /*RXPLL_PD  normal */
    bHDMI_WriteI2C_Byte 0x04 0xef
    bHDMI_WriteI2C_Byte 0x04 0xff        # /*RXPLL reset */
    # /*
    #  * bHDMI_WriteI2C_Byte(0x8C,0x10);// calibratin Enable
    #  * Tx Pll 2
    #  */
    ucHDMI_ReadI2C_Byte 0x97 ucread_data
    bHDMI_WriteI2C_Byte 0x97 $(( ucread_data | 0x24 ))
    ucHDMI_ReadI2C_Byte 0x30 ucread_data
    bHDMI_WriteI2C_Byte 0x35 $ucread_data
    bHDMI_WriteI2C_Byte 0x95 0xc0        # /*disable calibratin */
    bHDMI_WriteI2C_Byte 0x96 0x41        # /*RXPLL_VREF_EN */
    bHDMI_WriteI2C_Byte 0x9d 0x08        # /*RXPLL_VREG25_EN */
    bHDMI_WriteI2C_Byte 0x96 0xc1        # /*RXPLL_VREG_EN */
    bHDMI_WriteI2C_Byte 0x95 0x40        # /*RXPLL_BIAS normal */
    bHDMI_WriteI2C_Byte 0x95 0x00        # /*RXPLL_PD  normal */
    bHDMI_WriteI2C_Byte 0x04 0xdf
    bHDMI_WriteI2C_Byte 0x04 0xff        # /*RXPLL reset */
    # /*bHDMI_WriteI2C_Byte(0x95,0x10);// calibratin Enable */

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdRxRst( void )
vdRxRst() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdRxRst()"
    fi

    bHDMI_WriteI2C_Byte 0x02 0xfe
    bHDMI_WriteI2C_Byte 0x02 0xff

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdPiRst( void )
vdPiRst() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdPiRst()"
    fi

    bHDMI_WriteI2C_Byte 0x04 0xf0
    bHDMI_WriteI2C_Byte 0x04 0xff

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}

# void vdEqConfig( EqModeType eqmode )
vdEqConfig() {
    local eqmode=$1
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdEqConfig(${eqmode})"
    fi


    local uieq_errcnt_d0=0x0000
    local uieq_errcnt_d1=0x0000
    local uieq_errcnt_d2=0x0000
    local ucoptimal_eqcnt_d0=0x00
    local ucoptimal_eqcnt_d1=0x00
    local ucoptimal_eqcnt_d2=0x00
    local ucread_var=0x00
    local ucloop_num=0
    # /*u8  index_min_error; */
    local ucindex_min_error_d0
    local ucindex_min_error_d1
    local ucindex_min_error_d2
    local ucindex_start_d0
    # /*u8  num; */
    local a_Save_EQ_ERR_D0=$(for i in {0..15}; do echo 0; done)
    local a_Save_EQ_ERR_D1=$(for i in {0..15}; do echo 0; done)
    local a_Save_EQ_ERR_D2=$(for i in {0..15}; do echo 0; done)
    local a_EQ_index_D0=$(for i in {0..15}; do echo 0; done)
    local a_EQ_index_D1=$(for i in {0..15}; do echo 0; done)
    local a_EQ_index_D2=$(for i in {0..15}; do echo 0; done)

    vdRxRst
    vdPiRst
    if (( eqmode == AutoEq ))
    then
        ucHDMI_ReadI2C_Byte 0x23 ucread_var
        bHDMI_WriteI2C_Byte 0x23 $(( ucread_var | 0x02 ))                              # /*enable err detect register control. */

        # /*first part. */
        ucloop_num=0
        while true
        do
            bHDMI_WriteI2C_Byte 0xd5 ${l_aEQ_level[$ucloop_num]}                       # /*set eq register */

            sleep 0.001
            ucHDMI_ReadI2C_Byte 0x03 ucread_var                                        # /*//reset error count */
            ucread_var=$(( ucread_var & 0xdf ))
            bHDMI_WriteI2C_Byte 0x03 $ucread_var
            ucread_var=$(( ucread_var | 0x20 ))
            bHDMI_WriteI2C_Byte 0x03 $ucread_var
            sleep 0.001

            bHDMI_WriteI2C_Byte 0x5e $ucread_var
            if (( ucread_var & 0x07 != 0 ))                                            # /*(ucHDMI_ReadI2C_Byte(0x5c)&0x0e) == 0x0e */
            # /*
            #  * if any one of ERR_DET_OFF0/1/2 is 0,it means that channel
            #  * \has no valid data decoded. And move to next EQ loop.
            #  */
            then
                ucHDMI_ReadI2C_Byte 0x67 uieq_errcnt_d0
                uieq_errcnt_d0=$(( uieq_errcnt_d0 << 8 ))
                bHDMI_WriteI2C_Byte 0x66 $ucread_var
                uieq_errcnt_d0=$(( uieq_errcnt_d0 + $ucread_var ))                      # /*read D0 error count */
            else
                uieq_errcnt_d0=0xffff
            fi

            a_Save_EQ_ERR_D0[$ucloop_num]=$uieq_errcnt_d0

            (( ucloop_num++ ))

            if (( ucloop_num > 14 ))
            then
                break  # /*EQ scan is over. */
            fi
        done

        # /*find eq index which has minimum error count value. */
        ucindex_min_error_d0=0
        for (( ucloop_num = 1; ucloop_num < 15; ucloop_num++ ))
        do
            if (( a_Save_EQ_ERR_D0[$ucloop_num] <= a_Save_EQ_ERR_D0[$ucindex_min_error_d0] ))
            then
                ucindex_min_error_d0=$ucloop_num
            fi
        done

        # /*find EQ index that has same error count. */
        for (( ucloop_num = 0; ucloop_num < 15; ucloop_num++ ))
        do
            if (( a_Save_EQ_ERR_D0[$ucindex_min_error_d0] == a_Save_EQ_ERR_D0[$ucloop_num] ))
            then
                a_EQ_index_D0[$ucoptimal_eqcnt_d0]=$ucloop_num
                (( ucoptimal_eqcnt_d0++ ))
            fi
        done

        # /*calculate the optimal EQ index. */
        if (( ucoptimal_eqcnt_d0 % 2 == 0 ))
        then
            local ucoptimal_eqcnt_d0_tmp=$(( ucoptimal_eqcnt_d0 / 2 - 1 ))
            ucindex_min_error_d0=${a_EQ_index_D0[$ucoptimal_eqcnt_d0_tmp]}
        else
            local ucoptimal_eqcnt_d0_tmp=$(( (ucoptimal_eqcnt_d0 + 1) / 2 - 1 ))
            ucindex_min_error_d0=${a_EQ_index_D0[$ucoptimal_eqcnt_d0_tmp]}
        fi


        # /*second part. */
        if (( ucindex_min_error_d0 > 1 )) && (( ucindex_min_error_d0 < 14 ))
        then
            ucindex_start_d0=$(( ucindex_min_error_d0 - 2 ))

        elif (( ucindex_min_error_d0 > 1 ))
        then
            ucindex_start_d0=11

        else
            ucindex_start_d0=0

        fi

        for (( ucloop_num=ucindex_start_d0; ucloop_num < (ucindex_start_d0 + 5); ucloop_num++ ))
        do
            bHDMI_WriteI2C_Byte 0xd5 ${l_aEQ_level[$ucloop_num]}                      # /*set eq register */
            bHDMI_WriteI2C_Byte 0xd6 ${l_aEQ_level[$ucloop_num]}                      # /*set eq register */
            bHDMI_WriteI2C_Byte 0xd7 ${l_aEQ_level[$ucloop_num]}                      # /*set eq register */

            sleep 0.001
            ucHDMI_ReadI2C_Byte 0x03 ucread_var                                       # /*//reset error count */
            bHDMI_WriteI2C_Byte 0x03 $(( ucread_var & 0xdf ))

            ucread_var=$(( ucread_var | 0x20 ))
            bHDMI_WriteI2C_Byte 0x03 $ucread_var
            sleep 0.01

            # /*if any one of ERR_DET_OFF0/1/2 is 0,it means that channel */
            # /*\has no valid data decoded. And move to next EQ loop. */
            ucHDMI_ReadI2C_Byte 0x5e ucread_var
            if (( (ucread_var & 0x07) == 0x07 ))
            then
                ucHDMI_ReadI2C_Byte 0x67 ucread_var
                uieq_errcnt_d0=$(( ucread_var << 8 ))
                ucHDMI_ReadI2C_Byte 0x66 ucread_var
                uieq_errcnt_d0=$(( uieq_errcnt_d0 + ucread_var ))     # /*read D0 error count */

                ucHDMI_ReadI2C_Byte 0x69 ucread_var
                uieq_errcnt_d1=$(( ucread_var << 8 ))
                ucHDMI_ReadI2C_Byte 0x68 ucread_var
                uieq_errcnt_d1=$(( uieq_errcnt_d1 + ucread_var ))     # /*read D1 error count */

                ucHDMI_ReadI2C_Byte 0x6B ucread_var
                uieq_errcnt_d2=$(( ucread_var << 8 ))
                ucHDMI_ReadI2C_Byte 0x6A ucread_var
                uieq_errcnt_d2=$(( uieq_errcnt_d2 + ucread_var ))     # /*read D2 error count */
            else
                uieq_errcnt_d0=0xffff
                uieq_errcnt_d1=0xffff
                uieq_errcnt_d2=0xffff
            fi

            a_Save_EQ_ERR_D0[$ucloop_num]=$uieq_errcnt_d0
            a_Save_EQ_ERR_D1[$ucloop_num]=$uieq_errcnt_d1
            a_Save_EQ_ERR_D2[$ucloop_num]=$uieq_errcnt_d2
        done

        # /*find eq index which has minimum error count value. */
        ucindex_min_error_d0=$ucindex_start_d0
        ucindex_min_error_d1=$ucindex_start_d0
        ucindex_min_error_d2=$ucindex_start_d0
        for (( ucloop_num = ucindex_start_d0 + 1; ucloop_num < (ucindex_start_d0 + 5); ucloop_num++ ))
        do
            if (( a_Save_EQ_ERR_D0[ucloop_num] <= a_Save_EQ_ERR_D0[ucindex_min_error_d0] ))
            then
                ucindex_min_error_d0=$ucloop_num
            fi

            if (( a_Save_EQ_ERR_D1[ucloop_num] <= a_Save_EQ_ERR_D1[ucindex_min_error_d1] ))
            then
                ucindex_min_error_d=$ucloop_num
            fi

            if (( a_Save_EQ_ERR_D2[ucloop_num] <= a_Save_EQ_ERR_D2[ucindex_min_error_d2] ))
            then
                ucindex_min_error_d2=$ucloop_num
            fi
        done

        # /*find EQ index that has same error count. */
        ucoptimal_eqcnt_d0=0
        ucoptimal_eqcnt_d1=0
        ucoptimal_eqcnt_d2=0
        for (( ucloop_num = ucindex_start_d0; ucloop_num < (ucindex_start_d0 + 5); ucloop_num++ ))
        do
            if (( a_Save_EQ_ERR_D0[ucindex_min_error_d0] == a_Save_EQ_ERR_D0[ucloop_num] ))
            then
                a_EQ_index_D0[$ucoptimal_eqcnt_d0]=$ucloop_num
                (( ucoptimal_eqcnt_d0++ ))
            fi

            if (( a_Save_EQ_ERR_D1[ucindex_min_error_d1] == a_Save_EQ_ERR_D1[ucloop_num] ))
            then
                a_EQ_index_D1[$ucoptimal_eqcnt_d1]=$ucloop_num
                (( ucoptimal_eqcnt_d1++ ))
            fi

            if (( a_Save_EQ_ERR_D2[ucindex_min_error_d2] == a_Save_EQ_ERR_D2[ucloop_num] ))
            then
                a_EQ_index_D2[$ucoptimal_eqcnt_d2]=$ucloop_num
                (( ucoptimal_eqcnt_d2++ ))
            fi
        done

        # /*calculate the optimal EQ index. */
        if (( ucoptimal_eqcnt_d0 % 2 == 0 ))
        then
            local ucoptimal_eqcnt_d0_tmp=$(( ucoptimal_eqcnt_d0 / 2 - 1 ))
            ucindex_min_error_d0=${a_EQ_index_D0[$ucoptimal_eqcnt_d0_tmp]}
        else
            local ucoptimal_eqcnt_d0_tmp=$(( (ucoptimal_eqcnt_d0 + 1) / 2 - 1 ))
            ucindex_min_error_d0=${a_EQ_index_D0[$ucoptimal_eqcnt_d0_tmp]}
        fi

        if (( ucoptimal_eqcnt_d1 % 2 == 0 ))
        then
            local ucoptimal_eqcnt_d1_tmp=$(( ucoptimal_eqcnt_d1 / 2 - 1 ))
            ucindex_min_error_d1=${a_EQ_index_D1[$ucoptimal_eqcnt_d1_tmp]}
        else
            local ucoptimal_eqcnt_d1_tmp=$(( (ucoptimal_eqcnt_d1 + 1) / 2 - 1 ))
            ucindex_min_error_d1=${a_EQ_index_D1[$ucoptimal_eqcnt_d1_tmp]}
        fi

        if (( ucoptimal_eqcnt_d2 % 2 == 0 ))
        then
            local ucoptimal_eqcnt_d2_tmp=$(( ucoptimal_eqcnt_d2 / 2 - 1 ))
            ucindex_min_error_d2=${a_EQ_index_D2[$ucoptimal_eqcnt_d2_tmp]}
        else
            local ucoptimal_eqcnt_d2_tmp=$(( (ucoptimal_eqcnt_d2 + 1) / 2 - 1 ))
            ucindex_min_error_d2=${a_EQ_index_D2[$ucoptimal_eqcnt_d2_tmp]}
        fi

        bHDMI_WriteI2C_Byte 0xd5 ${l_aEQ_level[$ucindex_min_error_d0]}       # /*set eq register */
        bHDMI_WriteI2C_Byte 0xd6 ${l_aEQ_level[$ucindex_min_error_d1]}       # /*set eq register */
        bHDMI_WriteI2C_Byte 0xd7 ${l_aEQ_level[$ucindex_min_error_d2]}       # /*set eq register */

    else
        bHDMI_WriteI2C_Byte 0xd5 ${l_aEQ_level[0x04]}                     # /*set eq register */
        bHDMI_WriteI2C_Byte 0xd6 ${l_aEQ_level[0x04]}                     # /*set eq register */
        bHDMI_WriteI2C_Byte 0xd7 ${l_aEQ_level[0x04]}                     # /*set eq register */
    fi

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}


# void vdTX_Parameter_Set(void)
vdTX_Parameter_Set() {
    if [ "$DEBUG" = true ]; then
        IndentsIncrease
        echo "${_IndStr}vdTX_Parameter_Set()"
    fi

    local ulfreq_val
    local ucread_data
    local ucTX_SWING
    local ucTX_TAP_SWING
    local ucTX_PREEMPHASIS

    if [ "$g_bFlagHdmi20True" = true ]
    then
        ucTX_SWING=$TX_SWING_600MHZ
        ucTX_TAP_SWING=$TX_TAP_SWING_600MHZ
        ucTX_PREEMPHASIS=$TX_PREEMPHASIS
    else
        ulFreqIndicate ulfreq_val
        if (( ulfreq_val < FREQ_DIV_200MHZ ))
        then
            ucTX_SWING=$TX_SWING_200MHZ
            ucTX_TAP_SWING=$TX_TAP_SWING_200MHZ
            ucTX_PREEMPHASIS=$TX_NOPREEMPHASIS
        else
            ucTX_SWING=$TX_SWING_300MHZ
            ucTX_TAP_SWING=$TX_TAP_SWING_300MHZ
            ucTX_PREEMPHASIS=$TX_PREEMPHASIS
        fi
    fi

    bHDMI_WriteI2C_Byte 0xe4 0x60
    # //TX0
    ucHDMI_ReadI2C_Byte 0xa2 ucread_data
    bHDMI_WriteI2C_Byte 0xa2 $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xa4 ucread_data
    bHDMI_WriteI2C_Byte 0xa4 $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xa5 ucread_data
    bHDMI_WriteI2C_Byte 0xa5 $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    # //ucread_data = ucHDMI_ReadI2C_Byte 0xa6
    bHDMI_WriteI2C_Byte 0xa6 $ucTX_PREEMPHASIS
    # //TX1
    ucHDMI_ReadI2C_Byte 0xab ucread_data
    bHDMI_WriteI2C_Byte 0xab $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xad ucread_data
    bHDMI_WriteI2C_Byte 0xad $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xae ucread_data
    bHDMI_WriteI2C_Byte 0xae $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    # //ucread_data = ucHDMI_ReadI2C_Byte 0xaf
    bHDMI_WriteI2C_Byte 0xaf $ucTX_PREEMPHASIS
    # //TX2
    ucHDMI_ReadI2C_Byte 0xb4 ucread_data
    bHDMI_WriteI2C_Byte 0xb4 $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xb6 ucread_data
    bHDMI_WriteI2C_Byte 0xb6 $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xb7 ucread_data
    bHDMI_WriteI2C_Byte 0xb7 $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    # //ucread_data = ucHDMI_ReadI2C_Byte 0xb8
    bHDMI_WriteI2C_Byte 0xb8 $ucTX_PREEMPHASIS
    # //TX3
    ucHDMI_ReadI2C_Byte 0xbd ucread_data
    bHDMI_WriteI2C_Byte 0xbd $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xbf ucread_data
    bHDMI_WriteI2C_Byte 0xbf $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    ucHDMI_ReadI2C_Byte 0xc0 ucread_data
    bHDMI_WriteI2C_Byte 0xc0 $(( (ucread_data & 0xc0) | ucTX_TAP_SWING ))
    # //ucread_data = ucHDMI_ReadI2C_Byte 0xc1
    bHDMI_WriteI2C_Byte 0xc1 $ucTX_PREEMPHASIS

    bHDMI_WriteI2C_Byte 0x04 0x7f       # /*close for data syc */
    bHDMI_WriteI2C_Byte 0x04 0xff
    sleep 0.02
    bHDMI_WriteI2C_Byte 0x04 0x7f       # /*close for data syc */

    if [ "$DEBUG" = true ]; then
        IndentsDecrease
    fi
}



#################### MAIN ####################

# vdInit_MCU_GPIO
vdInit_System_State

loopCounter=0
#vdTX_Drive_Config # REMOVE
while true
do
    if [ "$DEBUG" = true ]; then
        echo "----- Begin loop ${loopCounter} -----"
    fi


    # /*Error_Counter(); */
    bRxHdmi5vStatusDet  # <- fixed to true
    bTxHpdStatusDet

    # /***************************************************/
    if [ "$g_ucRxHdmi5vStatus" = true ] && (( g_ucTxPlugNumber != 0 ))
    then
        chipIsOff=false

        if $g_bFlagSystemStart
        then
            g_bFlagSystemStart=false
            sleep 0.1  # sleep 0.1 seconds

            if [ "$DEBUG" = true ]
            then
                echo "Init HDMI Switch"
                echo "----- BEGIN -----"
            fi
            if [ "$DEBUG" = true ]
            then
                echo "-----  END  -----"
            fi

            sleep 0.1  # sleep 0.05 seconds  # DEBUG
            vdSet_RX_HPD HIGH
            g_bFlagChipConfig=false
        fi

        sleep 0.05  # sleep 0.05 seconds
        bTmdsClkStableDet ret_ClkStable
        if [ "$ret_ClkStable" == true ]
        then
            # /*******HDMI�汾�仯����HDMIʱ�ӱ仯********/
            bHdmiVerChangeDet ret_HdmiVerChange
            bTmdsClkChangeDet ret_TmdsClkChangeDet
            if $ret_HdmiVerChange || $ret_TmdsClkChangeDet
            then
                g_bFlagChipConfig=false
            fi
            # /*******************************************/

            # /*******��ʼ����LT86102UX********/
            if [ "$g_bFlagChipConfig" = false ]
            then
                if [ "$DEBUG" = true ]
                then
                    echo "Configure chip..."
                fi

                bHDMI_WriteI2C_Byte 0xe7 0x08
                vdHdmiVerStatusDet              # /*�жϵ�ǰHDMI�ź���2.0��ʽ����1.4��ʽ�� */
                vdTxScdcWriteHandle TX_All      # /*дSCDC */
                vdRxTermCalibration             # /*Rx TermУ׼ */
                vdTxTermCalibration             # /*Tx TermУ׼ */
                vdRxPllStepSet                  # /*����HDMI�ź�����PLL */
                vdRxPllCalibration              # /*Rx PllУ׼ */
                vdTxPllCalibration              # /*Tx PllУ׼ */
                vdEqConfig AutoEq               # /*ѡ���Զ�EQ���ֶ�EQ��дFixEq */
                g_bFlagChipConfig=true

                # /*******ʹ��TX�����TX�仯*******/
                if [ "$g_bFlagRxPllLock" = true ]
                then
                    # /*vdTxHpdChangeHandleEDID(); */
                    vdTX_Parameter_Set
                    vdTX_Drive_Config DRIVE_ON
                    # /*vdDelay_ms( 500 ); */
                else
                    g_bFlagChipConfig=false
                fi
            fi
        else
            sleep 0.1  # sleep 0.1 seconds
            bTmdsClkStableDet ret_ClkStable
            if [ "$ret_ClkStable" == false ]
            then
                g_bFlagChipConfig=false
            fi
        fi
    else
        if [ "$chipIsOff" == false ]
        then
            vdSet_RX_HPD LOW
            vdTX_Drive_Config DRIVE_OFF

            g_bFlagChipConfig=false
            g_bFlagSystemStart=false
            g_bFlagHdmi20True=false
            g_bFlagRxPllLock=false

            chipIsOff=true
        fi
    fi

    if $g_bFlagTxChanged
    then
        g_bFlagTxChanged=false
        vdTxScdcWriteHandle TX_All
        vdTX_Drive_Config DRIVE_ON
    else
        echo "Completed"
        exit 0
    fi

    (( loopCounter++ ))
    if [ "$DEBUG" = true ]; then
        sleep 3
    else
        sleep 0.5
    fi
done
