// Copyright 2024-2025 RnD Center "ELVEES", JSC

/*! \file
 *  \brief Заголовочный файл c функциями запусков вычислений
 *  \author Фролов Андрей
 */

#ifndef CALCULATION_RUNNERS_H
#define CALCULATION_RUNNERS_H

#include "elcore50-dsplib/tile_segmentation.hpp"

extern "C" {
#include "elcore50-dsplib/asm_func.h"
}

/*!
 *  \fn void calculation_neg16_tile(const TileSegConfig* config, int buf_num,
 *                                  int tile_num)
 *  \brief Отрицание элементов вектора типа int16 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_neg16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_neg32_tile(const TileSegConfig* config, int buf_num,
 *                                  int tile_num)
 *  \brief Отрицание элементов вектора типа int32 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_neg32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_neg_fl_tile(const TileSegConfig* config, int buf_num,
 *                                   int tile_num)
 *  \brief Отрицание элементов вектора типа float в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_neg_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_neg_db_tile(const TileSegConfig* config, int buf_num,
 *                                   int tile_num)
 *  \brief Отрицание элементов вектора типа double в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_neg_db_tile(const TileSegConfig* config, int buf_num, int tile_num);

/*!
 *  \fn void RunCalculationNeg16(const TileSegConfig* config)
 *  \brief Запуск вычислений Neg16
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationNeg16(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationNeg32(const TileSegConfig* config)
 *  \brief Запуск вычислений Neg32
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationNeg32(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationNegFl(const TileSegConfig* config)
 *  \brief Запуск вычислений NegFl
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationNegFl(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationNegDb(const TileSegConfig* config)
 *  \brief Запуск вычислений NegDb
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationNegDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_add16_tile(const TileSegConfig* config, int buf_num,
 *                                  int tile_num)
 *  \brief Сумма элементов вектора типа int16 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_add16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_add32_tile(const TileSegConfig* config, int buf_num,
 *                                  int tile_num)
 *  \brief Сумма элементов вектора типа int32 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_add32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_add_fl_tile(const TileSegConfig* config, int buf_num,
 *                                   int tile_num)
 *  \brief Сумма элементов вектора типа float в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_add_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_add_db_tile(const TileSegConfig* config, int buf_num,
 *                                   int tile_num)
 *  \brief Сумма элементов вектора типа double в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_add_db_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void RunCalculationAdd16(const TileSegConfig* config)
 *  \brief Запуск вычислений Add16
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationAdd16(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationAdd32(const TileSegConfig* config)
 *  \brief Запуск вычислений Add32
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationAdd32(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationAddFl(const TileSegConfig* config)
 *  \brief Запуск вычислений AddFl
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationAddFl(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationAddDb(const config* config)
 *  \brief Запуск вычислений AddDb
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationAddDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_maxval16_tile(const TileSegConfig* config,
 *                                     int buf_num, int tile_num)
 *  \brief Поиск значения максимального элемента в векторе типа int16 в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_maxval16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_maxval32_tile(const TileSegConfig* config,
 *                                     int buf_num, int tile_num)
 *  \brief Поиск значения максимального элемента в векторе типа int32 в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_maxval32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_maxval_fl_tile(const TileSegConfig* config,
 *                                      int buf_num, int tile_num)
 *  \brief Поиск значения максимального элемента в векторе типа float в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_maxval_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_maxval_db_tile(const TileSegConfig* config,
 *                                      int buf_num, int tile_num)
 *  \brief Поиск значения максимального элемента в векторе типа double в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_maxval_db_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn int16_t RunCalculationMaxVal16(const TileSegConfig* config)
 *  \brief Запуск вычислений MaxVal16
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int16 значение максимума
 */
int16_t RunCalculationMaxVal16(const TileSegConfig* config);
/*!
 *  \fn int32_t RunCalculationMaxVal32(const TileSegConfig* config)
 *  \brief Запуск вычислений MaxVal32
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int32 значение максимума
 */
int32_t RunCalculationMaxVal32(const TileSegConfig* config);
/*!
 *  \fn float RunCalculationMaxValFl(const TileSegConfig* config)
 *  \brief Запуск вычислений MaxValFl
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b float значение максимума
 */
float RunCalculationMaxValFl(const TileSegConfig* config);
/*!
 *  \fn double RunCalculationMaxValDb(const TileSegConfig* config)
 *  \brief Запуск вычислений MaxValDb
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b double значение максимума
 */
double RunCalculationMaxValDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_minval16_tile(const TileSegConfig* config,
 *                                     int buf_num, int tile_num)
 *  \brief Поиск значения минимального элемента в векторе типа int16 в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_minval16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_minval32_tile(const TileSegConfig* config,
 *                                     int buf_num, int tile_num)
 *  \brief Поиск значения минимального элемента в векторе типа int32 в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_minval32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_minval_fl_tile(const TileSegConfig* config,
 *                                     int buf_num, int tile_num)
 *  \brief Поиск значения минимального элемента в векторе типа float в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_minval_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_minval_db_tile(const TileSegConfig* config,
 *                                     int buf_num, int tile_num)
 *  \brief Поиск значения минимального элемента в векторе типа double в рамках
 *  одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_minval_db_tile(const TileSegConfig* config, int buf_num, int tile_num);

/*!
 *  \fn int16_t RunCalculationMinVal16(const TileSegConfig* config)
 *  \brief Запуск вычислений MinVal16
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int16 значение минимума
 */
int16_t RunCalculationMinVal16(const TileSegConfig* config);
/*!
 *  \fn int32_t RunCalculationMinVal32(const TileSegConfig* config)
 *  \brief Запуск вычислений MinVal32
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int32 значение минимума
 */
int32_t RunCalculationMinVal32(const TileSegConfig* config);
/*!
 *  \fn float RunCalculationMinValFl(const TileSegConfig* config)
 *  \brief Запуск вычислений MinValFl
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b float значение минимума
 */
float RunCalculationMinValFl(const TileSegConfig* config);
/*!
 *  \fn double RunCalculationMinValDb(const TileSegConfig* config)
 *  \brief Запуск вычислений MinValDb
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b double значение минимума
 */
double RunCalculationMinValDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_dotprod16_tile(const TileSegConfig* config,
 *                                      int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа int16 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_dotprod32_tile(const TileSegConfig* config,
 *                                      int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа int32 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_dotprod_fl_tile(const TileSegConfig* config,
 *                                       int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа float в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_dotprod_db_tile(const TileSegConfig* config,
 *                                       int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа double в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod_db_tile(const TileSegConfig* config, int buf_num, int tile_num);

/*!
 *  \fn int32_t RunCalculationDotprod16(const TileSegConfig* config)
 *  \brief Запуск вычислений Dotprod16
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int32 результат скалярного произведения
 */
int32_t RunCalculationDotprod16(const TileSegConfig* config);
/*!
 *  \fn int64_t RunCalculationDotprod32(const TileSegConfig* config)
 *  \brief Запуск вычислений Dotprod32
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int64_t результат скалярного произведения
 */
int64_t RunCalculationDotprod32(const TileSegConfig* config);
/*!
 *  \fn float RunCalculationDotprodFl(const TileSegConfig* config)
 *  \brief Запуск вычислений DotprodFl
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b float результат скалярного произведения
 */
float RunCalculationDotprodFl(const TileSegConfig* config);
/*!
 *  \fn double RunCalculationDotprodDb(const TileSegConfig* config)
 *  \brief Запуск вычислений DotprodDb
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b double результат скалярного произведения
 */
double RunCalculationDotprodDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_vecsumsq16_tile(const TileSegConfig* config,
 *                                       int buf_num, int tile_num)
 *  \brief Корень из суммы элементов вектора типа int16 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_vecsumsq16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_vecsumsq32_tile(const TileSegConfig* config,
 *                                       int buf_num, int tile_num)
 *  \brief Корень из суммы элементов вектора типа int32 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_vecsumsq32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_vecsumsq_fl_tile(const TileSegConfig* config,
 *                                        int buf_num, int tile_num)
 *  \brief Корень из суммы элементов вектора типа float в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_vecsumsq_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_vecsumsq_db_tile(const TileSegConfig* config,
 *                                        int buf_num, int tile_num)
 *  \brief Корень из суммы элементов вектора типа double в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_vecsumsq_db_tile(const TileSegConfig* config, int buf_num, int tile_num);

/*!
 *  \fn int32_t RunCalculationVecSumSq16(const TileSegConfig* config)
 *  \brief Запуск вычислений VecSumSq16
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int32_t корень из суммы элементов вектора
 */
int32_t RunCalculationVecSumSq16(const TileSegConfig* config);
/*!
 *  \fn int64_t RunCalculationVecSumSq32(const TileSegConfig* config)
 *  \brief Запуск вычислений VecSumSq32
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int64_t корень из суммы элементов вектора
 */
int64_t RunCalculationVecSumSq32(const TileSegConfig* config);
/*!
 *  \fn float RunCalculationVecSumSqFl(const TileSegConfig* config)
 *  \brief Запуск вычислений VecSumSqFl
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b float корень из суммы элементов вектора
 */
float RunCalculationVecSumSqFl(const TileSegConfig* config);
/*!
 *  \fn double RunCalculationVecSumSqDb(const TileSegConfig* config)
 *  \brief Запуск вычислений VecSumSqDb
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b double корень из суммы элементов вектора
 */
double RunCalculationVecSumSqDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_w_vec16_tile(const TileSegConfig* config, int buf_num,
 *                                    int tile_num)
 *  \brief Сложение элементов вектора с взвешенными элементами другого вектора
 *  для типа int16 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_w_vec16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_w_vec32_tile(const TileSegConfig* config, int buf_num,
 *                                    int tile_num)
 *  \brief Сложение элементов вектора с взвешенными элементами другого вектора
 *  для типа int32 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_w_vec32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_w_vec_fl_tile(const TileSegConfig* config, int buf_num,
 *                                     int tile_num)
 *  \brief Сложение элементов вектора с взвешенными элементами другого вектора
 *  для типа float в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_w_vec_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_w_vec_db_tile(const TileSegConfig* config, int buf_num,
 *                                     int tile_num)
 *  \brief Сложение элементов вектора с взвешенными элементами другого вектора
 *  для типа double в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_w_vec_db_tile(const TileSegConfig* config, int buf_num, int tile_num);

/*!
 *  \fn void RunCalculationWVec16(const TileSegConfig* config)
 *  \brief Запуск вычислений WVec16
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationWVec16(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationWVec32(const TileSegConfig* config)
 *  \brief Запуск вычислений WVec32
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationWVec32(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationWVecFl(const TileSegConfig* config)
 *  \brief Запуск вычислений WVecFl
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationWVecFl(const TileSegConfig* config);
/*!
 *  \fn void RunCalculationWVecDb(const TileSegConfig* config)
 *  \brief Запуск вычислений WVecDb
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationWVecDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_dotprod_sqr16_tile(const TileSegConfig* config,
 *                                          int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа int16 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod_sqr16_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_dotprod_sqr32_tile(const TileSegConfig* config,
 *                                          int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа int32 в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod_sqr32_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_dotprod_sqr_fl_tile(const TileSegConfig* config,
 *                                           int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа float в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod_sqr_fl_tile(const TileSegConfig* config, int buf_num, int tile_num);
/*!
 *  \fn void calculation_dotprod_sqr_db_tile(const TileSegConfig* config,
 *                                           int buf_num, int tile_num)
 *  \brief Скалярное произведение векторов типа double в рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_dotprod_sqr_db_tile(const TileSegConfig* config, int buf_num, int tile_num);

/*!
 *  \fn int32_t RunCalculationDotprodSqr16(const TileSegConfig* config)
 *  \brief Запуск вычислений DotprodSqr16
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int32_t сумма квадратов элементов второго вектора
 */
int32_t RunCalculationDotprodSqr16(const TileSegConfig* config);
/*!
 *  \fn int64_t RunCalculationDotprodSqr32(const TileSegConfig* config)
 *  \brief Запуск вычислений DotprodSqr32
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b int64_t сумма квадратов элементов второго вектора
 */
int64_t RunCalculationDotprodSqr32(const TileSegConfig* config);
/*!
 *  \fn float RunCalculationDotprodSqrFl(const TileSegConfig* config)
 *  \brief Запуск вычислений DotprodSqrFl
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b float сумма квадратов элементов второго вектора
 */
float RunCalculationDotprodSqrFl(const TileSegConfig* config);
/*!
 *  \fn double RunCalculationDotprodSqrDb(const TileSegConfig* config)
 *  \brief Запуск вычислений DotprodSqrDb
 *  \param[in] config структура с параметрами вычислений
 *  \return
 *      - \b double сумма квадратов элементов второго вектора
 */
double RunCalculationDotprodSqrDb(const TileSegConfig* config);

/*!
 *  \fn void calculation_recip16_tile(const TileSegConfig* config, int buf_num,
 *                                    int tile_num)
 *  \brief Обратная величина для чисел с фиксированной точкой типа int16 в
 *  рамках одного тайла
 *  \param[in] config структура с указателями на буферы во внут. памяти
 *  \param[in] buf_num номер буфера для текущих вычислений
 *  \param[in] tile_num номер обрабатываемого тайла
 */
void calculation_recip16_tile(const TileSegConfig* config, int buf_num, int tile_num);

/*!
 *  \fn void RunCalculationRecip16(const TileSegConfig* config)
 *  \brief Запуск вычислений Recip16
 *  \param[in] config структура с параметрами вычислений
 */
void RunCalculationRecip16(const TileSegConfig* config);

/// Для операций с фиксированной точкой числа записываются в int32,
/// Младшие 16 бит - дробная часть, старшие - целая.
/// Порядок следования байтов little-endian.

/// Умножение чисел с фиксированной точкой (i=16,f=16) в рамках одного тайла
void calculation_mul16i16f_tile(
    const TileSegConfig* config,  ///< [in]  структура с указателями на буферы во внут. памяти
    int buf_num,                  ///< [in]  номер буфера для текущих вычислений
    int tile_num                  ///< [in]  номер обрабатываемого тайла
);

/// Запуск вычислений Mul16I16F
void RunCalculationMul16I16F(const TileSegConfig* config  ///< [in]  структура с параметрами вычислений
);

#endif
