// Copyright 2018-2025 RnD Center "ELVEES", JSC

/*! \file
 *  \brief Тестирование функции vecsumsq
 *  \author Фролов Андрей
 */

#include "tests.h"

int main() {
  int failed_count = 0;
  int size[TEST_COUNT] = {512, 1024, 2048, 8192, 16384};
  int print = 0;

  print_table_header();

#ifndef LOCAL_MEM
  void* src0_pos = memalign(64, size[TEST_COUNT - 1] * sizeof(int64_t));
  void* src0_neg = memalign(64, size[TEST_COUNT - 1] * sizeof(int64_t));
  void* src0_rnd = memalign(64, size[TEST_COUNT - 1] * sizeof(int64_t));
#else
#ifdef BARE_METAL
  void* src0_pos = &__local_mem;
#else
  disable_l2_cache();
  void* src0_pos = &xyram_data;
#endif
  void* src0_neg = src0_pos + size[TEST_COUNT - 1] * sizeof(int64_t);
  void* src0_rnd = src0_neg + size[TEST_COUNT - 1] * sizeof(int64_t);
#endif

  create_vector_s16((int16_t*)src0_neg, size[TEST_COUNT - 1], -1);
  create_vector_s16((int16_t*)src0_pos, size[TEST_COUNT - 1], 1);
  create_vector_s16((int16_t*)src0_rnd, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(int16_t) * 1;
    int32_t ti_tics = size[i] / 4 + 25;

#ifndef DISABLE_NEG_TEST_DATA
    printf("| vecsumsq_short_neg     | %14d |", size[i]);
    failed_count += test_vecsumsq((int16_t*)src0_neg, size[i], print, input_bytes, ti_tics);
#endif

    printf("| vecsumsq_short_rnd     | %14d |", size[i]);
    failed_count += test_vecsumsq((int16_t*)src0_rnd, size[i], print, input_bytes, ti_tics);

#ifndef DISABLE_POS_TEST_DATA
    printf("| vecsumsq_short_pos     | %14d |", size[i]);
    failed_count += test_vecsumsq((int16_t*)src0_pos, size[i], print, input_bytes, ti_tics);
#endif
  }

  create_vector_s32((int32_t*)src0_neg, size[TEST_COUNT - 1], -1);
  create_vector_s32((int32_t*)src0_pos, size[TEST_COUNT - 1], 1);
  create_vector_s32((int32_t*)src0_rnd, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(int32_t) * 1;
    int32_t ti_tics = 0;

#ifndef DISABLE_NEG_TEST_DATA
    printf("| vecsumsq_int_neg       | %14d |", size[i]);
    failed_count += test_vecsumsq32((int32_t*)src0_neg, size[i], print, input_bytes, ti_tics);
#endif

    printf("| vecsumsq_int_rnd       | %14d |", size[i]);
    failed_count += test_vecsumsq32((int32_t*)src0_rnd, size[i], print, input_bytes, ti_tics);

#ifndef DISABLE_POS_TEST_DATA
    printf("| vecsumsq_int_pos       | %14d |", size[i]);
    failed_count += test_vecsumsq32((int32_t*)src0_pos, size[i], print, input_bytes, ti_tics);
#endif
  }

  create_vector_float((float*)src0_neg, size[TEST_COUNT - 1], -1);
  create_vector_float((float*)src0_pos, size[TEST_COUNT - 1], 1);
  create_vector_float((float*)src0_rnd, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(float) * 1;
    int32_t ti_tics = (int)(0.25 * size[i] + 51);

#ifndef DISABLE_NEG_TEST_DATA
    printf("| vecsumsq_fl_neg        | %14d |", size[i]);
    failed_count += test_vecsumsq_fl((float*)src0_neg, size[i], print, input_bytes, ti_tics);
#endif

    printf("| vecsumsq_fl_rnd        | %14d |", size[i]);
    failed_count += test_vecsumsq_fl((float*)src0_rnd, size[i], print, input_bytes, ti_tics);

#ifndef DISABLE_POS_TEST_DATA
    printf("| vecsumsq_fl_pos        | %14d |", size[i]);
    failed_count += test_vecsumsq_fl((float*)src0_pos, size[i], print, input_bytes, ti_tics);
#endif
  }

  create_vector_double((double*)src0_neg, size[TEST_COUNT - 1], -1);
  create_vector_double((double*)src0_pos, size[TEST_COUNT - 1], 1);
  create_vector_double((double*)src0_rnd, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(double) * 1;
    int32_t ti_tics = 0;
#ifndef DISABLE_NEG_TEST_DATA
    printf("| vecsumsq_db_neg        | %14d |", size[i]);
    failed_count += test_vecsumsq_db((double*)src0_neg, size[i], print, input_bytes, ti_tics);
#endif

    printf("| vecsumsq_db_rnd        | %14d |", size[i]);
    failed_count += test_vecsumsq_db((double*)src0_rnd, size[i], print, input_bytes, ti_tics);

#ifndef DISABLE_POS_TEST_DATA
    printf("| vecsumsq_db_pos        | %14d |", size[i]);
    failed_count += test_vecsumsq_db((double*)src0_pos, size[i], print, input_bytes, ti_tics);
#endif
  }

#ifndef LOCAL_MEM
  free(src0_rnd);
  free(src0_pos);
  free(src0_neg);
#else
#ifndef BARE_METAL
  enable_l2_cache(L2_CACHE_SIZE);
#endif
#endif

  return failed_count;
}
int test_vecsumsq(int16_t* src0, int size, int print, int32_t input_bytes, int32_t ti_tics) {
  int ret = 0;

  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  int32_t res = ref_vecsumsq(src0, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  int32_t c = vecsumsq(src0, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_s16(src0, size);
    printf("dsp_res:");
    printf("%d\n", c);
    printf("ref_res:");
    printf("%d\n", res);
  }

  if (c == res) {
    ret = 0;
  } else {
    ret = 1;
  }

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}

int test_vecsumsq32(int32_t* src0, int size, int print, int32_t input_bytes, int32_t ti_tics) {
  int ret = 0;

  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  int64_t res = ref_vecsumsq32(src0, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  int64_t c = vecsumsq32(src0, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_s32(src0, size);
    printf("dsp_res:");
    printf("%lld\n", (long long int)c);
    printf("ref_res:");
    printf("%lld\n", (long long int)res);
  }

  if (c == res) {
    ret = 0;
  } else {
    ret = 1;
  }

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}

int test_vecsumsq_fl(float* src0, int size, int print, int32_t input_bytes, int32_t ti_tics) {
  int ret = 0;

  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  float res = ref_vecsumsq_fl(src0, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  float c = vecsumsq_fl(src0, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_float(src0, size);
    printf("dsp_res:");
    printf("%f\n", c);
    printf("ref_res:");
    printf("%f\n", res);
  }

#ifdef EPS
  ret = compare_float_eps(&c, &res, 1, EPS);
#else
  ret = compare_float_eps(&c, &res, 1, 0.00001);
#endif

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}

int test_vecsumsq_db(double* src0, int size, int print, int32_t input_bytes, int32_t ti_tics) {
  int ret = 0;

  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  double res = ref_vecsumsq_db(src0, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  double c = vecsumsq_db(src0, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_double(src0, size);
    printf("dsp_res:");
    printf("%f\n", c);
    printf("ref_res:");
    printf("%f\n", res);
  }

  ret = compare_double_eps(&c, &res, 1, 0.000001);

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}
