// Copyright 2018-2025 RnD Center "ELVEES", JSC

/*! \file
 *  \brief Тестирование функции neg
 *  \author Фролов Андрей
 */

#include "tests.h"

int main() {
  int failed_count = 0;
  int size[TEST_COUNT] = {512, 1024, 2048, 8192, 16384};
  int print = 0;

  print_table_header();

#ifndef LOCAL_MEM
  void* src0 = memalign(64, size[TEST_COUNT - 1] * sizeof(int64_t));
  void* dst_ref = memalign(64, size[TEST_COUNT - 1] * sizeof(int64_t));
  void* dst_opt = memalign(64, size[TEST_COUNT - 1] * sizeof(int64_t));
#else
#ifdef BARE_METAL
  void* src0 = &__local_mem;
#else
  disable_l2_cache();
  void* src0 = &xyram_data;
#endif
  void* dst_ref = src0 + size[TEST_COUNT - 1] * sizeof(int64_t);
  void* dst_opt = dst_ref + size[TEST_COUNT - 1] * sizeof(int64_t);
#endif

  create_vector_s16((int16_t*)src0, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(int16_t) * 1;
    int32_t ti_tics = 0;
    printf("| neg_short              | %14d |", size[i]);
    failed_count +=
        test_neg16((int16_t*)src0, (int16_t*)dst_ref, (int16_t*)dst_opt, size[i], print, input_bytes, ti_tics);
  }

  create_vector_s32((int32_t*)src0, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(int32_t) * 1;
    int32_t ti_tics = size[i] / 2 + 21;
    printf("| neg_int                | %14d |", size[i]);
    failed_count +=
        test_neg32((int32_t*)src0, (int32_t*)dst_ref, (int32_t*)dst_opt, size[i], print, input_bytes, ti_tics);
  }

  create_vector_float((float*)src0, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(float) * 1;
    int32_t ti_tics = 0;
    printf("| neg_fl                 | %14d |", size[i]);
    failed_count += test_neg_fl((float*)src0, (float*)dst_ref, (float*)dst_opt, size[i], print, input_bytes, ti_tics);
  }

  create_vector_double((double*)src0, size[TEST_COUNT - 1], 0);
  for (int i = 0; i < TEST_COUNT; ++i) {
    int32_t input_bytes = size[i] * sizeof(double) * 1;
    int32_t ti_tics = 0;
    printf("| neg_db                 | %14d |", size[i]);
    failed_count +=
        test_neg_db((double*)src0, (double*)dst_ref, (double*)dst_opt, size[i], print, input_bytes, ti_tics);
  }

#ifndef LOCAL_MEM
  free(src0);
  free(dst_ref);
  free(dst_opt);
#else
#ifndef BARE_METAL
  enable_l2_cache(L2_CACHE_SIZE);
#endif
#endif

  return failed_count;
}

int test_neg16(int16_t* src0, int16_t* dst_ref, int16_t* dst_opt, int size, int print, int32_t input_bytes,
               int32_t ti_tics) {
  int ret = 0;

  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  ref_neg16(src0, dst_ref, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  neg16(src0, dst_opt, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_s16(src0, size);
    printf("dsp_res:");
    print_vector_s16(dst_opt, size);
    printf("ref_res:");
    print_vector_s16(dst_ref, size);
  }

  ret = compare_s16(dst_ref, dst_opt, size);

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}

int test_neg32(int32_t* src0, int32_t* dst_ref, int32_t* dst_opt, int size, int print, int32_t input_bytes,
               int32_t ti_tics) {
  int ret = 0;

  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  ref_neg32(src0, dst_ref, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  neg32(src0, dst_opt, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_s32(src0, size);
    printf("dsp_res:");
    print_vector_s32(dst_opt, size);
    printf("ref_res:");
    print_vector_s32(dst_ref, size);
  }

  ret = compare_s32(dst_ref, dst_opt, size);

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}

int test_neg_fl(float* src0, float* dst_ref, float* dst_opt, int size, int print, int32_t input_bytes,
                int32_t ti_tics) {
  int ret = 0;
  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  ref_neg_fl(src0, dst_ref, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  neg_fl(src0, dst_opt, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_float(src0, size);
    printf("dsp_res:");
    print_vector_float(dst_opt, size);
    printf("ref_res:");
    print_vector_float(dst_ref, size);
  }

  ret = compare_float(dst_ref, dst_opt, size);

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}

int test_neg_db(double* src0, double* dst_ref, double* dst_opt, int size, int print, int32_t input_bytes,
                int32_t ti_tics) {
  int ret = 0;
  uint32_t tic_count[2], instruction_count[2];
  uint32_t ref_tic_count[2], ref_instruction_count[2];

  count_tics(ref_tic_count, ref_instruction_count);
  ref_neg_db(src0, dst_ref, size);
  count_tics(&ref_tic_count[1], &ref_instruction_count[1]);

  count_tics(tic_count, instruction_count);
  neg_db(src0, dst_opt, size);
  count_tics(&tic_count[1], &instruction_count[1]);

  if (print) {
    printf("vect1:");
    print_vector_double(src0, size);
    printf("dsp_res:");
    print_vector_double(dst_opt, size);
    printf("ref_res:");
    print_vector_double(dst_ref, size);
  }

  ret = compare_double(dst_ref, dst_opt, size);

  print_performance(ref_tic_count, ref_instruction_count, tic_count, instruction_count, input_bytes, ti_tics);

  if (ret == 0)
    printf(" passed |\n");
  else
    printf(" failed |\n");

  return ret;
}
